/*************************************************************************
** interpcom-3.1    (command interpreter)                                **
** _mapm.c : expression evaluator based on mapm library                  **
**                                                                       **
** Copyright (C) 2003  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      UMR 7586 du CNRS                                                 **
**      173, rue du Chevaleret                                           **
**      75013 Paris                                                      **
**      France                                                           **
**                                                                       **
 *************************************************************************/

#include "interp.h"
#include <m_apm.h>
extern int _mapm_dec_places;



/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
_mapm_iszero(void *z)
{
    M_APM          *t;

    t = (M_APM *)z;
    if (m_apm_compare(*t, MM_Zero) == 0)
        return 1;
    else
        return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_Zero(void *z)
{
    M_APM          *t;

    t = (M_APM *)z;
    m_apm_copy(*t, MM_Zero);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_clear(void *z)
{
    M_APM          *t;

    t = (M_APM *) z;
    m_apm_free(*t);
    free(z);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_neg(void *z)
{
    M_APM          *t,
                    u;

    u = m_apm_init();
    t = (M_APM *)z;
    m_apm_negate(u, *t);
    m_apm_copy(*t, u);
    m_apm_free(u);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_add(void *res, void *a, void *b)
{
    M_APM          *t,
                   *u,
                   *v,
                    w,
                    z;

    t = (M_APM *)res;
    u = (M_APM *)a;
    v = (M_APM *)b;
    w = m_apm_init();
    z = m_apm_init();
    m_apm_add(w, *u, *v);
    m_apm_round(z, _mapm_dec_places, w);
    m_apm_copy(*t, z);
    m_apm_free(w);
    m_apm_free(z);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_sub(void *res, void *a, void *b)
{
    M_APM          *t,
                   *u,
                   *v,
                    w,
                    z;

    t = (M_APM *)res;
    u = (M_APM *)a;
    v = (M_APM *)b;
    w = m_apm_init();
    z = m_apm_init();
    m_apm_subtract(w, *u, *v);
    m_apm_round(z, _mapm_dec_places, w);
    m_apm_copy(*t, z);
    m_apm_free(w);
    m_apm_free(z);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_mul(void *res, void *a, void *b)
{
    M_APM          *t,
                   *u,
                   *v,
                    w,
                    z;

    t = (M_APM *)res;
    u = (M_APM *)a;
    v = (M_APM *)b;
    w = m_apm_init();
    z = m_apm_init();
    m_apm_multiply(w, *u, *v);
    m_apm_round(z, _mapm_dec_places, w);
    m_apm_copy(*t, z);
    m_apm_free(w);
    m_apm_free(z);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_pow(void *res, void *a, void *b)
{
    M_APM          *t,
                   *u,
                   *v,
                    w;

    t = (M_APM *)res;
    u = (M_APM *)a;
    v = (M_APM *)b;
    w = m_apm_init();
    m_apm_pow(w, _mapm_dec_places, *u, *v);
    m_apm_copy(*t, w);
    m_apm_free(w);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_div(void *res, void *a, void *b)
{
    M_APM          *t,
                   *u,
                   *v,
                    w,
                    z;

    t = (M_APM *)res;
    u = (M_APM *)a;
    v = (M_APM *)b;
    w = m_apm_init();
    z = m_apm_init();
    m_apm_divide(w, _mapm_dec_places, *u, *v);
    m_apm_round(z, _mapm_dec_places, w);
    m_apm_copy(*t, z);
    m_apm_free(w);
    m_apm_free(z);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_set(void *res, char *h)
{
    M_APM          *t,
                    z;

    t = (M_APM *)res;
    z = m_apm_init();
    m_apm_set_string(z, h);
    m_apm_round(*t, _mapm_dec_places, z);
    m_apm_free(z);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_Init(void **s)
{
    M_APM          *t;

    t = (M_APM *) malloc((size_t) sizeof(M_APM));
    t[0] = m_apm_init();
    s[0] = (void *) t;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
char *
_mapm_print(void *c)
{
    M_APM          *t;
    char           *h;

    t = (M_APM *)c;
    h = (char *)malloc((size_t)(_mapm_dec_places + 20) * sizeof(char));
    m_apm_to_string(h, _mapm_dec_places, *t);
    return h;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void *
_mapm_copy(void *c)
{
    M_APM          *t,
                   *u;

    t = (M_APM *)c;
    u = (M_APM *) malloc(sizeof(M_APM));
    u[0] = m_apm_init();
    m_apm_copy(*u, *t);
    return (void *) u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
_mapm_sign(void *c)
{
    M_APM          *t;

    t = (M_APM *)c;
    return m_apm_sign(*t);
}
/*------------------------------------------------------------------*/





char desc_mapm[] = "expression evaluator based on the mapm library with 50 digits";

EXPREVAL_GEN        _mapm_exprev = {
    _mapm_clear,
    _mapm_Zero,
    _mapm_neg,
    _mapm_iszero,
    _mapm_add,
    _mapm_sub,
    _mapm_mul,
    _mapm_pow,
    _mapm_div,
    _mapm_set,
    _mapm_Init,
    _mapm_print,
    _mapm_copy,
    _mapm_sign,
    0,
    NULL,
    NULL,
    1,
    desc_mapm
};




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_sin(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_sin(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_cos(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_cos(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_tan(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_tan(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_asin(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_arcsin(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_acos(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_arccos(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_atan(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_arctan(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_sinh(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_sinh(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_cosh(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_cosh(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_tanh(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_tanh(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_exp(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_exp(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_log(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_log(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_log10(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_log10(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_sqrt(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_sqrt(v, _mapm_dec_places, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_mapm_fabs(void **res, void **x)
{
    M_APM          *t,
                   *u,
                    v;

    t = (M_APM *) x[0];
    u = (M_APM *) res[0];
    v = m_apm_init();
    m_apm_absolute_value(v, *t);
    m_apm_copy(*u, v);
    m_apm_free(v);
}
/*------------------------------------------------------------------*/




FUNCTIONGen         Funcs_mapm[] =
{
   /* name, function to call */
   { "sin",     1,    _mapm_sin },            /* 0    */
   { "cos",     1,    _mapm_cos },            /* 1    */
   { "tan",     1,    _mapm_tan },            /* 2    */
   { "asin",    1,    _mapm_asin },           /* 3    */
   { "acos",    1,    _mapm_acos },           /* 4    */
   { "atan",    1,    _mapm_atan },           /* 5    */
   { "sinh",    1,    _mapm_sinh },           /* 6    */
   { "cosh",    1,    _mapm_cosh },           /* 7    */
   { "tanh",    1,    _mapm_tanh },           /* 8    */
   { "exp",     1,    _mapm_exp },            /* 9    */
   { "log",     1,    _mapm_log },            /* 10   */
   { "log10",   1,    _mapm_log10 },          /* 11   */
   { "sqrt",    1,    _mapm_sqrt },           /* 12   */
   { "abs",     1,    _mapm_fabs },           /* 13   */
   { 0 }                                      /* 14   */
};

int                 _mapm_NBFONC0=14;

