/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1995.  The Regents of the University of California.  All rights   */
/*   reserved.                                                             */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XDIR shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XDIR for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <Xm/SelectioB.h>
#include <Xm/Text.h>
#include "xdir.h"
#include "str.h"

static Widget w_mkdirDialog;

static char *help_make_dir[] = {
	"This dialog enables you to create a directory with the",
	"name specified in the \"Directory Name\" field.  The new",
	"directory is created as a subdirectory of the current directory.\n",
    "\n",
	"The newly created directory is automatically entered if the",
	"user preference ENTER DIR UPON CREATION is set to YES.\n",
	"\n",
    "Click on the OK button to perform the operation and",
    "remove the dialog.  Click on the CANCEL button to remove",
    "the dialog without performing the operation.",
    NULL
};

static char *msg1 = "Unable to redisplay directory.\n\nDirectory window might be out of date.";

extern struct st_host_info hinfo[];
extern int enter_dir_upon_creation;
extern int diagnostics;

void cb_mkdir_ok();
void cb_mkdir_cancel();
void cb_mkdir_help();
void cb_map_dialog();
char *merge_paths();


/*
 * cb_make_dir - Callback for popping up the "Make Directory" dialog.
 */
void
cb_make_dir(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	struct dirwin_st *dirwin = (struct dirwin_st *)client_data;
	Widget w_dir;

	/* Start operation */
	if (!start_op(False))
		return;

    /* Clear error flag */
    raise_okflag();

    /* Create "Make Directory" dialog */
    create_mkdir_dialog(dirwin);

	/* Pop up dialog */
	w_dir = XmSelectionBoxGetChild(w_mkdirDialog, XmDIALOG_TEXT);
    if (hinfo[dirwin->host].system == SYS_VMS) {
        XmTextSetString(w_dir, "[.]");
        XmTextSetInsertionPosition(w_dir, 2);
	} else
        XmTextSetString(w_dir, "");
	XtManageChild(w_mkdirDialog);
	traverse_to_widget(w_dir);
}


/*
 * create_mkdir_dialog - Create the "Make Directory" dialog.  "dirwin"
 *                       is the directory window to center the Make
 *                       Directory dialog over.
 */
create_mkdir_dialog(dirwin)
struct dirwin_st *dirwin;
{
    int i;
    Arg args[2];

    /* Create prompt dialog to get directory path */
    i = 0;
    XtSetArg(args[i], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); i++;
	XtSetArg(args[i], XmNdefaultPosition, False); i++;
    w_mkdirDialog = XmCreatePromptDialog(dirwin->w_shell, "makeDir", args, i);
    XtAddCallback(w_mkdirDialog, XmNokCallback, cb_mkdir_ok, (XtPointer)dirwin);
	XtAddCallback(w_mkdirDialog, XmNcancelCallback, cb_mkdir_cancel,
		(XtPointer)NULL);
	XtAddCallback(w_mkdirDialog, XmNhelpCallback, cb_mkdir_help,
		(XtPointer)NULL);
	XtAddCallback(w_mkdirDialog, XmNmapCallback, cb_map_dialog,
		(XtPointer)XtWindow(dirwin->w_shell));

	/* Add callback for the WM_DELETE_WINDOW protocol */
	add_wm_delete_window_cb(w_mkdirDialog, cb_mkdir_cancel, NULL, False);
}


/*
 * cb_mkdir_ok - This callback actually creates the directory.
 */
void
cb_mkdir_ok(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	struct dirwin_st *dirwin = (struct dirwin_st *)client_data;
	char *dirname;
	char *new_dir;
	char *text;
	int len;
	char msg[MAXPATHLEN+40];
	int retval;

	/* This might take some time */
	use_busy_cursor();

	/* Get directory name*/
	text = XmTextGetString(XmSelectionBoxGetChild(w_mkdirDialog,
		XmDIALOG_TEXT));

	/* Get rid of Make Directory Dialog */
	XtDestroyWidget(XtParent(w_mkdirDialog));

	/* Handle VMS names */
	if (hinfo[dirwin->host].system == SYS_VMS) {
		len = strlen(text);
		if (strchr(text, '[') != NULL) {
			if ((len < 4) || (text[0] != '[') || (text[1] != '.')
			    	      || (text[len-1] != ']') || strchr(&text[2], '.')) {
				warn("Bad VMS directory name.\nUnable to make directory.",
					dirwin->w_shell);
				restore_prev_cursor();
				XtFree(text);
				end_op();
				return;
			}
			dirname = XtNewString(text);
		} else if (strstr(text, ".dir"))
			dirname = XtNewString(text);
		else {
			dirname = XtMalloc(strlen(text)+5);
			strcpy(dirname, text);
			strcat(dirname, ".dir");
		}
	} else
		dirname = XtNewString(text);
	XtFree(text);

    /* Form full path name */
	new_dir = merge_paths(hinfo[dirwin->host].system, dirwin->dirname, dirname);

	/* Make new directory */
    if (dirwin->host == LOCAL) {
        if (local_mkdir(new_dir, 0777) < 0) {
			XtFree(new_dir);
			XtFree(dirname);
			restore_prev_cursor();
            record_and_alert("Unable to create directory.", dirwin->w_shell);
			end_op();
            return;
        }
    } else {
		show_stop_button(dirwin);
		retval = remote_mkdir(dirwin->host, new_dir);
		if (retval < 0) {
			XtFree(new_dir);
			XtFree(dirname);
			restore_prev_cursor();
			switch (retval) {
			case -1:
				record_and_alert("Unable to create directory.",dirwin->w_shell);
				break;
			case -3:
	            restore_lost_connection(dirwin->host, dirwin);
				break;
			case -6:
				hide_abort_dialog();
				dirwin_out_of_date_alert(dirwin);
			}
			hide_stop_button();
			end_op();
            return;
        }
    }

	/* Report success */
	if (diagnostics >= NORMAL) {
		sprintf(msg, "*** Successfully created:  %s\n", dirname);
		write_log(msg);
	}

    /* Update the display */
	retval = display_dir(dirwin->host, dirwin, dirwin->dirname, True, False,
		dirwin->cache_mode, False);
	switch (retval) {
	case -6:
		hide_abort_dialog();
		dirwin_out_of_date_alert(dirwin);
		break;
	case -3:
		restore_lost_connection(dirwin->host, dirwin);
		break;
	case -1:
		record_and_alert(msg1, dirwin->w_shell);
	}
    restore_prev_cursor();

	/* Free memory */
	XtFree(new_dir);
	XtFree(dirname);

	/* Operation is complete */
	if (dirwin->host != LOCAL)
		hide_stop_button();
	end_op();
}


/*
 * cb_mkdir_cancel - Callback to handle Make Directory "Cancel" button.
 */
void
cb_mkdir_cancel(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	/* Get rid of Make Directory Dialog */
	XtDestroyWidget(XtParent(w_mkdirDialog));

	/* End operation */
	end_op();
}


/*
 * cb_mkdir_help - Callback to display help information for "Make Directory"
 *                 dialog.
 */
void
cb_mkdir_help(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	help_dialog(widget, True, "Make Directory", help_make_dir);
}

