/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1995.  The Regents of the University of California.  All rights   */
/*   reserved.                                                             */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XDIR shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XDIR for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <Xm/PushBG.h>
#include <Xm/SelectioB.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/LabelG.h>
#include <Xm/SeparatoG.h>
#include "xdir.h"
#include "list.h"
#include "history.h"
#include "str.h"

static Widget w_gotoDirDialog;

static char *help_goto_dir[] = {
    "This dialog enables you to display the directory specified",
    "in the \"Directory\" field.\n",
    "\n",
    "Previously referenced directories are remembered and displayed",
    "in the scrollable list labeled \"Recently Referenced",
    "Directories\".  Double-click on a path in the list to",
    "display that directory.  Single-click on a path in the",
    "list to place that entry into the text field labeled",
    "\"Directory\", where it can be edited.\n",
    "\n",
    "The host's list of previously referenced directories",
	"is modified every time one of",
    "its directories is displayed, regardless of the",
    "mechanism used to specify a directory to display.  The user",
    "preference HISTORY SORT ORDER specifies whether previously referenced",
    "directory paths are displayed in alphabetical order or",
    "with the most-recently-referenced paths at the top of",
    "the list.  The user preference MAX DIR NAME HISTORY",
    "specifies the maximum number of directory paths to remember",
    "per host.  This directory information is preserved across LLNL XDIR",
    "sessions.\n",
    "\n",
    "Click on the OK button to perform the operation and",
    "remove the dialog.  Click on the CANCEL button to remove",
    "the dialog without performing the operation.",
    NULL
};

extern struct dirwin_st *dirwin_head;
extern struct st_host_info hinfo[];
extern int sort_caches;

void cb_map_dialog();
void cb_goto_dir_ok();
void cb_goto_dir_cancel();
void cb_goto_dir_help();
void HandleUpArrowEvents();
Widget vertical_sb();
char **path_to_links();
char *links_to_path();
char *cstring_to_text();


/*
 * cb_goto_dir - Display the "Go to Directory" dialog.
 */
void
cb_goto_dir(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	struct dirwin_st *dirwin = (struct dirwin_st *)client_data;
	Widget w_dir;

	/* Start operation */
	if (!start_op(False))
		return;

    /* Clear error flag */
    raise_okflag();

    create_goto_dir_dialog(dirwin);

    update_goto_dir_dialog(dirwin);

    XtManageChild(w_gotoDirDialog);
	w_dir = XmSelectionBoxGetChild(w_gotoDirDialog, XmDIALOG_TEXT);
	traverse_to_widget(w_dir);
}


/*
 * create_goto_dir_dialog - Create "Go to Directory" dialog.  "dirwin" is
 *                          the directory window to center the "Go To"
 *                          dialog over.
 */
create_goto_dir_dialog(dirwin)
struct dirwin_st *dirwin;
{
	Arg args[2];
	int i;
	Widget w_list;

	i = 0;
	XtSetArg(args[i], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); i++;
	XtSetArg(args[i], XmNdefaultPosition, False); i++;
	w_gotoDirDialog = XmCreateSelectionDialog( dirwin->w_shell, "gotoDir",
		args,i);
	XtAddCallback(w_gotoDirDialog, XmNokCallback, cb_goto_dir_ok,
		(XtPointer)dirwin);
	XtAddCallback(w_gotoDirDialog, XmNcancelCallback, cb_goto_dir_cancel,
		(XtPointer)NULL);
	XtAddCallback(w_gotoDirDialog, XmNhelpCallback, cb_goto_dir_help,
		(XtPointer)NULL);
	XtAddCallback(w_gotoDirDialog, XmNmapCallback, cb_map_dialog,
		(XtPointer)XtWindow(dirwin->w_shell));

	/* Add callback for the WM_DELETE_WINDOW protocol */
	add_wm_delete_window_cb(w_gotoDirDialog, cb_goto_dir_cancel, NULL, False);

	/* Enhance appearance of scrolled list widget */
	w_list = XmSelectionBoxGetChild(w_gotoDirDialog, XmDIALOG_LIST);

	/* Don't need "Apply" button */
	XtUnmanageChild(XmSelectionBoxGetChild(w_gotoDirDialog,
		XmDIALOG_APPLY_BUTTON));
}


/*
 * update_goto_dir_dialog - Display most-recently-used directory paths
 *                          in the "Go to Directory" display for the
 *                          host associated with "dirwin".
 */
update_goto_dir_dialog(dirwin)
struct dirwin_st *dirwin;
{
	struct sl_struct *list;
	XmStringTable items;
	int i;
	Widget widget;

	/* Enter directory paths into scrolled list */
	retrieve_history(DIRECTORY, hinfo[dirwin->host].hostname, &list,
		sort_caches);
	items = (XmStringTable)XtMalloc(sizeof(XmString)*list->nentries);
	for (i=0; i<list->nentries; i++)
		items[i] = XmStringCreateSimple(list->entries[i]);
	XtVaSetValues(
		w_gotoDirDialog,
		XmNlistItems,		items,
		XmNlistItemCount,	list->nentries,
		NULL
	);

	/* Free stuff */
	for (i=0; i<list->nentries; i++)
		XmStringFree(items[i]);
	XtFree((char *)items);
	release_array_list(list);

    /* Clear text */
    widget = XmSelectionBoxGetChild(w_gotoDirDialog, XmDIALOG_TEXT);
    if (hinfo[dirwin->host].system == SYS_VMS) {
		XmTextSetString(widget, "[.]");
		XmTextSetInsertionPosition(widget, 2);
	} else 
        XmTextSetString(widget, "");

	/* Clear selection */
	widget = XmSelectionBoxGetChild(w_gotoDirDialog, XmDIALOG_LIST);
	XtVaSetValues(widget, XmNselectedItemCount, 0, NULL);
}


/*
 * cb_goto_dir_ok - Callback for "Go to Directory" dialog's OK button.
 *                  This routine actually displays the directory.
 */
void
cb_goto_dir_ok(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	struct dirwin_st *dirwin = (struct dirwin_st *)client_data;
	char *path;
	char *full_path;
	int retval;

	/* Get directory path*/
	path = XmTextGetString(XmSelectionBoxGetChild(w_gotoDirDialog,
		XmDIALOG_TEXT));

	/* Check for linefeeds */
	if (strchr(path, '\n')) {
		record_and_alert(
			"Unable to go to directory - Path contains a linefeed character.",
			dirwin->w_shell);
		end_op();
		return;
	}

	/* Get rid of dialog */
	XtDestroyWidget(XtParent(w_gotoDirDialog));

	/* Make operation interruptable for remote hosts */
	if (dirwin->host != LOCAL)
		show_stop_button(dirwin);

	/* Convert directory path to full path */
	retval = convert_to_full_path(dirwin, path, &full_path);
	XtFree(path);
	if (retval < 0) {
		switch (retval) {
		case -6:
			hide_abort_dialog();
			break;
		case -1:
			record_and_alert("Unable to go to directory.", dirwin->w_shell);
			break;
		case -3:
			restore_lost_connection(dirwin->host, dirwin);
		}
		if (dirwin->host != LOCAL)
			hide_stop_button();
		end_op();
		return;
	}

    /* Try to cd to specified directory */
	retval = display_dir(dirwin->host, dirwin, full_path, False, True, 
		dirwin->cache_mode, dirwin->cache_mode);
	XtFree(full_path);
	switch (retval) {
	case -6:
		hide_abort_dialog();
		break;
	case -1:
		record_and_alert("Unable to go to directory.", dirwin->w_shell);
		break;
	case -3:
		restore_lost_connection(dirwin->host, dirwin);
	}

	/* Get rid of stop button */
	if (dirwin->host != LOCAL)
		hide_stop_button();

	/* End operation */
	end_op();
}


/*
 * cb_goto_dir_cancel - Callback to handle "Go to Directory" Dialog
 *                      "Cancel" button.
 */
void
cb_goto_dir_cancel(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	/* Get rid of "Go to Directory" Directory Dialog */
	XtDestroyWidget(XtParent(w_gotoDirDialog));

	/* End operation */
	end_op();
}


/*
 * cb_goto_dir_help - Callback that invokes the help package for the "Go to
 *                    Directory" dialog.
 */
void
cb_goto_dir_help(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
    help_dialog(widget, True, "Go to Directory", help_goto_dir);
}


/*
 * cb_goto_dir_from_menu - Callback to go to the directory selected in
 *                         either the "Go to Previous Directory" menu
 *                         or in one of the "Go to Ancestor Directory"
 *                         menus.
 */
void
cb_goto_dir_from_menu(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	XmString string;
	char *new_wd;
	struct dirwin_st *dirwin = (struct dirwin_st *)client_data;
	int retval;

	/* Start operation */
	if (!start_op(False))
		return;

	/* Clear error flag */
	raise_okflag();

	/* Get name of directory to cd to */
	XtVaGetValues(widget, XmNlabelString, &string, NULL);
	new_wd = cstring_to_text(string);
	XmStringFree(string);   /* Yes, this is necessary */

	/* Make operation interruptable if host is remote */
	if (dirwin->host != LOCAL)
		show_stop_button(dirwin);

    /* Try to cd to specified directory */
	retval = display_dir(dirwin->host, dirwin, new_wd, False, True, 
		dirwin->cache_mode, dirwin->cache_mode);
	XtFree(new_wd);
	switch (retval) {
	case -6:
		hide_abort_dialog();
		break;
	case -1:
		record_and_alert("Unable to go to directory.", dirwin->w_shell);
		break;
	case -3:
		restore_lost_connection(dirwin->host, dirwin);
	}

	/* Don't need stop button any longer */
	if (dirwin->host != LOCAL)
		hide_stop_button();

	/* End operation */
	end_op();
}


/*
 * initialize_goto_menu - Initialize the "Go to Previous Directory" menu
 *                        by creating a bunch of placeholder menu items.
 */
initialize_goto_menu(dirwin)
struct dirwin_st *dirwin;
{
	XmString string;
	int i;
	Widget w_pulldown;

	/* Get pulldown menu */
	XtVaGetValues(dirwin->w_gotoMenu, XmNsubMenuId, &w_pulldown, NULL);

	/* Now create the menu items, but leave them unmanaged */
	string = XmStringCreateSimple("Dummy");
	for (i=0; i<MAXDIRHISTORY; i++) {
		dirwin->w_gotoMenuItem[i] = XtVaCreateWidget(
            "",
            xmPushButtonGadgetClass,
            w_pulldown,
            XmNlabelString, string,
            NULL
        );
        XtAddCallback(dirwin->w_gotoMenuItem[i], XmNactivateCallback,
            cb_goto_dir_from_menu, (XtPointer)dirwin);
    }
    XmStringFree(string);
}


/*
 * update_goto_menus_for_host - Update the "Go To" menu in each of the 
 *                              specified host's directory windows.
 */
update_goto_menus_for_host(host)
int host;
{
	struct dirwin_st *dirwin;

	dirwin = dirwin_head;
	while (dirwin) {
		if (dirwin->host == host)
			update_goto_menu(dirwin);
		dirwin = dirwin->next;
	}
}


/*
 * update_all_goto_menus - Update the "Go To" menu in each directory window.
 */
update_all_goto_menus()
{
	struct dirwin_st *dirwin;

	dirwin = dirwin_head;
	while (dirwin) {
		update_goto_menu(dirwin);
		dirwin = dirwin->next;
	}
}


/*
 * update_goto_menu - Update the "Go to Previous Directory" menu with the
 *                    latest values from the directory history.
 */
update_goto_menu(dirwin)
struct dirwin_st *dirwin;
{
	int i;
	struct sl_struct *list;
	XmString label;

    /* Get rid of current "Dir" menu items */
    for (i=0; i<MAXDIRHISTORY; i++)
        XtUnmanageChild(dirwin->w_gotoMenuItem[i]);

    /* Retrieve recently used directory paths */
    retrieve_history(DIRECTORY, hinfo[dirwin->host].hostname, &list,
		sort_caches);

    /* Enter directory paths into "Ops" menu */
    for (i=0; i<list->nentries; i++) {
        label = XmStringCreateSimple(list->entries[i]);
        XtVaSetValues(dirwin->w_gotoMenuItem[i], XmNlabelString, label,
			NULL);
        XmStringFree(label);
        XtManageChild(dirwin->w_gotoMenuItem[i]);
    }

	/* Free stuff */
	release_array_list(list);
}


/*
 * convert_to_full_path - Convert "path" to the equivalent full path
 *                        relative the dirwin's directory path.   Returns
 *                        0 for success, -3 for lost connection, -6 for
 *                        stop button pressed, and -1 for other errors.
 *                        If successful, full path is returned in
 *                        "full_path", which can be released with XtFree().
 */
convert_to_full_path(dirwin, path, full_path)
struct dirwin_st *dirwin;
char *path;
char **full_path;
{
	int host = dirwin->host;
	int retval;

    /* This might take some time */
    use_busy_cursor();

    /* If VMS or if partial path, first cd into working directory */
    if ((hinfo[host].system == SYS_VMS)||(strlen(path) && path[0] != '/')) {
        if (host == LOCAL) {
            if ((retval = local_cd(dirwin->dirname, True)) < 0)
				goto done;
        } else if ((retval = remote_cd(host, dirwin->dirname, True)) < 0)
			goto done;
    }

    /* cd into new directory */
    if (host == LOCAL) {
        if ((retval = local_cd(path, True)) < 0)
			goto done;
        retval = local_pwd(full_path);
    } else {
		if ((retval = remote_cd(host, path, True)) < 0)
			goto done;
        retval = remote_pwd(host, full_path);
    }

done:

	restore_prev_cursor();
	return retval;
}


/*
 * cb_goto_parent - Callback to display directory's parent.
 */
void
cb_goto_parent(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
    struct dirwin_st *dirwin = (struct dirwin_st *)client_data;
    char **wd_links;
    int nlinks = 0;
    char *path;

	/* Start operation */
	if (!start_op(False))
		return;

    /* Clear error flag */
    raise_okflag();

    /* Parse current directory path. */
    wd_links = path_to_links(hinfo[dirwin->host].system, dirwin->dirname);

    /* Determine number of links in parent's path name */
    while (wd_links[nlinks])
        nlinks++;
    nlinks--;

    /* If no links, no parent to display */
    if (nlinks == 0) {
		beep();
		warn("No parent to display", dirwin->w_shell);
		end_op();
        return;
    }

    /* Build path name for parent directory */
	path = links_to_path(hinfo[dirwin->host].system, wd_links, nlinks);

    /* Free up memory returned by path_to_links */
    release_path_links(wd_links);

	/* Make operation interruptable if host is remote */
	if (dirwin->host != LOCAL)
		show_stop_button(dirwin);

    /* Try to cd to parent directory */
	switch (display_dir(dirwin->host, dirwin, path, False, True, 
		dirwin->cache_mode, dirwin->cache_mode)) {
	case -6:
		hide_abort_dialog();
		break;
	case -1:
		record_and_alert("Unable to go to directory.", dirwin->w_shell);
		break;
	case -3:
		restore_lost_connection(dirwin->host, dirwin);
	}
	XtFree(path);

	/* Get rid of stop button */
	if (dirwin->host != LOCAL)
		hide_stop_button();

	/* End operation */
	end_op();
}


/*
 * cb_goto_home - Callback to display home directory.
 */
void
cb_goto_home(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
    struct dirwin_st *dirwin = (struct dirwin_st *)client_data;
	int retval;

	/* Start operation */
	if (!start_op(False))
		return;

    /* Clear error flag */
    raise_okflag();

	/* Make operation interruptable if host is remote */
	if (dirwin->host != LOCAL)
		show_stop_button(dirwin);

    /* Try to cd to specified directory */
	retval = display_dir(dirwin->host, dirwin, hinfo[dirwin->host].homedir,
		False, True, dirwin->cache_mode, dirwin->cache_mode);
	switch (retval) {
	case -6:
		hide_abort_dialog();
		break;
	case -1:
		record_and_alert("Unable to go to directory.", dirwin->w_shell);
		break;
	case -3:
		restore_lost_connection(dirwin->host, dirwin);
	}

	/* Get rid of stop button */
	if (dirwin->host != LOCAL)
		hide_stop_button();

	/* End operation */
	end_op();
}


/*
 * init_ancestors_menus - Create the popup ancestor menu and then
 *                        initialize the popup and pulldown ancestor
 *                        menus by creating a bunch of placeholder menu
 *                        items.
 */
init_ancestors_menus(dirwin)
struct dirwin_st *dirwin;
{
	XmString string;
	int i;
	Widget w_pulldown;

	/* Get pulldown menu */
	XtVaGetValues(dirwin->w_ancestorsMenu, XmNsubMenuId, &w_pulldown, NULL);

	/* Create popup menu */
	dirwin->w_popupAncestorsMenu = XmCreatePopupMenu(dirwin->w_upArrowButton,
		"popupAncestorsMenu", NULL, 0);

	/* Add event handler to pick up press of button 3 */
	XtAddEventHandler(dirwin->w_upArrowButton, ButtonPressMask, False,
		HandleUpArrowEvents, (XtPointer)dirwin);

	/* Create header for popup menu */
	XtVaCreateManagedWidget("popupAncestorsMenuLabel", xmLabelGadgetClass,
		dirwin->w_popupAncestorsMenu, NULL);
	XtVaCreateManagedWidget("separator", xmSeparatorGadgetClass,
		dirwin->w_popupAncestorsMenu, NULL);
	XtVaCreateManagedWidget("separator", xmSeparatorGadgetClass,
		dirwin->w_popupAncestorsMenu, NULL);

	/* Now create the menu items, but leave them unmanaged */
	string = XmStringCreateSimple("Dummy");
	for (i=0; i<MAXLEVELS; i++) {
		dirwin->w_ancestorsMenuItem[i] = XtVaCreateWidget(
            "",
            xmPushButtonGadgetClass,
            w_pulldown,
            XmNlabelString, string,
            NULL
        );
        XtAddCallback(dirwin->w_ancestorsMenuItem[i], XmNactivateCallback,
            cb_goto_dir_from_menu, (XtPointer)dirwin);
   		dirwin->w_popupAncestorsMenuItem[i] = XtVaCreateWidget(
            "",
            xmPushButtonGadgetClass,
            dirwin->w_popupAncestorsMenu,
            XmNlabelString, string,
            NULL
        );
        XtAddCallback(dirwin->w_popupAncestorsMenuItem[i], XmNactivateCallback,
            cb_goto_dir_from_menu, (XtPointer)dirwin);
    }
    XmStringFree(string);
}


/*
 * update_ancestors_menus - Update the popup and pulldown ancestors menus
 *                          the latest values.
 */
update_ancestors_menus(dirwin)
struct dirwin_st *dirwin;
{
	int i;
	int nlinks = 0;
	char **wd_links;
	char *path;
	XmString label;

    /* Get rid of current ancestor menus items */
    for (i=0; i<MAXLEVELS; i++) {
        XtUnmanageChild(dirwin->w_ancestorsMenuItem[i]);
		XtUnmanageChild(dirwin->w_popupAncestorsMenuItem[i]);
	}

	/* Parse this window's directory path. */
	wd_links = path_to_links(hinfo[dirwin->host].system, dirwin->dirname);

	/* Determine number of links in directory path */
	while (wd_links[nlinks])
		nlinks++;
	if (nlinks > MAXLEVELS)
		fatal_error("Exceeded maximum path links in update_ancestors_menus()");

	/* Build path names for ancestors */
	for (i=0; i<nlinks-1; i++) {
		path = links_to_path(hinfo[dirwin->host].system, wd_links, nlinks-i-1);
		label = XmStringCreateSimple(path);
		XtFree(path);
		XtVaSetValues(dirwin->w_ancestorsMenuItem[i], XmNlabelString, label,
			NULL);
		XtManageChild(dirwin->w_ancestorsMenuItem[i]);
		XtVaSetValues(dirwin->w_popupAncestorsMenuItem[i], XmNlabelString,
			label, NULL);
		XtManageChild(dirwin->w_popupAncestorsMenuItem[i]);
		XmStringFree(label);
	}

	/* Free up memory returned by path_to_links */
	release_path_links(wd_links);

	/* Adjust sensitivity of "Go to Ancestor" pullright */
	if (nlinks == 1)
		XtSetSensitive(dirwin->w_ancestorsMenu, False);
	else
		XtSetSensitive(dirwin->w_ancestorsMenu, True);
}


/*
 * HandleUpArrowEvents - Handle presses of mouse button 3 on up arrow button.
 */
void
HandleUpArrowEvents(widget, client_data, event, continue_to_dispatch_return)
Widget widget;
XtPointer client_data;
XEvent *event;
Boolean *continue_to_dispatch_return;
{
	struct dirwin_st *dirwin = (struct dirwin_st *)client_data;

	/* Sanity check */
	if (event->type != ButtonPress)
		fatal_error("Bug in HandleUpArrowEvents()");
	
	/* Only want right mouse button */
	if (event->xbutton.button != Button3)
		return;

	/* Pop up "Go to Ancestors" menu */
	XmMenuPosition(dirwin->w_popupAncestorsMenu,
		(XButtonPressedEvent *)event);
	XtManageChild(dirwin->w_popupAncestorsMenu);
}

