/* TI-99/4A Emulator.
   GROM Emulator part.

   Functions:

   void grom_write(word address,word data);
   * Writes data into grom memory if (address&2)==0. This is the case when
     >9C00 is addressed. The data is assumed to be 16 bits, but only the
     MSbyte is written. The GromAdressCounter is incremented by one.
   * Updates the GromAdressCounter when (address&2)!=0. This is the case
     when >9C02 is addressed. The GromAddressCounter is shifted left 8
     bits, and the MSbyte of 'data' is the new LSbyte of the GAC.
     So to change the GAC, write the new MSbyte first, followed by the
     LSbyte.

   word grom_read(word address,word data);
   * Reads data from grom memory if (address&2)==0. The byte pointed to by
     the GAC is returned as MSbyte of a word. The GAC is incremented.
   * Returnes the MSbyte of the GAC as the MSbyte of a word. The GAC is
     shifted left 8 bits, so grom-read-data actions cannot take place until
     a new grom address is written.


   A GROM is essentially a mere ROM memory with a serial access mechanism.
   To this end the grom is equipped with a address counter. This counter
   must be set by the CPU. When the data is read-out through the read-data
   address the address counter is incremented automatically. Though a
   typical GROM produced by TI is only 6k (2k is wasted every 8k) and has
   its own counter, in the emulator the GROM bank is 32k (or more) of
   contiguous RAM. It is decided to make the GROM read-only, but it should
   be very easy to emulate a GRAM device.

   The adresses through which GROM is addressed are:

   >9800:	GROM read data
   >9802:	GROM read address
   >9C00:	GROM write data
   >9C02:	GROM write address

   These addresses are not fully decoded, so the same adressing is achieved
   throughout the >9800..>9FFF area. (e.g. >9800=>9804=>9808=... etc.)

*/

void init_grom(void)
{
	grom=(byte *)malloc(GROM_MEMSIZE);
	if (grom==NULL)
	{
		printf("Not enough memory for GROM.\n");
		exit(0);
	}
	GromAddressCounter=0;
}

word grom_read(word address)
{
	if(address&1)  /* GROM read address. Address will be destroyed. */
	{
		word dummy;
		GromAddressCounter++;
		dummy=(GromAddressCounter&0xff00);
		GromAddressCounter&=0xff; 	/* filter lsbyte */
		GromAddressCounter<<=8; 	/* shift left    */
		return(dummy);
	}

	else		/* GROM read data */
	return(word)(*(grom+GromAddressCounter++)<<8);
}

void grom_write(word gwaddress,word gdata)
{
	if (gwaddress&1)	/* write address */
	GromAddressCounter=(GromAddressCounter<<8)|(gdata>>8);

/*      	write data not (yet) supported
	else
	{
		grom[GromAddressCounter]=(byte)(gdata>>8);
		GromAddressCounter++;
	}
*/
}
