/*
	rt.c

	Runs process on realtime priority.

	$Id: rt.c,v 1.30 1999/11/06 11:39:34 fedorov Exp $

	tab size: 4
*/

#define VERSION "2.2"

#include <stdio.h>
#include <sched.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>			/* memset(3)  */
#include <ctype.h>			/* isdigit(3) */


#ifndef  SCHED_IDLE
# define SCHED_IDLE 3		/* from sched_idle patch */
#endif

#ifndef  __STRING
# define __STRING(x)	#x
#endif

#if defined(__GNUG__) && (__GNUG__>2 || (__GNUG__==2 && __GNUC_MINOR__>=6))
const bool On=true;
const bool Off=false;
#else
typedef enum bool { false=0, true=1, Off=false, On=true } bool;
#endif

#define BAD(a)       ( ~((__typeof__(a))0) )
#define ISBAD(a)     (a==BAD(a))
#define SETBAD(a)    (a=BAD(a))


#ifndef  EXIT_SUCCESS
# define EXIT_SUCCESS 0
#endif
#define EXIT_OK      EXIT_SUCCESS

#define EXIT_USAGE   1

#undef  EXIT_FAILURE
#define EXIT_FAILURE 2


/* QNX-scheduler patch tolerance */
union sched_param_extended
{
	struct sched_param param;
	int pad[16];
};


static void __attribute__((noreturn)) usage(void)
{
	fprintf(stdout,
"\n"
"Linux real time scheduling client, v" VERSION "\n"
"Copyright (C) 1996, Boris Tobotras <boris@xtalk.msk.su>\n"
"              1998, Dmitry Fedorov <fedorov@inp.nsk.su>\n"
"\n"
"This software may be used and distributed according to the terms\n"
"of the GNU Public License, incorporated herein by reference.\n"
"\n"
"Usage: rt [options] [command]\n"
"\n"
"Sets real-time priority, lowers euid,egid to real ones and execute command.\n"
"\n"
"Recognized options are:\n"
"\t-r[N]   Use SCHED_RR scheduling policy (default);\n"
"\t-f[N]   Use SCHED_FIFO scheduling policy;\n"
"\t-i[N]   Use SCHED_IDLE scheduling policy (patched kernel only);\n"
"\t   N    Priority level. Default is sched_get_priority_min()\n"
"\t-o      Use SCHED_OTHER scheduling policy (not real time);\n"
"\t-s[pid] Show priority of running process by pid;\n"
"\t-S[pid] Set priority of running process by pid;\n"
"\t-v      Be verbose;\n"
"\t-V      Print version;\n"
"\t-h      This help screen;\n"
"\t--      End of options.\n"
"\n"
		);

	exit(EXIT_USAGE);
}

static void __attribute__((noreturn)) version(void)
{
	printf(VERSION "\n");
	exit(EXIT_OK);
}

static void __attribute__((noreturn)) error(const char* where)
{
	fprintf(stderr, "rt: %s: %s\n", where, strerror(errno));
	exit(EXIT_FAILURE);
}


static int getintoptarg(void)
{
	if (!isdigit(*optarg))	/* positive integer only */
	{
		fprintf(stderr, "positive number expected.\n");
		exit(EXIT_USAGE);
	}

	return atoi(optarg);
}


static const char* strpolicy(int policy)
{
	static const char unknown_format[] = "UNKNOWN %d";
	static char unknown[ sizeof(unknown_format)+11 ];

	switch (policy)
	{
		case SCHED_OTHER:
			return __STRING(SCHED_OTHER);
		case SCHED_FIFO:
			return __STRING(SCHED_FIFO);
		case SCHED_RR:
			return __STRING(SCHED_RR);
		case SCHED_IDLE:
			return __STRING(SCHED_IDLE);
		default:
			sprintf(unknown, unknown_format, policy);
			return unknown;
	}
}


/* returns error flag */
static bool check_priority_limits(int policy, int priority,
	int prio_min, int prio_max)
{
	if ( priority<prio_min || priority>prio_max )
	{
		fprintf(stderr,
			"Bad priority value %d, should be %d..%d for %s policy.\n",
			priority, prio_min, prio_max, strpolicy(policy) );
		return true;
	}

	return false;
}


int main(int argc, char *argv[])
{
	union sched_param_extended xparam;
	int opt;
	int policy=BAD(policy), priority=BAD(priority),
		prio_min=BAD(prio_min), prio_max=BAD(prio_max);
	pid_t pid=0;	/* self */
	bool set=false, show=false, policy_present=false, prio_present=false,
		verbose=false;

	memset( &xparam, 0, sizeof(xparam) );

	/* + parse options */
	if (argc<2) usage();


#if 0	/* help2man(1) requirement */
	{
        int i;
		for (i=1; i<argc; i++)
		{
			if (!strcmp(argv[i],"--help"))
				usage();

			if (!strcmp(argv[i],"--version"))
				version();
		}
	}
#endif


	while ( (opt=getopt(argc, argv, "r::f::i::os::S::hvV")) != EOF )
	{
#ifdef DEBUG
		fprintf(stderr, "opt = '%c'  optind = %d  optarg = '%s'\n",
			opt, optind, optarg);
#endif
		switch (opt)
		{
			case 'r':
				policy=SCHED_RR;
				if (optarg!=NULL) priority=getintoptarg();
				break;
			case 'f':
				policy=SCHED_FIFO;
				if (optarg!=NULL) priority=getintoptarg();
				break;
			case 'i':
				policy=SCHED_IDLE;
				if (optarg!=NULL) priority=getintoptarg();
				break;
			case 'o':
				policy=SCHED_OTHER;
				priority=0;
				break;
			case 's':
				show=true;
				if (optarg!=NULL) pid=getintoptarg();
				break;
			case 'S':
				set=true;
				if (optarg!=NULL) pid=getintoptarg();
				break;
			default:
			case 'h':
				usage();
			case 'v':
				verbose=true;
				break;
			case 'V':
				version();
		}
	}

	if (set&&show)
	{
		fprintf(stderr, "Set and Show options are mutually exclusive.\n");
		return EXIT_USAGE;
	}

	if ( (set||show) && (argv[optind] != NULL) )
	{
		fprintf(stderr, "Set and Show options don't need extra command.\n");
		return EXIT_USAGE;
	}

	if (!ISBAD(policy))
		policy_present=true;

	if (!ISBAD(priority))
		prio_present=true;

	if ( show && (policy_present||prio_present) )
	{
		fprintf(stderr,
			"Show option don't need priority or policy options.\n");
		return EXIT_USAGE;
	}
	/* - parse options */

/*--------------------------------------------------------------------*/

	/* + get current policy or set default */
	if ( show || (set && !policy_present) )
	{
		int rc=sched_getscheduler(pid);		/* get policy */
		if (ISBAD(rc))
			error("sched_getscheduler()");
		policy=rc;
	}
	else
		if (!policy_present)
			policy=SCHED_RR;				/* default policy */
	/* - get current policy or set default */


	/* + get priority limits */
	{
		/* sched_idle patch don't touched
			sys_sched_get_priority_{min,max} () */
		int _policy = (policy==SCHED_IDLE) ? SCHED_RR : policy;

		prio_min=sched_get_priority_min(_policy);
		if (ISBAD(prio_min))
			error("sched_get_priority_min()");

		prio_max=sched_get_priority_max(_policy);
		if (ISBAD(prio_max))
			error("sched_get_priority_max()");
	}
	/* - get priority limits */

	/* + get current priority or set default */
	if ( show || (set && !prio_present) )
	{
		if (sched_getparam(pid, &xparam.param)!=0)	/* get priority */
			error("sched_getparam()");
		priority=xparam.param.sched_priority;
	}
	else
		if (!prio_present)
			priority=prio_min;	/* default priority */
	/* - get current priority or set default */

	xparam.param.sched_priority=priority;

/*--------------------------------------------------------------------*/

	if (show)
	{
		/* sched_idle patch bug workaround */
        int _priority = policy==SCHED_IDLE ? priority+1999 : priority;

		fprintf(stdout,
			"process id %d, policy %s, priority %d, priority range %d..%d\n",
			pid, strpolicy(policy), _priority, prio_min, prio_max);

		check_priority_limits(policy, _priority, prio_min, prio_max);

		return EXIT_OK;	/* show done */
	}
	else
		if (check_priority_limits(policy, priority, prio_min, prio_max))
			return EXIT_USAGE;


	if (verbose)
	{
		if (set)
			fprintf(stdout,
"Setting process %d with policy %s, priority %d.\n",
				pid, strpolicy(policy), priority );
		else
			fprintf(stdout,
"Starting %s with policy %s, priority %d, real uid %u, real gid %u.\n",
				argv[optind], strpolicy(policy), priority,
				getuid(), getgid() );
	}


	if (sched_setscheduler(set?pid:0, policy, &xparam.param))
		error("sched_setscheduler()");

	if (set) return EXIT_OK;		/* set done */


	if (setregid(-1, getgid()))		/* lower to real group id */
		error("setregid()");

	if (setreuid(-1, getuid()))		/* lower to real user id */
		error("setreuid()");

	execvp(argv[optind], argv+optind);
	fprintf(stderr, "Can't exec %s: %s\n", argv[optind],
			 strerror(errno) );
	return EXIT_FAILURE;
}


/*
$Log: rt.c,v $
Revision 1.30  1999/11/06 11:39:34  fedorov
tabs

Revision 1.29  1999/11/06 11:38:31  fedorov
Some output was going to stderr insted of stdout. Fixed.

Revision 1.27  1999/06/28 09:04:34  fedorov
printing version added

Revision 1.26  1999/06/27 15:09:37  fedorov
text sorted;
EXIT_* constants added;
BAD redefined, ISBAD,SETBAD macros added and used;
str_policy() renamed to strpolicy();
__STRING macros used in strpolicy();
some code moved to new check_priority_limits();
SCHED_IDLE patch bug with priority value workaround;

Revision 1.25  1999/02/04 13:39:40  fedorov
indentation

Revision 1.24  1999/02/04 07:13:41  fedorov
bool

Revision 1.22  1998/08/15 11:41:38  fedorov
experiment is out of time

Revision 1.21  1998/08/15 11:39:08  fedorov
indentation

Revision 1.20  1998/06/01 13:10:28  fedorov
correct description

Revision 1.19  1998/05/31 12:46:33  fedorov
lower euid sentence added to Usage;
getintoptarg() bug fixed;
-1 defined as BAD symbol;
conditional getopt debug code added;
check for extraneous command on -s,-S options presence;
force priority 0 on SCHED_OTHER policy;

Revision 1.18  1998/05/31 10:45:26  fedorov
usage(),error() is noreturn now

Revision 1.17  1998/05/31 10:16:54  fedorov
priority value option moved to policy option argument;
pid moved to '-s' and '-S' option argument;
getintoptarg() added to simplify;
get/set uid/gid code compacted;

Revision 1.16  1998/05/31 06:45:18  fedorov
formatting style changed; No negative priority wrapping

Revision 1.13  1998/05/19 23:07:58  fedorov
shorted license text

Revision 1.12  1998/05/19 21:37:31  fedorov
restore to real gid added

Revision 1.11  1998/05/15 14:06:47  fedorov
digit checking added on pid argument

Revision 1.10  1998/05/10 09:20:05  fedorov
QNX scheduler patch tolerance

Revision 1.9  1998/05/10 09:01:50  fedorov
Set/Show policy & priority of running process;
SCHED_IDLE & SCHED_OTHER policy added;
str_policy() convert integer policy to string representation;
typedefed bool used for some flags;

Revision 1.8  1998/02/25 16:19:21  fedorov
setreuid(old_user_id) added - return from root to user privilege

Revision 1.7  1998/02/25 15:37:46  fedorov
sys_errlist[errno] replaced by POSIX strerror(errno)

Revision 1.6  1998/02/25 15:28:59  fedorov
Usage() returns exit code now; blank lines added to usage text

Revision 1.5  1998/02/25 15:21:40  fedorov
Fedorov copyright fixed

Revision 1.4  1998/02/25 15:18:59  fedorov
Fedorov copyright added; default priority is min now

Revision 1.3  1998/02/22 16:08:54  fedorov
comment extra (c) message in Usage()

Revision 1.2  1998/02/22 15:46:05  fedorov
main() must return int
*/

