/* #Specification: dnsconf / ip range / strategy
	linuxconf allows one to qualify the different IP numbers of his
	network. For exemple, one managing a class C network such as
	192.168.1.0 may want to organised things this way

	#
	192.168.1.1-20		Servers
	192.168.1.21-40		IP for PPP dialin
	192.168.1.41-60		Virtual domain
	192.168.1.61-254	Reserved
	#

	This has two effects. First it documents the network and allows a
	new operator to have some clues. Next, it allows linuxconf to help
	you find an unused IP number.

	The user interface for managing IP number use this and will locate
	one unused IP number in every range (If possible).
*/
#include <string.h>
#include <misc.h>
#include "dnsconf.h"
#include "internal.h"
#include "dnsconf.m"

static DNSCONF_HELP_FILE help_iprange("iprange");

PUBLIC IPMAP::IPMAP(const char *line)
{
	line = iprange.copyword (line);
	line = str_skip(line);
	comment.setfrom (line);
}
PUBLIC IPMAP::IPMAP()
{
}

/*
	Parse the IP range and prepare for the search of an available IP.
	Return -1 if the IP range is invalid (not x.y.z.w1-w2)
*/
PUBLIC int IPMAP::setup()
{
	int ret = -1;
	char tmp[200];
	iprange.copy(tmp);
	char *pt = strchr(tmp,'-');
	over = 0;
	if (pt != NULL){
		*pt++ = '\0';
		minimum.setfrom(tmp);
		if (minimum.is_valid()){
			IP_ADDR part;
			part.setfrom(str_skip(pt));
			avail.setfrom (tmp);
			maximum.setfrom (tmp);
			part.shift_right();
			maximum.merge (part);
			if (maximum.is_valid()) ret = 0;
		}
	}
	return ret;
}

/*
	Record the fact that the IP "adr" is in use.
	This function assume that it will be called with adr always increasing.
*/
PUBLIC void IPMAP::setuse (const IP_ADDR *adr)
{
	if (minimum.cmp(adr)<=0
		&& maximum.cmp(adr)>=0){
		if (avail.cmp(adr)==0){
			if (avail.cmp(&maximum)==0){
				over = 1;
			}else{
				avail.increm();
			}
		}
	}
}

static char IPDB[]="IPDB";
static char RANGE[]="RANGE";

PUBLIC IPMAP *IPMAPS::getitem(int no)
{
	return (IPMAP*)ARRAY::getitem(no);
}

PUBLIC IPMAPS::IPMAPS()
{
	SSTRINGS lst;
	int nb = linuxconf_getall (IPDB,RANGE,lst,0);
	for (int i=0; i<nb; i++){
		add (new IPMAP(lst.getitem(i)->get()));
	}
}

/*
	Update /etc/conf.linuxconf
*/
PUBLIC int IPMAPS::write()
{
	linuxconf_removeall (IPDB,RANGE);
	int n = getnb();
	for (int i=0; i<n; i++){
		IPMAP *e = getitem(i);
		if (!e->iprange.is_empty()){
			char buf[200];
			sprintf (buf,"%s %s",e->iprange.get(),e->comment.get());
			linuxconf_add (IPDB,RANGE,buf);
		}
	}
	return linuxconf_save();
}


/*
	Compute the available IP number in each IP address range.
	adrs will be sort by this algoryth.
*/
PUBLIC void IPMAPS::setuse (IP_ADDRS &adrs)
{
	int n = getnb();
	int i;
	for (i=0; i<n; i++){
		getitem(i)->setup();
	}
	adrs.sort();
	int na = adrs.getnb();
	for (i=0; i < na; i++){
		IP_ADDR *adr = adrs.getitem(i);
		for (int j=0; j<n; j++){
			getitem(j)->setuse(adr);
		}
	}
	for (i=0; i<n; i++){
		getitem(i)->avail.reformat();
	}
}

/*
	Setup the comp field options helping in IP number allocation.
*/
PUBLIC void IPMAPS::setcombo (class FIELD_COMBO *comb)
{
	int n = getnb();
	for (int i=0; i<n; i++){
		IPMAP *e = getitem(i);
		const char *str = e->avail.get();
		if (e->over){
			str = MSG_U(F_RANGEFULL,"None available");
		}
		comb->addopt (str,e->comment.get());
	}
}

PRIVATE void IPMAPS::newdiaf (DIALOG &dia, IPMAP *e)
{
	dia.newf_str (MSG_U(F_IPRANGE,"One IP range"),e->iprange);
	dia.newf_str (MSG_U(F_RANGECOMMENT,"Comment"),e->comment);
}
	
PUBLIC int IPMAPS::edit()
{
	int no = 0;
	if (getnb()==0)	add (new IPMAP);
	DIALOG dia;
	int n = getnb();
	for (int i=0; i<n; i++) newdiaf (dia,getitem(i));
	while (1){
		MENU_STATUS code = dia.edit (MSG_U(T_IPRANGE,"IP range definitions")
			,MSG_U(I_IPRANGE
				,"You can define here the range of IP numbers available\n"
				 "to this DNS. It won't limit your ability of assigning\n"
				 "the IP number you want, but will allow you to find\n"
				 "an available one easily later.")
				,help_iprange
				,no
				,MENUBUT_ADD|MENUBUT_CANCEL|MENUBUT_ACCEPT);
		if (code == MENU_CANCEL || code == MENU_ESCAPE){
			dia.restore();
			break;
		}else if (code == MENU_ACCEPT){
			int err = 0;
			for (int j=0; j<n; j++){
				IPMAP *e = getitem(j);
				if (!e->iprange.is_empty()
					&& e->setup()==-1){
					no = j*2;
					err = 1;
					xconf_error (MSG_U(E_IVLDIPMAP
						,"Invalid IP range definition %s\n"
						 "Expected X.Y.Z.W1-W2\n"
						 "or       X.Y.Z1.W1-Z2.W2")
						,e->iprange.get());
					break;
				}
			}
			if (!err){
				write();
				break;
			}
		}else{
			IPMAP *e = new IPMAP;
			add (e);
			newdiaf(dia,e);
		}
	}
	return 0;
}

void ipmap_edit ()
{
	IPMAPS maps;
	maps.edit();
}

