/* $Id: volume.c,v 1.8 1996/03/27 15:29:04 brianp Exp $ */

/* Vis5D version 4.2 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990-1996 Bill Hibbard, Brian Paul, Dave Santek,
and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


/*
 * Volume rendering.  Requires alpha blending capability.  This is
 * supported in on SGI hardware with VGX or higher graphics.  Otherwise,
 * it's done in software on some other systems.
 *
 * Volume rendering is also supported on everthing running OpenGL!
 */


#include <math.h>
#include <stdlib.h>
#include "globals.h"
#include "graphics.h"
#include "grid.h"
#include "memory.h"
#include "proj.h"
#include "volume.h"
#ifdef OPENGL
#  include <GL/gl.h>
#endif
#if defined(SGI_GL) || defined(DENALI)
#  include <gl/gl.h>
#endif



#define ABS(A)  ( (A) < 0 ? -(A) : (A) )


/* Direction of slices: */
#define BOTTOM_TO_TOP  0
#define TOP_TO_BOTTOM  1
#define EAST_TO_WEST   2
#define WEST_TO_EAST   3
#define NORTH_TO_SOUTH 4
#define SOUTH_TO_NORTH 5

#define LUT_SIZE 255



struct volume {
	int		dir;		/* Direction of slices */
	int		valid;		/* Valid flag */
	int		slices;		/* Number of slices */
	int		rows, cols;	/* Size of each slice */
	float		*vertex;	/* slice vertices stored as: */
					/*    vertex[slices][rows][cols][3] */
	uint_1		*index;		/* color table index in [0,255] */
};



/*
 * Allocate a new volume structure and return a pointer to it.  If we
 * can't do volume rendering return NULL.
 * Input:  nr, nc, nl - dimensions of largest volume we'll encounter.
 * Return:  pointer to volume struct or NULL
 */
struct volume *alloc_volume( Context ctx, int nr, int nc, int nl )
{
   long alphabits;
   int volren = 0;
   struct volume *v = NULL;

   if (nl <= 1) return 0; /* no volume variables */

#if defined (SGI_GL) || defined (DENALI)
   alphabits = getgdesc( GD_BLEND ); 
   if (alphabits==0) {
      /* no alpha support means no volume rendering */
      return NULL;
   }
   volren = 1;
#endif

#if defined(OPENGL)
   volren = 1;
#endif

   if (volren) {
      v = (struct volume *) malloc( sizeof(struct volume) );
      /* initialize the volume struct */
      v->valid = 0;
      /* No matter which way we slice it, we need the same size arrays for */
      /* storing the vertices and colors. */
      v->vertex = (float *) pallocate( ctx, nl*nr*nc*3*sizeof(float) );
      v->index = (uint_1 *) pallocate( ctx, nl*nr*nc*sizeof(uint_1) );
      if (!v->vertex || !v->index) {
	 printf("WARNING:  insufficient memory for volume rendering (%d bytes needed)\n",
		nl * nr * nc * (3*sizeof(float)+sizeof(uint_1)) );
	 return NULL;
      }
   }

   return v;
}



/*
 * Compute a volume rendering of the given grid.
 * Input:  data - 3-D data grid.
 *         time, var - time step and variable
 *         nr, nc, nl - size of 3-D grid.
 *         min, max - min and max data value in data grid.
 *         dir - direction to do slicing
 *         v - pointer to a volume struct with the vertex and index
 *             fields pointing to sufficiently large buffers.
 * Output:  v - volume struct describing the computed volume.
 */
static int compute_volume( Context ctx, float data[],
                           int time, int var,
                           int nr, int nc, int nl, int lowlev,
                           float min, float max,
                           int dir,
                           struct volume *v )
{
   float zs[MAXLEVELS];
   register int ir, ic, il, i, j;
   register float x, y, dx, dy;
   register float dscale, val;
   register int ilnrnc, icnr;           /* to help speed up array indexing */

   /* compute graphics Z for each grid level */
   for (il=0; il<nl; il++) {
     zs[il] = gridlevel_to_z(ctx, time, var, (float) (il + lowlev));
   }

   /* compute some useful values */
   dx = (ctx->Xmax-ctx->Xmin) / (nc-1);
   dy = (ctx->Ymax-ctx->Ymin) / (nr-1);
   dscale = (float) (LUT_SIZE-1) / (max-min);

   v->dir = dir;

   switch (dir) {
      case BOTTOM_TO_TOP:
         /* compute slices from bottom to top */
         v->slices = nl;
	 v->rows = nr;
	 v->cols = nc;

	 i = 0;  /* index into vertex array */
	 j = 0;  /* index into index array */
	 for (il=0; il<nl; il++) {
	    y = ctx->Ymax;
	    ilnrnc = il * nr * nc;
	    for (ir=0;ir<nr;ir++) {
	       x = ctx->Xmin;
	       for (ic=0;ic<nc;ic++) {
		  /* compute vertex */
		  v->vertex[i++] = x;
		  v->vertex[i++] = y;
		  v->vertex[i++] = zs[il];

		  /* compute color table index */
		  val = data[ ilnrnc + ic * nr + ir ];
		  if (IS_MISSING(val))
		    v->index[j++] = 255;
		  else
		    v->index[j++] = (uint_1) (int) ((val-min) * dscale);

		  x += dx;
	       }
	       y -= dy;
	    }
	 }
	 break;

      case TOP_TO_BOTTOM:
	 /* compute slices from top to bottom */
	 v->slices = nl;
	 v->rows = nr;
	 v->cols = nc;

	 i = 0;  /* index into vertex array */
	 j = 0;  /* index into index array */
	 for (il=nl-1; il>=0; il--) {
	    y = ctx->Ymax;
	    ilnrnc = il * nr * nc;
	    for (ir=0;ir<nr;ir++) {
	       x = ctx->Xmin;
	       for (ic=0;ic<nc;ic++) {
		  /* compute vertex */
		  v->vertex[i++] = x;
		  v->vertex[i++] = y;
		  v->vertex[i++] = zs[il];

		  /* compute color table index */
		  val = data[ ilnrnc + ic * nr + ir ];
		  if (IS_MISSING(val))
		    v->index[j++] = 255;
		  else
		    v->index[j++] = (uint_1) (int) ((val-min) * dscale);

		  x += dx;
	       }
	       y -= dy;
	    }
	 }
	 break;

      case WEST_TO_EAST:
	 /* compute slices from west to east */
	 v->slices = nc;
	 v->rows = nl;
	 v->cols = nr;

	 i = 0;  /* index into vertex array */
	 j = 0;  /* index into index array */
	 x = ctx->Xmin;
	 for (ic=0; ic<nc; ic++) {
	    icnr = ic * nr;
	    for (il=nl-1;il>=0;il--) {
	       y = ctx->Ymin;
	       ilnrnc = il * nr * nc + icnr;
	       for (ir=nr-1;ir>=0;ir--) {
		  /* compute vertex */
		  v->vertex[i++] = x;
		  v->vertex[i++] = y;
		  v->vertex[i++] = zs[il];

		  /* compute color table index */
		  val = data[ ilnrnc + ir ];
		  if (IS_MISSING(val))
		    v->index[j++] = 255;
		  else
		    v->index[j++] = (uint_1) (int) ((val-min) * dscale);

		  y += dy;
	       }
	    }
	    x += dx;
	 }
	 break;

      case EAST_TO_WEST:
	 /* compute slices from east to west */
	 v->slices = nc;
	 v->rows = nl;
	 v->cols = nr;

	 i = 0;  /* index into vertex array */
	 j = 0;  /* index into index array */
	 x = ctx->Xmax;
	 for (ic=nc-1; ic>=0; ic--) {
	    icnr = ic*nr;
	    for (il=nl-1;il>=0;il--) {
	       y = ctx->Ymin;
	       ilnrnc = il * nr * nc + icnr;
	       for (ir=nr-1;ir>=0;ir--) {
		  /* compute vertex */
		  v->vertex[i++] = x;
		  v->vertex[i++] = y;
		  v->vertex[i++] = zs[il];

		  /* compute color table index */
		  val = data[ ilnrnc + ir ];
		  if (IS_MISSING(val))
		    v->index[j++] = 255;
		  else
		    v->index[j++] = (uint_1) (int) ((val-min) * dscale);

		  y += dy;
	       }
	    }
	    x -= dx;
	 }
	 break;

      case NORTH_TO_SOUTH:
	 /* compute slices from north to south */
	 v->slices = nr;
	 v->rows = nl;
	 v->cols = nc;

	 i = 0;  /* index into vertex array */
	 j = 0;  /* index into index array */
	 y = ctx->Ymax;
	 for (ir=0; ir<nr; ir++) {
	    for (il=nl-1;il>=0;il--) {
	       x = ctx->Xmin;
	       ilnrnc = il * nr * nc;
	       for (ic=0;ic<nc;ic++) {
		  /* compute vertex */
		  v->vertex[i++] = x;
		  v->vertex[i++] = y;
		  v->vertex[i++] = zs[il];

		  /* compute color table index */
		  val = data[ ilnrnc + ic*nr + ir ];
		  if (IS_MISSING(val))
		    v->index[j++] = 255;
		  else
		    v->index[j++] = (uint_1) (int) ((val-min) * dscale);

		  x += dx;
	       }
	    }
	    y -= dy;
	 }
	 break;

      case SOUTH_TO_NORTH:
	 /* compute slices from south to north */
	 v->slices = nr;
	 v->rows = nl;
	 v->cols = nc;

	 i = 0;  /* index into vertex array */
	 j = 0;  /* index into index array */
	 y = ctx->Ymin;
	 for (ir=nr-1; ir>=0; ir--) {
	    for (il=nl-1;il>=0;il--) {
	       x = ctx->Xmin;
	       ilnrnc = il * nr * nc;
	       for (ic=0;ic<nc;ic++) {
		  /* compute vertex */
		  v->vertex[i++] = x;
		  v->vertex[i++] = y;
		  v->vertex[i++] = zs[il];

		  /* compute color table index */
		  val = data[ ilnrnc + ic*nr + ir ];
		  if (IS_MISSING(val))
		    v->index[j++] = 255;
		  else
		    v->index[j++] = (uint_1) (int) ((val-min) * dscale);

		  x += dx;
	       }
	    }
	    y += dy;
	 }
	 break;

      default:
	 printf("Error in compute_volume!\n");

   } /* switch */

   v->valid = 1;
   return 1;
}



/*
 * Render the volume described by the given volume struct.
 * Return:  1 = ok
 *          0 = bad volume struct.
 */
static int render_volume( Context ctx,
                          struct volume *v, unsigned int ctable[] )
{
   register int rows, cols, slices, i, j, s;
   register uint_1 *cp0, *cp1;
   register float *vp0, *vp1;

   if (!v || !v->slices)
      return 0;

#if defined (SGI_GL) || defined (DENALI)
   lmcolor( LMC_COLOR );             /* no shading */
   blendfunction( BF_SA, BF_MSA );   /* enable alpha blending */
#endif
#ifdef OPENGL
   glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
   glEnable( GL_BLEND );
#endif

   /* put rows, cols, slices values into registers */
   rows = v->rows-1;  /* number of quad strips = number of data rows - 1 */
   cols = v->cols;
   slices = v->slices;
   /* setup color and vertex pointers */
   cp0 = v->index;
   cp1 = cp0 + cols;
   vp0 = v->vertex;
   vp1 = vp0 + cols * 3;   /* 3 floats per vertex */

   /* loop over slices */
   for (s=0;s<slices;s++) {

      /* draw 'rows' quadrilateral strips */
      for (i=0;i<rows;i++) {
#if defined(SGI_GL) || defined(DENALI)
	 bgnqstrip();
	 for (j=0;j<cols;j++) {
	    cpack( ctable[*cp0++] );
	    v3f( vp0 );
	    vp0 += 3;
	    cpack( ctable[*cp1++] );
	    v3f( vp1 );
	    vp1 += 3;
	 }
	 endqstrip();
#endif
#ifdef OPENGL
	 glBegin( GL_QUAD_STRIP );
	 for (j=0;j<cols;j++) {
	    glColor4ubv( (GLubyte *) &ctable[*cp0++] );
	    glVertex3fv( vp0 );
	    vp0 += 3;
	    glColor4ubv( (GLubyte *) &ctable[*cp1++] );
	    glVertex3fv( vp1 );
	    vp1 += 3;
	 }
	 glEnd();
#endif
      }

      /* skip a row after each slice */
      cp0 += cols;
      vp0 += cols*3;
      cp1 += cols;
      vp1 += cols*3;
   }

#if defined(SGI_GL) || defined(DENALI)
   blendfunction( BF_ONE, BF_ZERO );  /* disable alpha blending */
#endif
#ifdef OPENGL
   glDisable( GL_BLEND );
#endif
   return 1;
}



/*
 * Draw a volumetric rendering of the grid for timestep it and variable ip.
 * Input: it - timestep
 *        ip - variable
 */
void draw_volume( Context ctx, int it, int ip, unsigned int *ctable )
{
   float *data;
   static int prev_it = -1, prev_ip = -1;
   int dir;
   float x, y, z, ax, ay, az;
   MATRIX ctm, proj;

   /* Get 3rd column values from transformation matrix */
#if defined (SGI_GL) || defined (DENALI)
   /* Compute orientation of 3-D box with respect to current matrices */
   /* with no assumptions about the location of the camera.  This was */
   /* done for the CAVE. */
   mmode( MPROJECTION );
   getmatrix( proj );
   mmode( MVIEWING );
   getmatrix( ctm );
#endif
#if defined(OPENGL)
   glGetFloatv( GL_PROJECTION_MATRIX, (GLfloat *) proj );
   glGetFloatv( GL_MODELVIEW_MATRIX, (GLfloat *) ctm );
#endif

   /* compute third column values in the product of ctm*proj */
   x = ctm[0][0]*proj[0][2] + ctm[0][1]*proj[1][2]
     + ctm[0][2]*proj[2][2] + ctm[0][3]*proj[3][2];
   y = ctm[1][0]*proj[0][2] + ctm[1][1]*proj[1][2]
     + ctm[1][2]*proj[2][2] + ctm[1][3]*proj[3][2];
   z = ctm[2][0]*proj[0][2] + ctm[2][1]*proj[1][2]
     + ctm[2][2]*proj[2][2] + ctm[2][3]*proj[3][2];

   /* examine values to determine how to draw slices */
   ax = ABS(x);
   ay = ABS(y);
   az = ABS(z);
   if (ax>=ay && ax>=az) {
      /* draw x-axis slices */
      dir = (x<0.0) ? WEST_TO_EAST : EAST_TO_WEST;
   }
   else if (ay>=ax && ay>=az) {
      /* draw y-axis slices */
      dir = (y<0.0) ? SOUTH_TO_NORTH : NORTH_TO_SOUTH;
   }
   else {
      /* draw z-axis slices */
      dir = (z<0.0) ? BOTTOM_TO_TOP : TOP_TO_BOTTOM;
   }

   /* If this is a new time step or variable then invalidate old volumes */
   if (it!=prev_it || ip!=prev_ip) {
      ctx->Volume->valid = 0;
      prev_it = it;
      prev_ip = ip;
   }

   /* Determine if we have to compute a set of slices for the direction. */
   if (ctx->Volume->dir!=dir || ctx->Volume->valid==0) {
      data = get_grid( ctx, it, ip );
      if (data) {
         compute_volume( ctx, data, it, ip, ctx->Nr, ctx->Nc, ctx->Nl[ip],
                         ctx->LowLev[ip], ctx->MinVal[ip], ctx->MaxVal[ip],
                         dir, ctx->Volume );
         release_grid( ctx, it, ip, data );
      }
   }

   render_volume( ctx, ctx->Volume, ctable );
}

