/* $Id: topo.c,v 1.11 1996/03/12 19:07:41 billh Exp $ */

/* VIS-5D version 4.0 */

/*
VIS-5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990, 1991, 1992, 1993, 1994  Bill Hibbard, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "binio.h"
#include "globals.h"
#include "graphics.h"
#include "image.h"
#include "memory.h"
#include "proj.h"
#include "topo.h"



/* maximum rows, columns of topography vertices for high and low res. topo */
#define LO_RES_VERTS 5000
#define HI_RES_VERTS 50000





/**********************************************************************/
/***                     Topography file stuff                      ***/
/**********************************************************************/

/*
 * This struct directly maps to the first part of the topo file.
 * NO LONGER USED, just kept around for reference.
 */
struct topo_header {
   char id[40];        /* id string "TOPO2" */
   float westlon;      /* West longitude in degrees */
   float eastlon;      /* East longitude in degrees */
   float northlat;     /* North latitude in degrees */
   float southlat;     /* South latitude in degrees */
   int rows;           /* number of rows */
   int cols;           /* number of columns */
   /* Next is the topo data in the form:  float data[rows][cols] */
};




/*
 * Read a topography file and initialize Topo and TopoData.
 * Input:  filename - name of topo file.
 * Return:  0 if error, otherwise non-zero for success.
 */
static int read_topo( Context ctx, char *filename )
{
   int f;
   int n;
   char id[40];

   f = open( filename, O_RDONLY );
   if (f<0) {
      printf("Topo file %s not found\n", filename );
      return 0;
   }

   /* Read topo file header */
   read_bytes( f, id, 40 );
   read_float4( f, &ctx->Topo_westlon );
   read_float4( f, &ctx->Topo_eastlon );
   read_float4( f, &ctx->Topo_northlat );
   read_float4( f, &ctx->Topo_southlat );
   read_int4( f, &ctx->Topo_rows );
   read_int4( f, &ctx->Topo_cols );

   if (strcmp(id,"TOPO")==0) {
      /* OLD STYLE: bounds given as ints, convert to floats */
      int *p;
      p = (int *) &ctx->Topo_westlon;  ctx->Topo_westlon = (float) *p / 100.0;
      p = (int *) &ctx->Topo_eastlon;  ctx->Topo_eastlon = (float) *p / 100.0;
      p = (int *) &ctx->Topo_northlat; ctx->Topo_northlat = (float) *p / 100.0;
      p = (int *) &ctx->Topo_southlat; ctx->Topo_southlat = (float) *p / 100.0;
   }
   else if (strcmp(id,"TOPO2")==0) {
      /* OK */
   }
   else {
      printf("%s is not a TOPO file\n", filename);
      close(f);
      return 0;
   }

   ctx->TopoData = (short *) allocate( ctx, ctx->Topo_rows * ctx->Topo_cols
				       * sizeof(short) );
   if (!ctx->TopoData) {
      close(f);
      return 0;
   }


   n = ctx->Topo_rows * ctx->Topo_cols;
   if (read_int2_array( f, ctx->TopoData, n) < n) {
      deallocate( ctx, ctx->TopoData, n*sizeof(short) );
      ctx->TopoData = NULL;
      close(f);
      return 0;
   }

   close(f);
   return 1;
}



/*
 * Free the memory used to store the topography data
 */
void free_topo( ctx )
Context ctx;
{
   if (ctx->TopoData) {
      deallocate( ctx, ctx->TopoData, ctx->Topo_rows*ctx->Topo_cols*sizeof(short) );
      ctx->TopoData = NULL;
   }
}




/*
 * Return the elevation of the topography at location (lat,lon) and a
 * flag indicating water or land.
 * Input:  lat, lon - location in degrees
 *         water - pointer to integer
 * Output:  water - set to 1 if water, 0 if land.
 * Returned:  elevation in meters at (lat,lon) or 0 if error.
 */
float elevation( Context ctx, float lat, float lon, int *water )
{
   float fr, fc;
   int rowa, cola, rowb, colb;
   float hgt;
   int count, r, c;
   int val, watcount;

   /* make sure longitude is in [-180,180] */
   if (lon>ctx->Topo_westlon) {
      lon -= 360.0;
   }
   if (lon<ctx->Topo_eastlon) {
      lon += 360.0;
   }

   while (lat<-90.0) {
      lat += 180.0;
   }
   while (lat>90.0) {
      lat -= 180.0;
   }

   if (!ctx->TopoData || lon<ctx->Topo_eastlon || lon>ctx->Topo_westlon
       || lat<ctx->Topo_southlat || lat>ctx->Topo_northlat) {
      if (water)
         *water = 0;
      return 0.0;
   }


   /* Return elevation at (lat,lon) by sampling LatSample*LonSample */
   /* values centered at that location. */

   /* calculate range of rows */
   fr = (ctx->Topo_rows - 1) * (lat - ctx->Topo_northlat)
           / (ctx->Topo_southlat - ctx->Topo_northlat);
   rowa = (int) fr - ctx->LatSample/2;
   rowb = rowa + ctx->LatSample;
   if (rowa<0)
      rowa = 0;
   if (rowb>=ctx->Topo_rows)
      rowb = ctx->Topo_rows - 1;

   /* calculate range of columns */
   fc = (ctx->Topo_cols - 1) * (lon - ctx->Topo_westlon)
           / (ctx->Topo_eastlon - ctx->Topo_westlon);
   cola = (int) fc - ctx->LonSample/2;
   colb = cola + ctx->LonSample;
   if (cola<0)
      cola = 0;
   if (colb>=ctx->Topo_cols)
      colb = ctx->Topo_cols - 1;

   /* find average height in sample area */
   hgt = 0.0;
   count = watcount = 0;
   for (r=rowa;r<=rowb;r++) {
      for (c=cola;c<=colb;c++) {
	 val = ctx->TopoData[r*ctx->Topo_cols+c];
	 if (val&1)
	    watcount++;
	 hgt += (float) (val / 2);
	 count++;
      }
   }
   hgt = hgt / (float) count;

   /* calculate water flag */
   if (water) {
      if (watcount>count/2)
	*water = 1;
      else
	*water = 0;
   }

   return hgt;
}





/**********************************************************************/
/***                   Topography display stuff                     ***/
/**********************************************************************/


#define TRANS(XMAX,XMIN,YMAX,YMIN,XVAL) (YMAX-(YMAX-YMIN)*(XMAX-XVAL)/(XMAX-XMIN))

#define CLAMP(VAL,MIN,MAX)   ( (VAL<MIN) ? MIN : ((VAL>MAX) ? MAX : VAL) )

#define ABS( A )   ( (A) < 0.0 ? -(A) : (A) )


static int topo_object = 0;



/*** init_topo_color_table ********************************************
   Initialize the topography color table.
   Input:  ct - the color table
           size - number of entries in the table
**********************************************************************/
void init_topo_color_table( Context ctx, unsigned int ct[], int size )
{
#define OCEAN
#ifdef OCEAN
   /* Change submitted by Mike McCann to give better under-water */
   /* topography colors. */
   static float red[7]   = { 5.0, 45.0,  20.0, 20.0,  70.0, 165.0, 200.0};
   static float green[7] = {10.0, 50.0, 170.0,170.0, 200.0,  42.0, 200.0};
   static float blue[7]  = {30.0,150.0,  42.0, 42.0,   0.0,  42.0, 200.0};
   static float range[7] = {-5.0, -0.020,-0.015, 0.0,   0.1,  1.0,   2.8};
#else
   static float red[4]   = { 20.0,  70.0, 165.0, 200.0};
   static float green[4] = {170.0, 200.0,  42.0, 200.0};
   static float blue[4]  = { 42.0,   0.0,  42.0, 200.0};
   static float range[4] = {  0.0,   0.1,  1.0,   2.8};
#endif
   int i, j;
   float x0, x1;
   float r, g, b, dr, dg, db;

   /* initialize to all white to start */
   for (i=0;i<size-1;i++) {
      ct[i] = 0xffffffff;
   }
   ct[size-1] = PACK_COLOR( 25, 25, 255, 255 );  /* r=25, g=25, b=255 */

#ifdef OCEAN
   for (i=0;i<6;i++) {
#else
   for (i=0;i<3;i++) {
#endif
      if (ctx->MinTopoHgt==ctx->MaxTopoHgt) {
	 r = g = b = 0;
	 dr = dg = db = 0;
	 x0 = x1 = 0;
      }
      else {
	 x0 = (range[i] - ctx->MinTopoHgt)
	        / (ctx->MaxTopoHgt - ctx->MinTopoHgt) * (float)(size-1);
	 x1 = (range[i+1] - ctx->MinTopoHgt)
	        / (ctx->MaxTopoHgt - ctx->MinTopoHgt) * (float)(size-1);
	 dr = (red[i+1]-red[i]) / (x1-x0);
	 dg = (green[i+1]-green[i]) / (x1-x0);
	 db = (blue[i+1]-blue[i]) / (x1-x0);
	 r = red[i];
	 g = green[i];
	 b = blue[i];
      }
      for (j=(int) x0; j<(int) x1; j++) {
	 if (j>=0 && j<size-1) {
	    ct[j] = PACK_COLOR( (int) r, (int) g, (int) b, 0xff );
	 }
	 r += dr;
	 g += dg;
	 b += db;
      }
   }

}



void recolor_topo( Context ctx, unsigned int ct[], int size )
{
   int i;

   if (ctx->TopoColor) {
      for (i=0;i<ctx->qrows*ctx->qcols;i++) {
	 ctx->TopoColor[i] = ct[ ctx->TopoIndex[i] ];
      }
   }

   if (topo_object) {
      delete_object( topo_object );
      topo_object = 0;
   }
}




/*
 * Generate the topography quadmesh.  This must be called after the
 * grid data set has been loaded.
 * Input:  toponame - name of topography file
 *         textureflag - 1 = use texture mapping, 0 = don't texture map
 *         hi_res - 1=high resolution topography, 0=normal resolution
 * Return:  1 = ok,  0 = error
 */
int init_topo( Context ctx, char *toponame, int textureflag, int hi_res )
{
   float dx, dy;
   float lat, lon;
   float topo_dlat, topo_dlon;
   float *topoheight;
   int i, j;
   int topoflag;
   int qr, qc;

   topoflag = read_topo( ctx, toponame );
   if (!topoflag && !textureflag) {
      return 0;
   }
   
   /* qrows, qcols - size of topography quadmesh in rows and columns */
   if (ctx->Topo_cols==ctx->Nc && ctx->Topo_rows==ctx->Nr) {
      /* use same topography resolution as grid resolution */
      qc = ctx->Topo_cols;
      qr = ctx->Topo_rows;
   }
   else {
      int maxverts = hi_res ? HI_RES_VERTS : LO_RES_VERTS;
      float r = sqrt( (float) maxverts
		      / ((ctx->Xmax - ctx->Xmin)*(ctx->Ymax - ctx->Ymin)) );
      qc = (int) (r * (ctx->Xmax - ctx->Xmin) + 0.5);
      qr = (int) (r * (ctx->Ymax - ctx->Ymin) + 0.5);
   }

   /* allocate space for topography vertex and color arrays */
   ctx->TopoVertex     = (float *) malloc( qr*qc*3*sizeof(float) );
   ctx->TopoNormal     = (float *) malloc( qr*qc*3*sizeof(float) );
   ctx->TopoIndex      = (int *)   malloc( qr*qc*1*sizeof(int)   );
   ctx->TopoColor      = (uint *)  malloc( qr*qc*1*sizeof(uint)  );
   ctx->TopoTexcoord   = (float *) malloc( qr*qc*2*sizeof(float) );
   ctx->TopoFlatVertex = (float *) malloc( qr*qc*3*sizeof(float) );
   topoheight = (float *) allocate( ctx, qr*qc*sizeof(float) );

   /*
    * Compute topography vertices.
    */
   if (ctx->CurvedBox==0) {
      /* Rectangular box:  generate vertices in graphics coords */
      int k;
      float x, y, z;
      float texture_s, texture_t, delta_s, delta_t;

      dx = (ctx->Xmax-ctx->Xmin) / (float) (qc-1);
      dy = (ctx->Ymax-ctx->Ymin) / (float) (qr-1);

      delta_s = 1.0 / (float) (qc-1);
      delta_t = 1.0 / (float) (qr-1);

      /* calculate sampling size */
      if (ctx->Topo_cols==ctx->Nc && ctx->Topo_rows==ctx->Nr) {
	 ctx->LatSample = ctx->LonSample = 1;
      }
      else {
	 topo_dlat = (ctx->Topo_northlat-ctx->Topo_southlat) / ctx->Topo_rows;
	 ctx->LatSample = CLAMP( (int) (2.0*dy/topo_dlat), 2, 20 );
	 topo_dlon = (ctx->Topo_westlon-ctx->Topo_eastlon) / ctx->Topo_cols;
	 ctx->LonSample = CLAMP( (int) (2.0*dx/topo_dlon), 2, 20 );
      }

      k = 0;
      y = ctx->Ymax;
      texture_t = 0.0;
      for (i=0; i<qr; i++) {
	 x = ctx->Xmin;
         texture_s = 0.0;
	 for (j=0; j<qc; j++) {
	    int water;
	    float hgt;

	    xyz_to_geo( ctx, -1, -1, x, y, 0.0, &lat, &lon, &hgt );
	    hgt = elevation( ctx, lat, lon, &water ) / 1000.0;  /* hgt in km */
	    z = height_to_z( ctx, hgt );
	    z = ABS(ctx->Zmin - z) < 0.01 ? ctx->Zmin+0.01 : z;
	    ctx->TopoVertex[k*3+0] = x;
	    ctx->TopoVertex[k*3+1] = y;
	    ctx->TopoVertex[k*3+2] = z;

	    topoheight[k] = hgt;  /* save topo height at this vertex */
	    /* if water flag is set, index will be 255 */
	    ctx->TopoIndex[k] = (water) ? 255 : 0;

            ctx->TopoFlatVertex[k*3+0] = x;
            ctx->TopoFlatVertex[k*3+1] = y;
            ctx->TopoFlatVertex[k*3+2] = ctx->Zmax;

            ctx->TopoTexcoord[k*2+0] = texture_s;
            ctx->TopoTexcoord[k*2+1] = texture_t;

            k++;
	    x += dx;
            texture_s += delta_s;
	 }
	 y -= dy;
         texture_t += delta_t;
      }

   }
   else {
      /* Curved box:  generate vertices in geographic coordinates */

      float lat, lon;
      float dlat, dlon;
      int k;
      float texture_s, texture_t, delta_s, delta_t;

      dlat = (ctx->NorthBound - ctx->SouthBound) / (float) (qr-1);
      dlon = (ctx->WestBound - ctx->EastBound) / (float) (qc-1);

      delta_s = 1.0 / (float) (qc-1);
      delta_t = 1.0 / (float) (qr-1);

      k = 0;
      lat = ctx->NorthBound;
      texture_t = 0.0;
      for (i=0; i<qr; i++) {
	 lon = ctx->WestBound;
         texture_s = 0.0;
	 for (j=0; j<qc; j++) {
	    int water;
	    float hgt, x, y, z;

	    hgt = elevation( ctx, lat, lon, &water ) / 1000.0;  /* hgt in km */
	    geo_to_xyz( ctx, -1, -1, 1, &lat, &lon, &hgt, &x, &y, &z );
	    ctx->TopoVertex[k*3+0] = x;
	    ctx->TopoVertex[k*3+1] = y;
	    ctx->TopoVertex[k*3+2] = z;

	    topoheight[k] = hgt;
	    /* if water flag is set, index will be 255 */
	    ctx->TopoIndex[k] = (water) ? 255 : 0;

            hgt = ctx->BottomBound;
            geo_to_xyz( ctx, -1, -1, 1, &lat, &lon, &hgt, &x, &y, &z );
            ctx->TopoFlatVertex[k*3+0] = x;
            ctx->TopoFlatVertex[k*3+1] = y;
            ctx->TopoFlatVertex[k*3+2] = z;

            ctx->TopoTexcoord[k*2+0] = texture_s;
            ctx->TopoTexcoord[k*2+1] = texture_t;

            k++;
	    lon -= dlon;
            texture_s += delta_s;
	 }
	 lat -= dlat;
         texture_t += delta_t;
      }

   }

   /* Find MinTopoHgt and MaxTopoHgt */
   ctx->MinTopoHgt = 10000.0;
   ctx->MaxTopoHgt = -10000.0;
   for (i=0;i<qr*qc;i++) {
      if (topoheight[i]<ctx->MinTopoHgt) {
	 ctx->MinTopoHgt = topoheight[i];
      }
      if (topoheight[i]>ctx->MaxTopoHgt) {
	 ctx->MaxTopoHgt = topoheight[i];
      }
   }

   /* Compute topography color table indexes. */
   for (i=0;i<qr*qc;i++) {
      float hgt = topoheight[i];
      if (ctx->TopoIndex[i]!=255) {   /* if not water */
	 if (ctx->MinTopoHgt==ctx->MaxTopoHgt) {
	    ctx->TopoIndex[i] = 0;
	 }
	 else {
	    int index;
	    index = (int) ( (hgt-ctx->MinTopoHgt)
			   / (ctx->MaxTopoHgt-ctx->MinTopoHgt) * 254.0 );
	    ctx->TopoIndex[i] = CLAMP( index, 0, 254 );
	 }
      }
   }

   /* done with topoheight array */
   deallocate( ctx, topoheight, qr*qc*sizeof(float) );

   /* compute quadmesh normal vectors */
   {
      float *qnorm;

      qnorm = (float *) allocate( ctx, qc * qr * 3 * sizeof(float) );

      /* step 1: compute surface normal for each quadrilateral. */
      for (i=0;i<qr-1;i++) {
	 for (j=0;j<qc-1;j++) {
	    float a[3], b[3];
	    int index;

	    index = (i*qc+j)*3;
	    /* a is the down vector, b is the right vector */
	    a[0] = ctx->TopoVertex[index+qc*3+0] - ctx->TopoVertex[index+0];
	    a[1] = ctx->TopoVertex[index+qc*3+1] - ctx->TopoVertex[index+1];
	    a[2] = ctx->TopoVertex[index+qc*3+2] - ctx->TopoVertex[index+2];
	    b[0] = ctx->TopoVertex[index+3+0] - ctx->TopoVertex[index+0];
	    b[1] = ctx->TopoVertex[index+3+1] - ctx->TopoVertex[index+1];
	    b[2] = ctx->TopoVertex[index+3+2] - ctx->TopoVertex[index+2];
	    /* a cross b is the quad's facet normal */
	    qnorm[index+0] =  a[1]*b[2]-a[2]*b[1];
	    qnorm[index+1] = -a[0]*b[2]+a[2]*b[0];
	    qnorm[index+2] =  a[0]*b[1]-a[1]*b[0];
	 }
      }

      /* step 2: compute vertex normals by averaging adjacent */
      /* quadrilateral normals. */
      for (i=0;i<qr;i++) {
	 for (j=0;j<qc;j++) {
	    float n[3], mag;
	    int index;

	    index = (i*qc+j)*3;

	    n[0] = n[1] = n[2] = 0.0;
	    /* upper-left quad */
	    if (i>0 && j>0) {
	       n[0] += qnorm[ index-qc*3-3+0 ];
	       n[1] += qnorm[ index-qc*3-3+1 ];
	       n[2] += qnorm[ index-qc*3-3+2 ];
	    }
	    /* upper-right quad */
	    if (i>0 && j<qc-1) {
	       n[0] += qnorm[ index-qc*3+0 ];
	       n[1] += qnorm[ index-qc*3+1 ];
	       n[2] += qnorm[ index-qc*3+2 ];
	    }
	    /* lower-left quad */
	    if (i<qr-1 && j>0) {
	       n[0] += qnorm[ index-3+0 ];
	       n[1] += qnorm[ index-3+1 ];
	       n[2] += qnorm[ index-3+2 ];
	    }
	    /* lower-right quad */
	    if (i<qr-1 && j<qc-1) {
	       n[0] += qnorm[ index+0 ];
	       n[1] += qnorm[ index+1 ];
	       n[2] += qnorm[ index+2 ];
	    }

	    mag = sqrt( n[0]*n[0] + n[1]*n[1] + n[2]*n[2] );
	    if (mag>0.0) {
	       mag = 1.0 / mag;
	       ctx->TopoNormal[index+0] = n[0] * mag;
	       ctx->TopoNormal[index+1] = n[1] * mag;
	       ctx->TopoNormal[index+2] = n[2] * mag;
	    }
	 }
      }

      deallocate( ctx, qnorm, qc * qr * 3 * sizeof(float) );
   }

   ctx->qcols = qc;
   ctx->qrows = qr;

   /* Define the initial quadmesh vertex colors */
   init_topo_color_table( ctx, ctx->TopoColorTable, 256 );
   recolor_topo( ctx, ctx->TopoColorTable, 256 );

   return 1;
}




/*
 * Draw the topography.
 * Input:  time - the timestep number
 *         texture_flag - 0=no texture, 1=texture map
 *         flat_flag - 0=draw w/ topo heights, 1=draw flat
 */
void draw_topo( Context ctx, int time, int texture_flag, int flat_flag )
{
   set_color( 0xffffffff );

   if (flat_flag) {
      if (texture_flag) {
         /* flat texture map */
         use_texture( ctx, time );
         texture_quadmeshnorm( ctx->qrows, ctx->qcols,
                               (void*) ctx->TopoFlatVertex,
                               NULL,  /* use default normals */
                               (void*) ctx->TopoTexcoord );
      }
      else {
         /* draw nothing */
      }
   }
   else {
      if (texture_flag) {
         /* textured topo */
         use_texture( ctx, time );
         texture_quadmeshnorm( ctx->qrows, ctx->qcols,
                               (void*) ctx->TopoVertex,
                               (void*) ctx->TopoNormal,
                               (void*) ctx->TopoTexcoord );
      }
      else {
         /* untextured topo */
         if (topo_object==0) {
            topo_object = begin_object();
            quadmeshnorm( (void *) ctx->TopoVertex, (void *) ctx->TopoNormal,
                          ctx->TopoColor, ctx->qrows, ctx->qcols );
            end_object();
            call_object( topo_object );
         }
         else {
            call_object( topo_object );
         }
      }
   }
}

