/* graphics.denali.c */

/* Graphics functions using GL on the Kubota Denali system */

/* VIS-5D version 4.0 */

/*
VIS-5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990, 1991, 1992, 1993, 1994  Bill Hibbard, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <math.h>
#include <stdio.h>
#include <string.h>
#include "graphics.h"
#include "matrix.h"
#include "x.h"


/* API - to globals.c (or private indexed structure);
   except constants */

/*** Private Variables: ***/
static Window wmain;
static float Scale;
static Matrix ModelMat, InvModelMat, ProjMat;
static Matrix IdMat = { 1.0, 0.0, 0.0, 0.0,
			0.0, 1.0, 0.0, 0.0,
                        0.0, 0.0, 1.0, 0.0,
			0.0, 0.0, 0.0, 1.0 };
static int pretty_flag = 0;
static int Perspective = 0;



#include <fmclient.h>
/** set_pointer defined in NPGL **/
#define set_pointer st_pointer
#define DEFAULT_FONT "Times-Roman"
#define FLIP( A )  ( (A&0xff00ff00) | ((A>>16) & 0xff) | ((A&0xff) <<16) )
#define EYEDIST 3.0
#define FOV 400
#define VIEWOBJ 10
static Window GLwindow;
static fmfonthandle visfont;
static long zfar;
static unsigned int CurColor = 0xffffffff;
static GC gc1;
static unsigned int CurTrans = 0xff000000;
static float FrontClip;
static float xminplane[4], xmaxplane[4], yminplane[4], ymaxplane[4];

/* setup the default material surface description: */
static float mat1[] = {
      AMBIENT, 0.3, 0.3, 0.3,   /* surface ambient color */
      DIFFUSE, 0.7, 0.7, 0.7,   /* surface diffuse color */
      LMNULL
};

/* setup the light source descriptions: */
static float light1[] = {
      LCOLOR, 0.8, 0.8, 0.8,          /* light color */
      POSITION, 0.0, 0.0, 10.0, 0.0,  /* light location */
      LMNULL
};

static float light2[] = {
      LCOLOR, 0.8, 0.8, 0.8,          /* light color */
      POSITION, 0.0, 0.0, -10.0, 0.0,  /* light location */
      LMNULL
};

/* setup the lighting model description: */
static float lmodel[] = {
      AMBIENT, 0.2, 0.2, 0.2,         /* ambient color */
      LOCALVIEWER, 0.0,               /* infinite viewpoint */
      TWOSIDE, 0,
      LMNULL
};

/* Accumulation buffer antialiasing */
#define AA_INC  1.0
static float  xoffsets[AA_PASSES] =
     { -AA_INC, 0.0, AA_INC,  -AA_INC, 0.0, AA_INC,  -AA_INC, 0.0, AA_INC };
static float  yoffsets[AA_PASSES] =
     { -AA_INC, -AA_INC, -AA_INC,  0.0, 0.0, 0.0,  AA_INC, AA_INC, AA_INC };

unsigned short screen1[16] = { /* 75% */
      0x7777, 0xdddd, 0x7777, 0xdddd, 0x7777, 0xdddd, 0x7777, 0xdddd,
      0x7777, 0xdddd, 0x7777, 0xdddd, 0x7777, 0xdddd, 0x7777, 0xdddd
     };
unsigned short screen2[16] = { /* 50% */
      0xaaaa, 0x5555, 0xaaaa, 0x5555, 0xaaaa, 0x5555, 0xaaaa, 0x5555,
      0xaaaa, 0x5555, 0xaaaa, 0x5555, 0xaaaa, 0x5555, 0xaaaa, 0x5555
     };
unsigned short screen3[16] = { /* 25% */
      0x8888, 0x2222, 0x8888, 0x2222, 0x8888, 0x2222, 0x8888, 0x2222,
      0x8888, 0x2222, 0x8888, 0x2222, 0x8888, 0x2222, 0x8888, 0x2222
     };
static int screendoor;




void init_graphics2( alphaflag )
int alphaflag;
{
   HQR_available = 0;
   screendoor = 1;
   Perspec_available = 1;
}



void terminate_graphics()
{
}




Window make_window( title, xpos, ypos, width, height,
                    fontname, fontsize, NPGLwin )
char *title;
int xpos, ypos;
int width, height;
char *fontname;
int fontsize;
Window *NPGLwin;
{
   static long Black = 0xff000000;
   char title2[100];
   XSizeHints hints;

   strcpy( title2, title );  /* uniquify the window title */
   strcat( title2, "   " );

   foreground();

   if (width==ScrWidth && height==ScrHeight) {
      noborder();
      prefposition( 0, ScrWidth-1, 0, ScrHeight-1 );
   }
   else {
      prefposition( xpos, xpos+width-1,
		    ScrHeight-ypos-1, ScrHeight-ypos-height );
   }
   winopen( title2 );
   winconstraints();

   /* find X window pointer to the GL window */
   *NPGLwin = find_window( GfxDpy, GfxScr, RootWindow(GfxDpy,GfxScr), title2 );
   if (!*NPGLwin) {
      fprintf(stderr,"Unable to create graphics window\n");
      exit(1);
   }
   wintitle( title );   /* restore window title */
/*  Unable to access ButtonPress events from GL window directly.  Create */
/*  an InputOnly X-window ontop of GL window to intercept ButtonPress */
/*  events. */
   wmain = XCreateWindow ( GfxDpy, *NPGLwin,
				0, 0,      /* X and Y location */
				DisplayWidth(GfxDpy,GfxScr),
				DisplayHeight(GfxDpy,GfxScr), 
				0,         /* Border width */
				CopyFromParent,
				InputOnly,
				CopyFromParent,
				NULL, NULL);
   XMapWindow( GfxDpy, wmain );
   /* select which X events to receive */
   XSelectInput( GfxDpy, wmain, PointerMotionMask | KeyPressMask 
		| ButtonPressMask | ButtonReleaseMask
		| StructureNotifyMask | VisibilityChangeMask );
/*  InputOnly windows don't receive Exposure events.  Therefore enable */
/*  Exposure events on GL window. */
   XSelectInput( GfxDpy, *NPGLwin, ExposureMask );
   GLwindow = *NPGLwin;
   /* setup 24-bit, double buffer, z-buffered window */
   RGBmode();
   doublebuffer();
   gconfig();
   zbuffer( TRUE );
   zfunction( ZF_LESS );  /* improves transparency quality */
   zfar = getgdesc( GD_ZMAX );

   czclear( Black, zfar ); 
   swapbuffers();
   czclear( Black, zfar ); 

   mmode( MVIEWING );     /* multi-matrix mode */

   getsize( &WinWidth, &WinHeight );
   AspectRatio = (float) WinWidth / (float) WinHeight;
   ortho( -1.0, 1.0, -1.0, 1.0, 1.0, -1.0 );  /* this MUST be after mmode()! */
   viewport( 0, (Screencoord) WinWidth-1, 0, (Screencoord) WinHeight-1);

   lmdef( DEFMATERIAL, 10, 0, mat1 );   /* define material, light, model */
   lmdef( DEFLIGHT, 20, 0, light1 );
   lmdef( DEFLIGHT, 21, 0, light2 );
   lmdef( DEFLMODEL, 30, 0, lmodel );
   lmbind( MATERIAL, 10 );
   lmbind( LIGHT1, 20 );
   lmbind( LIGHT2, 21 );
   lmbind( LMODEL, 30 );

   /* Set font */
   fminit();
   if (fontname[0]==0)
      strcpy( fontname, DEFAULT_FONT );

   visfont = fmfindfont( fontname );
   if (!visfont)
      die("Unable to find font");
   if (fontsize)
      FontHeight = fontsize;
   else
      FontHeight = 20;

   visfont = fmscalefont( visfont, (double) FontHeight );
   fmsetfont( visfont );
   FontDescent = getdescender();

   /* init front clipping plane */
   FrontClip = 1.0;

   /* box clipping plane */
   clipplane( 1, CP_DEFINE, xminplane );
   clipplane( 1, CP_OFF, xminplane );
   clipplane( 2, CP_DEFINE, xmaxplane );
   clipplane( 2, CP_OFF, xmaxplane );
   clipplane( 3, CP_DEFINE, yminplane );
   clipplane( 3, CP_OFF, yminplane );
   clipplane( 4, CP_DEFINE, ymaxplane );
   clipplane( 4, CP_OFF, ymaxplane );

   /* screen door transparency patterns */
   defpattern( 1, 16, screen1 );
   defpattern( 2, 16, screen2 );
   defpattern( 3, 16, screen3 );
   setpattern(0);

   subpixel(TRUE);   /* for faster rendering on VGX */
   return wmain;
}



/*** set_pointer ******************************************************
   Set mouse pointer image to an arrow or 'busy' clock.
   Input: p - 0 = pointer image
              1 = clock image
**********************************************************************/
void set_pointer( p )
int p;
{
   static unsigned short  hourglass[16] = { 0x1ff0, 0x1ff0, 0x0820, 0x0820,
					    0x0820, 0x0c60, 0x06c0, 0x0100,
					    0x0100, 0x06c0, 0x0c60, 0x0820,
					    0x0820, 0x0820, 0x1ff0, 0x1ff0 };
   if (p) {
      /* make busy cursor */
      curstype( C16X1 );
      defcursor( 1, hourglass );
      setcursor( 1, 0, 0 );
   }
   else {
      /* regulay cursor */
      setcursor( 0, 0, 0 );
   }
}



/*** save_formats *****************************************************
   Return a set of flag bits indicating what image formats we can
   save as.
**********************************************************************/
int save_formats()
{
   return SAVE_XWD;
}



/*** save_window ******************************************************
   Save the current image in the 3-D window to the named file.
   Input:  filename - image filename
           format - SAVE_SGI = SGI .rgb
	            SAVE_GIF = .GIF
		    SAVE_XWD = .xwd   (X window dump)
   Return: 1 = no errors
           0 = error occured.
**********************************************************************/
int save_window( filename, format )
char filename[];
int format;
{
   FILE *f;

   set_pointer(1);

   /* Make an X window dump file (.xwd) */
   f = fopen(filename,"w");
   if (f) {
      /* Take a window dump using GLwindow ID instead of wmain */
      Window_Dump( GfxDpy, GfxScr, GLwindow, f );
      fclose(f);
      set_pointer(0);
      return 1;
   }
   else {
      printf("Error unable to open %s for writing\n", filename);
      set_pointer(0);
      return 0;
   }
}



/*** print_window *****************************************************
   Print the current image in the 3-D window to the default PostScript
   printer.
   Return: 1 = ok, 0 = error
**********************************************************************/
int print_window( )
{
   FILE *f;
   static char bin_file[] = "/usr/tmp/VIS-5D_image.bin";
   static char rgb_file[] = "/usr/tmp/VIS-5D_image.rgb";
   static char ps_file[] = "/usr/tmp/VIS-5D_image.ps";

   readsource( SRC_FRONT );
   set_pointer(1);

   /* write binary data to temporary file */
   f = fopen( bin_file, "w");
   if (f) {
      int i, j, k;
      char cmd[1000];
      printf("Writing %s...\n", bin_file);

      for (i=0;i<WinHeight;i++) {
	 unsigned int data[2000];
	 char rgbdata[2000*3];

	 /* get a row of pixels from frame buffer */
	 lrectread( (Screencoord) 0, (Screencoord) i,
		    (Screencoord) WinWidth-1, (Screencoord) i, data );

	 /* bit-twiddle data from data[] to rgbdata[] */
	 k = 0;
	 for (j=0;j<WinWidth;j++) {
	    rgbdata[k++] = (char) UNPACK_RED(data[j]);
	    rgbdata[k++] = (char) UNPACK_GREEN(data[j]);
	    rgbdata[k++] = (char) UNPACK_BLUE(data[j]);
	 }

	 /* write the row of RGB data */
	 fwrite( rgbdata, 1, k, f );
      }

      fclose(f);

      /* convert data from binary to .rgb format */
      if (!installed("frombin")) return 0;
      sprintf(cmd,"frombin %s %s %d %d 3", bin_file, rgb_file,
	      WinWidth,WinHeight);
      printf("Executing: %s\n", cmd );
      system(cmd);
      unlink( bin_file );

      /* We now have a SGI .rgb file named rgb_file */

      /* Convert .rgb file to PS */
      if (!installed("tops")) return 0;
      sprintf(cmd,"tops %s -B > %s", rgb_file, ps_file );
      printf("Executing: %s\n", cmd );
      system(cmd);
      /* delete .rgb file */
      unlink( rgb_file );

      /* Send ps_file to default printer */
      if (!installed("lpr")) return 0;
      sprintf(cmd,"lpr %s\n", ps_file );
      printf("Executing: %s\n", cmd );
      system(cmd);
      /* delete .ps file */
      unlink( ps_file );

      printf("Done.\n");
      set_pointer(0);
      return 1;
   }
   else {
      printf("Error unable to open %s for writing\n", bin_file);
      set_pointer(0);
      return 0;
   }
}





/*** clear_window *****************************************************
   Clear the graphics window.  This is called prior to rendering
   a frame.
**********************************************************************/
void clear_window( bgcolor )
unsigned int bgcolor;
{
   czclear( (unsigned long) bgcolor, zfar );
}



/*
 * Called when window size changes.
 */
void resize_window( width, height )
int width, height;
{
   WinWidth = width;
   WinHeight = height;
   AspectRatio = (float) WinWidth / (float) WinHeight;
   viewport( 0, (Screencoord) WinWidth, 0, (Screencoord) WinHeight);
}




/*** update_window ****************************************************
   Update the 3-D window.  This called when finished rendering a frame.
**********************************************************************/
void update_window()
{
   swapbuffers();
}



/*** set_2d ***********************************************************
   This is to be called prior to any 2-D rendering calls.
**********************************************************************/
void set_2d()
{
   loadmatrix( IdMat );
   ortho2( -0.5, WinWidth-0.5, -0.5, WinHeight-0.5 );
   zfunction( ZF_ALWAYS );
}



/*** set_3d ***********************************************************
   This is to be called prior to any 3-D rendering calls.
   Input:  ctm - the current transformation matrix.
           scale - scaling factor of the ctm.
	   perspec - 0 = parallel, 1 = perspective projection
**********************************************************************/
void set_3d( ctm, scale, perspec )
Matrix ctm;
float scale;
int perspec;
{
   static Matrix flip = { 1.0, 0.0, 0.0, 0.0,
			  0.0, 1.0, 0.0, 0.0,
			  0.0, 0.0,-1.0, 0.0,
			  0.0, 0.0, 0.0, 1.0 };
   float front, rear;

   gflush();   /* to prevent RE2 pipe hangup */

   mat_copy( ModelMat, ctm );
   mat_inv( InvModelMat, ctm );
   InvModelMat[3][0] = InvModelMat[3][1] = InvModelMat[3][2] = 0.0;

   if (perspec) {
      /* Perspective projection */
      front = EYEDIST - scale * 1.75 * FrontClip;
      if (front<0.01)  front = 0.01;
      rear = EYEDIST + scale * 1.75;
      perspective( FOV, AspectRatio, front, rear );
      mat_mul( ModelMat, ModelMat, flip );
      ModelMat[3][2] = -EYEDIST;
      loadmatrix( ModelMat );

      /* make an object, VIEWOBJ, so we can use mapw later */
      makeobj(VIEWOBJ);
      perspective( FOV, AspectRatio, front, rear );
      loadmatrix( ModelMat );
      closeobj();

      /* get projection matrix */
      mmode( MPROJECTION );
      getmatrix( ProjMat );
      mmode( MVIEWING );
   }
   else {
      /* parallel projection */
      /* Note reversed Z! */
      front = scale * 1.75 * FrontClip;
      rear = -scale * 1.75;
      ortho( -AspectRatio, AspectRatio, -1.0, 1.0, front, rear );
      loadmatrix( ModelMat );
   }
   Perspective = perspec;
   Scale = scale;
   zfunction( ZF_LESS );
}



/*** project **********************************************************
   Use current transformation and viewing information to project a
   point p from 3-D graphics coordinates to 2-D window coordinates
   in [0,WinWidth-1]x[0,WinHeight-1].
   Input:  p - 3-D point in graphics space
   Output:  x, y - 2-D point in window pixel coordinates.
**********************************************************************/
void project( p, x, y )
float p[3];
float *x, *y;
{
   float q[4], nx, ny;

   q[0] = p[0];
   q[1] = p[1];
   q[2] = p[2];
   q[3] = 1.0;
   if (Perspective) {
      /* Perspective Projection */
      mat_vecmul4( q, ModelMat );
      mat_vecmul4( q, ProjMat );
      nx = q[0] / q[3];   /* divide by w */
      ny = q[1] / q[3];
      *x = (nx+1.0) / 2.0 * WinWidth;
      *y = (1.0-ny) / 2.0 * WinHeight;
   }
   else {
      /* Parallel Projection */
      mat_vecmul4( q, ModelMat );
      nx = q[0];
      ny = q[1];
      *x = ((nx/AspectRatio+1.0) / 2.0 * WinWidth);
      *y = (1.0-ny) / 2.0 * WinHeight;
   }

}



/*** unproject ********************************************************
   Given a 2-D window coordinate in [0,WinWidth-1]x[0,WinHeight-1],
   return the parametric equation of a line in 3-D such that the
   projection of the line from 3-D to 2-D is a point at the window
   coordinate.
   Input:  x, y - window coordinate.
   Output:  p, d - parametric equation of line:  l = p + t*d
                   NOTE, d will have unit length
**********************************************************************/
void unproject( x, y, p, d )
float x, y;
float p[3], d[3];
{
   float mag;

   if (Perspective) {
      /* Perspective Projection */
      Screencoord sx, sy;
      float a0,a1,a2;
      sx = (Screencoord) x;
      sy = (Screencoord) (WinHeight-y-1);
      mapw( VIEWOBJ, sx, sy, &p[0], &p[1], &p[2], &a0, &a1, &a2 );
      d[0] = a0 - p[0];
      d[1] = a1 - p[1];
      d[2] = a2 - p[2];
   }
   else {
      /* Parallel Projection */
      d[0] = 0.0;
      d[1] = 0.0;
      d[2] = 1.0;
      mat_vecmul3( d, InvModelMat );
      p[0] = ((x/WinWidth) * 2.0 - 1.0) * AspectRatio;
      p[1] = 1.0 - (y/WinHeight) * 2.0;
      p[2] = 0.0;
      p[0] -= ModelMat[3][0];
      p[1] -= ModelMat[3][1];
      p[2] -= ModelMat[3][2];
      mat_vecmul3( p, InvModelMat );
   }
   mag = sqrt( d[0]*d[0] + d[1]*d[1] + d[2]*d[2] );
   d[0] /= mag;
   d[1] /= mag;
   d[2] /= mag;
}




/*** set_color ********************************************************
   Set the current drawing color.  The color is specified in packed
   ARGB format; 1-byte each of Alpha, Red, Green, Blue where Alpha
   is the most-significant byte and blue is the least-signficant byte.
**********************************************************************/
void set_color( c )
unsigned int c;
{
   CurColor = FLIP( c );
   CurTrans = c & 0xff000000;
   if (screendoor) {
      CurTrans = 3 - UNPACK_ALPHA(c) / 64;
   }
   cpack( (unsigned long) CurColor );
}



/*** set_depthcue *****************************************************
   Set the line depth cueing flag on or off.
**********************************************************************/
void set_depthcue( onoff )
int onoff;
{
   short r, g, b;
   if (onoff) {
      r = UNPACK_RED( CurColor );
      g = UNPACK_GREEN( CurColor );
      b = UNPACK_BLUE( CurColor );
      depthcue( TRUE );
      lRGBrange( r/5, g/5, b/5,  r, g, b, 0, zfar );
   }
   else {
      depthcue( FALSE );
   }
}



/*** set_line_width ***************************************************
   Set the width of lines.
   Input:  w - width in pixels.
**********************************************************************/
void set_line_width( w )
float w;
{
   linewidthf( w );
}



/*** set_frontclip ****************************************************
   Set the position of the front clipping plane.
   Input:  d - front clipping plane z position in [1,-1] where 1 is
               near viewer and -1 is away from viewer.
**********************************************************************/
void set_frontclip( d )
float d;
{
   if (d>1.0)
      FrontClip = 1.0;
   else if (d<-1.0)
      FrontClip = -1.0;
   else
      FrontClip = d;
   /* set_3d() will put new front clipping plane into effect */
}




/*** set_box_clipping *************************************************
   Enable or disable clipping of graphics to a box.
   Input:  onoff - 1 = enable, 0 = disable.
           xmin, xmax, ymin, ymax - clipping bounds in graphics space,
                                    note no Z clipping is done.
**********************************************************************/
void set_box_clipping( onoff, xmin, xmax, ymin, ymax )
int onoff;
float xmin, xmax, ymin, ymax;
{
   if (onoff) {
      xminplane[0]=  1.0;  xminplane[1]=xminplane[2]=0.0;  xminplane[3]= -xmin;
      xmaxplane[0]= -1.0;  xmaxplane[1]=xmaxplane[2]=0.0;  xmaxplane[3]=  xmax;
      yminplane[1]=  1.0;  yminplane[0]=yminplane[2]=0.0;  yminplane[3]= -ymin;
      ymaxplane[1]= -1.0;  ymaxplane[0]=ymaxplane[2]=0.0;  ymaxplane[3]=  ymax;

      clipplane( 1, CP_DEFINE, xminplane );  clipplane( 1, CP_ON, xminplane );
      clipplane( 2, CP_DEFINE, xmaxplane );  clipplane( 2, CP_ON, xmaxplane );
      clipplane( 3, CP_DEFINE, yminplane );  clipplane( 3, CP_ON, yminplane );
      clipplane( 4, CP_DEFINE, ymaxplane );  clipplane( 4, CP_ON, ymaxplane );
   }
   else {
      clipplane( 1, CP_OFF, xminplane );
      clipplane( 2, CP_OFF, xmaxplane );
      clipplane( 3, CP_OFF, yminplane );
      clipplane( 4, CP_OFF, ymaxplane );
   }
}




/*** set_pretty *******************************************************
   Set pretty rendering mode on or off.
   XFDI:  This function is used to select antialiasing and multi-pass
             transparency.
   SGI:  This function selects line antialiasing.
   Input: onoff - 1 = on, 0 = off.
**********************************************************************/
void set_pretty( onoff )
{
   if (onoff) {
      /* turn on */
      /*subpixel(TRUE);*/
      glcompat( GLC_OLDPOLYGON, 0 );
      acsize(16);
      gconfig();
      set_pointer(1);
      pretty_flag = 1;
   }
   else {
      /* turn off */
      /*subpixel(FALSE);*/
      glcompat( GLC_OLDPOLYGON, 1 );
      acsize(0);
      gconfig();
      set_pointer(0);
      pretty_flag = 0;
   }
}


/*** These functions are taken from the SGI GL Programming Guide, ch. 15 ***/
static void subpixwindow( left, right, bottom, top, near, far, pixdx, pixdy )
float left, right, bottom, top, near, far, pixdx, pixdy;
{
   short vleft, vright, vbottom, vtop;
   float xwsize, ywsize, dx, dy;
   int xpixels, ypixels;

   getviewport( &vleft, &vright, &vbottom, &vtop );
   xpixels = vright - vleft + 1;
   ypixels = vtop - vbottom + 1;
   xwsize = right - left;
   ywsize = top - bottom;
   dx = -pixdx * xwsize / xpixels;
   dy = -pixdy * ywsize / ypixels;
   window( left+dx,right+dx, bottom+dy, top+dy, near, far );
}


static void subpixperspective( fovy, aspect, near, far, pixdx, pixdy )
Angle fovy;
float aspect, near, far, pixdx, pixdy;
{
   float fov2, left, right, bottom, top;
   fov2 = ((fovy*M_PI)/1800)/2.0;
   top = near / (cosf(fov2) / sinf(fov2));
   bottom = -top;
   right = top * aspect;
   left = -right;
   subpixwindow( left, right, bottom, top, near, far, pixdx, pixdy );
}



void start_aa_pass( n )
int n;
{
   float w, h, front, rear;

   if (pretty_flag) {
      if (n==0) {
	 /* clear ACC buffer */
	 acbuf( AC_CLEAR, 0.0 );
      }
      if (Perspective) {
	 float front, rear;

	 front = EYEDIST - Scale * 1.75 * FrontClip;
	 if (front<0.01)  front = 0.01;
	 rear = EYEDIST + Scale * 1.75;
	 subpixperspective( FOV, AspectRatio, front, rear,
			   xoffsets[n]/2.0, yoffsets[n]/2.0 );
	 loadmatrix( ModelMat );
      }
      else {
	 w = xoffsets[n]/WinWidth;
	 h = yoffsets[n]/WinHeight;
	 front = Scale * 1.75 * FrontClip;
	 rear = -Scale * 1.75;
	 ortho( -AspectRatio+w, AspectRatio+w, -1.0+h, 1.0+h, front, rear );
	 loadmatrix( ModelMat );
      }
   }
}



void end_aa_pass( n )
int n;
{
   if (pretty_flag) {
      acbuf( AC_ACCUMULATE, 1.0 );
      if (n==AA_PASSES-1) {
	 acbuf( AC_RETURN, 1.0/(float) AA_PASSES );
      }
   }
}



void start_hqtrans_pass()
{
}


int end_hqtrans_pass()
{
   return 0;
}




/*** polytrinorm ******************************************************
   Draw a polytriangle strip with normals.
**********************************************************************/
void polytrinorm( vert, norm, n )
float vert[][3];
float norm[][3];
int n;
{
   register int i;

   lmcolor( LMC_AD );
   cpack( CurColor );
   /* enable transparency */
   if (screendoor)
     setpattern( CurTrans );
   else
     blendfunction( BF_SA, BF_MSA );

   bgntmesh();
   for (i=0;i<n;i++) {
      n3f( norm[i] );
      v3f( vert[i] );
   }
   endtmesh();

   /* turn off transparency */
   if (screendoor)
     setpattern(0);
   else
     blendfunction( BF_ONE, BF_ZERO );
}



/*** polytri **********************************************************
   Draw a polytriangle strip without shading.
**********************************************************************/
void polytri( vert, n )
float vert[][3];
int n;
{
   register int i;

   lmcolor( LMC_COLOR );
   cpack( CurColor );
   /* enable transparency */
   if (screendoor)
     setpattern( CurTrans );
   else
     blendfunction( BF_SA, BF_MSA );

   bgntmesh();
   for (i=0;i<n;i++) {
      v3f( vert[i] );
   }
   endtmesh();

   /* turn off transparency */
   if (screendoor)
     setpattern(0);
   else
     blendfunction( BF_ONE, BF_ZERO );
}



/*** polyline *********************************************************
   Draw a 3-D poly line.
**********************************************************************/
void polyline( vert, n )
float vert[][3];
int n;
{
   register int i;

   lmcolor( LMC_COLOR );
   cpack( CurColor );

   bgnline();
   for (i=0;i<n;i++) {
      v3f( vert[i] );
   }
   endline();
}




/*** disjointpolyline *************************************************
   Draw a series of disjoint 3-D lines.
**********************************************************************/
void disjointpolyline( vert, n )
float vert[][3];
int n;
{
   register int i;

   lmcolor( LMC_COLOR );
   cpack( CurColor );

   for (i=0;i<n;i+=2) {
/*  bgnlinelist is a Kubota GL extension */
      bgnlinelist();
      v3f( vert[i] );
      v3f( vert[i+1] );
      endlinelist();
   }
}





/*** quadmesh *********************************************************
   Draw a quadrilateral mesh with colors at each vertex.
**********************************************************************/
void quadmesh( vert, color, rows, cols )
float vert[][3];
unsigned int color[];
int rows, cols;
{
   register int i, j, base1, base2;

   lmcolor ( LMC_COLOR );
   /* enable transparency blending */
   if (screendoor)
     setpattern( CurTrans );
   else
     blendfunction( BF_SA, BF_MSA );

   /* break mesh into strips */
   for (i=0;i<rows-1;i++) {
      base1 = i * cols;
      base2 = (i+1) * cols;
      bgnqstrip();
      for (j=0;j<cols;j++) {
	 cpack( color[base1+j] );
	 v3f( vert[base1+j] );
	 cpack( color[base2+j] );
	 v3f( vert[base2+j] );
      }
      endqstrip();
   }

   /* turn off transparency */
   if (screendoor)
     setpattern(0);
   else
     blendfunction( BF_ONE, BF_ZERO );
}



/*** quadmeshnorm *****************************************************
   Draw a quadrilateral mesh with normals and colors at each vertex.
**********************************************************************/
void quadmeshnorm( vert, norm, color, rows, cols )
float vert[][3];
float norm[][3];
unsigned int color[];
int rows, cols;
{
   register int i, j, base1, base2;

   lmcolor( LMC_AD ); 
   /* enable transparency blending */
   if (screendoor)
     setpattern( CurTrans );
   else
     blendfunction( BF_SA, BF_MSA );

   /* break mesh into strips */
   for (i=0;i<rows-1;i++) {
      base1 = i * cols;
      base2 = (i+1) * cols;
      bgnqstrip();
      for (j=0;j<cols;j++) {
	 cpack( color[base1+j] ); 
	 n3f( norm[base1+j] );
	 v3f( vert[base1+j] );
	 cpack( color[base2+j] ); 
	 n3f( norm[base2+j] );
	 v3f( vert[base2+j] );
      }
      endqstrip();
   }

   /* turn off transparency */
   if (screendoor)
     setpattern(0);
   else
     blendfunction( BF_ONE, BF_ZERO );
}



void quadmeshnorm_translated( vert, norm, color, rows, cols, tx, ty )
float vert[][3];
float norm[][3];
unsigned int color[];
int rows, cols;
float tx, ty;
{
   pushmatrix();
   translate( tx, ty, 0.0 );

   quadmeshnorm( vert, norm, color, rows, cols );

   popmatrix();
}



/*** polyline2d *******************************************************
   Draw a 2-D poly line.  Coordinates are in pixels with the origin
   in the upper-left corner of the window.  NOTE:  vertices are shorts.
**********************************************************************/
void polyline2d( vert, n )
short vert[][2];
int n;
{
   short v[2];
   int i;

   lmcolor( LMC_COLOR );
   cpack( CurColor );
   bgnline();
   for (i=0;i<n;i++) {
      v[0] = vert[i][0];
      v[1] = WinHeight - vert[i][1];
      v2s( v );
   }
   endline();
}


/*** draw_rect ********************************************************
   Draw a rectangle in 2-D.
**********************************************************************/
void draw_rect( x1, y1, x2, y2 )
int x1, y1, x2, y2;
{
   short v[5][2];

   v[0][0] = x1;  v[0][1] = y1;
   v[1][0] = x2;  v[1][1] = y1;
   v[2][0] = x2;  v[2][1] = y2;
   v[3][0] = x1;  v[3][1] = y2;
   v[4][0] = x1;  v[4][1] = y1;
   polyline2d( v, 5 );
}



/*** draw_text ********************************************************
   Draw a text string.  Position coordinates are in pixels with the
   origin being in the upper-left corner.
**********************************************************************/
void draw_text( xpos, ypos, str )
int xpos, ypos;
char *str;
{
   lmcolor( LMC_COLOR );
   cpack( CurColor );
   cmovi( xpos, WinHeight-ypos, 1 );
   fmprstr( str );
}



/*** text_width *******************************************************
   Return the width of a text string in pixels.
**********************************************************************/
int text_width( str )
char *str;
{
   return fmgetstrwidth( visfont, str );
}



int begin_object()
{
   int n;

   n = genobj();
   makeobj( n );
   return n;
}


int end_object()
{
   closeobj();
}


int draw_object( n )
int n;
{
   callobj( n );
}

