/* $Id: work.c,v 1.42 1996/03/07 19:52:46 brianp Exp $ */

/* Vis5D version 4.2 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990-1996  Bill Hibbard, Brian Paul, Dave Santek,
and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#ifdef sgi
#  include <sys/types.h>
#  include <sys/prctl.h>
#endif
#include "analysis.h"
#include "api.h"
#include "contour.h"
#include "globals.h"
#include "grid.h"
#include "memory.h"
#include "misc.h"
#include "proj.h"
#include "queue.h"
#include "stream.h"
#include "sync.h"
#include "traj.h"
#include "vtmcP.h"
#include "work.h"


#define DEG2RAD(d)  ((d)*3.14159/180.0)
#define MIN(A,B)  ( (A) < (B) ? (A) : (B) )
#define MAX(A,B)  ( (A) > (B) ? (A) : (B) )


/* Maximum number of vertices... */
#ifdef BIG_GFX
#  define MAX_ISO_VERTS 120000    /* in an isosurface */
#else
#  define MAX_ISO_VERTS 65000     /* in an isosurface */
#endif
#define MAX_CONT_VERTS (MAXROWS*MAXCOLUMNS)   /* in a contour line slice */
#define MAX_WIND_VERTS (4*MAXROWS*MAXCOLUMNS) /* in a wind vector slice */
#define MAX_TRAJ_VERTS 5000                   /* in a wind trajectory */




/*
 * Compute the color table indexes for a colored isosurface.
 * Input:  ctx - the context
 *         time - which timestep
 *         isovar - the isosurface variable
 *         colorvar - the coloring variable
 */
static void color_isosurface( Context ctx, int time, int isovar, int colorvar )
{
   uint_1 *color_indexes;
   int i, n;
   float vscale = 1.0 / VERTEX_SCALE;
   float min, valscale;

   /* Free old color indexes if any */
   wait_write_lock( &ctx->SurfTable[isovar][time].lock );
   if (ctx->SurfTable[isovar][time].colors) {
      deallocate( ctx, ctx->SurfTable[isovar][time].colors,
                  ctx->SurfTable[isovar][time].numverts*sizeof(uint_1) );
      ctx->SurfTable[isovar][time].colors = NULL;
   }
   done_write_lock( &ctx->SurfTable[isovar][time].lock );

   if (colorvar!=-1) {
      /* Allocate storage for new color indexes */
      n = ctx->SurfTable[isovar][time].numverts;
      color_indexes = allocate( ctx, n*sizeof(uint_1) );
      if (!color_indexes) {
         return;
      }

      min = ctx->MinVal[colorvar];
      valscale = 1.0 / (ctx->MaxVal[colorvar] - ctx->MinVal[colorvar]);

      /* Compute color indexes */
      for (i=0;i<n;i++) {
         float x, y, z;
         float row, col, lev;
         float val;
         int index;

         x = ctx->SurfTable[isovar][time].verts[i*3+0] * vscale;
         y = ctx->SurfTable[isovar][time].verts[i*3+1] * vscale;
         z = ctx->SurfTable[isovar][time].verts[i*3+2] * vscale;

         xyz_to_grid( ctx, time, isovar, x, y, z, &row, &col, &lev );

         val = interpolate_grid_value( ctx, time, colorvar, row, col, lev );

         if (IS_MISSING(val)) {
            color_indexes[i] = 255;
         }
         else {
            int index = (val - min) * valscale * 254.0F;
            color_indexes[i] = (uint_1) index;
         }
      }
   }
   else {
      color_indexes = NULL;
   }

   /* save results */
   wait_write_lock( &ctx->SurfTable[isovar][time].lock );
   ctx->SurfTable[isovar][time].colors = color_indexes;
   ctx->SurfTable[isovar][time].colorvar = colorvar;
   done_write_lock( &ctx->SurfTable[isovar][time].lock );
}



/*
 * Calculate an isosurface and store it.
 * Input:  time - the time step.
 *         var - which variable.
 *         iso_level - level to construct contour at.
 *         colorvar - which variable to color with or -1
 *         threadnum - thread ID
 * Output:  resulting poly-triangle strip is saved in SurfTable.
 */
static void calc_isosurface( Context ctx, int time, int var,
                             float iso_level, int colorvar, int threadnum )
{
   /* marching cubes parameters */
   float arx=1.0, ary=1.0, arz=1.0;
   float vc[MAX_ISO_VERTS], vr[MAX_ISO_VERTS], vl[MAX_ISO_VERTS];
   float nx[MAX_ISO_VERTS], ny[MAX_ISO_VERTS], nz[MAX_ISO_VERTS];
   int vpts[2*MAX_ISO_VERTS];
   int numverts, numindexes, ipoly, itri;
   /* other vars */
   float *grid;
   int_2 *cverts;
   int_1 *cnorms;
#ifdef BIG_GFX
   uint_4 *index;
#else
   uint_2 *index;
#endif

   if (!ctx->SurfTable[var][time].valid ||
       ctx->SurfTable[var][time].isolevel != iso_level) {

      /* compute the isosurface */

      grid = get_grid( ctx, time, var );  /* get pointer to grid data */
      if (!grid) return;

      /* Pass number of levels of parameter. main_march is not changed */
      main_march( ctx, grid, ctx->Nc, ctx->Nr, ctx->Nl[var], ctx->LowLev[var],
              iso_level, arx, ary, arz, MAX_ISO_VERTS,
              vc,vr,vl, nx,ny,nz, 2*MAX_ISO_VERTS, vpts,
              &numverts, &numindexes, &ipoly, &itri );

      release_grid( ctx, time, var, grid );

      recent( ctx, ISOSURF, var );

      if (numindexes>MAX_ISO_VERTS)
         numindexes = MAX_ISO_VERTS;

      /*************************** Compress data ***************************/

      if (numverts>0 && numindexes>0) {
         int vbytes, nbytes, bytes, i;

         /* allocate memory to store compressed vertices */
         vbytes = 3*numverts*sizeof(int_2);
         cverts = (int_2 *) allocate_type( ctx, vbytes, CVX_TYPE );
         /* convert (r,c,l) coords to compressed (x,y,z) */
         grid_to_compXYZ( ctx, time, var, numverts, vr,vc,vl, (void*) cverts );

         /* allocate memory to store compressed normals */
         nbytes = 3*numverts*sizeof(int_1);
         cnorms = (int_1 *) allocate_type( ctx, nbytes, CNX_TYPE );
         /* cvt normals from floats in [-1,1] to 1-byte ints in [-125,125] */
         project_normals( ctx, numverts, vr,vc,vl, nx,ny,nz, (void*) cnorms );

         /* allocate memory to store index array */
#ifdef BIG_GFX
         bytes = numindexes * sizeof(uint_4);
         index = (uint_4 *) allocate_type( ctx, bytes, PTS_TYPE );
         memcpy( index, vpts, numindexes * sizeof(uint_4) );
#else
         bytes = numindexes * sizeof(uint_2);
         index = (uint_2 *) allocate_type( ctx, bytes, PTS_TYPE );
         /* convert 4-byte vpts[] elements to 2-byte index[] elements */
         for (i=0;i<numindexes;i++)
            index[i] = (uint_2) vpts[i];
#endif

      }
      else {
         cverts = NULL;
         cnorms = NULL;
         index = NULL;
         numverts = numindexes = 0;
      }

      /******************** Store the new surface ************************/

      wait_write_lock( &ctx->SurfTable[var][time].lock );

      /* deallocate existing surface, if any */
      free_isosurface( ctx, time, var );

      /* add surface to table */
      ctx->SurfTable[var][time].isolevel = iso_level;
      ctx->SurfTable[var][time].numverts = numverts;
      ctx->SurfTable[var][time].verts = cverts;
      ctx->SurfTable[var][time].norms = cnorms;
      ctx->SurfTable[var][time].numindex = numindexes;
      ctx->SurfTable[var][time].index = index;
      ctx->SurfTable[var][time].valid = 1;

      done_write_lock( &ctx->SurfTable[var][time].lock );

   }

   if (colorvar!=-1 || ctx->SurfTable[var][time].colorvar!=colorvar) {
      color_isosurface( ctx, time, var, colorvar );
   }

   if (time==ctx->CurTime) {
      ctx->Redraw = 1;
   }
}



/*
 * Extract a horizontal slice from a 3-D grid.
 * Input:  grid - pointer to nr * nc * nl 3-D grid of data
 *         nr, nc, nl - size of 3-D grid.
 *         level - position in [0,numlev-1] to extract slice from.
 *         colmajor - 1 = column major, 0 = row major.
 * Returned:  pointer to nr x nc array of floats
 */
static float* extract_hslice( Context ctx, float *grid,
                              int nr, int nc, int nl, int lowlev,
                              float level, int colmajor )
{
   float *slice;

   /* allocate buffer to put 2-D slice of data */
   slice = (float *) allocate_type( ctx, nr * nc * sizeof(float), HSLICE_TYPE );
   if (!slice) {
      return NULL;
   }

   /* extract the 2-D array from 3-D grid */
   if (nl==1) {
      /*
       * Special case:  the 3-D grid is really 2-D
       */
      float g1;
      int i, j;

      if (colmajor) {
         for (j=0; j<nc; j++) {
            for (i=0; i<nr; i++) {
               g1 = grid[j*nr+i];
               if (IS_MISSING(g1)) {
                  slice[j*nr+i] = MISSING;
               }
               else {
                  slice[j*nr+i] = g1;
               }
            }
         }
      }
      else {
         /* change from column-major to row-major order */
         for (i=0; i<nr; i++) {
            for (j=0; j<nc; j++) {
               g1 = grid[j*nr+i];
               if (IS_MISSING(g1)) {
                  slice[i*nc+j] = MISSING;
               }
               else {
                  slice[i*nc+j] = g1;
               }
            }
         }
      }
   }
   else {
      /*
       * extract (and interpolate) 2-D slice from 3-D grid
       */
      int upper, lower, above, below;
      float a, b, g1, g2;
      int i, j;

      /* WLH 6-30-95 */
      level -= lowlev;
      if (level < 0 || level > nl-1) {
        for (i=0; i<nr*nc; i++) slice[i] = MISSING;
        return slice;
      }

      lower = (int) level;
      upper = lower + 1;
      if (upper>nl-1)
	 upper = nl - 1;
      a = level - (float) lower;
      b = 1.0 - a;

      if (a==0.0) {
         /* If the location of the slice exactly corresponds to a discrete */
         /* grid level, don't interpolate with next higher level! */
         upper = lower;
      }

#ifdef LEVELTYPES 
      /* Correct if logaritmic interpolation required */
#include "logfrac.h"
      printf("linfrac=%f   ", a);
      CalcLinLogFrac(lower, a, b);
      printf("logfrac=%f\n", a);
#endif

      below = lower * nr * nc;
      above = upper * nr * nc;

      /* interpolate between layers */
      if (colmajor) {
	 for (j=0; j<nc; j++) {
	    for (i=0; i<nr; i++) {
	       g1 = grid[above+j*nr+i];
	       g2 = grid[below+j*nr+i];
	       if (IS_MISSING(g1) || IS_MISSING(g2)) {
		  slice[j*nr+i] = MISSING;
	       }
	       else {
		  slice[j*nr+i] = a * g1 + b * g2; 
	       }
	    }
	 }
      }
      else {
	 /* change from column-major to row-major order */
	 for (i=0; i<nr; i++) {
	    for (j=0; j<nc; j++) {
	       g1 = grid[above+j*nr+i];
	       g2 = grid[below+j*nr+i];
	       if (IS_MISSING(g1) || IS_MISSING(g2)) {
		  slice[i*nc+j] = MISSING;
	       }
	       else {
		  slice[i*nc+j] = a * g1 + b * g2;
	       }
	    }
	 }
      }
   }

   return slice;
}




/*** extract_vslice ***************************************************
   Extract a vertical slice from a 3-D grid.
   Input:  grid - pointer to Nl x ctx->Nr * ctx->Nc float array.
           r1,c1,r2,c2 - position in [0,ctx->Nr-1],[0,ctx->Nc-1] to extract slice.
	   cols, rows - size of slice returned.
	   colmajor - 1 = column major, 0 = row major.
   Returned:  pointer to ctx->Nr x ctx->Nc array (slice) of floats
              or NULL if error
**********************************************************************/
static float *extract_vslice( Context ctx, float *grid,
                              float r1, float c1, float r2, float c2,
                              int rows, int cols,
                              int colmajor )
{
   float gx,gy, stepx, stepy;
   int ic, ir, i, j, iii;
   float g1,g2,g3,g4, ei,ej;
   float *slice;

   /* allocate slice array */
   slice = (float *) allocate_type( ctx, rows * cols * sizeof(float), VSLICE_TYPE );
   if (!slice)
      return NULL;

   /* initialize gx and gy- the real grid indices */
   gx = c1;
   gy = r1;

   /* calculate step sizes for iterating from c1 to c2 and r1 to r2 */
   stepx = (c2-c1) / (float) (cols-1);
   stepy = (r2-r1) / (float) (cols-1);

   if (colmajor) {
      for (ic=0; ic<cols; ic++) {
	 /* calculate error terms */
	 ei = gx - (float) ( (int) gx );   /* in [0,1) */
	 ej = gy - (float) ( (int) gy );   /* in [0,1) */
	 /* convert real gx,gy to integer i,j for array indexing */
	 i = (int) gx;
	 j = (int) gy;
	 if (i > ctx->Nc-1) {
	    i = ctx->Nc-1;
	 }
	 if (j > ctx->Nr-1) {
	    j = ctx->Nr-1;
	 }
	 for (ir=0; ir<rows; ir++) {
	    iii = ir * ctx->Nr * ctx->Nc;
	    /* fetch grid data */
	    g1 = grid[iii + i*ctx->Nr + j];          /*  g1 -- g2    +-- j */
	    g3 = grid[iii + i*ctx->Nr + j+1];        /*  |      |    |     */
	    if (ei!=0.0) {                           /*  g3 -- g4    i     */
	       g2 = grid[iii + (i+1)*ctx->Nr + j];
	       g4 = grid[iii + (i+1)*ctx->Nr + j+1];
	    }
	    else {
	       g2 = g4 = 0.0;
	    }
	    if (IS_MISSING(g1) || IS_MISSING(g2)
		|| IS_MISSING(g3) || IS_MISSING(g4)) {
	       slice[ ic*rows + rows - ir - 1 ] = MISSING;
	    }
	    else {
	       slice[ ic*rows + rows - ir - 1 ] = g1 * (1.0-ei) * (1.0-ej)
				                + g2 *      ei  * (1.0-ej)
				                + g3 * (1.0-ei) *      ej
				                + g4 *      ei  *      ej;
	    }
	 }
	 /* increment grid indices */
	 gx += stepx;
	 gy += stepy;
      }
   }
   else {
      for (ic=0; ic<cols; ic++) {
	 /* calculate error terms */
	 ei = gx - (float) ( (int) gx );   /* in [0,1) */
	 ej = gy - (float) ( (int) gy );   /* in [0,1) */
	 /* convert real gx,gy to integer i,j for array indexing */
	 i = (int) gx;
	 j = (int) gy;
	 if (i > ctx->Nc-1) {
	    i = ctx->Nc-1;
	 }
	 if (j > ctx->Nr-1) {
	    j = ctx->Nr-1;
	 }
	 for (ir=0; ir<rows; ir++) {
	    iii = ir * ctx->Nr * ctx->Nc;
	    /* fetch grid data */
	    g1 = grid[iii + i*ctx->Nr + j];            /* g1 -- g2    +-- j */
	    g3 = grid[iii + i*ctx->Nr + j+1];          /* |      |    |     */
	    if (ei!=0.0) {                             /* g3 -- g4    i     */
	       g2 = grid[iii + (i+1)*ctx->Nr + j];
	       g4 = grid[iii + (i+1)*ctx->Nr + j+1];
	    }
	    else {
	       g2 = g4 = 0.0;
	    }
	    if (IS_MISSING(g1) || IS_MISSING(g2)
		|| IS_MISSING(g3) || IS_MISSING(g4)) {
	       slice[ ir*cols + ic ] = MISSING;
	    }
	    else {
	       slice[ ir*cols + ic ] = g1 * (1.0-ei) * (1.0-ej)
		                     + g2 *      ei  * (1.0-ej)
	                    	     + g3 * (1.0-ei) *      ej
		                     + g4 *      ei  *      ej;
	    }
	 }
	 /* increment grid indices */
	 gx += stepx;
	 gy += stepy;
      }
   }

   return slice;
}




/*
 * Generate a list of vertices for a drawing a horizontal bounding rectangle
 * around horizontal contour line slices or horizontal wind vector slices.
 * Input:  time, var - the timestep and variable
 *         level - the grid level in [0,Nl-1]
 *         curved - if non-zero, generate vertices for a curved box.
 * Output:  boxverts - pointer to array of vertices we've allocated.
 * Return:  number of vertices in the vertex list.
 */
static int make_horizontal_rectangle( Context ctx, int time, int var,
                                      int curved, float level,
                                      float **boxverts )
{
   int i, n;
   float *v;

   n = 0;

   if (curved==0) {
      v = (float *) allocate_type( ctx, 5 * 3 * sizeof(float), MHRECT_TYPE );
      if (v) {
	 n = 5;
	 v[0*3+0] = 0.0;
	 v[0*3+1] = 0.0;
	 v[0*3+2] = level;
	 v[1*3+0] = 0.0;
	 v[1*3+1] = (float) (ctx->Nc-1);
	 v[1*3+2] = level;
	 v[2*3+0] = (float) (ctx->Nr-1);
	 v[2*3+1] = (float) (ctx->Nc-1);
	 v[2*3+2] = level;
	 v[3*3+0] = (float) (ctx->Nr-1);
	 v[3*3+1] = 0.0;
	 v[3*3+2] = level;
	 v[4*3+0] = 0.0;
	 v[4*3+1] = 0.0;
	 v[4*3+2] = level;
      }
   }
   else {
      /* curved box */
      v = (float *) allocate_type( ctx, (2*ctx->Nr + 2*ctx->Nc - 3) * 3 * sizeof(float),
                                   MHRECT_TYPE );
      if (v) {
	 /* north edge */
	 for (i=0;i<ctx->Nc;i++) {
	    v[n++] = 0.0;
	    v[n++] = i;
	    v[n++] = level;
	 }
	 /* east edge */
	 for (i=1;i<ctx->Nr;i++) {
	    v[n++] = i;
	    v[n++] = ctx->Nc-1;
	    v[n++] = level;
	 }
	 /* south edge */
	 for (i=ctx->Nc-2;i>=0;i--) {
	    v[n++] = ctx->Nr-1;
	    v[n++] = i;
	    v[n++] = level;
	 }
	 /* west edge */
	 for (i=ctx->Nr-2;i>=0;i--) {
	    v[n++] = i;
	    v[n++] = 0;
	    v[n++] = level;
	 }
	 n /= 3;
	 assert( n == 2*ctx->Nr + 2*ctx->Nc - 3 );
      }
   }

   /* convert vertices from grid to graphics coords */
   for (i=0;i<n;i++) {
      float r = v[i*3+0];
      float c = v[i*3+1];
      float l = v[i*3+2];
      grid_to_xyz( ctx, time, var, 1, &r, &c, &l,
		   &v[i*3+0], &v[i*3+1], &v[i*3+2] );
   }

   *boxverts = v;
   return n;
}




static int make_vertical_rectangle( Context ctx, int time, int var,
                                    int curved,
                                    float r1, float c1, float r2, float c2,
                                    int segs, float **boxverts )
{
   int i, n;
   float *v;

   n = 0;

   if (curved==0) {
      v = (float *) allocate_type( ctx, 5 * 3 * sizeof(float), MVRECT_TYPE );
      if (v) {
	 n = 5;
	 v[0*3+0] = r1;
	 v[0*3+1] = c1;
	 v[0*3+2] = ctx->LowLev[var];
	 v[1*3+0] = r1;
	 v[1*3+1] = c1;
	 v[1*3+2] = ctx->Nl[var]-1 + ctx->LowLev[var];
	 v[2*3+0] = r2;
	 v[2*3+1] = c2;
	 v[2*3+2] = ctx->Nl[var]-1 + ctx->LowLev[var];;
	 v[3*3+0] = r2;
	 v[3*3+1] = c2;
	 v[3*3+2] = ctx->LowLev[var];
	 v[4*3+0] = r1;
	 v[4*3+1] = c1;
	 v[4*3+2] = ctx->LowLev[var];
      }
   }
   else {
      /* curved box */
      v = (float *) allocate_type( ctx, (2*ctx->Nl[var] + 2*segs - 3) * 3 * sizeof(float),
                                   MVRECT_TYPE );
      if (v) {
	 float r, c, dr, dc;
	 dr = (r2-r1) / (float) (segs-1);
	 dc = (c2-c1) / (float) (segs-1);
	 /* top */
	 r = r1;
	 c = c1;
	 for (i=0;i<segs;i++) {
	    v[n++] = r;
	    v[n++] = c;
	    v[n++] = ctx->Nl[var]-1 + ctx->LowLev[var];
	    r += dr;
	    c += dc;
	 }
	 /* right */
	 for (i=ctx->Nl[var]-2;i>=0;i--) {
	    v[n++] = r2;
	    v[n++] = c2;
	    v[n++] = i + ctx->LowLev[var];
	 }
	 /* bottom */
	 r = r2-dr;
	 c = c2-dc;
	 for (i=segs-2;i>=0;i--) {
	    v[n++] = r;
	    v[n++] = c;
	    v[n++] = ctx->LowLev[var];
	    r -= dr;
	    c -= dc;
	 }
	 /* left */
	 for (i=1;i<ctx->Nl[var];i++) {
	    v[n++] = r1;
	    v[n++] = c1;
	    v[n++] = i + ctx->LowLev[var];
	 }
	 n /= 3;
	 assert( n == 2*ctx->Nl[var] + 2*segs - 3 );
      }
   }

   /* convert vertices from grid to graphics coords */
   for (i=0;i<n;i++) {
      float r = v[i*3+0];
      float c = v[i*3+1];
      float l = v[i*3+2];
      grid_to_xyz( ctx, time, var, 1, &r, &c, &l,
		   &v[i*3+0], &v[i*3+1], &v[i*3+2] );
   }

   *boxverts = v;
   return n;
}




/*** calc_hslice ******************************************************
   Calculate a horizontal contour line slice and store it.
   Input:  time - the time step.
           var - which variable.
	   interval - interval between lines.
	   low, high - range of values to contour.
	   level - position of slice in [0..Nl-1].
	   threadnum - thread ID
   Output:  resulting poly-triangle strip is saved in HSliceTable.
**********************************************************************/
static void calc_hslice( Context ctx, int time, int var,
                         float interval, float low, float high, float level,
                         int threadnum )
{
   struct hslice *slice = &ctx->HSliceTable[var][time];
   float vr1[MAX_CONT_VERTS],vc1[MAX_CONT_VERTS];  /* contour line vertices */
   float vr2[MAX_CONT_VERTS/2],vc2[MAX_CONT_VERTS/2];  /* hidden line verts */
   float vr3[MAX_CONT_VERTS/2],vc3[MAX_CONT_VERTS/2];  /* label line verts */
   float vl[MAX_CONT_VERTS];   /* vertex level */
   float *grid, *slicedata;
   int num1, num2, num3, maxnum, bytes, i;
   float base;
   int_2 *cverts1, *cverts2, *cverts3;
   float *boxverts;
   int numboxverts;


   if (ctx->Nl[var]==1) {
      wait_write_lock( &slice->lock );
      if (slice->valid && !ctx->CurvedBox && slice->interval==interval
          && slice->lowlimit==low && slice->highlimit==high) {
         /* special case: just translate existing slice! */
         float z = gridlevel_to_z( ctx, time, var, level );
         int_2 compz = (int_2) (z * VERTEX_SCALE);
         int_2 *zptr;
         int n;
         /* translate verts1 */
         n = slice->num1;
         zptr = slice->verts1 + 2;
         for (i=0;i<n;i++) {
            *zptr = compz;
            zptr += 3;
         }
         /* translate verts2 */
         n = slice->num2;
         zptr = slice->verts2 + 2;
         for (i=0;i<n;i++) {
            *zptr = compz;
            zptr += 3;
         }
         /* translate verts3 */
         n = slice->num3;
         zptr = slice->verts3 + 2;
         for (i=0;i<n;i++) {
            *zptr = compz;
            zptr += 3;
         }
         /* update the bounding rectangle */
         numboxverts = make_horizontal_rectangle( ctx, time, var,
                                                  ctx->CurvedBox,
                                                  level, &boxverts );
         slice->numboxverts = numboxverts;
         slice->boxverts = boxverts;
         slice->level = level;
         recent( ctx, HSLICE, var );
         done_write_lock( &slice->lock );
         return;
      }
      done_write_lock( &slice->lock );
   }



   /* get the 3-D grid */
   grid = get_grid( ctx, time, var );
   if (!grid)
      return;

   /* extract the 2-D slice from the 3-D grid */
   slicedata = extract_hslice( ctx, grid, ctx->Nr, ctx->Nc, ctx->Nl[var],
                               ctx->LowLev[var], level, 1 );
   if (!slicedata)
      return;

   if (low==ctx->MinVal[var])
     base = 0.0;
   else
     base = low;

   /* call contouring routine */
   contour( ctx, slicedata, ctx->Nr, ctx->Nc, interval, low, high, base,
	    vr1, vc1, MAX_CONT_VERTS, &num1,
	    vr2, vc2, MAX_CONT_VERTS/2, &num2,
	    vr3, vc3, MAX_CONT_VERTS/2, &num3
	   );

   /* done with grid and slice */
   deallocate( ctx, slicedata, -1 );
   release_grid( ctx, time, var, grid );

   /* generate level coordinates array */
   if (num1>num2 && num1>num3) {
      maxnum = num1;
   }
   else if (num2>num1 && num2>num3) {
      maxnum = num2;
   }
   else {
      maxnum = num3;
   }
   for (i=0;i<maxnum;i++) {
      vl[i] = level;
   }

   /*
    * Transform vertices from grid coordinates in [0,Nr][0,Nc][0,Nl] to
    * compressed graphics coordinates in [-10000,10000]^3.
    */

   if (num1) {
      bytes = 3*num1*sizeof(int_2);
      cverts1 = (int_2 *) allocate_type( ctx, bytes, CVX1H_TYPE );
      if (cverts1) {
         grid_to_compXYZ( ctx, time, var, num1, vr1, vc1, vl, (void*)cverts1 );
      }
      else {
         num1 = 0;
      }
   }
   else {
      cverts1 = NULL;
   }

   if (num2) {
      bytes = 3*num2*sizeof(int_2);
      cverts2 = (int_2 *) allocate_type( ctx, bytes, CVX2H_TYPE );
      if (cverts2) {
         grid_to_compXYZ( ctx, time, var, num2, vr2, vc2, vl, (void*)cverts2 );
      }
      else {
         num2 = 0;
      }
   }
   else {
      cverts2 = NULL;
   }

   if (num3) {
      bytes = 3*num3*sizeof(int_2);
      cverts3 = (int_2 *) allocate_type( ctx, bytes, CVX3H_TYPE );
      if (cverts3) {
         grid_to_compXYZ( ctx, time, var, num3, vr3, vc3, vl, (void*)cverts3 );
      }
      else {
         num3 = 0;
      }
   }
   else {
      cverts3 = NULL;
   }

   /*
    * Generate bounding rectangle.
    */
   numboxverts = make_horizontal_rectangle( ctx, time, var, ctx->CurvedBox,
					    level, &boxverts );


   /************************ Store the new slice ************************/

   recent( ctx, HSLICE, var );

   wait_write_lock( &slice->lock );

   /* deallocate existing slice, if any */
   free_hslice( ctx, time, var );

   /* store new slice */
   slice->interval = interval;
   slice->lowlimit = low;
   slice->highlimit = high;
   slice->level = level;
   slice->num1 = num1;
   slice->verts1 = cverts1;
   slice->num2 = num2;
   slice->verts2 = cverts2;
   slice->num3 = num3;
   slice->verts3 = cverts3;
   slice->boxverts = boxverts;
   slice->numboxverts = numboxverts;
   slice->valid = 1;

   done_write_lock( &slice->lock );

   if (time==ctx->CurTime) {
      ctx->Redraw = 1;
   }
}




/*** calc_vslice ******************************************************
   Calculate a vertical contour line slice and store it.
   Input:  time - the time step.
           var - which variable.
	   interval - interval between lines.
	   low, high - range of values to contour
	   r1, c1 - position of 1st corner in [0..ctx->Nr-1],[0..ctx->Nc-1]
	   r2, c2 - position of 1st corner in [0..ctx->Nr-1],[0..ctx->Nc-1]
	   threadnum - thread ID
   Output:  resulting poly-triangle strip is saved in VSliceTable.
**********************************************************************/
static void calc_vslice( Context ctx, int time, int var,
                         float interval, float low, float high,
                         float r1, float c1, float r2, float c2,
                         int threadnum )
{
   float vr1[MAX_CONT_VERTS],vc1[MAX_CONT_VERTS],vl1[MAX_CONT_VERTS];
   float vr2[MAX_CONT_VERTS/2],vc2[MAX_CONT_VERTS/2],vl2[MAX_CONT_VERTS/2];
   float vr3[MAX_CONT_VERTS/2],vc3[MAX_CONT_VERTS/2],vl3[MAX_CONT_VERTS/2];
   float *grid;
   float *slice;
   int cols, rows;
   int i;
   int num1, num2, num3, bytes;
   float dr, dc, r, base;
   int_2 *cverts1, *cverts2, *cverts3;
   float *boxverts;
   int numboxverts;

   /* get the 3-D grid */
   grid = get_grid( ctx, time, var );
   if (!grid)
      return;

   /* extract the 2-D slice from the 3-D grid */
   rows = ctx->Nl[var];
   cols = (ctx->Nr>ctx->Nc) ? ctx->Nr : ctx->Nc;  /* kludge */
   slice = extract_vslice( ctx, grid, r1,c1,r2,c2, rows, cols, 1 );
   if (!slice)
      return;

   if (low==ctx->MinVal[var])
     base = 0.0;
   else
     base = low;

   /* call contouring routine */
   contour( ctx, slice, rows, cols, interval, low, high, base,
	    vr1, vc1, MAX_CONT_VERTS, &num1,
	    vr2, vc2, MAX_CONT_VERTS/2, &num2,
	    vr3, vc3, MAX_CONT_VERTS/2, &num3
	   );

   deallocate( ctx, slice, -1 );
   release_grid( ctx, time, var, grid );

   /*
    * Convert 2-D coordinates from [0,rows-1][0,cols-1] to 3-D coords
    * in [0,rows-1][0,cols-1][0,Nl-1].
    */

   dr = r2-r1;
   dc = c2-c1;
   for (i=0;i<num1;i++) {
      r = vc1[i] / (float) (cols-1);   /* r in [0,1] */
      vl1[i] = (float) (ctx->Nl[var]-1+ctx->LowLev[var]) - vr1[i];
      vr1[i] = r1 + r * dr;
      vc1[i] = c1 + r * dc;
   }
   for (i=0;i<num2;i++) {
      r = vc2[i] / (float) (cols-1);   /* r in [0,1] */
      vl2[i] = (float) (ctx->Nl[var]-1+ctx->LowLev[var]) - vr2[i];
      vr2[i] = r1 + r * dr;
      vc2[i] = c1 + r * dc;
   }
   for (i=0;i<num3;i++) {
      r = vc3[i] / (float) (cols-1);   /* r in [0,1] */
      vl3[i] = (float) (ctx->Nl[var]-1+ctx->LowLev[var]) - vr3[i];
      vr3[i] = r1 + r * dr;
      vc3[i] = c1 + r * dc;
   }

   recent( ctx, VSLICE, var );

   if (num1) {
      bytes = 3*num1*sizeof(int_2);
      cverts1 = (int_2 *) allocate_type( ctx, bytes, CVX1V_TYPE );
      if (cverts1) {
         grid_to_compXYZ( ctx, time, var, num1, vr1, vc1, vl1,
                          (void*) cverts1 );
      }
      else {
         num1 = 0;
      }
   }
   else {
      cverts1 = NULL;
   }

   if (num2) {
      bytes = 3*num2*sizeof(int_2);
      cverts2 = (int_2 *) allocate_type( ctx, bytes, CVX2V_TYPE );
      if (cverts2) {
         grid_to_compXYZ( ctx, time, var, num2, vr2, vc2, vl2,
                          (void*) cverts2 );
      }
      else {
         num2 = 0;
      }
   }
   else {
      cverts2 = NULL;
   }

   if (num3) {
      bytes = 3*num3*sizeof(int_2);
      cverts3 = (int_2 *) allocate_type( ctx, bytes, CVZ3V_TYPE );
      if (cverts3) {
         grid_to_compXYZ( ctx, time, var, num3, vr3, vc3, vl3,
                          (void*) cverts3 );
      }
      else {
         num3 = 0;
      }
   }
   else {
      cverts3 = NULL;
   }

   /*
    * Generate bounding rectangle.
    */
   numboxverts = make_vertical_rectangle( ctx, time, var, ctx->CurvedBox,
					  r1, c1, r2, c2, cols, &boxverts );


   /************************ Store the new slice ************************/

   wait_write_lock( &ctx->VSliceTable[var][time].lock );

   /* deallocate existing slice, if any */
   free_vslice( ctx, time, var );

   /* store new slice */
   ctx->VSliceTable[var][time].interval = interval;
   ctx->VSliceTable[var][time].lowlimit = low;
   ctx->VSliceTable[var][time].highlimit = high;
   ctx->VSliceTable[var][time].r1 = r1;
   ctx->VSliceTable[var][time].c1 = c1;
   ctx->VSliceTable[var][time].r2 = r2;
   ctx->VSliceTable[var][time].c2 = c2;
   ctx->VSliceTable[var][time].num1 = num1;
   ctx->VSliceTable[var][time].verts1 = cverts1;
   ctx->VSliceTable[var][time].num2 = num2;
   ctx->VSliceTable[var][time].verts2 = cverts2;
   ctx->VSliceTable[var][time].num3 = num3;
   ctx->VSliceTable[var][time].verts3 = cverts3;
   ctx->VSliceTable[var][time].boxverts = boxverts;
   ctx->VSliceTable[var][time].numboxverts = numboxverts;
   ctx->VSliceTable[var][time].valid = 1;

   done_write_lock( &ctx->VSliceTable[var][time].lock );

   if (time==ctx->CurTime) {
      ctx->Redraw = 1;
   }
}




/*** calc_chslice *****************************************************
   Calculate a horizontal colored slice and store it.
   Input:  time - the time step.
           var - which variable.
	   level - position of slice in [0..Nl-1].
	   threadnum - thread ID
   Output:  resulting data values are saved in CHSliceTable.
**********************************************************************/
static void calc_chslice( Context ctx, int time, int var,
                          float level, int threadnum )
{
   struct chslice *slice = &ctx->CHSliceTable[var][time];
   float vr[MAXROWS*MAXCOLUMNS];
   float vc[MAXROWS*MAXCOLUMNS];
   float vl[MAXROWS*MAXCOLUMNS];
   int_2 *cverts;
   float *grid, *slicedata, scale;
   uint_1 *indexes;
   int vbytes, ibytes;
   int slice_rows, slice_cols;
   float density = 1.0;  /* Make this a parameter someday */

   if (ctx->Nl[var]==1) {
      wait_write_lock( &slice->lock );
      if (slice->valid && !ctx->CurvedBox) {
         /* special case: just translate existing slice! */
         float z = gridlevel_to_z( ctx, time, var, level );
         int_2 compz = (int_2) (z * VERTEX_SCALE);
         int nrnc = ctx->Nr * ctx->Nc;
         int_2 *zptr = slice->verts + 2;
         int i;
         for (i=0;i<nrnc;i++) {
            *zptr = compz;
            zptr += 3;
         }
         slice->level = level;
         recent( ctx, CHSLICE, var );
         done_write_lock( &slice->lock );
         return;
      }
      done_write_lock( &slice->lock );
   }


   /* get the 3-D grid */
   grid = get_grid( ctx, time, var );
   if (!grid)
      return;

   /* extract the 2-D array from 3-D grid */
   slicedata = extract_hslice( ctx, grid, ctx->Nr, ctx->Nc, ctx->Nl[var],
                               ctx->LowLev[var], level, 0 );
   if (!slicedata)
      return;

   /* compute size of colored slice */
   slice_rows = ctx->Nr * density;
   slice_cols = ctx->Nc * density;

   /* allocate space for vertices and color indexes */
   vbytes = slice_rows * slice_cols * sizeof(int_2) * 3;
   cverts = (int_2 *) allocate_type( ctx, vbytes, VXH_TYPE );
   ibytes = slice_rows * slice_cols * sizeof(uint_1);
   indexes = (uint_1 *) allocate_type( ctx, ibytes, INDEXESH_TYPE );
   if (!cverts || !indexes) {
      if (cverts) deallocate(ctx, cverts, vbytes );
      if (indexes) deallocate(ctx, indexes, ibytes );
      return;
   }

   /* compute graphics coords vertices */
   if (density==1.0) {
      int i, j, n = 0;
      for (i=0;i<slice_rows;i++) {
         for (j=0;j<slice_cols;j++) {
            vr[n] = (float) i;
            vc[n] = (float) j;
            vl[n] = level;
            n++;
         }
      }
   }
   else {
      float rowscale = (float) (ctx->Nr-1) / (float) (slice_rows-1);
      float colscale = (float) (ctx->Nc-1) / (float) (slice_cols-1);
      int i, j, n = 0;
      for (i=0;i<slice_rows;i++) {
         int src_row = i * rowscale;
         for (j=0;j<slice_cols;j++) {
            int src_col = j * colscale;
            vr[n] = (float) src_row;
            vc[n] = (float) src_col;
            vl[n] = level;
            n++;
         }
      }
   }
   grid_to_compXYZ( ctx, time, var, slice_rows*slice_cols, vr, vc, vl,
                    (void*) cverts );


   /* scale data values to [0,254] with missing = 255 */
   if (ctx->MinVal[var]==ctx->MaxVal[var])
     scale = 0.0;
   else
     scale = 254.0 / (ctx->MaxVal[var]-ctx->MinVal[var]);
   if (density==1.0) {
      /* simple calculation */
      float minval = ctx->MinVal[var];
      int i;
      for (i=0;i<slice_rows*slice_cols;i++) {
         if (IS_MISSING(slicedata[i]))
            indexes[i] = 255;
         else
            indexes[i] = (uint_1) (int) ((slicedata[i]-minval) * scale);
      }
   }
   else {
      /* resampling needed */
      int row, col;
      float minval = ctx->MinVal[var];
      float rowscale = (float) (ctx->Nr-1) / (float) (slice_rows-1);
      float colscale = (float) (ctx->Nc-1) / (float) (slice_cols-1);
      int i = 0;
      for (row=0; row<slice_rows; row++) {
         int src_row = row * rowscale;
         for (col=0; col<slice_cols; col++) {
            int src_col = col * colscale;
            float val = slicedata[ src_row * ctx->Nc + src_col ];
            if (IS_MISSING(val))
              indexes[i] = 255;
            else
              indexes[i] = (uint_1) (int) ((val-minval) * scale);
            i++;
         }
      }
   }


   /* done with the 3-D grid and 2-D slice */
   release_grid( ctx, time, var, grid );
   deallocate( ctx, slicedata, -1 );


   /**************************** Store ********************************/
   recent( ctx, CHSLICE, var );

   wait_write_lock( &slice->lock );

   /* deallocate existing slice, if any */
   free_chslice( ctx, time, var );

   /* store new slice */
   slice->level = level;
   slice->rows = slice_rows;
   slice->columns = slice_cols;
   slice->verts = cverts;
   slice->color_indexes = indexes;
   slice->valid = 1;

   done_write_lock( &slice->lock );

   if (time==ctx->CurTime) {
      ctx->Redraw = 1;
   }
}




/*** calc_cvslice ******************************************************
   Calculate a vertical colored slice and store it.
   Input:  time - the time step.
           var - which variable.
	   r1, c1 - position of 1st corner in [0..ctx->Nr-1],[0..ctx->Nc-1]
	   r2, c2 - position of 1st corner in [0..ctx->Nr-1],[0..ctx->Nc-1]
	   threadnum - thread ID
   Output:  resulting poly-triangle strip is saved in CVSliceTable.
**********************************************************************/
static void calc_cvslice( Context ctx, int time, int var,
                          float r1, float c1, float r2, float c2,
                          int threadnum )
{
   float vr[MAXROWS*MAXLEVELS];
   float vc[MAXROWS*MAXLEVELS];
   float vl[MAXROWS*MAXLEVELS];
   float *grid, *slice;
   int_2 *cverts;
   uint_1 *indexes;
   int cols, rows, i, j, n;
   float scale, r, c, dr, dc;
   float mr, mc, ml;
   float x, y, z;
   int vbytes, ibytes;

   /* size of quadmesh: */
   rows = ctx->Nl[var];
   cols = MAX(ctx->Nr,ctx->Nc);

   /* get the 3-D grid */
   grid = get_grid( ctx, time, var );
   if (!grid)
      return;

   /* extract the 2-D slice from the 3-D grid */
   slice = extract_vslice( ctx, grid, r1,c1,r2,c2, rows, cols, 0 );

   /* allocate space for vertices and color indexes */
   vbytes = rows * cols * sizeof(int_2) * 3;
   ibytes = rows*cols*sizeof(uint_1);
   cverts = (int_2 *) allocate_type( ctx, vbytes, VXV_TYPE );
   indexes = (uint_1 *) allocate_type( ctx, ibytes, INDEXESV_TYPE );
   if (!cverts || !indexes) {
      if (cverts) deallocate(ctx, cverts, vbytes );
      if (indexes) deallocate(ctx, indexes, ibytes );
      return;
   }

   /* compute the vertices */
   n = 0;
   dr = (r2-r1) / (cols-1);
   dc = (c2-c1) / (cols-1);
   for (i=0;i<rows;i++) {
      r = r1;
      c = c1;
      for (j=0;j<cols;j++) {
	 vr[n] = r;
	 vc[n] = c;
	 vl[n] = i+ctx->LowLev[var];
	 r += dr;
	 c += dc;
	 n++;
      }
   }
   grid_to_compXYZ( ctx, time, var, rows*cols, vr, vc, vl, (void*) cverts );

   /* scale grid values to [0,254] with missing = 255 */
   if (ctx->MinVal[var]==ctx->MaxVal[var])
      scale = 0.0;
   else
      scale = 254.0 / (ctx->MaxVal[var]-ctx->MinVal[var]);
   for (i=0;i<rows*cols;i++) {
      if (IS_MISSING(slice[i]))
	 indexes[i] = 255;
      else
	 indexes[i] = (uint_1) (int) ((slice[i]-ctx->MinVal[var]) * scale);
   }

   /* done with the 3-D grid and 2-D slice */
   release_grid( ctx, time, var, grid );
   deallocate( ctx, slice, -1 );

   /* make the tick mark at midpoint of top edge */
   mr = (r1+r2)/2.0;
   mc = (c1+c2)/2.0;
   ml = (float) (ctx->Nl[var]-1+ctx->LowLev[var]);
   grid_to_xyz( ctx, time, var, 1, &mr, &mc, &ml, &x, &y, &z );

   ctx->CVSliceTable[var][time].mark[0][0] = x;
   ctx->CVSliceTable[var][time].mark[0][1] = y;
   ctx->CVSliceTable[var][time].mark[0][2] = z+0.02;
   ctx->CVSliceTable[var][time].mark[1][0] = x;
   ctx->CVSliceTable[var][time].mark[1][1] = y;
   ctx->CVSliceTable[var][time].mark[1][2] = z-0.02;


   /************************* Store **********************************/
   recent( ctx, CVSLICE, var );

   wait_write_lock( &ctx->CVSliceTable[var][time].lock );

   /* deallocate existing slice, if any */
   free_cvslice( ctx, time, var );

   /* store new slice */
   ctx->CVSliceTable[var][time].r1 = r1;
   ctx->CVSliceTable[var][time].c1 = c1;
   ctx->CVSliceTable[var][time].r2 = r2;
   ctx->CVSliceTable[var][time].c2 = c2;
   ctx->CVSliceTable[var][time].rows = rows;
   ctx->CVSliceTable[var][time].columns = cols;
   ctx->CVSliceTable[var][time].color_indexes = indexes;
   ctx->CVSliceTable[var][time].verts = cverts;
   ctx->CVSliceTable[var][time].valid = 1;

   done_write_lock( &ctx->CVSliceTable[var][time].lock );

   if (time==ctx->CurTime) {
      ctx->Redraw = 1;
   }
}



/*
 * Macros for calc_hwindslice and calc_vwindslice:
 */

#define CROSS( c, a, b )  { c[0] =  a[1]*b[2]-a[2]*b[1]; \
			    c[1] = -a[0]*b[2]+a[2]*b[0]; \
			    c[2] =  a[0]*b[1]-a[1]*b[0]; \
			  }

#define MAGNITUDE( a )    sqrt( a[0]*a[0] + a[1]*a[1] + a[2]*a[2] )

#define NORMALIZE( v ) { float mag = 1.0/sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]); \
			 v[0] *= mag; \
		         v[1] *= mag; \
			 v[2] *= mag; \
		       }

#define SCALE_VEC( v, d )  {  v[0] = v[0] / (d); \
			      v[1] = v[1] / (d); \
			      v[2] = v[2] / (d); \
		           }



/*
 * Compute vectors in a horizontal wind slice.
 * Input:  time - which timestep
 *         ws - which wind slice [0,WINDSLICES-1]
 *         level - grid level of slice in [0,Nl-1]
 *         scale - user scaling factor  (1.0 is typical)
 *         density - user density factor  1.0, 0.5, 0.25, etc.
 *         threadnum - which thread
 */
static void calc_hwindslice( Context ctx, int time, int ws,
                             float level, float scale, float density,
                             int threadnum )
{
   float *grid, *ugrid, *vgrid, *wgrid;
   float mpcol, mprow, mplev;   /* mp = meters per */
   float colscale, rowscale, levscale;
   float midlat, midlon;
   int row, col, drow, dcol, vcount;
   float deltatime;
   float vr[MAX_WIND_VERTS], vc[MAX_WIND_VERTS], vl[MAX_WIND_VERTS];
   int_2 *cverts;
   int uvar, vvar, wvar;
   float *boxverts;
   int numboxverts;

   uvar = ctx->Uvar[ws];
   vvar = ctx->Vvar[ws];
   wvar = ctx->Wvar[ws];

   if (uvar<0 || vvar<0) {
      /* no wind variables specified */
      return;
   }

   /* Get U, V, W 2-D grids */
   grid = get_grid( ctx, time, uvar );
   ugrid = extract_hslice( ctx, grid, ctx->Nr, ctx->Nc, ctx->WindNl,
                           ctx->WindLow, level, 0 );
   release_grid( ctx, time, uvar, grid );

   grid = get_grid( ctx, time, vvar );
   vgrid = extract_hslice( ctx, grid, ctx->Nr, ctx->Nc, ctx->WindNl,
                           ctx->WindLow, level, 0 );
   release_grid( ctx, time, vvar, grid );

   if (wvar>-1) {
      grid = get_grid( ctx, time, wvar );
      wgrid = extract_hslice( ctx, grid, ctx->Nr, ctx->Nc, ctx->WindNl,
                              ctx->WindLow, level, 0 );
      release_grid( ctx, time, wvar, grid );
   }

   /* Compute distance in meters between grid points in each direction. */
   switch (ctx->Projection) {
      case PROJ_GENERIC:
         mpcol = ctx->ColInc;
	 mprow = ctx->RowInc;
	 break;
      case PROJ_LINEAR:
      case PROJ_CYLINDRICAL:
      case PROJ_SPHERICAL:
	 midlat = (ctx->NorthBound+ctx->SouthBound) / 2.0;
	 mpcol = (ctx->WestBound-ctx->EastBound) * 111120.0
		 * cos( DEG2RAD(midlat) ) / (float) (ctx->Nc-1);
	 mprow = (ctx->NorthBound-ctx->SouthBound) * 111120.0 / (float) (ctx->Nr-1);
	 break;
      case PROJ_ROTATED:
         /* WLH 4-21-95 */
	 midlat = (ctx->NorthBound+ctx->SouthBound) / 2.0;
	 midlon = (ctx->WestBound+ctx->EastBound) / 2.0;
/*
         pandg_back(&midlat, &midlon, ctx->CentralLat, ctx->CentralLon, ctx->Rotation);
*/
	 mpcol = (ctx->WestBound-ctx->EastBound) * 111120.0
		 * cos( DEG2RAD(midlat) ) / (float) (ctx->Nc-1);
	 mprow = (ctx->NorthBound-ctx->SouthBound) * 111120.0 / (float) (ctx->Nr-1);
	 break;
      case PROJ_LAMBERT:
	 mpcol = ctx->ColInc * 1000.0;
	 mprow = ctx->ColInc * 1000.0;  /* TODO: is this OK? */
	 break;
      case PROJ_STEREO:
	 mpcol = ctx->ColInc * 1000.0;
	 mprow = ctx->ColInc * 1000.0;  /* TODO: is this OK? */
	 break;
   }

   switch (ctx->VerticalSystem) {
      case VERT_GENERIC:
         mplev = ctx->LevInc;
	 break;
      case VERT_EQUAL_KM:
         mplev = ctx->LevInc * 1000.0;
	 break;
      case VERT_NONEQUAL_MB:
      case VERT_NONEQUAL_KM:
	 /* TODO: is this sufficient? */
	 if (ctx->MaxNl > 1) mplev = (ctx->TopBound-ctx->BottomBound) *
                                     1000.0 / (float) (ctx->MaxNl-1);
         else mplev = 0.0;
	 break;
	 /* TODO: What about ctx->LogFlag? */
   }


   /* Compute time interval: */
   deltatime = (ctx->Nc * mpcol) / 750.0;

   /* Compute scaling from <U,V,W> to <deltaC, deltaR, deltaL>: */
   colscale = scale * deltatime / mpcol;
   rowscale = scale * deltatime / mprow;
   levscale = scale * deltatime / mplev;

   /* Density: */
   if (density>1.0 || density<0.0)
     density = 1.0;
   drow = (int) (1.0 / density);
   dcol = (int) (1.0 / density);

   /* calculate vector vertices */
   vcount = 0;
   for (row=0; row<ctx->Nr && vcount+4<MAX_WIND_VERTS; row+=drow) {
      for (col=0; col<ctx->Nc && vcount+4<MAX_WIND_VERTS; col+=dcol) {
	 float u, v, w;
	 float a[3], dir[3], up[3], len, alen;
	 float pr, pc, pl;
	 float tr, tc, tl;

	 /* get <U,V,W> vector */
	 u = ugrid[row*ctx->Nc+col];
	 v = vgrid[row*ctx->Nc+col];
	 w = (wvar>-1) ? wgrid[row*ctx->Nc+col] : 0.0;

	 if (IS_MISSING(u) || IS_MISSING(v) || IS_MISSING(w)) {
	    /* if u, v, or w is missing, draw no vector */
	 }
	 else {
	    dir[0] = -v * rowscale;
	    dir[1] = u * colscale;
	    dir[2] = w * levscale;
	    len = MAGNITUDE( dir );    /* len's units are grid boxes */
	    if (len>=0.05) {   /* was 0.01 */
	       /* tail of arrow */
	       vr[vcount] = (float) row;
	       vc[vcount] = (float) col;
	       vl[vcount] = level;
/*	       vcount++;*/

	       /* head of arrow */
	       vr[vcount+1] = tr = (float) row + dir[0];
	       vc[vcount+1] = tc = (float) col + dir[1];
	       vl[vcount+1] = tl = level + dir[2];
/*	       vcount++;*/

	       /* compute arrow head lines */
	       if (dir[2]>len/2.0 || dir[2]<-len/2.0) {
		  up[0] = 0.0;  up[1] = 1.0;  up[2] = 0.0;
	       }
	       else {
		  up[0] = 0.0;  up[1] = 0.0;  up[2] = 1.0;
	       }

	       CROSS( a, dir, up );
	       alen = MAGNITUDE( a );
	       a[0] = a[0] / alen * len * 0.1;
	       a[1] = a[1] / alen * len * 0.1;
	       a[2] = a[2] / alen * len * 0.1;

	       pr = tr - dir[0] * 0.2;
	       pc = tc - dir[1] * 0.2;
	       pl = tl - dir[2] * 0.2;

	       vr[vcount+2] = pr + a[0];
	       vc[vcount+2] = pc + a[1];
	       vl[vcount+2] = pl + a[2];
/*	       vcount++;*/

	       vr[vcount+3] = pr - a[0];
	       vc[vcount+3] = pc - a[1];
	       vl[vcount+3] = pl - a[2];
/*	       vcount++;*/

	       vcount+=4;

/* Used to debug R8000 -O2/O3 bug:
	       if (vcount==12) {
		  int ii;
		  printf("\n");
		  for (ii=0;ii<12;ii++) {
		     printf("%g %g %g\n", vr[ii], vc[ii], vl[ii] );
		  }
		  printf("  %g %g %g  %g %g %g\n", pr, pc, pl, a[0], a[1], a[2] );
	       }
*/
	    }
	 }
      }
   }

   /* deallocate 2-D grids */
   deallocate( ctx, ugrid, -1 );
   deallocate( ctx, vgrid, -1 );
   if (wvar>-1)
     deallocate( ctx, wgrid, -1 );

   /*
    * Bounding rectangle
    */
   numboxverts = make_horizontal_rectangle( ctx, time, uvar, ctx->CurvedBox,
					    level, &boxverts );

   /************************ Compress ********************************/

   if (vcount>0) {
      int bytes = 3*vcount*sizeof(int_2);
      cverts = (int_2 *) allocate_type( ctx, bytes, WINDXH_TYPE );
      if (!cverts) {
         deallocate( ctx, cverts, bytes);
         cverts = NULL;
         vcount = 0;
      }
      else {
         grid_to_compXYZ( ctx, time, uvar, vcount, vr, vc, vl, (void*)cverts );
      }
   }
   else {
      vcount = 0;
      cverts = NULL;
   }

   /************************* Store wind slice ***********************/
   recent( ctx, HWIND, ws );

   wait_write_lock( &ctx->HWindTable[ws][time].lock );

   /* deallocate existing slice, if any */
   free_hwind( ctx, time, ws );

   /* store new slice */
   ctx->HWindTable[ws][time].uvar = ctx->Uvar[ws];
   ctx->HWindTable[ws][time].vvar = ctx->Vvar[ws];
   ctx->HWindTable[ws][time].wvar = ctx->Wvar[ws];
   ctx->HWindTable[ws][time].level = level;
   ctx->HWindTable[ws][time].density = density;
   ctx->HWindTable[ws][time].scale = scale;
   ctx->HWindTable[ws][time].nvectors = vcount / 4;  /* 4 vertices / vector */
   ctx->HWindTable[ws][time].verts = cverts;
   ctx->HWindTable[ws][time].boxverts = boxverts;
   ctx->HWindTable[ws][time].numboxverts = numboxverts;
   ctx->HWindTable[ws][time].valid = 1;

   done_write_lock( &ctx->HWindTable[ws][time].lock );

   if (time==ctx->CurTime) {
      ctx->Redraw = 1;
   }
}



/*
 * Compute vectors in a vertical wind slice.
 * Input:  time - which timestep
 *         ws - which wind slice [0,WINDSLICES-1]
 *         r1, c1 - row, column of left end of slice
 *         r2, c2 - row, column of right end of slice
 *         scale - user scaling factor  (1.0 is typical)
 *         density - user density factor  1.0, 0.5, 0.25, etc.
 *         threadnum - which thread
 */
static void calc_vwindslice( Context ctx, int time, int ws,
                             float r1, float c1, float r2, float c2,
                             float scale, float density,
                             int threadnum )
{
   float *grid, *ugrid, *vgrid, *wgrid;
   int row, col, rows, cols, drow;
   float vr[MAX_WIND_VERTS], vc[MAX_WIND_VERTS], vl[MAX_WIND_VERTS];
   float midlat, midlon, mprow, mpcol, mplev;
   int vcount;
   float deltatime;
   int_2 *cverts;
   int uvar, vvar, wvar;
   float dr, dc;
   float rowscale, colscale, levscale;
   int numboxverts;
   float *boxverts;
   float up[3];

   /* Determine which variables to use for U,V,W */
   uvar = ctx->Uvar[ws];
   vvar = ctx->Vvar[ws];
   wvar = ctx->Wvar[ws];

   if (uvar<0 || vvar<0) {
      /* no wind variables specified */
      return;
   }

   /* Density: */
   if (density>1.0 || density<0.0)
     density = 1.0;

   /* size of 2-D slice */
   rows = ctx->WindNl;
   cols = MAX(ctx->Nr,ctx->Nc) * density;

   /* get u, v, w grid slices */
   grid = get_grid( ctx, time, uvar );
   ugrid = extract_vslice( ctx, grid, r1,c1, r2,c2, rows, cols, 0 );
   release_grid( ctx, time, uvar, grid );

   grid = get_grid( ctx, time, vvar );
   vgrid = extract_vslice( ctx, grid, r1,c1, r2,c2, rows, cols, 0 );
   release_grid( ctx, time, vvar, grid );

   if (wvar>-1) {
      grid = get_grid( ctx, time, wvar );
      wgrid = extract_vslice( ctx, grid, r1,c1, r2,c2, rows, cols, 0 );
      release_grid( ctx, time, wvar, grid );
   }

   /* Compute distance in meters between grid points in each direction. */
   switch (ctx->Projection) {
      case PROJ_GENERIC:
         mpcol = ctx->ColInc;
	 mprow = ctx->RowInc;
	 break;
      case PROJ_LINEAR:
      case PROJ_CYLINDRICAL:
      case PROJ_SPHERICAL:
	 midlat = (ctx->NorthBound+ctx->SouthBound) / 2.0;
	 mpcol = (ctx->WestBound-ctx->EastBound) * 111120.0
		 * cos( DEG2RAD(midlat) ) / (float) (ctx->Nc-1);
	 mprow = (ctx->NorthBound-ctx->SouthBound) * 111120.0 / (float) (ctx->Nr-1);
	 break;
      case PROJ_ROTATED:
         /* WLH 4-21-95 */
	 midlat = (ctx->NorthBound+ctx->SouthBound) / 2.0;
	 midlon = (ctx->WestBound+ctx->EastBound) / 2.0;
         pandg_back(&midlat, &midlon, ctx->CentralLat, ctx->CentralLon, ctx->Rotation);
	 mpcol = (ctx->WestBound-ctx->EastBound) * 111120.0
		 * cos( DEG2RAD(midlat) ) / (float) (ctx->Nc-1);
	 mprow = (ctx->NorthBound-ctx->SouthBound) * 111120.0 / (float) (ctx->Nr-1);
	 break;
      case PROJ_LAMBERT:
	 mpcol = ctx->ColInc * 1000.0;
	 mprow = ctx->ColInc * 1000.0;  /* TODO: is this OK? */
	 break;
      case PROJ_STEREO:
	 mpcol = ctx->ColInc * 1000.0;
	 mprow = ctx->ColInc * 1000.0;  /* TODO: is this OK? */
	 break;
   }

   switch (ctx->VerticalSystem) {
      case VERT_GENERIC:
         mplev = ctx->LevInc;
	 break;
      case VERT_EQUAL_KM:
         mplev = ctx->LevInc * 1000.0;
	 break;
      case VERT_NONEQUAL_MB:
      case VERT_NONEQUAL_KM:
	 if (ctx->MaxNl > 1) mplev = (ctx->TopBound-ctx->BottomBound) * 1000.0 / (float) (ctx->MaxNl-1);
         else mplev = 0.0;
	 break;
         /* What about ctx->LogFlag? */
   }

   /* Compute time interval: */
   deltatime = (ctx->Nc * mpcol) / 750.0;

   /* Compute scaling from <U,V,W> to <deltaC, deltaR, deltaL>: */
   colscale = scale * deltatime / mpcol;
   rowscale = scale * deltatime / mprow;
   levscale = scale * deltatime / mplev;

   drow = (int) (1.0 / density);     /* in slice coords */
   dr = (r2-r1) / (float) (cols-1);  /* delta row and column in */
   dc = (c2-c1) / (float) (cols-1);  /* 3-d grid coords */

   /* "upward" vector */
   up[0] = dc;  up[1] = dr;  up[2] = 0.0;
   NORMALIZE( up );

   /* Compute vectors */
   vcount = 0;
   for (row=0; row<rows && vcount+4<MAX_WIND_VERTS; row+=drow) {
      float gr = (float) r1;
      float gc = (float) c1;
/* WLH 6-30-95
      float gl = (float) row;
*/
      float gl = (float) (row + ctx->WindLow);   /*(ctx->WindNl-row-1); this is wrong */
      for (col=0; col<cols && vcount+4<MAX_WIND_VERTS; col++) {
	 float u, v, w;
	 float dir[3];  /* wind vector direction */
	 float a[3];  /* up & across vectors */
	 float pr, pc, pl, len, alen;
	 float tr, tc, tl;

	 u = ugrid[row*cols+col];
	 v = vgrid[row*cols+col];
	 w = (wvar>-1) ? wgrid[row*cols+col] : 0.0;

	 if (IS_MISSING(u) || IS_MISSING(v) || IS_MISSING(w)) {
	    /* if u, v, or w is missing, draw no vector */
	 }
	 else {
	    dir[0] = -v * rowscale;
	    dir[1] = u * colscale;
	    dir[2] = w * levscale;
	    len = MAGNITUDE( dir );    /* len's units are grid boxes */
	    if (len>=0.01) {
	       /* tail of arrow */
	       vr[vcount] = gr;
	       vc[vcount] = gc;
	       vl[vcount] = gl;
	       vcount++;

	       /* head of arrow */
	       vr[vcount] = tr = gr + dir[0];
	       vc[vcount] = tc = gc + dir[1];
	       vl[vcount] = tl = gl + dir[2];
	       vcount++;

	       /* compute arrow head lines */
	       CROSS( a, dir, up );
	       alen = MAGNITUDE( a );
	       a[0] = a[0] / alen * len * 0.1;
	       a[1] = a[1] / alen * len * 0.1;
	       a[2] = a[2] / alen * len * 0.1;
	       pr = tr - dir[0] * 0.2;
	       pc = tc - dir[1] * 0.2;
	       pl = tl - dir[2] * 0.2;
	       vr[vcount] = pr + a[0];
	       vc[vcount] = pc + a[1];
	       vl[vcount] = pl + a[2];
	       vcount++;
	       vr[vcount] = pr - a[0];
	       vc[vcount] = pc - a[1];
	       vl[vcount] = pl - a[2];
	       vcount++;
	    }
         } /*if missing*/
	 gr += dr;
	 gc += dc;
      } /*for col*/
   } /* for row */

   /* deallocate 2-D slices */
   deallocate( ctx, ugrid, -1 );
   deallocate( ctx, vgrid, -1 );
   if (wvar>-1)
     deallocate( ctx, wgrid, -1 );

   /*
    * Bounding rectangle
    */
   numboxverts = make_vertical_rectangle( ctx, time, uvar, ctx->CurvedBox,
					  r1, c1, r2, c2, cols, &boxverts );

   /*************************** Compress *******************************/

   if (vcount>0) {
      int bytes = 3*vcount*sizeof(int_2);
      cverts = (int_2 *) allocate_type( ctx, bytes, WINDXV_TYPE );
      if (!cverts) {
         deallocate( ctx, cverts, bytes);
         cverts = NULL;
         vcount = 0;
      }
      else {
         grid_to_compXYZ( ctx, time, uvar, vcount, vr, vc, vl, (void*)cverts );
      }
   }
   else {
      vcount = 0;
      cverts= NULL;
   }

   /************************ Store wind slice ***************************/
   recent( ctx, VWIND, ws );

   wait_write_lock( &ctx->VWindTable[ws][time].lock );

   /* deallocate existing slice, if any */
   free_vwind( ctx, time, ws );

   /* store new slice */
   ctx->VWindTable[ws][time].uvar = ctx->Uvar[ws];
   ctx->VWindTable[ws][time].vvar = ctx->Vvar[ws];
   ctx->VWindTable[ws][time].wvar = ctx->Wvar[ws];
   ctx->VWindTable[ws][time].r1 = r1;
   ctx->VWindTable[ws][time].c1 = c1;
   ctx->VWindTable[ws][time].r2 = r2;
   ctx->VWindTable[ws][time].c2 = c2;
   ctx->VWindTable[ws][time].density = density;
   ctx->VWindTable[ws][time].scale = scale;
   ctx->VWindTable[ws][time].nvectors = vcount / 4;   /* 4 vertices / vector */
   ctx->VWindTable[ws][time].verts = cverts;
   ctx->VWindTable[ws][time].numboxverts = numboxverts;
   ctx->VWindTable[ws][time].boxverts = boxverts;
   ctx->VWindTable[ws][time].valid = 1;

   done_write_lock( &ctx->VWindTable[ws][time].lock );

   if (time==ctx->CurTime) {
      ctx->Redraw = 1;
   }
}





/*
 * Compute streamlines on a horizontal slice.
 * Input:  time - which timestep
 *         ws - which wind slice [0,WINDSLICES-1]
 *         level - grid level of slice in [0,Nl-1]
 *         density - user density factor  1.0, 0.5, 0.25, etc.
 *         threadnum - which thread
 */
static void calc_streamslice( Context ctx, int time, int ws,
                             float level, float density, int threadnum )
{
   float *grid, *ugrid, *vgrid;
   float mpcol, mprow, mplev;   /* mp = meters per */
   float colscale, rowscale, levscale;
   float midlat, midlon;
   int i, row, col, drow, dcol, num;
   float deltatime;
   float vr[MAX_WIND_VERTS], vc[MAX_WIND_VERTS], vl[MAX_WIND_VERTS];
   int_2 *cverts;
   int uvar, vvar;
   float *boxverts;
   int numboxverts;

   /* Determine which variables to use for U,V,W */
   uvar = ctx->Uvar[ws];
   vvar = ctx->Vvar[ws];

   if (uvar<0 || vvar<0) {
      /* no wind variables specified */
      return;
   }

   /* Get U, V 2-D grids */
   grid = get_grid( ctx, time, uvar );
   ugrid = extract_hslice( ctx, grid, ctx->Nr, ctx->Nc, ctx->WindNl,
                           ctx->WindLow, level, 0 );
   release_grid( ctx, time, uvar, grid );

   grid = get_grid( ctx, time, vvar );
   vgrid = extract_hslice( ctx, grid, ctx->Nr, ctx->Nc, ctx->WindNl,
                           ctx->WindLow, level, 0 );
   release_grid( ctx, time, vvar, grid );


   /* call contouring routine */
   stream( ctx, ugrid, vgrid, ctx->Nr, ctx->Nc, density,
            vr, vc, MAX_WIND_VERTS, &num);

   for(i=0; i<num; i++) vl[i] = level;

   /* deallocate 2-D grids */
   deallocate( ctx, ugrid, -1 );
   deallocate( ctx, vgrid, -1 );

   /*
    * Bounding rectangle
    */
   numboxverts = make_horizontal_rectangle( ctx, time, uvar, ctx->CurvedBox,
					    level, &boxverts );

   /************************ Compress ********************************/
   /*
    * Transform vertices from grid coordinates in [0,Nr][0,Nc][0,Nl] to
    * compressed graphics coordinates in [-10000,10000]^3.
    */

   if (num > 0) {
      int bytes = 3 * num * sizeof(int_2);
      cverts = (int_2 *) allocate_type( ctx, bytes, STREAM1_TYPE );
      if (cverts) {
         grid_to_compXYZ( ctx, time, uvar, num, vr, vc, vl, (void*)cverts );
      }
      else {
         deallocate( ctx, cverts, bytes);
         num = 0;
         cverts = NULL;
      }
   }
   else {
      num = 0;
      cverts = NULL;
   }

   /************************* Store wind slice ***********************/
   recent( ctx, STREAM, ws );

   wait_write_lock( &ctx->StreamTable[ws][time].lock );

   /* deallocate existing slice, if any */
   free_stream( ctx, time, ws );

   /* store new slice */
   ctx->StreamTable[ws][time].uvar = ctx->Uvar[ws];
   ctx->StreamTable[ws][time].vvar = ctx->Vvar[ws];
   ctx->StreamTable[ws][time].level = level;
   ctx->StreamTable[ws][time].density = density;
   ctx->StreamTable[ws][time].nlines = num;
   ctx->StreamTable[ws][time].verts = cverts;
   ctx->StreamTable[ws][time].boxverts = boxverts;
   ctx->StreamTable[ws][time].numboxverts = numboxverts;
   ctx->StreamTable[ws][time].valid = 1;

   done_write_lock( &ctx->StreamTable[ws][time].lock );

   if (time==ctx->CurTime) {
      ctx->Redraw = 1;
   }
}




/*
 * Compute the color table indexes for a colored trajectory.
 * Input:  ctx - the context
 *         t - the trajectory struct
 *         colorvar - the coloring variable
 */
static void color_traj( Context ctx, struct traj *t, int colorvar )
{
   uint_1 *color_indexes;
   int i, n;
   float vscale = 1.0 / VERTEX_SCALE;
   float min, valscale;
   int time;


   /* Free old color indexes if any */
   wait_write_lock( &t->lock );
   if (t->colors) {
      deallocate( ctx, t->colors, t->length*sizeof(uint_1) );
   }
   t->colors = NULL;
   t->colorvar = -1;
   done_write_lock( &t->lock );

   if (colorvar!=-1) {
      /* Allocate storage for new color indexes */
      n = t->length;
      color_indexes = allocate( ctx, n*sizeof(uint_1) );
      if (!color_indexes) {
         return;
      }

      min = ctx->MinVal[colorvar];
      valscale = 1.0F / (ctx->MaxVal[colorvar] - ctx->MinVal[colorvar]);

      /* Compute color indexes */
      time = 0;
      for (i=0;i<n;i++) {
         float x, y, z;
         float row, col, lev;
         float val;
         int index;

         x = t->verts[i*3+0] * vscale;
         y = t->verts[i*3+1] * vscale;
         z = t->verts[i*3+2] * vscale;

         xyz_to_grid( ctx, 0, ctx->TrajU, x, y, z, &row, &col, &lev );

         /* find the timestep which corresponds to this traj pos */
         while (t->start[time]<i && time<ctx->NumTimes-1) {
            time++;
         }

         val = interpolate_grid_value( ctx, time, colorvar, row, col, lev );

         if (IS_MISSING(val)) {
            color_indexes[i] = 255;
         }
         else {
            int index = (val - min) * valscale * 254.0F;
            color_indexes[i] = (uint_1) index;
         }
      }
   }
   else {
      color_indexes = NULL;
   }

   /* save results */
   wait_write_lock( &t->lock );
   t->colors = color_indexes;
   t->colorvar = colorvar;
   done_write_lock( &t->lock );
}



/*
 * Recolor a set of trajectores to the variable specified by
 * TrajColorVar[traj_set].
 */
static void recolor_traj_set( Context ctx, int traj_set )
{
   int i;

   for (i=0;i<ctx->NumTraj;i++) {
      struct traj *t = ctx->TrajTable[i];
      if (t->group==traj_set) {
         if (t->colorvar != ctx->TrajColorVar[traj_set]) {
            color_traj( ctx, t, ctx->TrajColorVar[traj_set] );
         }
      }
   }
}



/*
 * Compute a wind trajectory.
 * Input:  row, col, lev - start position in grid coords.
 *         time - start time in [0..NumTimes-1].
 *         id - trajectory id (group) number
 *         ribbon - 1 = make ribbon traj, 0 = make line segment traj
 *         step_mult - integration step size multiplier (default 1.0)
 *         len_mult - trajectory length multiplier (default 1.0)
 *  Output:  list of coordinates and times are saved in TrajTable.
 */
static void calc_traj( Context ctx, float row, float col, float lev,
                       int time, int id, int ribbon,
                       float step_mult, float len_mult,
                       int colorvar )
{
   int len, i;
   float vr[MAX_TRAJ_VERTS], vc[MAX_TRAJ_VERTS], vl[MAX_TRAJ_VERTS];
   float vx[MAX_TRAJ_VERTS], vy[MAX_TRAJ_VERTS], vz[MAX_TRAJ_VERTS];
   float nx[MAX_TRAJ_VERTS], ny[MAX_TRAJ_VERTS], nz[MAX_TRAJ_VERTS];
   int tt[MAX_TRAJ_VERTS];  /* traj times */
   int vbytes, nbytes;
   int_2 *cverts;
   int_1 *cnorms;
   struct traj *t;


   if (ctx->NumTraj>=MAXTRAJ) {
      /* out of trajectory space */
      return;
   }

   if (Debug)
      printf("calc_traj( %f %f %f %d %d )\n", row, col, lev, time, id );
   len = trace( ctx, row, col, lev, time, (int) (step_mult * ctx->TrajStep),
	       MAX_TRAJ_VERTS, vr, vc, vl, tt );

   if (len==0)
      return;

   /* convert coords from grid to graphics */
   grid_to_xyz( ctx, time, ctx->Uvar[0], len, vr, vc, vl,  vx, vy, vz );

   if (ribbon) {
      /* convert to ribbon, calc normals */
      len = to_ribbon( len, vx,vy,vz, tt,  nx,ny,nz );
   }
   if (len==0)
      return;

   /****************************** Compress ***************************/
   vbytes = 3 * len * sizeof(int_2);
   cverts = (int_2 *) allocate_type( ctx, vbytes, TRAJX_TYPE );
   if (cverts) {
      /* convert floats to int_2 */
      for (i=0;i<len;i++) {
         cverts[i*3+0] = (int_2) (vx[i] * VERTEX_SCALE);
         cverts[i*3+1] = (int_2) (vy[i] * VERTEX_SCALE);
         cverts[i*3+2] = (int_2) (vz[i] * VERTEX_SCALE);
      }
   }
   else {
      len = 0;
   }

   if (ribbon & len>0) {
      nbytes = 3 * len * sizeof(int_1);
      cnorms = (int_1 *) allocate_type( ctx, nbytes, TRAJXR_TYPE );
      if (!cnorms) {
         deallocate( ctx, cverts, vbytes );
         cverts = NULL;
         len = 0;
      }
      else {
         /* compress normals to 1-byte ints */
         for (i=0;i<len;i++) {
            cnorms[i*3+0] = (int_1) (int) (-nx[i] * NORMAL_SCALE);
            cnorms[i*3+1] = (int_1) (int) ( ny[i] * NORMAL_SCALE);
            cnorms[i*3+2] = (int_1) (int) ( nz[i] * NORMAL_SCALE);
         }
      }
   }
   else {
      cnorms = NULL;
   }

   /***************************** Store ******************************/

   t = allocate( ctx, sizeof(struct traj) );
   if (!t) {
      return;
   }

   t->lock = 0;
   t->row = row;
   t->col = col;
   t->lev = lev;
   t->timestep = time;
   t->stepmult = step_mult;
   t->lengthmult = len_mult;
   t->length = len;
   t->verts = cverts;
   t->norms = cnorms;
   t->start = (uint_2 *) allocate_type( ctx,
                                 ctx->NumTimes * sizeof(uint_2), START_TYPE );
   t->len = (uint_2 *) allocate_type( ctx,
                                 ctx->NumTimes * sizeof(uint_2), LEN_TYPE );

   /* calculate start and len array values */
   if (len>0) {
      int tlen = (int) (len_mult * ctx->TrajLength);
      for (i=0;i<ctx->NumTimes;i++) {
         int t0, t1, j, startj;

         t0 = ctx->Elapsed[i] - tlen;
         t1 = ctx->Elapsed[i];

         /* find segment of traj extending in time from [t0,t1] */
         j = 0;
         while (tt[j]<t0 && j<len)
           j++;
         if (j>=len) {
            t->start[i] = 0;
            t->len[i] = 0;
         }
         else {
            t->start[i] = startj = j;
            while (tt[j]<=t1 && j<len)
              j++;
            if (j-startj<=1)
              t->len[i] = 0;
            else
              t->len[i] = j - startj;
         }
      }
   }

   t->group = id;
   t->kind = ribbon;
   t->colors = NULL;
   if (colorvar>=0) {
      color_traj( ctx, t, colorvar );
   }
   else {
      t->colorvar = -1;
   }
   t->lock = 0;

   LOCK_ON( TrajLock );
   ctx->TrajTable[ctx->NumTraj] = t;
   ctx->NumTraj++;
   LOCK_OFF( TrajLock );

   recent( ctx, TRAJ, id );

   ctx->Redraw = 2;
}







/*
 * Call once this to initialize this module.
 */
int init_work( void )
{
   return 0;
}


/*
 * Call this when ready to exit vis5d.
 */
int terminate_work( void )
{
   return 0;
}



/*
 * Return 1 if there are any background tasks pending, else return 0.
 */
int any_work_pending( void )
{
   int size, waiters;

   get_queue_info( &size, &waiters );
   if (size==0 && waiters==NumThreads-1) {
      return 0;
   }
   else {
      return 1;
   }
}



/*
 * Take one job off the queue and do it.
 * Input:  threadnum - thread number
 * Return:  1 = one job done, 0 = time to exit
 */
int do_one_task( int threadnum )
{
   int type;
   int i1, i2, i3;
   float f1, f2, f3, f4, f5;
   int time, var, colorvar;
   Context ctx;

   get_qentry( &ctx, &type, &i1, &i2, &i3, &f1, &f2, &f3, &f4, &f5 );

   if (Debug) printf("got entry: %d\n", type );
   time = i1;
   var = i2;
   if (Debug)  printf("\ntask type=%d\n", type );
   switch (type) {
      case TASK_ISOSURFACE:
         /* compute an isosurface. */
         colorvar = i3;
         calc_isosurface( ctx, time, var, ctx->IsoLevel[var],
                          ctx->IsoColorVar[var], threadnum );
	 break;
      case TASK_HSLICE:
	 /* calculate a horizontal contour line slice. */
	 calc_hslice( ctx, time, var, ctx->HSliceInterval[var],
		      ctx->HSliceLowLimit[var], ctx->HSliceHighLimit[var],
		      ctx->HSliceLevel[var], threadnum );
	 break;
      case TASK_VSLICE:
	 /* calculate a horizontal contour line slice. */
	 calc_vslice( ctx, time, var, ctx->VSliceInterval[var],
		      ctx->VSliceLowLimit[var], ctx->VSliceHighLimit[var],
		      ctx->VSliceR1[var], ctx->VSliceC1[var],
		      ctx->VSliceR2[var], ctx->VSliceC2[var], threadnum );
	 break;
      case TASK_CHSLICE:
	 /* calculate a horizontal colored slice */
	 calc_chslice( ctx, time, var, ctx->CHSliceLevel[var], threadnum );
	 break;
      case TASK_CVSLICE:
	 /* calculate a vertical colored slice */
	 calc_cvslice( ctx, time, var, ctx->CVSliceR1[var], ctx->CVSliceC1[var],
		      ctx->CVSliceR2[var], ctx->CVSliceC2[var], threadnum );
	 break;
      case TASK_HWIND:
	 /* calculate a horizontal wind slice */
	 calc_hwindslice( ctx, time, var, ctx->HWindLevel[var],
			 ctx->HWindScale[var], ctx->HWindDensity[var], threadnum );
	 break;
      case TASK_VWIND:
	 /* calculate a vertical wind slice */
	 calc_vwindslice( ctx, time, var, ctx->VWindR1[var], ctx->VWindC1[var],
			 ctx->VWindR2[var], ctx->VWindC2[var],
			 ctx->VWindScale[var], ctx->VWindDensity[var], threadnum );
	 break;
      case TASK_STREAM:
	 /* calculate a horizontal streamline slice */
	 calc_streamslice( ctx, time, var, ctx->StreamLevel[var],
			   ctx->StreamDensity[var], threadnum );
	 break;
      case TASK_TRAJ:
	 /* calculate a wind trajectory */
	 calc_traj( ctx, f1, f2, f3, time, i2, i3, f4, f5,
                    ctx->TrajColorVar[i2] );
	 break;
      case TASK_TRAJ_RECOLOR:
         recolor_traj_set( ctx, i1 );
         break;
      case TASK_EXT_FUNC:
	 calc_ext_func( ctx, time, var, threadnum );
	 break;
      case TASK_QUIT:
	 if (Debug) {
	    printf("TASK_QUIT\n");
	    return 0;
	 }
	 break;
      case TASK_NULL:
	 break;
      default:
	 printf("VIS-5D INTERNAL ERROR:  Undefined task code!!\n");
   } /*switch*/

   return 1;
}



/*
 * Multiple instances of this function execute in parallel on the SGI
 * and Stellar.  The function takes entries out of the queue and calls
 * the appropriate function to do the work of computing isosurfaces,
 * contour slices, trajectories, etc.
 * Input:  threadnum - integer thread ID in [1..NumThreads-1]
 */
void work( void *threadnum )
{
#ifdef sgi
   /* make this process die if the parent dies */
   prctl( PR_TERMCHILD );
#endif

   while (do_one_task( (int) threadnum ))
     ;
}
