/* $Id: script.c,v 1.32 1996/04/09 17:58:12 brianp Exp $ */

/* Vis5D version 4.2 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1996 Bill Hibbard, Brian Paul, Dave Santek,
and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




/* Tcl script interface for Vis5D */



#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/resource.h>
#include "api.h"
#include "globals.h"
#include "graphics.h"
#include "gui.h"
#include "tclsave.h"



#define ABS(X)  ( (X) < 0 ? -(X) : (X) )


#ifdef TCL
#  include <tcl.h>
#else

/*
 * If Tcl isn't available we use the following replacement functions which
 * emulate Tcl's most basic functions.  With these functions we can at
 * least parse and execute simple Vis5D "Tcl save" files.
 */

#define MAX_COMMANDS 200
#define MAX_ARGS     100
#define MAX_ARG_LEN  100
#define MAX_VARS     100

#define TCL_OK        0
#define TCL_ERROR     1
#define TCL_RETURN    2
#define TCL_BREAK     3
#define TCL_CONTINUE  4


typedef void *ClientData;


typedef struct {
   char            *result;      /* result string */
   struct private  *p;           /* private interpreter data */
} Tcl_Interp;


typedef int (Tcl_CmdProc) (ClientData clientData,
                           Tcl_Interp *interp,
                           int argc, char *argv[] );

typedef int *Tcl_Command;

typedef void (Tcl_CmdDeleteProc) (ClientData clientData);


struct private {
   int         num_cmd;                        /* number of commands */
   char        cmd_name[MAX_COMMANDS][100];    /* command names */
   Tcl_CmdProc *cmd_func[MAX_COMMANDS];        /* command C func pointers */
   ClientData  client_data[MAX_COMMANDS];      /* client's data */

   int         num_vars;                       /* number of variables */
   char        var_name[MAX_VARS][100];        /* names of variables */
   char        var_value[MAX_VARS][100];       /* values of variables */
};


static int Tcl_EvalFile( Tcl_Interp *interp, char *filename );


/*
 * Assign a value to a variable.
 */
static void assign_var( Tcl_Interp *interp, char *varname, char *value )
{
   int i;

   for (i=0;i<interp->p->num_vars;i++) {
      if (strcmp(interp->p->var_name[i], varname)==0) {
         /* var already defined, replace its value */
         strcpy( interp->p->var_value[i], value );
         return;
      }
   }
   /* new variable */
   i = interp->p->num_vars;
   strcpy( interp->p->var_name[i], varname );
   strcpy( interp->p->var_value[i], value );
   interp->p->num_vars++;
}



/*
 * Replace a variable with its value.  Note that varname and result may
 * point to the same dataspace.
 */
static int eval_var( Tcl_Interp *interp, char *varname, char *result )
{
   int i;

   for (i=0;i<interp->p->num_vars;i++) {
      if (strcmp(interp->p->var_name[i],varname)==0) {
         strcpy( result, interp->p->var_value[i] );
         return 1;
      }
   }
   /* var not found */
   return 0;
}



/*
 * Implements Tcl's set command.
 */
static int cmd_set( ClientData client_data, Tcl_Interp *interp,
                    int argc, char *argv[] )
{
   if (argc!=3) {
      sprintf( interp->result, "Error: set requires 2 arguments" );
      return TCL_ERROR;
   }
   assign_var( interp, argv[1], argv[2] );
   return TCL_OK;
}



/*
 * Implements Tcl's source command.
 */
static int cmd_source( ClientData client_data, Tcl_Interp *interp,
                       int argc, char *argv[] )
{
   if (argc!=2) {
      sprintf( interp->result, "Error: source requires a filename argument" );
      return TCL_ERROR;
   }
   return Tcl_EvalFile( interp, argv[1] );
}



/*
 * Create an interpreter
 */
static Tcl_Interp *Tcl_CreateInterp( void )
{
   Tcl_Interp *interp;
   interp = (Tcl_Interp *) calloc( 1, sizeof(Tcl_Interp) );
   if (interp) {
      interp->p = (struct private *) calloc( 1, sizeof(struct private) );
      /* register the set command */
      strcpy( interp->p->cmd_name[0], "set" );
      interp->p->cmd_func[0] = cmd_set;
      /* register the sourc command */
      strcpy( interp->p->cmd_name[1], "source" );
      interp->p->cmd_func[1] = cmd_source;
      interp->p->num_cmd = 2;
   }
   return interp;
}


/*
 * Delete an interpreter.
 */
static void Tcl_DeleteInterp( Tcl_Interp *interp )
{
   free( interp->p );
   free( interp );
}


/*
 * Evaluate a Tcl/Vis5D command.
 */
static int Tcl_Eval( Tcl_Interp *interp, char *cmd )
{
   int argc;
   char *argv[MAX_ARGS];
   char arg[MAX_ARGS][MAX_ARG_LEN];
   char error[1000];
   int i;
   char *cp;
   int inquote;  /* inside a quoted string? */
   int inlist;   /* inside a {...} list? */

   /* Reset results string */
   interp->result = NULL;

   if (cmd[0]=='#') {
      /* comment */
      return TCL_OK;
   }

   /* break cmd string into an argv list */

   cp = cmd;
   argc = 0;
   i = 0;
   inquote = 0;
   inlist = 0;
   while (*cp) {
      if (*cp=='\"') {
         if (!inquote) {
            /* just skip the opening quote */
            inquote = 1;
         }
         else {
            /* end of quoted string */
            if (i>0) {
               /* finish token */
               arg[argc][i] = 0;
               argc++;
               i = 0;
            }
            inquote = 0;
         }
      }
      else if (*cp=='{' && !inquote) {
         /* begining of a list */
         inlist = 1;
         /* ignore '{' char */
      }
      else if (*cp=='}' && !inquote) {
         /* end of list */
         if (i>0) {
            /* finish token */
            arg[argc][i] = 0;
            argc++;
            i = 0;
         }
         inlist = 0;
      }
      else if ((*cp==' ' || *cp=='\t' || *cp=='\n') && !inquote && !inlist) {
         if (i>0) {
            /* end of a token */
            arg[argc][i] = 0;
            argc++;
            i = 0;
         }
      }
      else {
         /* add char to current token */
         arg[argc][i++] = *cp;
      }
      cp++;
   }
   /* finish last token */
   if (i>0) {
      arg[argc][i] = 0;
      argc++;
   }

   if (argc==0) {
      /* no arguments is OK */
      return TCL_OK;
   }

   /* setup pointers */
   for (i=0;i<argc;i++) {
      argv[i] = arg[i];
   }

   /* Perform variable substitution for args with $ prefix */
   for (i=0;i<argc;i++) {
      if (arg[i][0]=='$') {
         if (!eval_var( interp, arg[i]+1, arg[i] )) {
            printf("can't read \"%s\": no such variable\n", arg[i]+1 );
            return TCL_ERROR;
         }
      }
   }

   /* Now search for function which matches arg[0] */
   for (i=0;i<interp->p->num_cmd;i++) {
      if (strcmp(interp->p->cmd_name[i],arg[0])==0) {
         /* call the user-function */
         int code;
         code = (*interp->p->cmd_func[i])
                           ( interp->p->client_data[i], interp, argc, argv );
         return code;
      }
   }

   /* command not found! */
   sprintf(error,"invalid command name \"%s\"", arg[0]);
   interp->result = strdup(error);  /*TODO: is this right?*/
   return TCL_ERROR;
}



/*
 * Evaluate Tcl/Vis5D commands from a text file.
 */
static int Tcl_EvalFile( Tcl_Interp *interp, char *filename )
{
#define MAXLINE 1000
   FILE *f;
   char line[MAXLINE];
   int len, code;

   f = fopen( filename, "r" );
   if (!f) {
      interp->result = NULL;
      return TCL_ERROR;
   }

   while (!feof(f)) {
      fgets( line, MAXLINE, f );
      if (!feof(f)) {
         len = strlen(line);
         if (len>0) {
            /* if length > 0 then remove trailing newline and eval it */
            if (line[len-1]=='\n') {
               line[len-1] = 0;
            }
            code = Tcl_Eval( interp, line );
            if (code!=TCL_OK) {
               fclose(f);
               return code;
            }
         }
      }
   }

   fclose(f);
   return TCL_OK;
}


/*
 * Add a new Tcl command.
 */
static Tcl_Command Tcl_CreateCommand( Tcl_Interp *interp,
                                      char *cmdName,
                                      Tcl_CmdProc *proc,
                                      ClientData clientData,
                                      Tcl_CmdDeleteProc *deleteProc )
{
   if (interp->p->num_cmd<MAX_COMMANDS) {
      strcpy( interp->p->cmd_name[interp->p->num_cmd], cmdName );
      interp->p->cmd_func[interp->p->num_cmd] = proc;
      interp->p->num_cmd++;
   }
   else {
      printf("Fatal error in Tcl_CreateCommand!\n");
      abort();
   }
   return (Tcl_Command) proc;
}



#endif /*TCL*/



#define MAXPROJARGS 100



/*
 * Given a string which contains numbers separated by white space return
 * an array of those values as floats.
 * Input:  str - the input string
 *         max - size of x[] array
 * Output:  x - array of values
 * Return:  number of values found
 */
static int string_to_float_array( char *str, int max, float x[] )
{
   char buffer[100];
   int i, j, n;

   j = 0;
   n = 0;
   for (i=0;str[i];i++) {
      if (!isspace(str[i])) {
         buffer[j++] = str[i];
      }
      else {
         /* found white space */
         if (j>0) {
            buffer[j] = 0;
            x[n] = atof(buffer);
            n++;
            j = 0;
         }
      }
   }
   if (j>0) {
      buffer[j] = 0;
      x[n] = atof(buffer);
      n++;
   }
   return n;
}



/*
 * Like above function but return an array of integers.
 */
static int string_to_int_array( char *str, int max, int x[] )
{
   char buffer[100];
   int i, j, n;

   j = 0;
   n = 0;
   for (i=0;str[i];i++) {
      if (!isspace(str[i])) {
         buffer[j++] = str[i];
      }
      else {
         /* found white space */
         if (j>0) {
            buffer[j] = 0;
            x[n] = atoi(buffer);
            n++;
            j = 0;
         }
      }
   }
   if (j>0) {
      buffer[j] = 0;
      x[n] = atoi(buffer);
      n++;
   }
   return n;
}




/*
 * Return the variable number for s where s is either a variable name
 * or the number itself.  Return -1 if s is an unknown variable name.
 */
static int varnum( int index, char *s )
{
   if (isdigit(s[0])) {
      /* s is a number */
      return atoi( s );
   }
   else {
      /* s is a string */
      int var, numvars;
      vis5d_get_numvars( index, &numvars );
      for (var=0;var<numvars;var++) {
         char name[20];
         vis5d_get_var_name( index, var, name );
         if (strcmp(name,s)==0) {
            return var;
         }
      }
      return -1;
   }
}



/*
 * Check that the number of arguments is acceptable.
 * Input:  intepr - the Tcl interpreter
 *         func - name of calling function
 *         argc - number of arguments received, INCLUDING FUNCTION NAME!
 *         min - min number of args expected
 *         max - max number of args expected (1000+ == infinity)
 * Return:  1 = ok, 0 = wrong number of args.
 */
static int arg_check( Tcl_Interp *interp, char *func,
                      int argc, int min, int max )
{
   argc--;  /* skip the function name */

   if (argc>=min && argc<=max) {
      /* OK */
      return 1;
   }
   else if (min==max && min==1) {
      sprintf( interp->result, "Error in %s: one argument expected", func );
      return 0;
   }
   else if (min==max) {
      sprintf( interp->result, "Error in %s: %d arguments expected",
               func, min );
      return 0;
   }
   else if (min==max-1) {
      sprintf( interp->result, "Error in %s: %d or %d arguments expected",
               func, min, max );
      return 0;
   }
   else if (max<1000) {
      sprintf( interp->result, "Error in %s: from %d to %d arguments expected",
               func, min, max );
      return 0;
   }
   else {
      sprintf( interp->result,"Error in %s: at least %d arguments expected",
               func, min );
      return 0;
   }
}


/*
 * If errno is non-zero put an error message into the intepreter's
 * result string and return TCL_ERROR, else return TCL_OK.
 */
static int error_check( Tcl_Interp *interp, char *func, int errno )
{
   switch (errno) {
      case VIS5D_BAD_CONTEXT:
         sprintf( interp->result, "Error in %s: bad context", func );
         break;
      case VIS5D_BAD_CONSTANT:
         sprintf( interp->result, "Error in %s: undefined constant", func );
         break;
      case VIS5D_BAD_MODE:
         sprintf( interp->result, "Error in %s: bad mode parameter", func );
         break;
      case VIS5D_BAD_VALUE:
         sprintf( interp->result, "Error in %s: bad parameter value", func );
         break;
      case VIS5D_BAD_VAR_NUMBER:
         sprintf( interp->result, "Error in %s: invalid variable number",func);
         break;
      case VIS5D_BAD_TIME_STEP:
         sprintf( interp->result, "Error in %s: invalid timestep number",func);
         break;
      case VIS5D_FAIL:
         sprintf( interp->result, "Error in %s: function failed", func );
         break;
      case VIS5D_OUT_OF_MEMORY:
         sprintf( interp->result, "Error in %s: out of memory", func );
         break;
      default:
         return TCL_OK;
   }
   return TCL_ERROR;
}


/*** Initialization Functions ***/


static int cmd_initialize( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int result;
   if (argc==1) {
      result = vis5d_initialize( 0 );
   }
   else {
      result = vis5d_initialize( atoi( argv[1] ) );
   }
   return error_check( interp, "vis5d_initialize", result );
}


static int cmd_terminate( ClientData client_data, Tcl_Interp *interp,
                          int argc, char *argv[] )
{
   int result;
   result = vis5d_terminate( 1 );
   return error_check( interp, "vis5d_terminate", result );
}


static int cmd_workers( ClientData client_data, Tcl_Interp *interp,
                        int argc, char *argv[] )
{
   if (!arg_check( interp, "vis5d_workers", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   else {
      vis5d_workers( atoi( argv[1] ) );
      return TCL_OK;
   }
}


static int cmd_do_work( ClientData client_data, Tcl_Interp *interp,
                        int argc, char *argv[] )
{
   if (!arg_check( interp, "vis5d_do_work", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   vis5d_do_work();
   return TCL_OK;
}




/*** Context Initialization Functions ***/


static int cmd_init_begin( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_begin", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_begin( atoi(argv[1]) );
   return error_check( interp, "vis5d_init_begin", result );
}


static int cmd_init_end( ClientData client_data, Tcl_Interp *interp,
                         int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_end", argc, 1,1 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_end( atoi(argv[1]) );
   return error_check( interp, "vis5d_init_begin", result );
}



static int cmd_open_gridfile( ClientData client_data, Tcl_Interp *interp,
                              int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_open_gridfile", argc, 3, 3 )) {
      return TCL_ERROR;
   }
   result = vis5d_open_gridfile( atoi(argv[1]), argv[2], atoi(argv[3]) );
   return error_check( interp, "vis5d_open_gridfile", result );
}


static int cmd_init_window( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   if (!arg_check( interp, "vis5d_init_window", argc, 6, 6 )) {
      return TCL_ERROR;
   }
   else {
      int index = atoi(argv[1]);
      char *title = argv[2];
      int x = atoi(argv[3]);
      int y = atoi(argv[4]);
      int width = atoi(argv[5]);
      int height = atoi(argv[6]);
      int result;
      result = vis5d_init_window( index, title, x, y, width, height );
      return error_check( interp, "vis5d_init_window", result );
   }
}


static int cmd_init_map( ClientData client_data, Tcl_Interp *interp,
                          int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_map", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_map( atoi(argv[1]), argv[2] );
   return error_check( interp, "vis5d_init_map", result );
}


static int cmd_init_topo( ClientData client_data, Tcl_Interp *interp,
                          int argc, char *argv[] )
{
   int result;
   int highres = 0;
   if (!arg_check( interp, "vis5d_init_topo", argc, 2, 3 )) {
      return TCL_ERROR;
   }

   if (argc==4) {
      highres = atoi(argv[3]);
   }
   result = vis5d_init_topo( atoi(argv[1]), argv[2], highres );
   return error_check( interp, "vis5d_init_topo", result );
}


static int cmd_init_texture( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_texture", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_texture( atoi(argv[1]), argv[2] );
   return error_check( interp, "vis5d_init_texture", result );
}


static int cmd_init_firstarea( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_firstarea", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_firstarea( atoi(argv[1]), atoi(argv[2]) );
   return error_check( interp, "vis5d_init_firstarea", result );
}


static int cmd_init_sequence( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_sequence", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_sequence( atoi(argv[1]), argv[2] );
   return error_check( interp, "vis5d_init_sequence", result );
}


static int cmd_init_log( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_log", argc, 3, 3 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_log( atoi(argv[1]), atof(argv[2]), atof(argv[3]) );
   return error_check( interp, "vis5d_init_log", result );
}


static int cmd_init_box( ClientData client_data, Tcl_Interp *interp,
                         int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_box", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_box( atoi(argv[1]), atof(argv[2]),
                            atof(argv[3]), atof(argv[4]) );
   return error_check( interp, "vis5d_init_box", result );
}


static int cmd_init_memory( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_memory", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_memory( atoi(argv[1]), atoi(argv[2]) );
   return error_check( interp, "vis5d_init_memory", result );
}


static int cmd_init_path( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_path", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_init_path( atoi(argv[1]), argv[2] );
   return error_check( interp, "vis5d_init_path", result );
}


static int cmd_init_projection( ClientData client_data, Tcl_Interp *interp,
                                int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_projection", argc, 2, 1000 )) {
      return TCL_ERROR;
   }
   if (argc>3) {
      float projargs[MAXPROJARGS];
      int i;
      for (i=3;i<argc;i++) {
         projargs[i-3] = atof( argv[i] );
      }
      result = vis5d_init_projection( atoi(argv[1]), atoi(argv[2]), projargs );
   }
   else {
      result = vis5d_init_projection( atoi(argv[1]), atoi(argv[2]), NULL );
   }
   return error_check( interp, "vis5d_init_projection", result );
}


static int cmd_init_vertical( ClientData client_data, Tcl_Interp *interp,
                                int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_init_vertical", argc, 2, 1000 )) {
      return TCL_ERROR;
   }
   if (argc>3) {
      float vertargs[MAXPROJARGS];
      int i;
      for (i=3;i<argc;i++) {
         vertargs[i-3] = atof( argv[i] );
      }
      result = vis5d_init_vertical( atoi(argv[1]), atoi(argv[2]), vertargs );
   }
   else {
      result = vis5d_init_vertical( atoi(argv[1]), atoi(argv[2]), NULL );
   }
   return error_check( interp, "vis5d_init_vertical", result );
}


/*
 * Post-initialization functions
 */

/*** Time Functions ***/


static int cmd_get_numtimes( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int numtimes, result;
   if (!arg_check( interp, "vis5d_get_numtimes", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_get_numtimes( atoi(argv[1]), &numtimes );
   sprintf( interp->result, "%d", numtimes );
   return error_check( interp, "vis5d_get_numtimes", result );
}


static int cmd_get_time_stamp( ClientData client_data, Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   int date, time, result;
   if (!arg_check( interp, "vis5d_get_time_stamp", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_get_time_stamp( atoi(argv[1]), atoi(argv[2]), &date, &time );
   sprintf( interp->result, "%05d %06d", date, time );
   return error_check( interp, "vis5d_get_time_stamp", result );
}


static int cmd_set_time_stamp( ClientData client_data, Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   int date, time, result;
   if (!arg_check( interp, "vis5d_set_time_stamp", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   result = vis5d_set_time_stamp( atoi(argv[1]), atoi(argv[2]),
                                  atoi(argv[3]), atoi(argv[4]) );
   return error_check( interp, "vis5d_set_time_stamp", result );
}


static int cmd_set_timestep( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_set_timestep", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_set_timestep( atoi(argv[1]), atoi(argv[2]) );
   return error_check( interp, "vis5d_set_timestep", result );
}


static int cmd_get_timestep( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int time, result;
   if (!arg_check( interp, "vis5d_get_timestep", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_get_timestep( atoi(argv[1]), &time );
   sprintf( interp->result, "%d", time );
   return error_check( interp, "vis5d_get_timestep", result );
}


/*** Variable functions ***/

static int cmd_get_numvars( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int numvars, result;
   if (!arg_check( interp, "vis5d_get_numvars", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_get_numvars( atoi(argv[1]), &numvars );
   if (result==0) {
      sprintf( interp->result, "%d", numvars );
      return TCL_OK;
   }
   else {
      return error_check( interp, "vis5d_get_numvars", result );
   }
}


static int cmd_get_var_name( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int result, index, var;
   if (!arg_check( interp, "vis5d_get_var_name", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index,argv[2]);
   result = vis5d_get_var_name( index, var, interp->result );
   return error_check( interp, "vis5d_get_var_name", result );
}


static int cmd_get_var_units( ClientData client_data, Tcl_Interp *interp,
                              int argc, char *argv[] )
{
   int result, index, var;
   if (!arg_check( interp, "vis5d_get_var_units", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index,argv[2]);
   result = vis5d_get_var_units( index, var, interp->result );
   return error_check( interp, "vis5d_get_var_units", result );
}


static int cmd_get_var_type( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int type, result, index, var;
   if (!arg_check( interp, "vis5d_get_var_type", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum( index, argv[2] );
   result = vis5d_get_var_type( index, var, &type );
   if (result==0) {
      if (type==VIS5D_REGULAR) {
         interp->result = "VIS5D_REGULAR";
      }
      else if (type==VIS5D_CLONE) {
         interp->result = "VIS5D_CLONE";
      }
      else if (type==VIS5D_EXT_FUNC) {
         interp->result = "VIS5D_EXT_FUNC";
      }
      else if (type==VIS5D_EXPRESSION) {
         interp->result = "VIS5D_EXPRESSION";
      }
      return TCL_OK;
   }
   else {
      return error_check( interp, "vis5d_get_var_type", result );
   }
}



static int cmd_get_var_range( ClientData client_data, Tcl_Interp *interp,
                              int argc, char *argv[] )
{
   float min, max;
   int result, var, index;
   if (!arg_check( interp, "vis5d_get_range", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index, argv[2]);
   result = vis5d_get_var_range( index, var, &min, &max );
   if (result==0) {
      sprintf( interp->result, "%g %g", min, max );
      return TCL_OK;
   }
   else {
      return error_check( interp, "vis5d_get_var_range", result );
   }
}


static int cmd_get_wind_vars( ClientData client_data, Tcl_Interp *interp,
                              int argc, char *argv[] )
{
   int uvar, vvar, wvar;
   int u2var, v2var, w2var;
   int traju, trajv, trajw;
   int result;
   if (!arg_check( interp, "vis5d_get_wind_vars", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_get_wind_vars( atoi(argv[1]), &uvar, &vvar, &wvar,
                        &u2var, &v2var, &w2var,
                        &traju, &trajv, &trajw );
   sprintf( interp->result, "%d %d %d %d %d %d %d %d %d",
            uvar, vvar, wvar, u2var, v2var, w2var, traju, trajv, trajw );
   return error_check( interp, "vis5d_get_wind_vars", result );
}


/*** Grid functions ***/

static int cmd_get_grid_rows( ClientData client_data, Tcl_Interp *interp,
                              int argc, char *argv[] )
{
   int result, nr;
   if (!arg_check( interp, "vis5d_get_grid_rows", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_get_size( atoi(argv[1]), &nr, NULL, NULL, NULL,
                            NULL, NULL, NULL, NULL );
   sprintf( interp->result, "%d", nr );
   return error_check( interp, "vis5d_get_grid_rows", result );
}


static int cmd_get_grid_columns( ClientData client_data, Tcl_Interp *interp,
                                 int argc, char *argv[] )
{
   int result, nc;
   if (!arg_check( interp, "vis5d_get_grid_columns", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_get_size( atoi(argv[1]), NULL, &nc, NULL, NULL,
                            NULL, NULL, NULL, NULL );
   sprintf( interp->result, "%d", nc );
   return error_check( interp, "vis5d_get_grid_columns", result );
}


static int cmd_get_grid_levels( ClientData client_data, Tcl_Interp *interp,
                                int argc, char *argv[] )
{
   int result, nl[MAXVARS];
   int index, var;
   index = atoi(argv[1]);
   if (!arg_check( interp, "vis5d_get_grid_levels", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   var = varnum( index, argv[2] );
   result = vis5d_get_size( index, NULL, NULL, nl, NULL,
                            NULL, NULL, NULL, NULL );
   sprintf( interp->result, "%d", nl[var] );
   return error_check( interp, "vis5d_get_grid_levels", result );
}



/* TODO: vis5d_get_size */
/* TODO: vis5d_get_grid */
/* TODO: vis5d_put_grid */
/* TODO: vis5d_verylarge_mode */


/*** Map projection and VCS functions ***/

/* TODO: vis5d_get_projection */
/* TODO: vis5d_get_vertical */
/* TODO: vis5d_get_curved */


/*** Topograhy and map functions ***/

/* TODO: vis5d_check_topo */
/* TODO: vis5d_check_map */
/* TODO: vis5d_check_texture */
/* TODO: vis5d_get_topo_range */


static int cmd_recolor_topo( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int index, reset, result;

   if (!arg_check( interp, "vis5d_recolor_topo", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   reset = atoi(argv[2]);
   result = vis5d_recolor_topo( index, reset );
   return error_check( interp, "vis5d_recolor_topo", result );
}



/*** Cloning, Ext funcs, and expression functions ***/

static int cmd_make_clone_variable( ClientData client_data, Tcl_Interp *interp,
                                    int argc, char *argv[] )
{
   int index, var_to_clone, result, newvar;
   char *newname;

   if (!arg_check( interp, "vis5d_make_clone_variable", argc, 3, 3 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   newname = argv[2];
   var_to_clone = varnum( index, argv[3] );
   result = vis5d_make_clone_variable( index, var_to_clone, newname, &newvar );
   return error_check( interp, "vis5d_make_clone_variable", result );
}


static int cmd_compute_ext_func( ClientData client_data, Tcl_Interp *interp,
                                 int argc, char *argv[] )
{
   int index, result;
   int newvar;

   if (!arg_check( interp, "vis5d_compute_ext_func", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   result = vis5d_compute_ext_func( index, argv[2], &newvar );
   return error_check( interp, "vis5d_compute_ext_func", result );
}



static int cmd_make_expr_var( ClientData client_data, Tcl_Interp *interp,
                              int argc, char *argv[] )
{
   int index, result;
   int newvar, recompute;
   char newname[20], msg[2000];

   if (!arg_check( interp, "vis5d_make_expr_var", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   result = vis5d_make_expr_var( index, argv[2], newname, msg, &newvar,
                                 &recompute );
   return error_check( interp, "vis5d_make_expr_var", result );
}




/*** Rendering functions ***/


/* TODO: is this really needed? */
static int cmd_signal_redraw( ClientData client_data, Tcl_Interp *interp,
                              int argc, char *argv[] )
{
   if (!arg_check( interp, "vis5d_signal_redraw", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   if (vis5d_signal_redraw( atoi(argv[1]), atoi(argv[2]) )==0) {
      return TCL_OK;
   }
   else {
      return TCL_ERROR;
   }
}


static int cmd_check_redraw( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int redraw, result;
   if (!arg_check( interp, "vis5d_signal_redraw", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_check_redraw( atoi(argv[1]), &redraw );
   if (redraw) {
      interp->result = "1";
   }
   else {
      interp->result = "0";
   }
   return error_check( interp, "vis5d_check_redraw", result );
}


static int cmd_draw_frame( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int ctx;
   if (!arg_check( interp, "vis5d_draw_frame", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   ctx = atoi(argv[1]);
   vis5d_signal_redraw( ctx, 1 );
   vis5d_draw_frame( ctx, 1 );
   vis5d_swap_frame( ctx );
   return TCL_OK;
}


static int cmd_draw_3d_only( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   if (!arg_check( interp, "vis5d_draw_3d_only", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   if (vis5d_draw_3d_only( atoi(argv[1]), atoi(argv[2]) )==0) {
      return TCL_OK;
   }
   else {
      return TCL_ERROR;
   }
}


/* TODO: IS THIS REALLY NEEDED? */
static int cmd_swap_frame( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   if (!arg_check( interp, "vis5d_swap_frame", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   if (vis5d_swap_frame( atoi(argv[1]) )==0) {
      return TCL_OK;
   }
   else {
      return TCL_ERROR;
   }
}


static int cmd_invalidate_frames( ClientData client_data, Tcl_Interp *interp,
                                  int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_invalidate_frames", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_invalidate_frames( atoi(argv[1]) );
   return error_check( interp, "vis5d_invalidate_frames", result );
}


static int cmd_set_pointer( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_set_pointer", argc, 3, 3 )) {
      return TCL_ERROR;
   }
   result = vis5d_set_pointer( atoi(argv[1]), atoi(argv[2]), atoi(argv[3]) );
   return error_check( interp, "vis5d_set_pointer", result );
}


static int cmd_graphics_mode( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int what, mode, n;

   if (!arg_check( interp, "vis5d_graphics_mode", argc, 3, 3 )) {
      return TCL_ERROR;
   }
   if (strcmp(argv[2],"VIS5D_BOX")==0) {
      what = VIS5D_BOX;
   }
   else if (strcmp(argv[2],"VIS5D_CLOCK")==0) {
      what = VIS5D_CLOCK;
   }
   else if (strcmp(argv[2],"VIS5D_MAP")==0) {
      what = VIS5D_MAP;
   }
   else if (strcmp(argv[2],"VIS5D_TOPO")==0) {
      what = VIS5D_TOPO;
   }
   else if (strcmp(argv[2],"VIS5D_PERSPECTIVE")==0) {
      what = VIS5D_PERSPECTIVE;
   }
   else if (strcmp(argv[2],"VIS5D_CONTOUR_NUMBERS")==0) {
      what = VIS5D_CONTOUR_NUMBERS;
   }
   else if (strcmp(argv[2],"VIS5D_GRID_COORDS")==0) {
      what = VIS5D_GRID_COORDS;
   }
   else if (strcmp(argv[2],"VIS5D_PRETTY")==0) {
      what = VIS5D_PRETTY;
   }
   else if (strcmp(argv[2],"VIS5D_INFO")==0) {
      what = VIS5D_INFO;
   }
   else if (strcmp(argv[2],"VIS5D_PROBE")==0) {
      what = VIS5D_PROBE;
   }
   else if (strcmp(argv[2],"VIS5D_CURSOR")==0) {
      what = VIS5D_CURSOR;
   }
   else if (strcmp(argv[2],"VIS5D_ANIMRECORD")==0) {
      what = VIS5D_ANIMRECORD;
   }
   else if (strcmp(argv[2],"VIS5D_TEXTURE")==0) {
      what = VIS5D_TEXTURE;
   }
   else if (strcmp(argv[2],"VIS5D_DEPTHCUE")==0) {
      what = VIS5D_DEPTHCUE;
   }
   else {
      interp->result = "vis5d_graphics_mode: invalid argument";
      return TCL_ERROR;
   }
   
   if (strcmp(argv[3],"VIS5D_ON")==0) {
      mode = VIS5D_ON;
   }
   else if (strcmp(argv[3],"VIS5D_OFF")==0) {
      mode = VIS5D_OFF;
   }
   else if (strcmp(argv[3],"VIS5D_TOGGLE")==0) {
      mode = VIS5D_TOGGLE;
   }
   else if (strcmp(argv[3],"VIS5D_GET")==0) {
      mode = VIS5D_GET;
   }
   else {
      interp->result = "vis5d_graphics_mode: invalid argument";
      return TCL_ERROR;
   }

   n = vis5d_graphics_mode( atoi(argv[1]), what, mode );
   if (n>=0) {
      interp->result = n ? "1" : "0";
      return TCL_OK;
   }
   else {
      return error_check( interp, "vis5d_graphics_mode", n );
   }
}


static int cmd_enable_graphics( ClientData client_data, Tcl_Interp *interp,
                                int argc, char *argv[] )
{
   int index, what, mode, n, result, varflag;

   varflag = 0;

   if (!arg_check( interp, "vis5d_graphics", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   if (strcmp(argv[2],"VIS5D_ISOSURF")==0) {
      what = VIS5D_ISOSURF;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_HSLICE")==0) {
      what = VIS5D_HSLICE;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_VSLICE")==0) {
      what = VIS5D_VSLICE;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_CHSLICE")==0) {
      what = VIS5D_CHSLICE;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_CVSLICE")==0) {
      what = VIS5D_CVSLICE;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_VOLUME")==0) {
      what = VIS5D_VOLUME;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_TRAJ")==0) {
      what = VIS5D_TRAJ;
   }
   else if (strcmp(argv[2],"VIS5D_HWIND")==0) {
      what = VIS5D_HWIND;
   }
   else if (strcmp(argv[2],"VIS5D_VWIND")==0) {
      what = VIS5D_VWIND;
   }
   else if (strcmp(argv[2],"VIS5D_STREAM")==0) {
      what = VIS5D_STREAM;
   }
   else {
      interp->result = "Error in vis5d_enable_graphics: bad constant";
      return TCL_ERROR;
   }

   if (strcmp(argv[4],"VIS5D_ON")==0) {
      mode = VIS5D_ON;
   }
   else if (strcmp(argv[4],"VIS5D_OFF")==0) {
      mode = VIS5D_OFF;
   }
   else if (strcmp(argv[4],"VIS5D_TOGGLE")==0) {
      mode = VIS5D_TOGGLE;
   }
   else {
      interp->result = "Error in vis5d_enable_graphics: bad mode";
      return TCL_ERROR;
   }

   index = atoi(argv[1]);
   if (varflag) {
      n = varnum( index, argv[3] );
   }
   else {
      n = atoi( argv[3] );
   }

   result = vis5d_enable_graphics( index, what, n, mode );
   return error_check( interp, "vis5d_enable_graphics", result );
}


static int cmd_get_volume( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int vol, result;
   if (!arg_check( interp, "vis5d_get_volume", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_get_volume( atoi(argv[1]), &vol );
   sprintf( interp->result, "%d", vol );
   return error_check( interp, "vis5d_get_volume", result );
}


static int cmd_set_volume( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int vol, result, index, var;
   if (!arg_check( interp, "vis5d_set_volume", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum( index, argv[2] );
   result = vis5d_set_volume( index, var );
   return error_check( interp, "vis5d_set_volume", result );
}


static int cmd_set_color( ClientData client_data, Tcl_Interp *interp,
                          int argc, char *argv[] )
{
   int what, result;
   float r, g, b, a;
   int varflag;
   int n;
   int index;

   if (!arg_check( interp, "vis5d_set_color", argc, 7, 7 )) {
      return TCL_ERROR;
   }

   varflag = 0;

   if (strcmp(argv[2],"VIS5D_ISOSURF")==0) {
      what = VIS5D_ISOSURF;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_HSLICE")==0) {
      what = VIS5D_HSLICE;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_VSLICE")==0) {
      what = VIS5D_VSLICE;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_CHSLICE")==0) {
      what = VIS5D_CHSLICE;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_CVSLICE")==0) {
      what = VIS5D_CVSLICE;
      varflag = 1;
   }
   else if (strcmp(argv[2],"VIS5D_TRAJ")==0) {
      what = VIS5D_TRAJ;
   }
   else if (strcmp(argv[2],"VIS5D_HWIND")==0) {
      what = VIS5D_HWIND;
   }
   else if (strcmp(argv[2],"VIS5D_VWIND")==0) {
      what = VIS5D_VWIND;
   }
   else if (strcmp(argv[2],"VIS5D_STREAM")==0) {
      what = VIS5D_STREAM;
   }
   else if (strcmp(argv[2],"VIS5D_BOX")==0) {
      what = VIS5D_BOX;
   }
   else if (strcmp(argv[2],"VIS5D_MAP")==0) {
      what = VIS5D_MAP;
   }
   else if (strcmp(argv[2],"VIS5D_BACKGROUND")==0) {
      what = VIS5D_BACKGROUND;
   }
   else if (strcmp(argv[2],"VIS5D_LABEL")==0) {
      what = VIS5D_LABEL;
   }
   else {
      interp->result = "error in vis5d_set_color: bad constant";
      return TCL_ERROR;
   }

   r = atof( argv[4] );
   g = atof( argv[5] );
   b = atof( argv[6] );
   a = atof( argv[7] );

   index = atoi(argv[1]);

   if (varflag) {
      n = varnum( index, argv[3] );
   }
   else {
      n = atoi( argv[3] );
   }
   result = vis5d_set_color( index, what, n, r, g, b, a );
   return error_check( interp, "vis5d_set_color", result );
}



static int cmd_set_color_table_entry( ClientData client_data,
                                      Tcl_Interp *interp,
                                      int argc, char *argv[] )
{
   int what, var, result, entry;
   int r, g, b, a;
   int index;
   unsigned int *ctable;

   if (!arg_check( interp, "vis5d_set_color_table_entry", argc, 8, 8 )) {
      return TCL_ERROR;
   }

   if (strcmp(argv[2],"VIS5D_ISOSURF")==0) {
      what = VIS5D_ISOSURF;
   }
   else if (strcmp(argv[2],"VIS5D_CHSLICE")==0) {
      what = VIS5D_CHSLICE;
   }
   else if (strcmp(argv[2],"VIS5D_CVSLICE")==0) {
      what = VIS5D_CVSLICE;
   }
   else if (strcmp(argv[2],"VIS5D_TRAJ")==0) {
      what = VIS5D_TRAJ;
   }
   else if (strcmp(argv[2],"VIS5D_TOPO")==0) {
      what = VIS5D_TOPO;
   }
   else if (strcmp(argv[2],"VIS5D_VOLUME")==0) {
      what = VIS5D_VOLUME;
   }
   else {
      interp->result = "error in vis5d_set_color_table_entry: bad constant";
      return TCL_ERROR;
   }

   index = atoi( argv[1] );
   var = varnum( index, argv[3] );
   entry = atoi( argv[4] );
   r = atoi( argv[5] );
   g = atoi( argv[6] );
   b = atoi( argv[7] );
   a = atoi( argv[8] );
   vis5d_get_color_table_address( index, what, var, &ctable );
   if (entry>=0 && entry<=255) {
      ctable[entry] = PACK_COLOR( r, g, b, a );
   }
   return TCL_OK;
}


static int cmd_set_color_table_params( ClientData client_data,
                                       Tcl_Interp *interp,
                                       int argc, char *argv[] )
{
   int index, var, what;
   float p[4];

   if (!arg_check( interp, "vis5d_set_color_table_params", argc, 7, 7 )) {
      return TCL_ERROR;
   }

   if (strcmp(argv[2],"VIS5D_ISOSURF")==0) {
      what = VIS5D_ISOSURF;
   }
   else if (strcmp(argv[2],"VIS5D_CHSLICE")==0) {
      what = VIS5D_CHSLICE;
   }
   else if (strcmp(argv[2],"VIS5D_CVSLICE")==0) {
      what = VIS5D_CVSLICE;
   }
   else if (strcmp(argv[2],"VIS5D_TRAJ")==0) {
      what = VIS5D_TRAJ;
   }
   else if (strcmp(argv[2],"VIS5D_TOPO")==0) {
      what = VIS5D_TOPO;
   }
   else if (strcmp(argv[2],"VIS5D_VOLUME")==0) {
      what = VIS5D_VOLUME;
   }
   else {
      interp->result = "error in vis5d_set_color_table_params: bad constant";
      return TCL_ERROR;
   }
   index = atoi( argv[1] );
   var = varnum( index, argv[3] );
   p[0] = atof( argv[4] );
   p[1] = atof( argv[5] );
   p[2] = atof( argv[6] );
   p[3] = atof( argv[7] );

   vis5d_set_colorbar_params( index, what, var, p );
   return TCL_OK;
}
   

static int cmd_alpha_mode( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_alpha_mode", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_alpha_mode( atoi(argv[1]), atoi(argv[2]) );
   return error_check( interp, "vis5d_alpha_mode", result );
}


static int cmd_font( ClientData client_data, Tcl_Interp *interp,
                     int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_font", argc, 3, 4 )) {
      return TCL_ERROR;
   }
   result = vis5d_font( atoi(argv[1]), argv[2], atoi(argv[3]) );
   return error_check( interp, "vis5d_font", result );
}


static int cmd_linewidth( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_linewidth", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_linewidth( atoi(argv[1]), atof(argv[2]) );
   return error_check( interp, "vis5d_linewidth", result );
}


/* TODO: vis5d_texture_image */



/*** 3-D View Functions ***/

static int cmd_set_matrix( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int result;
   int i, j;
   float mat[4][4];

   if (!arg_check( interp, "vis5d_set_matrix", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   string_to_float_array( argv[2], 16, (float *) mat );

   result = vis5d_set_matrix( atoi(argv[1]), mat );
   return error_check( interp, "vis5d_set_matrix", result );
}


static int cmd_get_matrix( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   /* TODO:  return a list of 16 floats??? */
   return TCL_ERROR;
}


/* TODO: vis5d_set_ortho_view */


static int cmd_set_view( ClientData client_data, Tcl_Interp *interp,
                         int argc, char *argv[] )
{
   int n, r;
   int ctx;
   float xrot, yrot, zrot, scale, xtrans, ytrans, ztrans;

   if (!arg_check( interp, "vis5d_set_view", argc, 4, 8 )) {
      return TCL_ERROR;
   }
   ctx = atoi(argv[1]);
   xrot = atof(argv[2]);
   yrot = atof(argv[3]);
   zrot = atof(argv[4]);
   scale = (argc>5) ? atof(argv[5]) : 1.0;
   xtrans = (argc>6) ? atof(argv[6]) : 0.0;
   ytrans = (argc>7) ? atof(argv[7]) : 0.0;
   ztrans = (argc>8) ? atof(argv[8]) : 0.0;

   r = vis5d_set_view( ctx, xrot, yrot, zrot, scale, xtrans, ytrans, ztrans );
   return error_check( interp, "vis5d_set_view", r );
}


static int cmd_set_camera( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int n, r;
   int ctx, perspec;
   float front, zoom;

   if (!arg_check( interp, "vis5d_set_camera", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   ctx = atoi(argv[1]);
   perspec = atoi(argv[2]);
   front = atof(argv[3]);
   zoom = atof(argv[4]);

   r = vis5d_set_camera( ctx, perspec, front, zoom );
   return error_check( interp, "vis5d_set_camera", r );
}




/*** Coordinate conversion ***/

static int cmd_xyz_to_grid( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int n, index, time, var;
   float xyz[3], row, col, lev;
   if (!arg_check( interp, "vis5d_xyz_to_grid", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   time = atoi(argv[2]);
   var = varnum(index, argv[3] );
   string_to_float_array( argv[4], 3, xyz );
   n = vis5d_xyz_to_grid( index, time, var, xyz[0], xyz[1], xyz[2],
                          &row, &col, &lev );
   if (n) {
      return error_check( interp, "vis5d_xyz_to_grid", n );
   }
   else {
      sprintf( interp->result, "%f %f %f", row, col, lev );
      return 0;
   }
}



static int cmd_grid_to_xyz( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int n, index, time, var;
   float x, y, z, grid[3];
   if (!arg_check( interp, "vis5d_grid_to_xyz", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   time = atoi(argv[2]);
   var = varnum(index, argv[3] );
   string_to_float_array( argv[4], 3, grid );
   n = vis5d_grid_to_xyz( index, time, var, grid[0], grid[1], grid[2],
                          &x, &y, &z );
   if (n) {
      return error_check( interp, "vis5d_grid_to_xyz", n );
   }
   else {
      sprintf( interp->result, "%f %f %f", x, y, z );
      return 0;
   }
}



static int cmd_xyz_to_geo( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int n, index, time, var;
   float xyz[3], lat, lon, hgt;
   if (!arg_check( interp, "vis5d_xyz_to_geo", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   time = atoi(argv[2]);
   var = varnum(index, argv[3] );
   string_to_float_array( argv[4], 3, xyz );
   n = vis5d_xyz_to_geo( index, time, var, xyz[0], xyz[1], xyz[2],
                         &lat, &lon, &hgt );
   if (n) {
      return error_check( interp, "vis5d_xyz_to_geo", n );
   }
   else {
      sprintf( interp->result, "%f %f %f", lat, lon, hgt );
      return 0;
   }
}



static int cmd_geo_to_xyz( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int n, index, time, var;
   float x, y, z, geo[3];
   if (!arg_check( interp, "vis5d_geo_to_xyz", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   time = atoi(argv[2]);
   var = varnum(index, argv[3] );
   string_to_float_array( argv[4], 3, geo );
   n = vis5d_geo_to_xyz( index, time, var, geo[0], geo[1], geo[2],
                         &x, &y, &z );
   if (n) {
      return error_check( interp, "vis5d_geo_to_xyz", n );
   }
   else {
      sprintf( interp->result, "%f %f %f", x, y, z );
      return 0;
   }
}




/*** Isosurface, Slice, and Trajectory Functions ***/

static int cmd_set_isosurface( ClientData client_data, Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   int n, index, var;
   if (!arg_check( interp, "vis5d_set_isosurface", argc, 3, 3 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index, argv[2] );
   n = vis5d_set_isosurface( index, var, atof(argv[3]) );
   return error_check( interp, "vis5d_set_isosurface", n );
}


static int cmd_get_isosurface( ClientData client_data, Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   int n, index, var;
   float isolevel;
   if (!arg_check( interp, "vis5d_get_isosurface", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index, argv[2] );
   n = vis5d_get_isosurface( index, var, &isolevel );
   sprintf( interp->result, "%g", isolevel );
   return error_check( interp, "vis5d_get_isosurface", n );
}

      
static int cmd_set_isosurface_color_var( ClientData client_data,
                                         Tcl_Interp *interp,
                                         int argc, char *argv[] )
{
   int n, index, var, colorvar;
   if (!arg_check( interp, "vis5d_set_isosurface_color_var", argc, 3, 3 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index, argv[2]);
   colorvar = varnum(index, argv[3]);
   n = vis5d_set_isosurface_color_var( index, var, colorvar );
   return error_check( interp, "vis5d_set_isosurface_color_var", n );
}


static int cmd_make_isosurface( ClientData client_data, Tcl_Interp *interp,
                                int argc, char *argv[] )
{
   int n, index, var;
   if (!arg_check( interp, "vis5d_make_isosurface", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index, argv[3] );
   if (strcmp(argv[2],"VIS5D_ALL_TIMES")==0) {
      int numtimes, i;
      vis5d_get_numtimes( index, &numtimes );
      for (i=0;i<numtimes;i++) {
         n = vis5d_make_isosurface( index, i, var, atoi(argv[4]) );
         if (n!=0) {
            break;
         }
      }
   }
   else {
      /* make one timestep */
      n = vis5d_make_isosurface( index, atoi(argv[2]), var, atoi(argv[4]) );
   }
   if (NumThreads==1) {
      vis5d_finish_work();
   }
   return error_check( interp, "vis5d_make_isosurface", n );
}

      
static int cmd_set_hslice( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int n, index, var;
   float interval, low, high, level, min, max;
   if (!arg_check( interp, "vis5d_set_hslice", argc, 6, 6 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index, argv[2] );
   interval = atof(argv[3]);
   low = atof(argv[4]);
   high = atof(argv[5]);
   level = atof(argv[6]);
   vis5d_get_var_range( index, var, &min, &max );
   /* If low and high are very nearly equal to min and max, then set */
   /* low and high equal to min and max.  This takes care of the problem */
   /* of lost precision when converting from binary to ASCII and back. */
   if (ABS(low-min)<ABS(min/100.0)) {
      low = min;
   }
   if (ABS(high-max)<ABS(max/100.0)) {
      high = max;
   }
   n = vis5d_set_hslice( index, var, interval, low, high, level );
   return error_check( interp, "vis5d_set_hslice", n );
}


static int cmd_get_hslice( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   /* TODO: how to return results??? */
   return TCL_ERROR;
}


static int cmd_make_hslice( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int n, index, var;
   if (!arg_check( interp, "vis5d_make_hslice", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index, argv[3] );
   if (strcmp(argv[2],"VIS5D_ALL_TIMES")==0) {
      int numtimes, i;
      vis5d_get_numtimes( index, &numtimes );
      for (i=0;i<numtimes;i++) {
         n = vis5d_make_hslice( index, i, var, atoi(argv[4]) );
         if (n!=0) {
            break;
         }
      }
   }
   else {
      /* make one timestep */
      n = vis5d_make_hslice( index, atoi(argv[2]), var, atoi(argv[4]) );
   }
   if (NumThreads==1) {
      vis5d_finish_work();
   }
   return error_check( interp, "vis5d_make_hslice", n );
}



static int cmd_set_vslice( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int n, index, var;
   float interval, low, high, min, max;
   if (!arg_check( interp, "vis5d_set_vslice", argc, 9, 9 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum( index, argv[2] );
   interval = atof(argv[3]);
   low = atof(argv[4]);
   high = atof(argv[5]);
   vis5d_get_var_range( index, var, &min, &max );
   /* If low and high are very nearly equal to min and max, then set */
   /* low and high equal to min and max.  This takes care of the problem */
   /* of lost precision when converting from binary to ASCII and back. */
   if (ABS(low-min)<ABS(min/100.0)) {
      low = min;
   }
   if (ABS(high-max)<ABS(max/100.0)) {
      high = max;
   }
   n = vis5d_set_vslice( index, var, interval, low, high,
                         atof(argv[6]), atof(argv[7]),
                         atof(argv[8]), atof(argv[9]) );
   return error_check( interp, "vis5d_set_vslice", n );
}


static int cmd_get_vslice( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   /* TODO: how to return results??? */
   return TCL_ERROR;
}


static int cmd_make_vslice( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int n, index, var;
   if (!arg_check( interp, "vis5d_make_vslice", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index, argv[3] );
   if (strcmp(argv[2],"VIS5D_ALL_TIMES")==0) {
      int numtimes, i;
      vis5d_get_numtimes( index, &numtimes );
      for (i=0;i<numtimes;i++) {
         n = vis5d_make_vslice( index, i, var, atoi(argv[4]) );
         if (n!=0) {
            break;
         }
      }
   }
   else {
      /* make one timestep */
      n = vis5d_make_vslice( index, atoi(argv[2]), var, atoi(argv[4]) );
   }
   if (NumThreads==1) {
      vis5d_finish_work();
   }
   return error_check( interp, "vis5d_make_vslice", n );
}


static int cmd_make_chslice( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int n, index, var;
   if (!arg_check( interp, "vis5d_make_chslice", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum( index, argv[3] );
   if (strcmp(argv[2],"VIS5D_ALL_TIMES")==0) {
      int numtimes, i;
      vis5d_get_numtimes( index, &numtimes );
      for (i=0;i<numtimes;i++) {
         n = vis5d_make_chslice( index, i, var, atoi(argv[4]) );
         if (n) {
            break;
         }
      }
   }
   else {
      /* make one timestep */
      n = vis5d_make_chslice( index, atoi(argv[2]), var, atoi(argv[4]) );
   }
   if (NumThreads==1) {
      vis5d_finish_work();
   }
   return error_check( interp, "vis5d_make_chslice", n );
}


static int cmd_set_chslice( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int n, index, var;
   if (!arg_check( interp, "vis5d_set_chslice", argc, 3, 3 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum( index, argv[2] );
   n = vis5d_set_chslice( index, var, atof(argv[3]) );
   return error_check( interp, "vis5d_set_chslice", n );
}


static int cmd_get_chslice( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   /* TODO: how to return results??? */
   return TCL_ERROR;
}


static int cmd_make_cvslice( ClientData client_data, Tcl_Interp *interp,
                             int argc, char *argv[] )
{
   int n, index, var;
   if (!arg_check( interp, "vis5d_make_cvslice", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum( index, argv[3] );
   if (strcmp(argv[2],"VIS5D_ALL_TIMES")==0) {
      int numtimes, i;
      vis5d_get_numtimes( index, &numtimes );
      for (i=0;i<numtimes;i++) {
         n = vis5d_make_cvslice( index, i, var, atoi(argv[4]) );
         if (n) {
            break;
         }
      }
   }
   else {
      /* make one timestep */
      n = vis5d_make_cvslice( index, atoi(argv[2]), var, atoi(argv[4]) );
   }
   if (NumThreads==1) {
      vis5d_finish_work();
   }
   return error_check( interp, "vis5d_make_cvslice", n );
}


static int cmd_set_cvslice( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   int n, index, var;
   if (!arg_check( interp, "vis5d_set_cvslice", argc, 6, 6 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum( index, argv[2] );
   n = vis5d_set_cvslice( index, var,
                          atof(argv[3]), atof(argv[4]),
                          atof(argv[5]), atof(argv[6]) );
   return error_check( interp, "vis5d_set_cvslice", n );
}


static int cmd_get_cvslice( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   /* TODO: how to return results??? */
   return TCL_ERROR;
}


static int cmd_make_hwindslice( ClientData client_data, Tcl_Interp *interp,
                                int argc, char *argv[] )
{
   int n, index;
   if (!arg_check( interp, "vis5d_make_hwindslice", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   if (strcmp(argv[2],"VIS5D_ALL_TIMES")==0) {
      int numtimes, i;
      vis5d_get_numtimes( index, &numtimes );
      for (i=0;i<numtimes;i++) {
         n = vis5d_make_hwindslice( index, i, atoi(argv[3]), atoi(argv[4]) );
         if (n) {
            break;
         }
      }
   }
   else {
      /* make one timestep */
      n = vis5d_make_hwindslice( index, atoi(argv[2]), atoi(argv[3]),
                            atoi(argv[4]) );
   }
   if (NumThreads==1) {
      vis5d_finish_work();
   }
   return error_check( interp, "vis5d_make_hwindslice", n );
}


static int cmd_set_hwindslice( ClientData client_data, Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   int n, index;
   if (!arg_check( interp, "vis5d_set_hwindslice", argc, 5, 5 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   n = vis5d_set_hwindslice( index, atoi(argv[2]),
                             atof(argv[3]), atof(argv[4]),
                             atof(argv[5]) );
   return error_check( interp, "vis5d_set_hwindslice", n );
}


static int cmd_get_hwindslice( ClientData client_data, Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   /* TODO: how to return results??? */
   return TCL_ERROR;
}


static int cmd_make_vwindslice( ClientData client_data, Tcl_Interp *interp,
                                int argc, char *argv[] )
{
   int n;
   if (!arg_check( interp, "vis5d_make_vwindslice", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   if (strcmp(argv[2],"VIS5D_ALL_TIMES")==0) {
      int numtimes, i;
      int ctx = atoi(argv[1]);
      vis5d_get_numtimes( ctx, &numtimes );
      for (i=0;i<numtimes;i++) {
         n = vis5d_make_vwindslice( ctx, i, atoi(argv[3]), atoi(argv[4]) );
         if (n) {
            break;
         }
      }
   }
   else {
      /* make one timestep */
      n = vis5d_make_vwindslice( atoi(argv[1]), atoi(argv[2]), atoi(argv[3]),
                            atoi(argv[4]) );
   }
   if (NumThreads==1) {
      vis5d_finish_work();
   }
   return error_check( interp, "vis5d_make_vwindslice", n );
}


static int cmd_set_vwindslice( ClientData client_data, Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   int n;
   if (!arg_check( interp, "vis5d_set_vwindslice", argc, 8, 8 )) {
      return TCL_ERROR;
   }
   n = vis5d_set_vwindslice( atoi(argv[1]), atoi(argv[2]),
                             atof(argv[3]), atof(argv[4]),
                             atof(argv[5]), atof(argv[6]),
                             atof(argv[7]), atof(argv[8]) );
   return error_check( interp, "vis5d_set_vwindslice", n );
}


static int cmd_get_vwindslice( ClientData client_data, Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   /* TODO: how to return results??? */
   return TCL_ERROR;
}


static int cmd_make_streamslice( ClientData client_data, Tcl_Interp *interp,
                                 int argc, char *argv[] )
{
   int n;
   if (!arg_check( interp, "vis5d_make_streamslice", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   if (strcmp(argv[2],"VIS5D_ALL_TIMES")==0) {
      int numtimes, i;
      int ctx = atoi(argv[1]);
      vis5d_get_numtimes( ctx, &numtimes );
      for (i=0;i<numtimes;i++) {
         n = vis5d_make_streamslice( ctx, i, atoi(argv[3]), atoi(argv[4]) );
         if (n) {
            break;
         }
      }
   }
   else {
      /* make one timestep */
      n = vis5d_make_streamslice( atoi(argv[1]), atoi(argv[2]), atoi(argv[3]),
                                  atoi(argv[4]) );
   }
   if (NumThreads==1) {
      vis5d_finish_work();
   }
   return error_check( interp, "vis5d_make_streamslice", n );
}


static int cmd_set_streamslice( ClientData client_data, Tcl_Interp *interp,
                                int argc, char *argv[] )
{
   int n;
   if (!arg_check( interp, "vis5d_set_streamslice", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   n = vis5d_set_streamslice( atoi(argv[1]), atoi(argv[2]),
                              atof(argv[3]), atof(argv[4]));
   return error_check( interp, "vis5d_set_streamslice", n );
}


static int cmd_get_streamslice( ClientData client_data, Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   /* TODO: how to return results??? */
   return TCL_ERROR;
}


static int cmd_make_traj( ClientData client_data, Tcl_Interp *interp,
                          int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_make_traj", argc, 6, 6 )) {
      return TCL_ERROR;
   }
   result = vis5d_make_traj( atoi(argv[1]), atof(argv[2]), atof(argv[3]),
                             atof(argv[4]), atoi(argv[5]), atoi(argv[6]) );
   if (NumThreads==1) {
      vis5d_do_work();
   }
   return error_check( interp, "vis5d_make_traj", result );
}


static int cmd_set_traj( ClientData client_data, Tcl_Interp *interp,
                         int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_set_traj", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   result = vis5d_set_traj( atoi(argv[1]), atof(argv[2]), atof(argv[3]),
                            atoi(argv[4]));
   return error_check( interp, "vis5d_set_traj", result );
}


static int cmd_get_traj( ClientData client_data, Tcl_Interp *interp,
                         int argc, char *argv[] )
{
   /* TODO: how to return results? */
   return TCL_ERROR;
}


static int cmd_set_trajectory_color_var( ClientData client_data,
                                         Tcl_Interp *interp,
                                         int argc, char *argv[] )
{
   int result, index, var;
   if (!arg_check( interp, "vis5d_set_trajectory_color_var", argc, 3,3 )) {
      return TCL_ERROR;
   }
   index = atoi(argv[1]);
   var = varnum(index,argv[3]);
   result = vis5d_set_trajectory_color_var( index, atoi(argv[2]), var );
   return error_check( interp, "vis5d_set_trajectory_color_var", result );
}


static int cmd_delete_last_traj( ClientData client_data, Tcl_Interp *interp,
                                 int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_delete_last_traj", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_delete_last_traj( atoi(argv[1]) );
   return error_check( interp, "vis5d_delete_last_traj", result );
}


static int cmd_delete_traj_set( ClientData client_data, Tcl_Interp *interp,
                                int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_delete_traj_set", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_delete_traj_set( atoi(argv[1]), atoi(argv[2]) );
   return error_check( interp, "vis5d_delete_traj_set", result );
}



/*** Text Label Functions ***/


static int cmd_make_label( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_make_label", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   result = vis5d_make_label( atoi(argv[1]), atoi(argv[2]), atoi(argv[3]),
                              argv[4] );
   return error_check( interp, "vis5d_make_label", result );
}







static int cmd_make_timestep_graphics( ClientData client_data,
                                       Tcl_Interp *interp,
                                       int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_make_timestep_graphics", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   result = vis5d_make_timestep_graphics( atoi(argv[1]), atoi(argv[2]) );
   if (NumThreads==1) {
      vis5d_finish_work();
   }
   return error_check( interp, "vis5d_make_timestep_graphics", result );
}


static int cmd_free_graphics( ClientData client_data,
                               Tcl_Interp *interp,
                               int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_free_graphics", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   vis5d_finish_work();
   result = vis5d_free_graphics( atoi(argv[1]) );
   return error_check( interp, "vis5d_free_graphics", result );
}





/*** 3-D Cursor Functions ***/

static int cmd_set_cursor( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   int result;
   if (!arg_check( interp, "vis5d_set_cursor", argc, 4, 4 )) {
      return TCL_ERROR;
   }
   result = vis5d_set_cursor( atoi(argv[1]), atof(argv[2]), atof(argv[3]),
                              atof(argv[4]));
   return error_check( interp, "vis5d_set_cursor", result );
}


static int cmd_get_cursor( ClientData client_data, Tcl_Interp *interp,
                           int argc, char *argv[] )
{
   float x, y, z;
   int result;
   if (!arg_check( interp, "vis5d_get_cursor", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   result = vis5d_get_cursor( atoi(argv[1]), &x, &y, &z );
   sprintf( interp->result, "%g %g %g", x, y, z );
   return error_check( interp, "vis5d_get_cursor", result );
}



/*** 3-D viewing window functions ***/

/* TODO: vis5d_get_window */

/* TODO: vis5d_resize_window */

static int cmd_get_image_formats( ClientData client_data, Tcl_Interp *interp,
                                  int argc, char *argv[] )
{
   int formats;
   char result[1000];

   formats = vis5d_get_image_formats();

   /* form a string list of image formats supported */
   strcpy( result, "{ " );
   if (formats & VIS5D_RGB) {
      strcat( result, "VIS5D_RGB " );
   }
   if (formats & VIS5D_GIF) {
      strcat( result, "VIS5D_GIF " );
   }
   if (formats & VIS5D_XWD) {
      strcat( result, "VIS5D_XWD " );
   }
   if (formats & VIS5D_PS) {
      strcat( result, "VIS5D_PS " );
   }
   if (formats & VIS5D_COLOR_PS) {
      strcat( result, "VIS5D_COLOR_PS " );
   }
   strcat( result, "}" );
   interp->result = result;
   return TCL_OK;
}


static int cmd_save_window( ClientData client_data, Tcl_Interp *interp,
                            int argc, char *argv[] )
{
   if (!arg_check( interp, "vis5d_save_window", argc, 3, 3 )) {
      return TCL_ERROR;
   }
   else {
      int index = atoi( argv[1] );
      char *name = argv[2];
      int format;
      int result;
      if (strcmp(argv[3],"VIS5D_GIF")==0) {
         format = VIS5D_GIF;
      }
      else if (strcmp(argv[3],"VIS5D_RGB")==0) {
         format = VIS5D_RGB;
      }
      else if (strcmp(argv[3],"VIS5D_XWD")==0) {
         format = VIS5D_XWD;
      }
      else if (strcmp(argv[3],"VIS5D_PS")==0) {
         format = VIS5D_PS;
      }
      else if (strcmp(argv[3],"VIS5D_COLOR_PS")==0) {
         format = VIS5D_COLOR_PS;
      }
      else {
         interp->result = "vis5d_save_window: bad format";
         return TCL_ERROR;
      }
      result = vis5d_save_window( index, name, format );
      return error_check( interp, "vis5d_save_window", result );
   }
}

/* TODO: vis5d_print_window */



/*** Coordinate Conversion Functions ***/


/*** Save and Restore Functions ***/

static int cmd_save( ClientData client_data, Tcl_Interp *interp,
                     int argc, char *argv[] )
{
   int n;
   if (!arg_check( interp, "vis5d_save", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   n = tcl_save( atoi(argv[1]), argv[2] );
   return error_check( interp, "vis5d_save", n );
}


static int cmd_restore( ClientData client_data, Tcl_Interp *interp,
                        int argc, char *argv[] )
{
   int n;
   if (!arg_check( interp, "vis5d_restore", argc, 2, 2 )) {
      return TCL_ERROR;
   }
   n = vis5d_restore( atoi(argv[1]), argv[2] );
   return error_check( interp, "vis5d_restore", n );
}



static int cmd_sleep( ClientData client_data, Tcl_Interp *interp,
                      int argc, char *argv[] )
{
   struct timeval tp;
   long time0, time;
   long milliseconds;

   if (!arg_check( interp, "vis5d_sleep", argc, 1, 1 )) {
      return TCL_ERROR;
   }
   milliseconds = atoi( argv[1] );

   /* Get the current time */
   gettimeofday( &tp, (struct timezone *) 0 );
   time0 = tp.tv_sec * 1000 + tp.tv_usec / 1000;

   /* Loop until milliseconds elapses.  This is a poor implementation
    * but I don't know a better way that's portable.
    */
   do {
      gettimeofday( &tp, (struct timezone *) 0 );
      time = tp.tv_sec * 1000 + tp.tv_usec / 1000;
   } while (time < time0 + milliseconds);

   return TCL_OK;
}


/*** Miscellaneous Functions ***/



static void register_commands( Tcl_Interp *interp )
{
#define REGISTER( NAME, FUNC ) Tcl_CreateCommand(interp,NAME,FUNC,NULL,NULL)

   REGISTER( "vis5d_initialize", cmd_initialize );
   REGISTER( "vis5d_terminate", cmd_terminate );
   REGISTER( "vis5d_workers", cmd_workers );
   REGISTER( "vis5d_do_work", cmd_do_work );
   REGISTER( "vis5d_get_image_formats", cmd_get_image_formats );

   /* Context init functions */
   REGISTER( "vis5d_init_begin", cmd_init_begin );
   REGISTER( "vis5d_init_end", cmd_init_end );
   REGISTER( "vis5d_init_window", cmd_init_window );
   REGISTER( "vis5d_open_gridfile", cmd_open_gridfile );
   REGISTER( "vis5d_init_map", cmd_init_map );
   REGISTER( "vis5d_init_topo", cmd_init_topo );
   REGISTER( "vis5d_init_texture", cmd_init_texture );
   REGISTER( "vis5d_init_firstarea", cmd_init_firstarea );
   REGISTER( "vis5d_init_sequence", cmd_init_sequence );
   REGISTER( "vis5d_init_log", cmd_init_log );
   REGISTER( "vis5d_init_box", cmd_init_box );
   REGISTER( "vis5d_init_memory", cmd_init_memory );
   REGISTER( "vis5d_init_path", cmd_init_path );
   REGISTER( "vis5d_init_projection", cmd_init_projection );
   REGISTER( "vis5d_init_vertical", cmd_init_vertical );

   /* Time functions */
   REGISTER( "vis5d_get_numtimes", cmd_get_numtimes );
   REGISTER( "vis5d_get_time_stamp", cmd_get_time_stamp );
   REGISTER( "vis5d_set_time_stamp", cmd_set_time_stamp );
   REGISTER( "vis5d_get_timestep", cmd_get_timestep );
   REGISTER( "vis5d_set_timestep", cmd_set_timestep );

   /* Variable functions */
   REGISTER( "vis5d_get_numvars", cmd_get_numvars );
   REGISTER( "vis5d_get_var_name", cmd_get_var_name );
   REGISTER( "vis5d_get_var_units", cmd_get_var_units );
   REGISTER( "vis5d_get_var_type", cmd_get_var_type );
   REGISTER( "vis5d_get_var_range", cmd_get_var_range );
   REGISTER( "vis5d_get_wind_vars", cmd_get_wind_vars );

   /* Grid functions */
   REGISTER( "vis5d_get_grid_rows", cmd_get_grid_rows );
   REGISTER( "vis5d_get_grid_columns", cmd_get_grid_columns );
   REGISTER( "vis5d_get_grid_levels", cmd_get_grid_levels );

   /* Topography and map functions */
   REGISTER( "vis5d_recolor_topo", cmd_recolor_topo );

   /* New variables */
   REGISTER( "vis5d_make_clone_variable", cmd_make_clone_variable );
   REGISTER( "vis5d_compute_ext_func", cmd_compute_ext_func );
   REGISTER( "vis5d_make_expr_var", cmd_make_expr_var );

   /* Rendering functions */
   REGISTER( "vis5d_signal_redraw", cmd_signal_redraw );
   REGISTER( "vis5d_check_redraw", cmd_check_redraw );
   REGISTER( "vis5d_draw_frame", cmd_draw_frame );
   REGISTER( "vis5d_draw_3d_only", cmd_draw_3d_only );
   REGISTER( "vis5d_swap_frame", cmd_swap_frame );
   REGISTER( "vis5d_invalidate_frames", cmd_invalidate_frames );
   REGISTER( "vis5d_set_pointer", cmd_set_pointer );
   REGISTER( "vis5d_set_view", cmd_set_view );
   REGISTER( "vis5d_set_camera", cmd_set_camera );

   REGISTER( "vis5d_graphics_mode", cmd_graphics_mode );
   REGISTER( "vis5d_enable_graphics", cmd_enable_graphics );
   REGISTER( "vis5d_set_color", cmd_set_color );
   REGISTER( "vis5d_set_color_table_entry", cmd_set_color_table_entry );
   REGISTER( "vis5d_set_color_table_params", cmd_set_color_table_params );
   REGISTER( "vis5d_alpha_mode", cmd_alpha_mode );
   REGISTER( "vis5d_linewidth", cmd_linewidth );
   REGISTER( "vis5d_font", cmd_font );

   /* Isosurface, Slice, and Trajectory Functions */
   REGISTER( "vis5d_make_isosurface", cmd_make_isosurface );
   REGISTER( "vis5d_set_isosurface", cmd_set_isosurface );
   REGISTER( "vis5d_get_isosurface", cmd_get_isosurface );
   REGISTER( "vis5d_set_isosurface_color_var", cmd_set_isosurface_color_var );
   REGISTER( "vis5d_make_hslice", cmd_make_hslice );
   REGISTER( "vis5d_set_hslice", cmd_set_hslice );
   REGISTER( "vis5d_get_hslice", cmd_get_hslice );
   REGISTER( "vis5d_make_vslice", cmd_make_vslice );
   REGISTER( "vis5d_set_vslice", cmd_set_vslice );
   REGISTER( "vis5d_get_vslice", cmd_get_vslice );
   REGISTER( "vis5d_make_chslice", cmd_make_chslice );
   REGISTER( "vis5d_set_chslice", cmd_set_chslice );
   REGISTER( "vis5d_get_chslice", cmd_get_chslice );
   REGISTER( "vis5d_make_cvslice", cmd_make_cvslice );
   REGISTER( "vis5d_set_cvslice", cmd_set_cvslice );
   REGISTER( "vis5d_get_cvslice", cmd_get_cvslice );
   REGISTER( "vis5d_make_hwindslice", cmd_make_hwindslice );
   REGISTER( "vis5d_set_hwindslice", cmd_set_hwindslice );
   REGISTER( "vis5d_get_hwindslice", cmd_get_hwindslice );
   REGISTER( "vis5d_make_vwindslice", cmd_make_vwindslice );
   REGISTER( "vis5d_set_vwindslice", cmd_set_vwindslice );
   REGISTER( "vis5d_get_vwindslice", cmd_get_vwindslice );
   REGISTER( "vis5d_make_streamslice", cmd_make_streamslice );
   REGISTER( "vis5d_set_streamslice", cmd_set_streamslice );
   REGISTER( "vis5d_get_streamslice", cmd_get_streamslice );
   REGISTER( "vis5d_make_traj", cmd_make_traj );
   REGISTER( "vis5d_set_traj", cmd_set_traj );
   REGISTER( "vis5d_get_traj", cmd_get_traj );
   REGISTER( "vis5d_set_trajectory_color_var", cmd_set_trajectory_color_var );
   REGISTER( "vis5d_delete_last_traj", cmd_delete_last_traj );
   REGISTER( "vis5d_delete_traj_set", cmd_delete_traj_set );

   REGISTER( "vis5d_make_timestep_graphics", cmd_make_timestep_graphics );
   REGISTER( "vis5d_free_graphics", cmd_free_graphics );


   /* Text Label Functions */
   REGISTER( "vis5d_make_label", cmd_make_label );

   /* 3-D Cursor Functions */
   REGISTER( "vis5d_set_cursor", cmd_set_cursor );
   REGISTER( "vis5d_get_cursor", cmd_get_cursor );

   /* 3-D window functions */
   REGISTER( "vis5d_save_window", cmd_save_window );

   REGISTER( "vis5d_set_matrix", cmd_set_matrix );

   /* Coordinate Conversion Functions */
   REGISTER( "vis5d_xyz_to_grid", cmd_xyz_to_grid );
   REGISTER( "vis5d_grid_to_xyz", cmd_grid_to_xyz );
   REGISTER( "vis5d_xyz_to_geo", cmd_xyz_to_geo );
   REGISTER( "vis5d_geo_to_xyz", cmd_geo_to_xyz );

   /* Save and Restore Functions */
   REGISTER( "vis5d_save", cmd_save );
   REGISTER( "vis5d_restore", cmd_restore );

   REGISTER( "vis5d_sleep", cmd_sleep );
/*
   REGISTER( "vis5d_", cmd_ );
   REGISTER( "vis5d_", cmd_ );
*/
#undef REGISTER
}


/*
 * Execute a Tcl Vis5D script from a file.
 * Input:  index - the context index
 *         filename - name of Tcl script
 * Return:  1 = success, 0 = error
 */
int execute_script( int index, char *filename )
{
   Tcl_Interp *interp;
   int code;
   char setup_cmd[100];

   /* Create an interpreter */
   interp = Tcl_CreateInterp();

   register_commands( interp );
   sprintf( setup_cmd, "set ctx %d", index );
   code = Tcl_Eval( interp, setup_cmd );

   /* Execute the script */
   code = Tcl_EvalFile( interp, filename );
   if (interp->result && interp->result[0]) {
      printf("Tcl_EvalFile result: %s\n", interp->result);
   }

   Tcl_DeleteInterp( interp );

   if (code==TCL_OK) {
      return 1;
   }
   else {
      return 0;
   }
}


int interpret( int index )
{
   Tcl_Interp *interp = NULL;
   char setup_cmd[100];
   int code;
   FILE *f;

   /* Create an interpreter */
   interp = Tcl_CreateInterp();

   register_commands( interp );
   sprintf( setup_cmd, "set ctx %d", index );
   code = Tcl_Eval( interp, setup_cmd );

   /* Execute the TCL_STARTUP_FILE, if it exists. */
   f = fopen( TCL_STARTUP_FILE, "r" );
   if (f) {
      fclose(f);
      code = Tcl_EvalFile( interp, TCL_STARTUP_FILE );
      if (code!=TCL_OK) {
         if (interp->result && interp->result[0]) {
            printf("Error executing %s: %s\n",
                   TCL_STARTUP_FILE, interp->result);
         }
         else {
            printf("Couldn't execute %s\n", TCL_STARTUP_FILE);
         }
      }
   }

   printf("Vis5D interpreter: type exit when finished.\n");
   while (1) {
      char cmd[1000];
      int code;
      printf("vis5d> ");
      cmd[0] = 0;
      gets( cmd );
      if (strcmp(cmd,"exit")==0) {
         break;
      }
      code = Tcl_Eval( interp, cmd );
      if (interp->result && interp->result[0]) {
         printf("%s\n", interp->result);
      }
   }

   Tcl_DeleteInterp( interp );

   return 0;
}


