/* $Id: rgbsliders.c,v 1.9 1996/03/14 21:40:52 brianp Exp $ */

/* Vis5D version 4.2 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1995  Bill Hibbard, Brian Paul, Dave Santek,
and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




/*
 * Widget for RGB color sliders.
 */



#include <stdio.h>
#include <string.h>
#include "X11/Xlib.h"
#include "../lui5/lui.h"
#include "api.h"
#include "graphics.h"
#include "gui.h"
#include "mwmborder.h"
#include "rgbsliders.h"


static int current_ctx = -1;
static int current_graphic = -1;
static int current_var = -1;


static int rgb_bottom, rgba_bottom;



/*
 * Change the color of a GUI button to (r,g,b).
 */
static void update_button_color( GuiContext gtx, float r, float g, float b )
{
   if (current_graphic==VIS5D_HWIND) {
      LUI_ButtonColor( gtx->hwind_button[current_var], r, g, b );
   }
   else if (current_graphic==VIS5D_VWIND) {
      LUI_ButtonColor( gtx->vwind_button[current_var], r, g, b );
   }
   else if (current_graphic==VIS5D_STREAM) {
      LUI_ButtonColor( gtx->stream_button[current_var], r, g, b );
   }
   else if (current_graphic==VIS5D_HSLICE) {
      LUI_ButtonMatrixSetColor( gtx->ButtonMatrix, current_var, 1, r, g, b );
   }
   else if (current_graphic==VIS5D_VSLICE) {
      LUI_ButtonMatrixSetColor( gtx->ButtonMatrix, current_var, 2, r, g, b );
   }
   else if (current_graphic==VIS5D_CHSLICE) {
      LUI_ButtonMatrixSetColor( gtx->ButtonMatrix, current_var, 3, r, g, b );
   }
   else if (current_graphic==VIS5D_CVSLICE) {
      LUI_ButtonMatrixSetColor( gtx->ButtonMatrix, current_var, 4, r, g, b );
   }
   else if (current_graphic==VIS5D_MAP) {
      LUI_ButtonSetColor( gtx->map_button, (int)(r*255),
                          (int)(g*255), (int)(b*255) );
   }
   else if (current_graphic==VIS5D_TRAJ) {
      LUI_ButtonSetColor( gtx->TrajButton[current_var], (int)(r*255),
                          (int)(g*255), (int)(b*255) );
   }
}


/* called when red slider is changed */
static int red_slider_cb( LUI_NEWSLIDER *s, float value )
{
   GuiContext gtx = get_gui_gtx(current_ctx);
   float r, g, b, a;

   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );
   r = value;
   vis5d_set_color( current_ctx, current_graphic, current_var, r,g,b,a );

   vis5d_signal_redraw( current_ctx, 1 );
   update_button_color( gtx, r, g, b );
   return 0;
}



/* called when green slider is changed */
static int green_slider_cb( LUI_NEWSLIDER *s, float value )
{
   GuiContext gtx = get_gui_gtx(current_ctx);
   float r, g, b, a;

   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );
   g = value;
   vis5d_set_color( current_ctx, current_graphic, current_var, r,g,b,a );

   vis5d_signal_redraw( current_ctx, 1 );
   update_button_color( gtx, r, g, b );
   return 0;
}



/* called when blue slider is changed */
static int blue_slider_cb( LUI_NEWSLIDER *s, float value )
{
   GuiContext gtx = get_gui_gtx(current_ctx);
   float r, g, b, a;

   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );
   b = value;
   vis5d_set_color( current_ctx, current_graphic, current_var, r,g,b,a );

   vis5d_signal_redraw( current_ctx, 1 );
   update_button_color( gtx, r, g, b );
   return 0;
}



/* called when alpha slider is changed */
static int alpha_slider_cb( LUI_NEWSLIDER *s, float value )
{
   GuiContext gtx = get_gui_gtx(current_ctx);
   float r, g, b, a;

   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );
   a = value;
   vis5d_set_color( current_ctx, current_graphic, current_var, r,g,b,a );

   /* change the alpha value in the color lookup table */
   if (current_graphic==VIS5D_CHSLICE) {
      int ia = (int)(a * 255.0);
      set_slice_alpha( current_ctx, VIS5D_CHSLICE, current_var, ia );
   }
   else if (current_graphic==VIS5D_CVSLICE) {
      int ia = (int)(a * 255.0);
      set_slice_alpha( current_ctx, VIS5D_CVSLICE, current_var, ia );
   }

   vis5d_signal_redraw( current_ctx, 1 );
   return 0;
}



static int close_cb( LUI_NEWBUTTON *b, int state )
{
   GuiContext gtx = get_gui_gtx( current_ctx );

   XUnmapWindow( GuiDpy, gtx->rgb_sliders_window );
   current_ctx = current_graphic = current_var = -1;
   return 0;
}


void make_rgb_sliders( GuiContext gtx )
{
   Window w;
   LUI_NEWSLIDER *s;
   LUI_NEWBUTTON *b;

   /* make the window */
   w = LUI_CreateWindowAt( LUI_RootWindow, 10, 800, 380, 144 );
   set_mwm_border( GuiDpy, w, MWM_DECOR_BORDER | MWM_DECOR_TITLE );

   gtx->rgb_sliders_window = w;

   /* make the label */
   gtx->rgb_sliders_label = LUI_NewLabelCreate( w, LUI_LEFT, LUI_TOP, 311, 20,
                                                "default label" );

   /* make the "Close" button */
   b = LUI_PushButtonCreate( w, LUI_NEXT_X, LUI_SAME_Y, 60, 20, "Close" );
   LUI_ButtonCallback( b, close_cb );

   /* make red slider */
   s = LUI_NewSliderCreate( w, LUI_LEFT, LUI_NEXT_Y, 374 );
   gtx->rgb_red_slider = s;
   LUI_NewSliderChange( s, "Red", 0.0, 1.0, 0.5 );
   LUI_NewSliderCallback( s, red_slider_cb );

   /* make green slider */
   s = LUI_NewSliderCreate( w, LUI_LEFT, LUI_NEXT_Y, 374 );
   gtx->rgb_green_slider = s;
   LUI_NewSliderChange( s, "Green", 0.0, 1.0, 0.5 );
   LUI_NewSliderCallback( s, green_slider_cb );

   /* make blue slider */
   s = LUI_NewSliderCreate( w, LUI_LEFT, LUI_NEXT_Y, 374 );
   gtx->rgb_blue_slider = s;
   LUI_NewSliderChange( s, "Blue", 0.0, 1.0, 0.5 );
   LUI_NewSliderCallback( s, blue_slider_cb );

   rgb_bottom = LUI_LayoutGet( LUI_NEXT_Y );

   /* make alpha slider */
   s = LUI_NewSliderCreate( w, LUI_LEFT, LUI_NEXT_Y, 374 );
   gtx->rgb_alpha_slider = s;
   LUI_NewSliderChange( s, "Transparency", 0.0, 1.0, 0.5 );
   LUI_NewSliderCallback( s, alpha_slider_cb );

   rgba_bottom = LUI_LayoutGet( LUI_NEXT_Y );
}




/*
 * Hide the RGB sliders window.
 */
void hide_rgb_sliders( GuiContext gtx )
{
   current_ctx = current_graphic = current_var = -1;
   XUnmapWindow( GuiDpy, gtx->rgb_sliders_window );
}




/*
 * Display the RGB sliders window.
 */
void show_rgb_sliders( GuiContext gtx, int graphic, int var )
{
   float r, g, b, a;
   char label[100];

   if (current_ctx==gtx->context_index &&
       current_graphic==graphic &&
       current_var==var) {
      /* toggle the window off */
      XUnmapWindow( GuiDpy, gtx->rgb_sliders_window );
      current_ctx = current_graphic = current_var = -1;
      return;
   }
   else if (current_ctx>=0 && current_graphic>=0 && current_var>=0) {
      /* unmap window with old settings before showing new */
      XUnmapWindow( GuiDpy, gtx->rgb_sliders_window );
   }

   current_ctx = gtx->context_index;
   current_graphic = graphic;
   current_var = var;

   vis5d_get_color( current_ctx, current_graphic, current_var, &r,&g,&b,&a );

   if (graphic==VIS5D_MAP) {
      strcpy( label, "Map Color:" );
   }
   else if (graphic==VIS5D_HWIND) {
      sprintf( label, "Horizontal Wind slice %d color:", var+1 );
   }
   else if (graphic==VIS5D_VWIND) {
      sprintf( label, "Vertical Wind slice %d color:", var+1 );
   }
   else if (graphic==VIS5D_STREAM) {
      sprintf( label, "Stream slice %d color:", var+1 );
   }
   else if (graphic==VIS5D_HSLICE) {
      char varname[20];
      vis5d_get_var_name( gtx->context_index, var, varname );
      sprintf( label, "%s Horizontal Contour Slice color:", varname );
   }
   else if (graphic==VIS5D_VSLICE) {
      char varname[20];
      vis5d_get_var_name( gtx->context_index, var, varname );
      sprintf( label, "%s Vertical Contour Slice color:", varname );
   }
   else if (graphic==VIS5D_CHSLICE) {
      char varname[20];
      vis5d_get_var_name( gtx->context_index, var, varname );
      sprintf( label, "%s Horizontal Color Slice tickmark:", varname );
   }
   else if (graphic==VIS5D_CVSLICE) {
      char varname[20];
      vis5d_get_var_name( gtx->context_index, var, varname );
      sprintf( label, "%s Vertical Color Slice tickmark:", varname );
   }

   LUI_NewLabelChangeText( gtx->rgb_sliders_label, label );

   LUI_NewSliderSetValue( gtx->rgb_red_slider, r );
   LUI_NewSliderSetValue( gtx->rgb_green_slider, g );
   LUI_NewSliderSetValue( gtx->rgb_blue_slider, b );
   LUI_NewSliderSetValue( gtx->rgb_alpha_slider, a );

   if (graphic==VIS5D_CHSLICE || graphic==VIS5D_CVSLICE) {
      /* show alpha slider too */
      LUI_ResizeWindow( gtx->rgb_sliders_window, 380, rgba_bottom );
   }
   else {
      /* don't show alpha slider */
      LUI_ResizeWindow( gtx->rgb_sliders_window, 380, rgb_bottom );
   }

   XMapWindow( GuiDpy, gtx->rgb_sliders_window );
}


void refresh_rgb_sliders( GuiContext gtx )
{
   if (current_ctx>=0 && current_graphic>=0 && current_var>=0) {
      int gr = current_graphic, var = current_var;
      hide_rgb_sliders( gtx );
      show_rgb_sliders( gtx, gr, var );
   }
}

