/* $Id: render.c,v 1.47 1996/04/09 17:57:49 brianp Exp $ */

/* Vis5D version 4.2 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990-1996 Bill Hibbard, Brian Paul, Dave Santek,
and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include "anim.h"
#include "api.h"
#include "box.h"
#include "matrix.h"
#include "globals.h"
#include "graphics.h"
#include "grid.h"
#include "labels.h"
#include "map.h"
#include "memory.h"
#include "misc.h"
#include "proj.h"
#include "queue.h"
#include "sync.h"
#include "topo.h"
#include "vis5d.h"
#include "volume.h"
#include "v5d.h"



#define MAX(A,B)  ( (A) > (B) ? (A) : (B) )
#define MIN(A,B)  ( (A) < (B) ? (A) : (B) )

#define TICK_SIZE 0.05

#define VERT(Z) (ctx->VerticalSystem==VERT_NONEQUAL_MB ? height_to_pressure(Z) : (Z))



/*** float2string *****************************************************
   Convert a float into an ascii string.
**********************************************************************/
void float2string( float f, char *str )
{
   if (f==0.0 || fabs(f)<0.01)
      strcpy( str, "0.0" );
   else if (f>=100.0 || f<=-100.0)
      sprintf(str, "%d", (int) f );
   else
      sprintf(str, "%4.2f", f );
}



/*** plot_string  ******************************************************
   Plot (draw) a string in 3-D.  At this time, only strings of digits,
   periods, and dashes are implemented.
   Input: f - the string to plot.
          startx, y, z - the point in 3-D to start at.
          base - vector indicating text baseline.
          up - vector indicating upright direction for text.
          rjustify - non-zero value indicates right justify the text.
**********************************************************************/
void plot_string( char *str, float startx, float starty, float startz,
                  float base[], float up[], int rjustify )
{
   static float zero[] = { 0,0, 0,.8, .4,.8, .4,0, 0,0 },
      one[] = { 0,0, 0,.8 },
      two[] = { .4,0, 0,0, 0,.4, .4,.4, .4,.8, 0,.8 },
      three[] = { 0,0, .4,0, .4,.4, 0,.4, .4,.4, .4,.8, 0,.8 },
      four[] = { 0,.8, 0,.4, .4,.4, .4,.8, .4,0 },
      five[] = { 0,0, .4,0, .4,.4, 0,.4, 0,.8, .4,.8 },
      six[] = { .4,.8, 0,.8, 0,0, .4,0, .4,.4, 0,.4 },
      seven[] = { 0,.7, 0,.8, .4,.8, .4,0 },
      eight[] = { 0,0, 0,.8, .4,.8, .4,0, 0,0, 0,.4, .4,.4 },
      nine[] = { .4,.4, 0,.4, 0,.8, .4,.8, .4,0 },
      dash[] = { 0,.4, .4,.4 },
      dot[] = { 0,0, 0,.1, .1,.1, .1,0, 0,0 };

   static float *index[12] = { zero, one, two, three, four, five, six,
                               seven, eight, nine, dash, dot };

   static float width[12] = { 0.6, 0.2, 0.6, 0.6, 0.6, 0.6, 0.6, 0.6,
      0.6, 0.6, 0.6, 0.3 };
   static int verts[12] = { 5, 2, 6, 7, 5, 6, 6, 4, 7, 5, 2, 5 };

   float *temp, plot[100][3];
   float cx, cy, cz;
   int i, j, k, len;

   cx = startx;  cy = starty;  cz = startz;
   len = strlen(str);

   if (rjustify) {
      /* draw right justified text */
      for (i=len-1; i>=0; i--) {
         if (str[i]=='-')
            k = 10;
         else if (str[i]=='.')
            k = 11;
         else if (str[i]>='0' && str[i]<='9')
            k = str[i] - '0';
         else
            continue;
         /* calculate position for this char */
         cx += width[k]*base[0];
         cy += width[k]*base[1];
         cz += width[k]*base[2];
         /* make the vertex array for this character */
         temp = index[k];
         for (j=0; j<verts[k]; j++) {
            float x, y;
            x = *temp++;
            y = *temp++;
            plot[j][0] = cx - x*base[0] + y*up[0];
            plot[j][1] = cy - x*base[1] + y*up[1];
            plot[j][2] = cz - x*base[2] + y*up[2];
         }
         polyline( plot, verts[k] );
      }

   }
   else {
      /* draw left justified text */
      for (i=0; i<len; i++) {
         if (str[i]=='-')
            k = 10;
         else if (str[i]=='.')
            k = 11;
         else if (str[i]>='0' && str[i]<='9')
            k = str[i] - '0';
         else
            continue;
         /* make the vertex array for this character */
         temp = index[k];
         for (j=0; j<verts[k]; j++) {
            float x, y;
            x = *temp++;
            y = *temp++;
            plot[j][0] = cx + x*base[0] + y*up[0];
            plot[j][1] = cy + x*base[1] + y*up[1];
            plot[j][2] = cz + x*base[2] + y*up[2];
         }
         polyline( plot, verts[k] );
         /* calculate position for next char */
         cx += width[k]*base[0];
         cy += width[k]*base[1];
         cz += width[k]*base[2];
      }
   }

}



/*
 * Draw the tick mark for a horizontal slice.
 * Input:  level - grid level
 *         z - graphics z coord
 *         height - geographic height coord
 */
static void draw_horizontal_slice_tick( Context ctx, float level,
                                        float z, float height )
{
   float v[2][3];
   static float base[3] = { 0.035, -0.035, 0.0 };
   static float up[3] = { 0.0, 0.0, 0.07 };
   char str[1000];

   /* vertices for tick mark */
   v[0][0] = ctx->Xmax;
   v[0][1] = ctx->Ymin;
   v[0][2] = z;
   v[1][0] = ctx->Xmax + 0.05;
   v[1][1] = ctx->Ymin - 0.05;
   v[1][2] = z;
   polyline( v, 2 );

   /* the label */
   if (ctx->CoordFlag) {
      float2string( level+1.0, str );
   }
   else {
      float2string( VERT(height), str );
   }
   plot_string( str, ctx->Xmax+0.07, ctx->Ymin-0.07, z, base,up, 0 );
}




/*
 * Draw a tick mark for a vertical slice.
 * Input:  row, col - position in grid coords
 *         x, y - position in graphics coords
 *         lat,lon - position in geographic coords
 */
static void draw_vertical_slice_tick( Context ctx, float row, float col,
                                      float x, float y, float lat, float lon )
{
   float v[2][3];
   /* base and up vectors for drawing 3-D text */
   static float b2[3] = { 0.05, 0.0, 0.0 }, u2[3] = { 0.0, 0.05, 0.0 };
   static float b3[3] = { -0.05, 0.0, 0.0 }, u3[3] = { 0.0, 0.05, 0.0 };
   char str[1000];

   if (col==0.0) {
      /* draw on top-west edge */
      v[0][0] = x;
      v[0][1] = y;
      v[0][2] = ctx->Zmax;
      v[1][0] = x-0.05;
      v[1][1] = y;
      v[1][2] = ctx->Zmax;
      polyline( v, 2 );
      if (ctx->CoordFlag) {
         float2string( row+1, str );
      }
      else {
         float2string( lat, str );
      }
      plot_string( str, x-0.07, y, ctx->Zmax, b3, u3, 1 );
   }
   else if (col==(float)(ctx->Nc-1)) {
      /* draw on top-east edge */
      v[0][0] = x;
      v[0][1] = y;
      v[0][2] = ctx->Zmax;
      v[1][0] = x+0.05;
      v[1][1] = y;
      v[1][2] = ctx->Zmax;
      polyline( v, 2 );
      if (ctx->CoordFlag) {
         float2string( row+1, str );
      }
      else {
         float2string( lat, str );
      }
      plot_string( str, x+0.07, y, ctx->Zmax, b2, u2, 0 );
   }
   else if (row==0.0) {
      /* draw on top-north edge */
      v[0][0] = x;
      v[0][1] = y;
      v[0][2] = ctx->Zmax;
      v[1][0] = x;
      v[1][1] = y+0.05;
      v[1][2] = ctx->Zmax;
      polyline( v, 2 );
      if (ctx->CoordFlag) {
         float2string( col+1.0, str );
      }
      else {
         float2string( lon, str );
      }
      plot_string( str, x-0.07, y+0.07, ctx->Zmax, b2,u2, 0 );
   }
   else {
      /* draw on top-south edge */
      v[0][0] = x;
      v[0][1] = y;
      v[0][2] = ctx->Zmax;
      v[1][0] = x;
      v[1][1] = y-0.05;
      v[1][2] = ctx->Zmax;
      polyline( v, 2 );
      if (ctx->CoordFlag) {
         float2string( col+1.0, str );
      }
      else {
         float2string( lon, str );
      }
      plot_string( str, x-0.07, y-0.12, ctx->Zmax, b2,u2, 0 );
   }
}




/*
 * Print the current cursor position.
 */
static void print_cursor_position( Context ctx, int it )
{
   static float bx[3] = { 0.05, 0.0, 0.0 },      ux[3] = { 0.0, 0.05, 0.05 };
   static float by[3] = { -0.035, 0.0, -0.035 },  uy[3] = { 0.0, 0.07, 0.0 };
   static float bz[3] = { -0.035, -0.035, 0.0 }, uz[3] = { 0.0, 0.0, 0.07 };
   float v[6][3];
   float x, y, z;
   char str[100];

   set_color( *ctx->CursorColor );

   if (ctx->Projection==PROJ_LINEAR || ctx->Projection==PROJ_GENERIC) {
      /* Rectangular box:  put labels along edge of box in 3-D */

      set_depthcue( ctx->DepthCue );

      /* draw tick marks */
      v[0][0] = v[1][0] = ctx->CursorX;
      v[0][1] = ctx->Ymin;  v[1][1] = ctx->Ymin-0.05;
      v[0][2] = ctx->Zmin;  v[1][2] = ctx->Zmin-0.05;
      v[2][0] = ctx->Xmin;  v[3][0] = ctx->Xmin-0.05;
      v[2][1] = v[3][1] = ctx->CursorY;
      v[2][2] = ctx->Zmin;  v[3][2] = ctx->Zmin-0.05;
      v[4][0] = ctx->Xmin;  v[5][0] = ctx->Xmin-0.05;
      v[4][1] = ctx->Ymin;  v[5][1] = ctx->Ymin-0.05;
      v[4][2] = v[5][2] = ctx->CursorZ;
      disjointpolyline( v, 6 );

      /* draw position labels */
      if (ctx->CoordFlag) {
         /* display cursor position in grid coordinates */
         xyz_to_grid( ctx, it, -1,
                      ctx->CursorX, ctx->CursorY, ctx->CursorZ, &y, &x, &z );
         x += 1.0;
         y += 1.0;
         z += 1.0;
      }
      else {
         /* display cursor position in geographic coordinates */
         xyz_to_geo( ctx, it, -1,
                     ctx->CursorX, ctx->CursorY, ctx->CursorZ, &y, &x, &z );
         z = VERT(z);
      }

      float2string( x, str );
      plot_string( str, ctx->CursorX-0.04, ctx->Ymin-0.1,
                   ctx->Zmin-0.125, bx, ux, 0 );
      float2string( y, str );
      plot_string( str, ctx->Xmin-0.075, ctx->CursorY-0.02,
                   ctx->Zmin-0.075, by, uy, 1 );
      float2string( z, str );
      plot_string( str, ctx->Xmin-0.07, ctx->Ymin-0.07,
                   ctx->CursorZ+0.005, bz, uz, 1 );

      set_depthcue( 0 );
   }


   /* draw labels in upper-right corner of window in 2-D */
   if (ctx->CoordFlag) {
      float row, col, lev;
      /* display cursor position in grid coordinates */
      xyz_to_grid( ctx, it, -1, ctx->CursorX, ctx->CursorY, ctx->CursorZ,
                   &row, &col, &lev );
      sprintf( str, "Row: %g", row+1.0 );
      draw_text( ctx->WinWidth-200, ctx->FontHeight+10, str );
      sprintf( str, "Col: %g", col+1.0 );
      draw_text( ctx->WinWidth-200, 2*ctx->FontHeight+20, str );
      sprintf( str, "Lev: %g", lev+1.0 );
      draw_text( ctx->WinWidth-200, 3*ctx->FontHeight+30, str );
   }
   else {
      float lat, lon, hgt;
      /* display cursor position in geographic coordinates */
      xyz_to_geo( ctx, it, -1, ctx->CursorX, ctx->CursorY, ctx->CursorZ,
                  &lat, &lon, &hgt );
      sprintf( str, "Lat: %g", lat );
      draw_text( ctx->WinWidth-200, ctx->FontHeight+10, str );
      sprintf( str, "Lon: %g", lon );
      draw_text( ctx->WinWidth-200, 2*ctx->FontHeight+20, str );
      sprintf( str, "Hgt: %g", VERT(hgt) );
      draw_text( ctx->WinWidth-200, 3*ctx->FontHeight+30, str );
   }

}



/*
 * Examine the entries of a color lookup table to determine the alpha value.
 * If the alpha is the same for all entries return that value.  If the
 * alpha varies between entries return -1.
 * Input:  color_table - array of color values
 *         size - number of entries in table
 * Return:  -1 or constant alpha value
 */
static int get_alpha( unsigned int color_table[], int size )
{
   int alpha, i;

   /* Check for variable vs. constant alpha */
   alpha = UNPACK_ALPHA( color_table[0] );
   for (i=0;i<size;i++) {
      if (UNPACK_ALPHA(color_table[i])!=alpha) {
         return -1;
      }
   }
   return alpha;
}




/*
 * Render all isosurfaces selected for display.
 * Input:  ctx - the context
 *         time - which timestep
 *         tf - transparency flag: 1=only draw opaque surfaces
 *                                 0=only draw transparent surfaces
 */
static void render_isosurfaces( Context ctx, int time, int tf, int animflag )
{
   int var, alpha, lock;

   for (var=0;var<ctx->NumVars;var++) {
      if (ctx->DisplaySurf[var] && ctx->SurfTable[var][time].valid) {
         if (animflag) {
            lock = cond_read_lock( &ctx->SurfTable[var][time].lock );
         }
         else {
            wait_read_lock( &ctx->SurfTable[var][time].lock );
            lock = 1;
         }
         if (lock) {
            recent( ctx, ISOSURF, var );

            /* Determine alpha for surface:  -1=variable, 0..255=constant */
            if (ctx->SurfTable[var][time].colors) {
               int colorvar = ctx->SurfTable[var][time].colorvar;
               alpha = get_alpha( ctx->IsoColors[colorvar], 255 );
            }
            else {
               alpha = UNPACK_ALPHA( ctx->Color[var][ISOSURF] );
            }

            if ( (tf && alpha==255) || (tf==0 && alpha<255) ) {
               if (ctx->SurfTable[var][time].colors) {
                  int colorvar = ctx->SurfTable[var][time].colorvar;
                  draw_colored_isosurface(
                                   ctx->SurfTable[var][time].numindex,
                                   ctx->SurfTable[var][time].index,
                                   (void *) ctx->SurfTable[var][time].verts,
                                   (void *) ctx->SurfTable[var][time].norms,
                                   (void *) ctx->SurfTable[var][time].colors,
                                   ctx->IsoColors[colorvar],
                                   alpha );
               }
               else {
                  draw_isosurface( ctx->SurfTable[var][time].numindex,
                                   ctx->SurfTable[var][time].index,
                                   (void *) ctx->SurfTable[var][time].verts,
                                   (void *) ctx->SurfTable[var][time].norms,
                                   ctx->Color[var][0] );
               }

            }
            done_read_lock( &ctx->SurfTable[var][time].lock );
         }
      }
   }
}




/*
 * Render all horizontal contour slices selected for display.
 * Input:  ctx - the context
 *         time - the time step
 *         labels - draw labels flag.
 */
static void render_hslices( Context ctx, int time, int labels, int animflag )
{
   int var, lock;

   for (var=0;var<ctx->NumVars;var++) {
      if (ctx->DisplayHSlice[var] && ctx->HSliceTable[var][time].valid) {
         if (animflag) {
            lock = cond_read_lock(&ctx->HSliceTable[var][time].lock);
         }
         else {
            wait_read_lock(&ctx->HSliceTable[var][time].lock);
            lock = 1;
         }
         if (lock) {
            recent( ctx, HSLICE, var );

            /* draw main contour lines */
            draw_disjoint_lines( ctx->HSliceTable[var][time].num1,
                                 (void *) ctx->HSliceTable[var][time].verts1,
                                 ctx->Color[var][HSLICE] );
            
            if (labels) {
               /* draw contour labels */
               draw_disjoint_lines( ctx->HSliceTable[var][time].num3,
                                    (void *)ctx->HSliceTable[var][time].verts3,
                                    ctx->Color[var][HSLICE] );
            }
            else {
               /* draw hidden contour lines */
               draw_disjoint_lines( ctx->HSliceTable[var][time].num2,
                                    (void *)ctx->HSliceTable[var][time].verts2,
                                    ctx->Color[var][HSLICE] );
            }

            /* draw the bounding box */
            polyline( (void *) ctx->HSliceTable[var][time].boxverts,
                           ctx->HSliceTable[var][time].numboxverts );

            done_read_lock( &ctx->HSliceTable[var][time].lock );
         }

         /* draw position label */
         if (ctx->DisplayBox && !ctx->CurvedBox) {
            draw_horizontal_slice_tick( ctx,
                                        ctx->HSliceLevel[var],
                                        ctx->HSliceZ[var],
                                        ctx->HSliceHgt[var] );
         }
      }
   }
}




/*
 * Render all vertical contour slices selected for display.
 * Input:  ctx - the context
 *         time - the time step
 *         labels - draw labels flag.
 */
static void render_vslices( Context ctx, int time, int labels, int animflag )
{
   int var, lock;

   for (var=0;var<ctx->NumVars;var++) {
      if (ctx->DisplayVSlice[var] && ctx->VSliceTable[var][time].valid) {
         if (animflag) {
            lock = cond_read_lock(&ctx->VSliceTable[var][time].lock);
         }
         else {
            wait_read_lock(&ctx->VSliceTable[var][time].lock);
            lock = 1;
         }
         if (lock) {
            recent( ctx, VSLICE, var );

            /* draw main contour lines */
            draw_disjoint_lines( ctx->VSliceTable[var][time].num1,
                                 (void*) ctx->VSliceTable[var][time].verts1,
                                 ctx->Color[var][VSLICE] );

            if (labels) {
               /* draw contour labels */
               draw_disjoint_lines( ctx->VSliceTable[var][time].num3,
                                    (void*) ctx->VSliceTable[var][time].verts3,
                                    ctx->Color[var][VSLICE] );
            }
            else {
               /* draw hidden contour lines */
               draw_disjoint_lines( ctx->VSliceTable[var][time].num2,
                                    (void*) ctx->VSliceTable[var][time].verts2,
                                    ctx->Color[var][VSLICE] );
            }

            /* draw the bounding box */
            polyline( (void *) ctx->VSliceTable[var][time].boxverts,
                           ctx->VSliceTable[var][time].numboxverts );

            done_read_lock( &ctx->VSliceTable[var][time].lock );
         }

         if (ctx->DisplayBox && !ctx->CurvedBox) {
            /* draw position labels */
            float vert[4][3];
            float zbot, ztop;
            zbot = gridlevel_to_z(ctx, time, var, (float) ctx->LowLev[var]);
            ztop = gridlevel_to_z(ctx, time, var,
                                  (float) (ctx->Nl[var]-1+ctx->LowLev[var]));
            set_color( ctx->Color[var][VSLICE] );
            draw_vertical_slice_tick( ctx, ctx->VSliceR1[var],
                                      ctx->VSliceC1[var],
                                      ctx->VSliceX1[var],
                                      ctx->VSliceY1[var],
                                      ctx->VSliceLat1[var],
                                      ctx->VSliceLon1[var] );
            draw_vertical_slice_tick( ctx, ctx->VSliceR2[var],
                                      ctx->VSliceC2[var],
                                      ctx->VSliceX2[var],
                                      ctx->VSliceY2[var],
                                      ctx->VSliceLat2[var],
                                      ctx->VSliceLon2[var] );
            /* draw small markers at midpoint of top and bottom edges */
            vert[0][0] = vert[1][0] = vert[2][0] = vert[3][0] =
                              (ctx->VSliceX1[var] + ctx->VSliceX2[var]) * 0.5;
            vert[0][1] = vert[1][1] = vert[2][1] = vert[3][1] =
                              (ctx->VSliceY1[var] + ctx->VSliceY2[var]) * 0.5;
            vert[0][2] = ztop+TICK_SIZE;
            vert[1][2] = ztop;
            vert[2][2] = zbot;
            vert[3][2] = zbot-TICK_SIZE;
            set_line_width(5); /* WLH 3-5-96 */
            disjointpolyline( vert, 4 );
            set_line_width(ctx->LineWidth); /* WLH 3-5-96 */
         }
      }
   }
}





/*
 * Render all horizontal colored slices selected for display.
 * Input:  ctx - the context
 *         time - the timestep
 *         tf - transparency flag: 1=only draw opaque slices
 *                                 0=only draw transparent slices
 *         animflag - 1=animating, 0=not animating
 */
static void render_chslices( Context ctx, int time, int tf, int animflag )
{
   int var, alpha, lock;

   for (var=0;var<ctx->NumVars;var++) {
      if (ctx->DisplayCHSlice[var]) {
         if (ctx->CHSliceTable[var][time].valid) {
            if (animflag) {
               lock = cond_read_lock( &ctx->CHSliceTable[var][time].lock );
            }
            else {
               wait_read_lock( &ctx->CHSliceTable[var][time].lock );
               lock = 1;
            }
            if (lock) {
               recent( ctx, CHSLICE, var );

               alpha = get_alpha( ctx->CHSliceColors[var], 255 );

               if ( (tf && alpha==255) || (tf==0 && alpha<255) ) {
                  draw_color_quadmesh( ctx->CHSliceTable[var][time].rows,
                                    ctx->CHSliceTable[var][time].columns,
                                    (void *)ctx->CHSliceTable[var][time].verts,
                                    ctx->CHSliceTable[var][time].color_indexes,
                                    ctx->CHSliceColors[var], alpha );
               }

               done_read_lock( &ctx->CHSliceTable[var][time].lock );
            }

            /* draw position label */
            if (tf && ctx->DisplayBox && !ctx->CurvedBox) {
               set_color( ctx->Color[var][CHSLICE] );
               draw_horizontal_slice_tick( ctx, ctx->CHSliceLevel[var],
                                           ctx->CHSliceZ[var],
                                           ctx->CHSliceHgt[var] );
            }
         }
      }
   }
}




/*
 * Render all vertical colored slices selected for display.
 * Input:  ctx - the context
 *         time - the timestep
 *         tf - transparency flag: 1=only draw opaque slices
 *                                 0=only draw transparent slices
 */
static void render_cvslices( Context ctx, int time, int tf, int animflag )
{
   int var, alpha, lock;

   for (var=0;var<ctx->NumVars;var++) {
      if (ctx->DisplayCVSlice[var] && ctx->CVSliceTable[var][time].valid) {
         if (animflag) {
            lock = cond_read_lock(&ctx->CVSliceTable[var][time].lock);
         }
         else {
            wait_read_lock(&ctx->CVSliceTable[var][time].lock);
            lock = 1;
         }
         if (lock) {
            recent( ctx, CVSLICE, var );

            alpha = get_alpha( ctx->CVSliceColors[var], 255 );
            if ( (tf && alpha==255) || (tf==0 && alpha<255) ) {
               draw_color_quadmesh( ctx->CVSliceTable[var][time].rows,
                                    ctx->CVSliceTable[var][time].columns,
                                    (void *)ctx->CVSliceTable[var][time].verts,
                                    ctx->CVSliceTable[var][time].color_indexes,
                                    ctx->CVSliceColors[var], alpha );
            }
            done_read_lock( &ctx->CVSliceTable[var][time].lock );
         }

         if (tf & ctx->DisplayBox && !ctx->CurvedBox) {
            /* draw position labels */
            float zbot, ztop;
            float vert[4][3];
            zbot = gridlevel_to_z(ctx, time, var, (float) ctx->LowLev[var]);
            ztop = gridlevel_to_z(ctx, time, var,
                                  (float) (ctx->Nl[var]-1+ctx->LowLev[var]));
            set_color( ctx->Color[var][CVSLICE] );
            draw_vertical_slice_tick( ctx, ctx->CVSliceR1[var],
                                      ctx->CVSliceC1[var],
                                      ctx->CVSliceX1[var],
                                      ctx->CVSliceY1[var],
                                      ctx->CVSliceLat1[var],
                                      ctx->CVSliceLon1[var] );
            draw_vertical_slice_tick( ctx, ctx->CVSliceR2[var],
                                      ctx->CVSliceC2[var],
                                      ctx->CVSliceX2[var],
                                      ctx->CVSliceY2[var],
                                      ctx->CVSliceLat2[var],
                                      ctx->CVSliceLon2[var] );
            /* draw small markers at midpoint of top and bottom edges */
            vert[0][0] = vert[1][0] = vert[2][0] = vert[3][0] =
                           (ctx->CVSliceX1[var] + ctx->CVSliceX2[var]) * 0.5;
            vert[0][1] = vert[1][1] = vert[2][1] = vert[3][1] =
                           (ctx->CVSliceY1[var] + ctx->CVSliceY2[var]) * 0.5;
            vert[0][2] = ztop+TICK_SIZE;
            vert[1][2] = ztop;
            vert[2][2] = zbot;
            vert[3][2] = zbot-TICK_SIZE;
            set_line_width(5); /* WLH 3-5-96 */
            disjointpolyline( vert, 4 );
            set_line_width(ctx->LineWidth); /* WLH 3-5-96 */
         }
      }
   }
}




/*
 * Render all horizontal wind vector slices which are selected for display.
 * Input:  ctx - the context
 *         time - the timestep
 */
static void render_hwind_slices( Context ctx, int time, int animflag )
{
   int w, lock;

   for (w=0;w<VIS5D_WIND_SLICES;w++) {
      if (ctx->DisplayHWind[w] && ctx->HWindTable[w][time].valid) {
         if (animflag) {
            lock = cond_read_lock( &ctx->HWindTable[w][time].lock );
         }
         else {
            wait_read_lock( &ctx->HWindTable[w][time].lock );
            lock = 1;
         }
         if (lock) {
            recent( ctx, HWIND, w );

            /* draw the bounding box */
            set_color( ctx->HWindColor[w] );
            polyline( (void *) ctx->HWindTable[w][time].boxverts,
                      ctx->HWindTable[w][time].numboxverts );

            /* draw wind vectors */
            draw_wind_lines( ctx->HWindTable[w][time].nvectors,
                             (void *) ctx->HWindTable[w][time].verts,
                             ctx->HWindColor[w] );

            done_read_lock( &ctx->HWindTable[w][time].lock );
         }

         /* draw position label */
         if (ctx->DisplayBox && !ctx->CurvedBox) {
            draw_horizontal_slice_tick( ctx, ctx->HWindLevel[w],
                                        ctx->HWindZ[w], ctx->HWindHgt[w]);
         }
      }
   }
}



/*
 * Render all vertical wind vector slices which are selected for display.
 * Input:  ctx - the context
 *         time - the timestep
 */
static void render_vwind_slices( Context ctx, int time, int animflag )
{
   int w, lock;

   for (w=0;w<VIS5D_WIND_SLICES;w++) {
      if (ctx->DisplayVWind[w] && ctx->VWindTable[w][time].valid) {
         if (animflag) {
            lock = cond_read_lock(&ctx->VWindTable[w][time].lock);
         }
         else {
            wait_read_lock(&ctx->VWindTable[w][time].lock);
            lock = 1;
         }
         if (lock) {
            recent( ctx, VWIND, w );

            /* draw the bounding box */
            set_color( ctx->VWindColor[w] );
            polyline( (void *) ctx->VWindTable[w][time].boxverts,
                      ctx->VWindTable[w][time].numboxverts );

            /* draw wind vectors */
            draw_wind_lines( ctx->VWindTable[w][time].nvectors,
                             (void *) ctx->VWindTable[w][time].verts,
                             ctx->VWindColor[w] );

            done_read_lock( &ctx->VWindTable[w][time].lock );
         }

         if (ctx->DisplayBox && !ctx->CurvedBox) {
            /* position labels */
            float zbot, ztop;
            float vert[4][3];
            zbot = gridlevel_to_z(ctx, time, ctx->Uvar[0],
                                  (float) ctx->LowLev[ctx->Uvar[0]]);
            ztop = gridlevel_to_z(ctx, time, ctx->Uvar[0],
                                  (float) (ctx->Nl[ctx->Uvar[0]]-1
                                           +ctx->LowLev[ctx->Uvar[0]]));
            draw_vertical_slice_tick( ctx, ctx->VWindR1[w], ctx->VWindC1[w],
                                      ctx->VWindX1[w], ctx->VWindY1[w],
                                      ctx->VWindLat1[w], ctx->VWindLon1[w] );
            draw_vertical_slice_tick( ctx, ctx->VWindR2[w], ctx->VWindC2[w],
                                      ctx->VWindX2[w], ctx->VWindY2[w],
                                      ctx->VWindLat2[w], ctx->VWindLon2[w] );
            /* draw small markers at midpoint of top and bottom edges */
            vert[0][0] = vert[1][0] = vert[2][0] = vert[3][0] =
                                  (ctx->VWindX1[w] + ctx->VWindX2[w]) * 0.5;
            vert[0][1] = vert[1][1] = vert[2][1] = vert[3][1] =
                                  (ctx->VWindY1[w] + ctx->VWindY2[w]) * 0.5;
            vert[0][2] = ztop+TICK_SIZE;
            vert[1][2] = ztop;
            vert[2][2] = zbot;
            vert[3][2] = zbot-TICK_SIZE;
            set_line_width(5); /* WLH 3-5-96 */
            disjointpolyline( vert, 4 );
            set_line_width(ctx->LineWidth); /* WLH 3-5-96 */
         }
      }
   }
}



/*
 * Render all horizontal stream vector slices which are selected for display.
 * Input:  ctx - the context
 *         time - the timestep
 */
static void render_stream_slices( Context ctx, int time, int animflag )
{
   int w, lock;

   for (w=0;w<VIS5D_WIND_SLICES;w++) {

      if (ctx->DisplayStream[w] && ctx->StreamTable[w][time].valid) {
         if (animflag) {
            lock = cond_read_lock(&ctx->StreamTable[w][time].lock);
         }
         else {
            wait_read_lock(&ctx->StreamTable[w][time].lock);
            lock = 1;
         }
         if (lock) {
            recent( ctx, STREAM, w );

            /* draw the bounding box */
            set_color( ctx->StreamColor[w] );
            polyline( (void *) ctx->StreamTable[w][time].boxverts,
                      ctx->StreamTable[w][time].numboxverts );


            /* draw main contour lines */
            draw_disjoint_lines( ctx->StreamTable[w][time].nlines,
                                 (void *) ctx->StreamTable[w][time].verts,
                                 ctx->StreamColor[w] );
            
            done_read_lock( &ctx->StreamTable[w][time].lock );
         }

         /* draw position label */
         if (ctx->DisplayBox && !ctx->CurvedBox) {
            draw_horizontal_slice_tick( ctx, ctx->StreamLevel[w],
                                        ctx->StreamZ[w], ctx->StreamHgt[w]);
         }
      }
   }
}



static void render_trajectories( Context ctx, int it )
{
   int i, len, start;
   
   for (i=0;i<ctx->NumTraj;i++) {
      struct traj *t = ctx->TrajTable[i];

      if (ctx->DisplayTraj[t->group]
           && cond_read_lock(&t->lock) ) {

         assert( t->lock==1 );

         recent( ctx, TRAJ, t->group );

         start = t->start[it];
         len = t->len[it];
         if (start!=0xffff && len>0) {
            if (t->kind==0) {
               /* draw as line segments */
               int colorvar = t->colorvar;
               if (colorvar>=0) {
                  /* draw colored trajectory */
                  draw_colored_polylines( len,
                                  (void *) (t->verts + start*3),
                                  (void*)(t->colors + start),
                                  ctx->TrajColors[colorvar]);
               }
               else {
                  /* monocolored */
                  draw_polylines( len,
                                  (void *) (t->verts + start*3),
                                  ctx->TrajColor[t->group] );
               }
            }
            else {
               /* draw as triangle strip */
               int colorvar = t->colorvar;
               if (colorvar>=0) {
                  /* draw colored triangles */
                  draw_colored_triangle_strip( len,
                                    (void*)(t->verts + start*3),
                                    (void*)(t->norms + start*3),
                                    (void*)(t->colors + start),
                                    ctx->TrajColors[colorvar] );
               }
               else {
                  /* monocolor */
                  draw_triangle_strip( len,
                                    (void*)(t->verts + start*3),
                                    (void*)(t->norms + start*3),
                                    ctx->TrajColor[t->group] );
               }
            }
         }
         done_read_lock( &t->lock );
      }
   }
}




/*
 * Draw the clock in the upper-left corner of the 3-D window.
 * Input:  ctx - the vis5d context
 *         c - the color to use.
 */
static void draw_clock( Context ctx, unsigned int c )
{
   static char day[7][20] = {"Sunday", "Monday", "Tuesday", "Wednesday",
                             "Thursday", "Friday", "Saturday" };
   static float twopi = 2.0 * 3.141592;
   short pp[8][2];
   float ang;
   char str[12];
   int i;

   /* Draw the clock. */
   if (ctx->NumTimes)
      ang = (float) ctx->CurTime / (float) ctx->NumTimes;
   else
      ang = 0.0;

   pp[0][1] = 50;
   pp[0][0] = 50;
   pp[1][1] = 50 - 40 * cos(twopi * ang);
   pp[1][0] = 50 + 40 * sin(twopi * ang);
   pp[2][1] = pp[1][1] + 1;
   pp[2][0] = pp[1][0] + 1;
   pp[3][1] = pp[0][1] + 1;
   pp[3][0] = pp[0][0] + 1;
   pp[4][1] = pp[0][1] - 1;
   pp[4][0] = pp[0][0] + 1;
   pp[5][1] = pp[1][1] - 1;
   pp[5][0] = pp[1][0] + 1;

   set_color( c );
   polyline2d( pp, 6 );

   i = ctx->TimeStamp[ctx->CurTime];
   sprintf( str, "%02d:%02d:%02d", i/3600, (i/60)%60, i%60 );
   draw_text( 100, ctx->FontHeight+5, str );

   sprintf( str, "%05d", v5dDaysToYYDDD( ctx->DayStamp[ctx->CurTime] ) );
   draw_text( 100, 2*ctx->FontHeight+10, str );

   sprintf( str, "%d of %d", ctx->CurTime+1, ctx->NumTimes );
   draw_text( 100, 3*ctx->FontHeight+15, str );

   if (ctx->NumTimes == 1 ||
       ((ctx->Elapsed[ctx->NumTimes-1] - ctx->Elapsed[0])
         / (ctx->NumTimes - 1)) < 48*3600 ) {
     /* Print day of week */
     draw_text( 100, 4*ctx->FontHeight+20,
                day[ (ctx->DayStamp[ctx->CurTime]+0) % 7 ] );
   }

}



/*
 * Render all the 2-D text labels.
 */
static void render_text_labels( Context ctx )
{
   struct label *lab;

  for (lab=ctx->FirstLabel; lab; lab=lab->next) {
    draw_text( lab->x, lab->y, lab->text );
    if (lab->state) {
       /* being edited -> draw cursor */
       short verts[4][2];
       verts[0][0] = lab->x2;     verts[0][1] = lab->y1;
       verts[1][0] = lab->x2;     verts[1][1] = lab->y2;
       verts[2][0] = lab->x2+1;   verts[2][1] = lab->y2;
       verts[3][0] = lab->x2+1;   verts[3][1] = lab->y1;
       polyline2d( verts, 4 );
    }
  }
}




static void draw_fake_pointer( Context ctx )
{
   short pp[8][2];

   pp[0][0] = ctx->PointerX;       pp[0][1] = ctx->PointerY;
   pp[1][0] = ctx->PointerX+15;    pp[1][1] = ctx->PointerY+5;
   pp[2][0] = ctx->PointerX+5;     pp[2][1] = ctx->PointerY+15;
   pp[3][0] = ctx->PointerX;       pp[3][1] = ctx->PointerY;
   pp[4][0] = ctx->PointerX+20;    pp[4][1] = ctx->PointerY+20;

   polyline2d( pp, 5 );

}



/*
 * Print status info at bottom of window.
 */
static void print_info( Context ctx )
{
   char str[1000];
   int m, size, waiters;

   m = mem_used( ctx );
   get_queue_info( &size, &waiters );
   if (m>=0)
      sprintf(str, "Pending: %d   Memory Used: %d", size, m );
   else
      sprintf(str, "Pending: %d", size );

   draw_text( 10, ctx->WinHeight - ctx->FontHeight, str );
}



/*
 * Print the numeric value of each variable at the probe's current location.
 */
static void draw_probe( Context ctx )
{
   float val;
   char str[1000];
   int y, var;
   static int x = -1;

   /* find widest parameter name, but only once */
   if (x==-1) {
      for (var=0;var<ctx->NumVars;var++) {
         int w = text_width( ctx->VarName[var] );
         if (w>x)
            x = w;
      }
   }

   set_color( ctx->BoxColor );

   /* Draw from bottom of window upward */
   y = ctx->WinHeight - ctx->FontHeight;
   for (var=ctx->NumVars-1;var>=0;var--) {
      float r, c, l;
      xyz_to_grid( ctx, ctx->CurTime, var,
                   ctx->CursorX, ctx->CursorY, ctx->CursorZ, &r, &c, &l );
/* WLH 6-30-95
      if (l>ctx->Nl[var]-1) {
*/
      if (l < ctx->LowLev[var] || l > ctx->Nl[var]-1 + ctx->LowLev[var]) {
         val = MISSING;
      }
      else {
         if (ctx->CoordFlag==1) {
            /* discrete grid position */
            int row = (int) (r+0.01);
            int col = (int) (c+0.01);
            int lev = (int) (l+0.01);
            val = get_grid_value( ctx, ctx->CurTime, var, row, col, lev );
         }
         else {
            val = interpolate_grid_value( ctx, ctx->CurTime, var, r, c, l );
         }
      }
      sprintf( str, "%-4s", ctx->VarName[var] );
      draw_text( 10, y, str );
      if (IS_MISSING(val))
        sprintf( str, "=MISSING" );
      else
        sprintf( str, "=%f %s", val, ctx->Units[var] );
      draw_text( x+10, y, str );
      y -= (ctx->FontHeight+5);
   }
}



#ifdef COLORLEGENDS

#define LEGEND_SPACE   20
#define LEGEND_WIDTH   32 
#define LEGEND_HEIGHT 128 
#define TICK_LENGTH     4

/*** draw_legend ***********************************************************
   Draw a color legend.
   Input: var    = parameter index the color slice belongs to for which to
                  draw the legend,
          type  = CHSLICE or CVSLICE,
          xleft = x-position for left side of color bar,
          ybot  = y-position for bottom of color bar.
   Return: width of bar + numbers drawn.
***************************************************************************/

int draw_legend(ctx, var, type, xleft, ybot)
Context;
int var, type, xleft, ybot;
{
   int   y, yoffset,
         width,
         lutindex,
         tickspacing,
         textwidth;
   short cline[2][2];
   uint  *lut;
   char  scrap[100], format[20];
   float label;

   switch(type) {
      case CHSLICE:
         lut = CHSliceColors[var];
         break;
      case CVSLICE:
         lut = ctx->CVSliceColors[var];
         break;
      default:
         printf("draw_legend called for unappropriate graphics type (%d)\n", type);
         /* Nothing to draw */
         return 0;
   }

   /* These line values never change */
   cline[0][0] = xleft;
   cline[1][0] = xleft + LEGEND_WIDTH;

   /* Get the color for every line from the LUT */
   for (y = 0; y<LEGEND_HEIGHT; y++) {
      lutindex = (255 * y)/LEGEND_HEIGHT;
      cline[0][1] = cline[1][1] = ybot - y;

      set_color(lut[lutindex]);
      polyline2d(cline, 2);
   }


   /* Determine largest value physical variable can have */
   label = fabs(ctx->MaxVal[var]);
   if (abs(ctx->MinVal[var]) > label)
      label = fabs(ctx->MinVal[var]);

   /* Create 'pretty' formatting string */
   sprintf(scrap, "% .0f", label); 
   sprintf(format, "%% %d.2f", strlen(scrap)+3);

   /* Draw values and tick marks on the right hand side of the legend */
   textwidth = 0;

   tickspacing = 2*LEGEND_HEIGHT/ctx->FontHeight;

   /* Make sure we have a tick at the top of the legend @@ */

   cline[0][0] += TICK_LENGTH + LEGEND_WIDTH;
   set_color( ctx->BoxColor );
   for (y=0; y<LEGEND_HEIGHT; y+=tickspacing) {
      cline[0][1] = cline[1][1] = ybot - y;
      polyline2d(cline, 2);
      
      sprintf(scrap, format,
              ctx->MinVal[var] + (ctx->MaxVal[var] - ctx->MinVal[var])
                  * (float)y/LEGEND_HEIGHT);

      draw_text(ctx, xleft + LEGEND_WIDTH + TICK_LENGTH + 5,
                ybot - y - 2 + ctx->FontHeight/2,
                scrap);
      if (strwidth(scrap) > textwidth)
         textwidth = strwidth(scrap);
   }

   /* Print name of physical variable above legend */
   draw_text( xleft, ybot - LEGEND_HEIGHT - ctx->FontHeight/2,
              ctx->VarName[var]);

   return LEGEND_WIDTH + TICK_LENGTH + 5 + textwidth + LEGEND_SPACE;
}


/*** draw_colorlegends *****************************************************
   Draws color legends of activated color slices.
   Since the space in the 3-D window is restricted only one row of legends
   is drawn in the bottom of the 3-D window. The order of drawing is:
   first the legends of the horizontal slices for parameter 0..NumVars-1
   and then the vertical slices for parameter 0..NumVars-1.
***************************************************************************/
int draw_colorlegends(ctx)
Context ctx;
{
   int ip,
        xleft,     /* Left x position of current legend */
        ybot;      /* Bottom y position of current legend */ 
 
   xleft = 150;
   ybot  = WinHeight - 20;

   /* Find activated horizontal color slices */
   for (ip=0; ip<ctx->NumVars; ip++) {
      if (ctx->DisplayCHSlice[ip]) {
         /* Draw legend at position (xstart, ystart) = upper left corner */

         xleft += draw_legend(ip, CHSLICE, xleft, ybot); 
         if (xleft > WinWidth - 150)
            return;
      }
   }

   /* Find activated vertical color slices */
   for (ip=0; ip<ctx->NumVars; ip++) {
      if (ctx->DisplayCVSlice[ip]) {
         /* Draw legend at position (xstart, ystart) = upper left corner */

         xleft += draw_legend(ctx,ip, CVSLICE, xleft, ybot); 
         if (xleft > WinWidth - 150)
            return;
      }
   }
}
#endif



/*
 * Draw anything the user wants in 3D.
 * Drawing bounds are (Xmin,Ymin,Zmin) - (Xmax,Ymax,Zmax)
 */
static void draw_user_3d_graphics( ctx, time )
Context ctx;
int time;
{
}



/*
 * Draw anything the user wants in 2D.
 * Drawing bounds are (0,0) - (Width-1, Height-1), origin in upper-left corner.
 */
static void draw_user_2d_graphics( Context ctx, int time )
{
}




/*
 * Only draw the 3-D elements of the scene.  No matrix, viewport, etc
 * operations are done here.  This function is useful for the CAVE
 * since it controls the viewing parameters.
 * Input:  ctx - the context
 *         animflag - 1=animating, 0=not animating
 */
void render_3d_only( Context ctx, int animflag )
{
   int labels, i;

   set_current_window( ctx );

   if (animflag)
      labels = !ctx->ContnumFlag;
   else
      labels = ctx->ContnumFlag;

   /* Loop over antialiasing passes */
   for (i=0; i < (ctx->PrettyFlag ? AA_PASSES : 1); i++) {

      start_aa_pass(i);
#ifndef CAVE
      /* don't clear window if using the cave */
      clear_color( ctx->BgColor );
      clear_3d_window();
#endif

      /*** Draw 3-D lines ***/

      render_trajectories( ctx, ctx->CurTime );

      if (ctx->DisplayBox) {
         draw_box( ctx, ctx->CurTime );
      }

      if (ctx->DisplayCursor) {
         draw_cursor( ctx->RibbonFlag,
                      ctx->CursorX, ctx->CursorY, ctx->CursorZ,
                      *ctx->CursorColor );
         if (ctx->DisplayBox) {
            print_cursor_position( ctx, ctx->CurTime );
         }
      }

      render_hslices( ctx, ctx->CurTime, labels, animflag );
      render_vslices( ctx, ctx->CurTime, labels, animflag );

      render_hwind_slices( ctx, ctx->CurTime, animflag );
      render_vwind_slices( ctx, ctx->CurTime, animflag );
      render_stream_slices( ctx, ctx->CurTime, animflag );

      /*** Draw opaque 3-D graphics ***/

      if (ctx->TopoFlag && ctx->DisplayTopo) {
         draw_topo( ctx, ctx->CurTime, ctx->DisplayTexture, 0 );
         if (ctx->MapFlag && ctx->DisplayMap) {
            /* topo-following map */
            set_color( PACK_COLOR( 0, 0, 0, 255 ) );
            draw_map( ctx, ctx->CurTime, 0 );
         }
      }
      else if (ctx->MapFlag && ctx->DisplayMap) {
         if (ctx->DisplayTexture) {
            draw_topo( ctx, ctx->CurTime, 1, 1 );  /* flat texture map */
            set_depthcue( 0 );
            set_color( PACK_COLOR( 0, 0, 0, 255 ) ); /* black map lines */
         }
         else {
            set_color( ctx->MapColor );
            set_depthcue( ctx->DepthCue );
         }
         draw_map( ctx,ctx->CurTime, 1 );
         set_depthcue(0);
      }
      else if (ctx->DisplayTexture) {
         /* just draw flat textured image */
         draw_topo( ctx, ctx->CurTime, 1, 1 );
      }

      render_isosurfaces( ctx, ctx->CurTime, 1, animflag );

      render_chslices( ctx, ctx->CurTime, 1, animflag );

      render_cvslices( ctx, ctx->CurTime, 1, animflag );

      /*draw_user_3d_graphics( ctx, ctx->CurTime );*/

      /*** Draw transparent 3-D objects ***/
      render_isosurfaces( ctx, ctx->CurTime, 0, animflag );
      render_chslices( ctx, ctx->CurTime, 0, animflag );
      render_cvslices( ctx, ctx->CurTime, 0, animflag );
      if (ctx->CurrentVolume!=-1) {
         draw_volume( ctx, ctx->CurTime, ctx->CurrentVolume,
                      ctx->VolumeColors[ctx->CurrentVolume] );
      }

      end_aa_pass(i);

   } /* aa passes */

}



/*
 * Only draw the 2-D elements of the scene.  No matrix, viewport, etc
 * operations are done here.
 * Input:  ctx - the context
 */
void render_2d_only( Context ctx )
{
   if (ctx->DisplayClock) {
      draw_clock( ctx, ctx->BoxColor );
      draw_logo( ctx, ctx->BoxColor );
   }
   if (ctx->DisplayInfo) {
      print_info(ctx);
   }
   if (ctx->DisplayProbe) {
      draw_probe(ctx);
   }
   if (ctx->DisplayCursor && ctx->DisplayBox) {
      print_cursor_position( ctx, ctx->CurTime );
   }

   if (ctx->PointerX>=0 && ctx->PointerY>=0)
      draw_fake_pointer(ctx);  /* for remote widget mode */

   set_color( ctx->LabelColor );
   render_text_labels(ctx);

#ifdef COLORLEGENDS
   /* Draw color map legends of color slices (as much as fit in window) */
   draw_colorlegends(ctx);
#endif   

   /*draw_user_2d_graphics( ctx->CurTime );*/
}



/*
 * Redraw everything in the 3-D window but don't display it yet.  Call
 * swap_3d_window() to do that.
 * Input:  ctx - the vis5d context
 *         animflag - 1=animating, 0=notanimating
 * Return:  0 = ok, -1 = error.
 */
void render_everything( Context ctx, int animflag )
{
   if (get_frame(ctx, ctx->CurTime)) {
      return;
   }

   /*** Draw 3-D Objects ***/
   set_3d( ctx->GfxProjection, ctx->FrontClip, ctx->Zoom, (float*) ctx->CTM );
   render_3d_only( ctx, animflag );

   /*** Draw 2-D objects ***/
   set_2d();
   render_2d_only( ctx );

   if (ctx->AnimRecord) {
      save_frame( ctx, ctx->CurTime );
   }
}

