/* queue.c */

/* VIS-5D version 4.2 */

/*
VIS-5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1995 Bill Hibbard, Brian Paul, Dave Santek,
and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/



#include <stdio.h>
#include <unistd.h>
#include "analysis.h"
#include "globals.h"
#include "queue.h"
#include "sync.h"



#define QSIZE 3000


struct entry {
   Context ctx;			/* the vis5d context */
   int type;			/* the type of entry */
   int i1, i2, i3;		/* integer arguments */
   float f1, f2, f3, f4, f5;	/* float arguments */
};



static struct entry queue[QSIZE];
static int qsize;
static int qhead, qtail;      /* remove from head, add to tail */
static int qwaiters;
static LOCK qlock;
static SEMAPHORE qnotempty;




void init_queue( void )
{
   ALLOC_LOCK( qlock );
   ALLOC_SEM( qnotempty, 0 );
   qsize = 0;
   qhead = qtail = 0;
   qwaiters = 0;
}



void terminate_queue( void )
{
   FREE_LOCK( qlock );
   FREE_SEM( qnotempty );
}



/*
 * Return number of entries still in the queue and the number of threads
 * waiting for work to do.
 * If size==0 and waiters==NumWorkThreads then we're idle.
 */
void get_queue_info( int *size, int *waiters )
{
   LOCK_ON( qlock );
   *size = qsize;
   *waiters = qwaiters;
   LOCK_OFF( qlock );
}




static void add_qentry( Context ctx, int urgent, int type,
                        int i1, int i2, int i3,
                        float f1, float f2, float f3, float f4, float f5 )
{
   int pos, i, found=0;

   LOCK_ON( qlock );

   while (qsize==QSIZE-2) {
      if (Debug)
	 printf("QUEUE FULL!!!\n");
      LOCK_OFF( qlock );
      sleep(1);  /* wait a second for the queue to empty a bit */
      LOCK_ON( qlock );
   }

   /* check if already in the queue */
   pos = qhead;
   found = 0;

   for (i=0;i<qsize;i++) {
      /* check if request is already in the queue */
      if (queue[pos].ctx==ctx &&
          type!=TASK_TRAJ &&
	  queue[pos].type==type &&     /* only need to test */
	  queue[pos].i1==i1 &&         /* the first four fields */
	  queue[pos].i2==i2) {
	 /* already in queue, cancel it if urgent */
	 found = 1;
	 if (urgent)
	   queue[pos].type = TASK_NULL;
	 break;
      }
      else {
	 pos++;
	 if (pos==QSIZE)
	   pos = 0;
      }
   }

   if (urgent) {
      /* insert at head */
      if (qhead==0)
	 qhead = QSIZE-1;
      else
	 qhead--;
      pos = qhead;
      qsize++;
      SIGNAL_SEM( qnotempty );
   }
   else if (!found) {
      /* insert at tail */
      pos = qtail;
      qtail++;
      if (qtail==QSIZE)
	qtail = 0;
      qsize++;
      SIGNAL_SEM( qnotempty );
   }

   queue[pos].ctx = ctx;
   queue[pos].type = type;
   queue[pos].i1 = i1;
   queue[pos].i2 = i2;
   queue[pos].i3 = i3;
   queue[pos].f1 = f1;
   queue[pos].f2 = f2;
   queue[pos].f3 = f3;
   queue[pos].f4 = f4;
   queue[pos].f5 = f5;

   if (Debug) {
      if (urgent)
	printf("**URGENT** **URGENT** **URGENT** **URGENT** ");
      printf("ADDED AT POS=%d\n", pos );
   }

   LOCK_OFF( qlock );
}



/*** get_qentry *******************************************************
   Return an entry from the work queue.  If the queue is empty, this
   fuction blocks.
**********************************************************************/
void get_qentry( Context *ctx, int *type,
                 int *i1, int *i2, int *i3,
                 float *f1, float *f2, float *f3, float *f4, float *f5 )
{
   if (Debug) printf("get_qentry\n");

   LOCK_ON( qlock );

   while (qsize==0) {
      qwaiters++;
      LOCK_OFF( qlock );
      WAIT_SEM( qnotempty );
      LOCK_ON( qlock );
      qwaiters--;
   }


   if (qsize>0) {
      /* remove from head */
      *ctx = queue[qhead].ctx;
      *type = queue[qhead].type;
      *i1 = queue[qhead].i1;
      *i2 = queue[qhead].i2;
      *i3 = queue[qhead].i3;
      *f1 = queue[qhead].f1;
      *f2 = queue[qhead].f2;
      *f3 = queue[qhead].f3;
      *f4 = queue[qhead].f4;
      *f5 = queue[qhead].f5;

      if (Debug) printf("REMOVED FROM POS=%d\n", qhead );

      if (*type!=TASK_QUIT) {
	 qhead++;
	 if (qhead==QSIZE)
	   qhead = 0;

	 qsize--;
      }

   }
   else {
      *type = TASK_NULL;
   }
   LOCK_OFF( qlock );
   if (Debug) printf("return\n");
}



/*** request_quit *****************************************************
   This is called when we want to exit the program.  It puts a message
   on the queue that all 'work' threads are to exit.
**********************************************************************/
void request_quit( Context ctx )
{
   add_qentry( ctx, 1, TASK_QUIT, 0,0,0, 0.0,0.0,0.0,0.0,0.0 );
}



/*
 * Request that an isosurface is to be computed.
 * Input:  ctx - the context
 *         time - the timestep number
 *         var - which variable
 *         urgent - 1 = put request at front of queue, 0 = put at back of queue
 */
void request_isosurface( Context ctx, int time, int var, int urgent )
{
   if (ctx->SurfTable[var][time].valid==0 ||
       ctx->SurfTable[var][time].isolevel!=ctx->IsoLevel[var] ||
       ctx->SurfTable[var][time].colorvar!=ctx->IsoColorVar[var]) {
      add_qentry( ctx, urgent, TASK_ISOSURFACE, time, var, 0,
                  0.0, 0.0, 0.0, 0.0, 0.0 );
   }
}



/*
 * Request that a horizontal contour line slice is to be computed.
 * Input:  ctx - the context
 *         time - the timestep number
 *         var - which variable
 *         urgent - 1 = put request at front of queue, 0 = put at back of queue
 */
void request_hslice( Context ctx, int time, int var, int urgent )
{
   if (ctx->HSliceTable[var][time].valid==0 ||
       ctx->HSliceTable[var][time].level!=ctx->HSliceLevel[var] ||
       ctx->HSliceTable[var][time].interval!=ctx->HSliceInterval[var] ||
       ctx->HSliceTable[var][time].lowlimit!=ctx->HSliceLowLimit[var] ||
       ctx->HSliceTable[var][time].highlimit!=ctx->HSliceHighLimit[var] ) {
      add_qentry( ctx, urgent, TASK_HSLICE, time, var, 0,
		 0.0, 0.0, 0.0, 0.0, 0.0 );
   }
}



/*
 * Request that a vertical contour line slice is to be computed.
 * Input:  ctx - the context
 *         time - the timestep number
 *         var - which variable
 *         urgent - 1 = put request at front of queue, 0 = put at back of queue
 */
void request_vslice( Context ctx, int time, int var, int urgent )
{
   if (ctx->VSliceTable[var][time].valid==0 ||
       ctx->VSliceTable[var][time].r1!=ctx->VSliceR1[var] ||
       ctx->VSliceTable[var][time].c1!=ctx->VSliceC1[var] ||
       ctx->VSliceTable[var][time].r2!=ctx->VSliceR2[var] ||
       ctx->VSliceTable[var][time].c2!=ctx->VSliceC2[var] ||
       ctx->VSliceTable[var][time].interval!=ctx->VSliceInterval[var] ||
       ctx->VSliceTable[var][time].lowlimit!=ctx->VSliceLowLimit[var] ||
       ctx->VSliceTable[var][time].highlimit!=ctx->VSliceHighLimit[var] ) {
      add_qentry( ctx, urgent, TASK_VSLICE, time, var, 0,
		 0.0, 0.0, 0.0, 0.0, 0.0 );
   }
}



/*
 * Request that a horizontal color slice is to be computed.
 * Input:  ctx - the context
 *         time - the timestep number
 *         var - which variable
 *         urgent - 1 = put request at front of queue, 0 = put at back of queue
 */
void request_chslice( Context ctx, int time, int var, int urgent )
{
   if (ctx->CHSliceTable[var][time].valid==0 ||
       ctx->CHSliceTable[var][time].level!=ctx->CHSliceLevel[var] ) {
      add_qentry( ctx, urgent, TASK_CHSLICE, time, var, 0,
		 0.0, 0.0, 0.0, 0.0, 0.0 );
   }
}



/*
 * Request that a vertical colored slice is to be computed.
 * Input:  ctx - the context
 *         time - the timestep number
 *         var - which variable
 *         urgent - 1 = put request at front of queue, 0 = put at back of queue
 */
void request_cvslice( Context ctx, int time, int var, int urgent )
{
   if (ctx->CVSliceTable[var][time].valid==0 ||
       ctx->CVSliceTable[var][time].r1!=ctx->CVSliceR1[var] ||
       ctx->CVSliceTable[var][time].c1!=ctx->CVSliceC1[var] ||
       ctx->CVSliceTable[var][time].r2!=ctx->CVSliceR2[var] ||
       ctx->CVSliceTable[var][time].c2!=ctx->CVSliceC2[var] ) {
      add_qentry( ctx, urgent, TASK_CVSLICE, time, var, 0,
		 0.0, 0.0, 0.0, 0.0, 0.0 );
   }
}



/*
 * Request that a horizontal wind vector slice is to be computed.
 * Input:  ctx - the context
 *         time - the timestep number
 *         ws - which wind slice
 *         urgent - 1 = put request at front of queue, 0 = put at back of queue
 */
void request_hwindslice( Context ctx, int time, int ws, int urgent )
{
   if (ctx->HWindTable[ws][time].valid==0 ||
       ctx->HWindTable[ws][time].uvar!=ctx->Uvar[ws] ||
       ctx->HWindTable[ws][time].vvar!=ctx->Vvar[ws] ||
       ctx->HWindTable[ws][time].wvar!=ctx->Wvar[ws] ||
       ctx->HWindTable[ws][time].level!=ctx->HWindLevel[ws] ||
       ctx->HWindTable[ws][time].scale!=ctx->HWindScale[ws] ||
       ctx->HWindTable[ws][time].density!=ctx->HWindDensity[ws] ) {
      add_qentry( ctx, urgent, TASK_HWIND, time, ws, 0,
		 0.0, 0.0, 0.0, 0.0, 0.0 );
   }
}



/*
 * Request that a vertical wind vector slice is to be computed.
 * Input:  ctx - the context
 *         time - the timestep number
 *         ws - which wind slice
 *         urgent - 1 = put request at front of queue, 0 = put at back of queue
 */
void request_vwindslice( Context ctx, int time, int ws, int urgent )
{
   if (ctx->VWindTable[ws][time].valid==0 ||
       ctx->VWindTable[ws][time].uvar!=ctx->Uvar[ws] ||
       ctx->VWindTable[ws][time].vvar!=ctx->Vvar[ws] ||
       ctx->VWindTable[ws][time].wvar!=ctx->Wvar[ws] ||
       ctx->VWindTable[ws][time].r1!=ctx->VWindR1[ws] ||
       ctx->VWindTable[ws][time].c1!=ctx->VWindC1[ws] ||
       ctx->VWindTable[ws][time].r2!=ctx->VWindR2[ws] ||
       ctx->VWindTable[ws][time].c2!=ctx->VWindC2[ws] ||
       ctx->VWindTable[ws][time].scale!=ctx->VWindScale[ws] ||
       ctx->VWindTable[ws][time].density!=ctx->VWindDensity[ws] ) {
      add_qentry( ctx, urgent, TASK_VWIND, time, ws, 0,
		 0.0, 0.0, 0.0, 0.0, 0.0 );
   }
}



/*
 * Request that a horizontal streamline slice is to be computed.
 * Input:  ctx - the context
 *         time - the timestep number
 *         ws - which slice
 *         urgent - 1 = put request at front of queue, 0 = put at back of queue
 */
void request_streamslice( Context ctx, int time, int ws, int urgent )
{
   if (ctx->StreamTable[ws][time].valid==0 ||
       ctx->StreamTable[ws][time].uvar!=ctx->Uvar[ws] ||
       ctx->StreamTable[ws][time].vvar!=ctx->Vvar[ws] ||
       ctx->StreamTable[ws][time].level!=ctx->StreamLevel[ws] ||
       ctx->StreamTable[ws][time].density!=ctx->StreamDensity[ws] ) {
      add_qentry( ctx, urgent, TASK_STREAM, time, ws, 0,
                 0.0, 0.0, 0.0, 0.0, 0.0 );
   }
}



/*** request_traj *****************************************************
   This is called to compute a new trajectory.
   Input:  row, col, lev - start pos in [0..Nr-1],[0..Nc-1],[0..Nl-1].
           time - start time in [0..NumTimes-1].
           group - which traj group in [0..TRAJGROUPS-1].
           rib - non-zero means make ribbon instead of line traj.
           step - integration step multiplier.
           len - trajectory length multiplier.
**********************************************************************/
void request_traj( Context ctx, float row, float col, float lev,
                   int time, int group, int rib,
                   float step, float len )
{
   add_qentry( ctx, 1, TASK_TRAJ,
	       time, group, rib, row, col, lev, step, len );
}



/*
 * Request that all trajectories in a group be recolored according to
 * the current TrajColorVar variable.
 */
void request_traj_recoloring( Context ctx, int traj_set )
{
   add_qentry( ctx, 1, TASK_TRAJ_RECOLOR,
               traj_set, 0, 0, 0.0, 0.0, 0.0, 0.0, 0.0 );
}




/*
 * Put a job into the queue for computing an external function
 */
void request_ext_func( Context ctx, int time, int var )
{
#ifdef SINGLE_TASK
   calc_ext_func( ctx, time, var, 1 );
#else
   add_qentry( ctx, 0, TASK_EXT_FUNC, time, var, 0,
	       0.0, 0.0, 0.0, 0.0, 0.0 );
#endif
}

