/* graphics.c */

/* Graphics functions for Stellar XFDI */

/* VIS-5D version 4.0 */

/*
VIS-5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990, 1991, 1992, 1993, 1994  Bill Hibbard, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/



#include <math.h>
#include <stdio.h>
#include <string.h>
#include <xfdi.h>
#include <X11/Xlib.h>
#include <X11/Xcm.h>
#include <X11/Xdb.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include "graphics.h"
#include "foo.h"   /* this is a test */


typedef float Matrix[4][4];


/* API - to globals.c (or private indexed structure);
   except constants */

/*** Private Variables: ***/
static Window wmain;
static float Scale;
static Matrix ModelMat, InvModelMat, ProjMat;
static Matrix IdMat = { 1.0, 0.0, 0.0, 0.0,
			0.0, 1.0, 0.0, 0.0,
                        0.0, 0.0, 1.0, 0.0,
			0.0, 0.0, 0.0, 1.0 };
static int pretty_flag = 0;
static int Perspective = 0;


#define DEFAULT_FONT "vr-25"
#define EYEDIST 3.0
#define FOV 400
#define cot(X)  (sin(X)/cos(X))

static XFontStruct *visfont;
static XFDIGC gc0;
static GC gc1;
static GC xgc;       /* extra GC for transparent surfaces */
static XdbBufferAttributes  att;
static unsigned long mask;
static Drawable buff0, aabuff0, sorta, sorti, sortz;
static float LightColor[3] = { 0.7, 0.7, 0.7 };
static float AmbientColor[3] = { 1.0, 1.0, 1.0 };
static float CurTrans;
static float linewidth;

/* Pretty rendering stuff: */
#define AA_INC  (1.1/2)
static float  xoffsets[AA_PASSES] =
  { 0, 0, 0, AA_INC, -AA_INC, AA_INC, -AA_INC, -AA_INC, AA_INC };
static float  yoffsets[AA_PASSES] =
  { 0, -AA_INC, AA_INC, 0, 0, AA_INC, -AA_INC, AA_INC, -AA_INC };
static float  blendcoeffs[AA_PASSES] =
  { 0.99999, 0.722/1.472, 0.722/2.194, 0.722/2.916, 0.722/3.638,
    0.639/4.277, 0.639/4.916, 0.639/5.555, 0.639/6.194 };




void init_graphics2( alphaflag )
int alphaflag;
{
   HQR_available = 1;
   Perspec_available = 1;
}


void terminate_graphics()
{
}



Window make_window( title, xpos, ypos, width, height,
                    fontname, fontsize, NPGLwin )
char *title;
int xpos, ypos;
int width, height;
char *fontname;
int fontsize;
Window *NPGLwin;
{
   XColor background, foreground;
   XSetWindowAttributes xswa;
   XGCValues gcv;
   XSizeHints sizehints;
   XEvent pe;
   int depth;
   int swa_flags;
   int borderwidth;
   static float zero[3]  = {0.0, 0.0, 0.0};
   static unsigned long fgpix, bgpix;

   WinWidth = width;
   WinHeight = height;

   if (GfxVisual->class!=TrueColor)
      die("Display is not TrueColor.\n");

   /* Allocate foreground and background colors */
   background.red   = 0x0;
   background.green = 0x0;
   background.blue  = 0x0;
   background.flags = DoRed | DoGreen | DoBlue;
   if (XAllocColor(GfxDpy, GfxColormap, &background))
      bgpix = background.pixel;
   else {
      fprintf(stderr, "Can't allocate background color\n");
      bgpix = 0;
   }
   foreground.red   = 0xffff;
   foreground.green = 0xffff;
   foreground.blue  = 0xffff;
   foreground.flags = DoRed | DoGreen | DoBlue;
   if (XAllocColor(GfxDpy, GfxColormap, &foreground))
      fgpix = foreground.pixel;
   else {
      fprintf(stderr, "Can't allocate foreground color\n");
      fgpix = 0xffffff;
   }

   /* set window attributes */
   xswa.colormap = GfxColormap;
   xswa.event_mask = ExposureMask | PointerMotionMask | KeyPressMask
                | ButtonPressMask | ButtonReleaseMask | StructureNotifyMask;
   xswa.background_pixel = bgpix;
   xswa.border_pixel = fgpix;
   swa_flags = CWColormap | CWEventMask | CWBackPixel | CWBorderPixel;

   if (width==ScrWidth && height==ScrHeight)
     borderwidth = 0;
   else
     borderwidth = 5;

   /* Create the window */
   wmain = XCreateWindow( GfxDpy, RootWindow(GfxDpy, GfxScr), xpos, ypos,
                        width, height, borderwidth,
		        depth, InputOutput, GfxVisual, swa_flags, &xswa);

   if (!wmain)
      die("Unable to open 3-D window");

   /* Set size hints for window manager */
   sizehints.x = xpos;
   sizehints.y = ypos;
   sizehints.width  = width;
   sizehints.height = height;
   sizehints.flags = USSize | USPosition;

   XSetNormalHints(GfxDpy, wmain, &sizehints);
   XSetStandardProperties( GfxDpy, wmain, title, title,
		           None, (char **)NULL, 0, &sizehints);

   /* make gc */
   gcv.foreground = fgpix;
   gcv.background = bgpix;
   gc1 = XCreateGC(GfxDpy, wmain, GCForeground | GCBackground, &gcv);

   xgc = XCreateGC(GfxDpy, wmain, 0, NULL );

   /* set font */
   if (fontname[0]==0)
      strcpy( fontname, DEFAULT_FONT );

   if ( !(visfont = XLoadQueryFont( GfxDpy, fontname )) ) {
      fprintf( stderr, "Unable to load font: %s\n", fontname );
      exit(-1);
   }
   XSetFont( GfxDpy, gc1, visfont->fid );
   FontHeight = visfont->ascent + visfont->descent;
   FontDescent = visfont->descent;

   /* set pointer shape */
   XDefineCursor( GfxDpy, wmain, XCreateFontCursor(GfxDpy,XC_top_left_arrow) );
   /* wait for window to be mapped */
   XMapWindow(GfxDpy, wmain);
   do {
      XNextEvent(GfxDpy, &pe);
   } while (pe.type!=Expose);
   XSync(GfxDpy, 0);


   /*Initialize our offscreen pixmap.  This consists of the following
     objects: A buffer == (offscreen pixmap associated with a window),
     Renderable = (buffer made "renderable"), Z-buffer = (initially assoc'd
     with a renderable), gc = (3D graphics context - contains assoc of
     renderable).
   */
   att.update_action = XdbRetained;  /* Don't clear pixmap between redraws */
   att.update_mode = XdbFrequently;  /* Hint to server about # of updates  */
   att.resize_mode = XdbUndefinedBits;
   mask = DBUpdateAction | DBUpdateMode;

   /* Create a Renderable */
   buff0 = XdbCreateBuffer(GfxDpy, wmain, &att, mask, NULL);
   XFDIMakeRenderable(GfxDpy, buff0);
   XFDICreateZbuffer(GfxDpy, buff0);

   /* create XFDI GC */
   gc0 = XFDICreateGc3(GfxDpy, buff0);
   XFDISetVectorFlag(GfxDpy, gc0, XFDIObjectMatrix, XFDIVectorNoHomo);
   XFDISetClipPlane( GfxDpy, gc0, 0, XFDIClipAfterTransform,
                     0.0, 0.0, 1.0, 0.0, 1.0 );
   XFDISetClipLevel( GfxDpy, gc0, (XFDIClipClipPlanes ^ XFDIClipPlane0)
                     | XFDIClipZmin );
   XFDISetRenderable(GfxDpy, gc0, (Drawable) buff0);

   XFDISetSurfaceBackground(GfxDpy, gc0, bgpix);
   XFDISetLineBackground(GfxDpy, gc0, bgpix);

   /* Set surface properties */
   XFDISetSurfaceLightingModel(GfxDpy, gc0, XFDIDiffuse);
   XFDISetSurfaceShading(GfxDpy, gc0, XFDIShadeColor);
   XFDISetTransparencyMode( GfxDpy, gc0, XFDITransparencyFast );
   XFDISetSurfaceTransparency(GfxDpy, gc0, 1.0);

   /* Set the drawing mode - z buffered */
   XFDISetDrawMode(GfxDpy, gc0, XFDIDrawZ);

   /* set depth cueing numbers */
   XFDISetDepthCueing(GfxDpy, gc0, 1.0, -1.0, 0.5, 1.0);
   XFDISetDepthCueingFlag(GfxDpy, gc0, 1 );

   /* Setup light sources */
   XFDISetLight(GfxDpy, gc0, 0, XFDIAmbLight, zero, zero, AmbientColor);
   XFDISetLightStatus(GfxDpy, gc0, 0, XFDIOn);

   XFDISetLightStatus(GfxDpy, gc0, 1, XFDIOn);
   XFDISetLightStatus(GfxDpy, gc0, 2, XFDIOn);

   return wmain;
}



/*** set_pointer ******************************************************
   Set mouse pointer image to an arrow or 'busy' clock.
   Input: p - 0 = pointer image
              1 = clock image
**********************************************************************/
void set_pointer( p )
int p;
{
   if (p) {
      /* make busy cursor */
      XDefineCursor( GfxDpy, wmain, XCreateFontCursor(GfxDpy,XC_watch) );
   }
   else {
      XDefineCursor( GfxDpy, wmain, XCreateFontCursor(GfxDpy,XC_top_left_arrow) );
   }
}



/*** save_formats *****************************************************
   Return a set of flag bits indicating what image formats we can
   save as.
**********************************************************************/
int save_formats()
{
   return SAVE_XWD;
}



/*** save_window ******************************************************
   Save the current image in the 3-D window to the named file.
   Input:  filename - image filename
           format - SAVE_SGI = SGI .rgb
	            SAVE_GIF = .GIF
		    SAVE_XWD = .xwd   (X window dump)
   Return: 1 = no errors
           0 = error occured.
**********************************************************************/
int save_window( filename, format )
char filename[];
int format;
{
   FILE *f;

   set_pointer(1);

   /* Make an X window dump file (.xwd) */
   f = fopen(filename,"w");
   if (f) {
      Window_Dump( GfxDpy, GfxScr, wmain, f );
      fclose(f);
      set_pointer(0);
      return 1;
   }
   else {
      printf("Error unable to open %s for writing\n", filename);
      set_pointer(0);
      return 0;
   }
}



/*** print_window *****************************************************
   Print the current image in the 3-D window to the default PostScript
   printer.
   Return: 1 = ok, 0 = error
**********************************************************************/
int print_window( )
{
   static char xwd_file[] = "/usr/tmp/VIS-5D_image.xwd";
   char cmd[1000];
   FILE *f;

   set_pointer(1);

   /* Make an X window dump file (.xwd) */
   f = fopen(xwd_file,"w");
   if (f) {
      printf("Writing X window dump: %s\n", xwd_file);
      Window_Dump( GfxDpy, GfxScr, wmain, f );
      fclose(f);

      /* Use xpr to print the window dump */
      if (!installed("xpr"))  return 0;
      sprintf(cmd,"xpr -device ps %s | lpr\n", xwd_file );
      printf("Executing: %s\n", cmd );
      system(cmd);

      /* delete xwd file */
      unlink( xwd_file );

      printf("Done.\n");
      set_pointer(0);
      return 1;
   }
   else {
      printf("Error unable to open %s for writing\n", xwd_file);
      set_pointer(0);
      return 0;
   }
}




/*** clear_window *****************************************************
   Clear the graphics window.  This is called prior to rendering
   a frame.
**********************************************************************/
void clear_window( bgcolor )
unsigned int bgcolor;
{
   XFDIClear(GfxDpy, gc0, (unsigned long) bgcolor,
                 (unsigned long) 0x0, (XRectangle *) NULL);
   XFDISetLineBackground( GfxDpy, gc0, bgcolor );

}



/*
 * Called when window size changes.
 */
void resize_window( width, height )
int width, height;
{
   /* just save new values */
   WinWidth = width;
   WinHeight = height;
   AspectRatio = 1.0;   /* is this right??? */
}



/*** update_window ****************************************************
   Update the 3-D window.  This called when finished rendering a frame.
**********************************************************************/
void update_window()
{
   if (pretty_flag)
     XdbUpdateWindows(GfxDpy, &aabuff0, NULL, 1);
   else
     XdbUpdateWindows(GfxDpy, &buff0, NULL, 1);
}



/*** set_2d ***********************************************************
   This is to be called prior to any 2-D rendering calls.
**********************************************************************/
void set_2d()
{
   /* nothing */
}



/*** set_3d ***********************************************************
   This is to be called prior to any 3-D rendering calls.
   Input:  ctm - the current transformation matrix.
           scale - scaling factor of the ctm.
	   perspec - 0 = parallel, 1 = perspective projection
**********************************************************************/
void set_3d( ctm, scale, perspec )
Matrix ctm;
float scale;
int perspec;
{
   static float zero[3] = {0.0, 0.0, 0.0};
   float dir[3];
   float mag;

   mat_copy( ModelMat, ctm );
   mat_inv( InvModelMat, ctm );
   InvModelMat[3][0] = InvModelMat[3][1] = InvModelMat[3][2] = 0.0;

   if (perspec) {
      /* Perspective projection */
      Matrix m;
      float front, rear;

      front = EYEDIST - scale * 1.75 * 1.0;
      if (front<0.01)  front = 0.01;
      rear = EYEDIST + scale * 1.75;
      mat_copy( ProjMat, IdMat );
      ProjMat[2][3] = 1.0 / EYEDIST;

      XFDISetVectorFlag( GfxDpy, gc0, XFDIObjectMatrix, XFDIVectorPoint );
      XFDISetVectorFlag( GfxDpy, gc0, XFDILightMatrix,  XFDIVectorPoint );

      mat_mul( m, ModelMat, ProjMat );
      XFDISetMatrix(GfxDpy, gc0, XFDIObjectMatrix, m, XFDIMatrix4x4);
   }
   else {
      /* parallel projection */
      XFDISetVectorFlag( GfxDpy, gc0, XFDIObjectMatrix, XFDIVectorNoHomo );
      XFDISetMatrix(GfxDpy, gc0, XFDIObjectMatrix, ModelMat, XFDIMatrix4x4);
   }
   dir[0] = ModelMat[0][2];  dir[1] = ModelMat[1][2];  dir[2] = ModelMat[2][2];
   mag = (float) sqrt( dir[0]*dir[0] + dir[1]*dir[1] + dir[2]*dir[2] );
   dir[0] /= mag;
   dir[1] /= mag;
   dir[2] /= mag;
   XFDISetLight(GfxDpy, gc0, 1, XFDIDirLight, zero, dir, LightColor );

   dir[0] = -dir[0];   dir[1] = -dir[1];   dir[2] = -dir[2];
   XFDISetLight(GfxDpy, gc0, 2, XFDIDirLight, zero, dir, LightColor );

   Perspective = perspec;
   Scale = scale;
}



/*** project **********************************************************
   Use current transformation and viewing information to project a
   point p from 3-D graphics coordinates to 2-D window coordinates
   in [0,WinWidth-1]x[0,WinHeight-1].
   Input:  p - 3-D point in graphics space
   Output:  x, y - 2-D point in window pixel coordinates.
**********************************************************************/
void project( p, x, y )
float p[3];
float *x, *y;
{
   float q[4], nx, ny;

   q[0] = p[0];
   q[1] = p[1];
   q[2] = p[2];
   q[3] = 1.0;
   if (Perspective) {
      /* Perspective Projection */
      mat_vecmul4( q, ModelMat );
      mat_vecmul4( q, ProjMat );
      nx = q[0] / q[3];   /* divide by w */
      ny = q[1] / q[3];
      *x = (nx+1.0) / 2.0 * WinWidth;
      *y = (1.0-ny) / 2.0 * WinHeight;
   }
   else {
      /* Parallel Projection */
      mat_vecmul4( q, ModelMat );
      nx = q[0];
      ny = q[1];
      *x = ((nx/AspectRatio+1.0) / 2.0 * WinWidth);
      *y = (1.0-ny) / 2.0 * WinHeight;
   }

}



/*** unproject ********************************************************
   Given a 2-D window coordinate in [0,WinWidth-1]x[0,WinHeight-1],
   return the parametric equation of a line in 3-D such that the
   projection of the line from 3-D to 2-D is a point at the window
   coordinate.
   Input:  x, y - window coordinate.
   Output:  p, d - parametric equation of line:  l = p + t*d
                   NOTE, d will have unit length
**********************************************************************/
void unproject( x, y, p, d )
float x, y;
float p[3], d[3];
{
   float mag;

   if (Perspective) {
      /* Perspective Projection */
      float p2[3];
      Matrix inv;
      /* z = 0: */
      p[0] = (float) (x/WinWidth) * 2.0 - 1.0;
      p[1] = (float) 1.0 - (y/WinHeight) * 2.0;
      p[2] = 0.0;
      /* z = eyedist */
      p2[0] = p[0] * 2.0;
      p2[1] = p[1] * 2.0;
      p2[2] = 1.0 / ProjMat[2][3];
      mat_inv( inv, ModelMat );
      /*inv[3][0] = inv[3][1] = inv[3][2] = 0.0;*/
      inv[3][0] = -ModelMat[3][0];
      inv[3][1] = -ModelMat[3][1];
      inv[3][2] = -ModelMat[3][2];
      inv[3][3] = 1.0;
      mat_vecmul3( p, inv );
      mat_vecmul3( p2, inv );
      d[0] = p2[0] - p[0];
      d[1] = p2[1] - p[1];
      d[2] = p2[2] - p[2];
   }
   else {
      /* Parallel Projection */
      d[0] = 0.0;
      d[1] = 0.0;
      d[2] = 1.0;
      mat_vecmul3( d, InvModelMat );
      p[0] = ((x/WinWidth) * 2.0 - 1.0) * AspectRatio;
      p[1] = 1.0 - (y/WinHeight) * 2.0;
      p[2] = 0.0;
      p[0] -= ModelMat[3][0];
      p[1] -= ModelMat[3][1];
      p[2] -= ModelMat[3][2];
      mat_vecmul3( p, InvModelMat );
   }
   mag = sqrt( d[0]*d[0] + d[1]*d[1] + d[2]*d[2] );
   d[0] /= mag;
   d[1] /= mag;
   d[2] /= mag;
}




/*** set_color ********************************************************
   Set the current drawing color.  The color is specified in packed
   ARGB format; 1-byte each of Alpha, Red, Green, Blue where Alpha
   is the most-significant byte and blue is the least-signficant byte.
**********************************************************************/
void set_color( c )
unsigned int c;
{
   XFDISetSurfaceColor( GfxDpy, gc0, (unsigned long) c );
   XFDISetLineColor( GfxDpy, gc0, (unsigned long) c );
   CurTrans = (float) UNPACK_ALPHA(c) / 255.0;
   XFDISetSurfaceTransparency( GfxDpy, gc0, CurTrans );
   XSetForeground( GfxDpy, gc1, (unsigned long) c );
}



/*** set_depthcue *****************************************************
   Set the line depth cueing flag on or off.
**********************************************************************/
void set_depthcue( onoff )
int onoff;
{
   if (onoff)
      XFDISetDepthCueingFlag(GfxDpy, gc0, 1 );
   else
      XFDISetDepthCueingFlag(GfxDpy, gc0, 0 );
}



/*** set_line_width ***************************************************
   Set the width of lines.
   Input:  w - width in pixels.
**********************************************************************/
void set_line_width( w )
float w;
{
   linewidth = w;
   XFDISetLineWidth( GfxDpy, gc0, w );
}



/*** set_frontclip ****************************************************
   Set the position of the front clipping plane.
   Input:  d - front clipping plane z position in [1,-1] where 1 is
               near viewer and -1 is away from viewer.
**********************************************************************/
void set_frontclip( d )
float d;
{
   XFDISetClipPlane(GfxDpy, gc0, 0, XFDIClipAfterTransform,
		    0.0, 0.0, 1.0, 0.0, d );
}




/*** set_box_clipping *************************************************
   Enable or disable clipping of graphics to a box.
   Input:  onoff - 1 = enable, 0 = disable.
           xmin, xmax, ymin, ymax - clipping bounds in graphics space,
                                    note no Z clipping is done.
**********************************************************************/
void set_box_clipping( onoff, xmin, xmax, ymin, ymax )
int onoff;
float xmin, xmax, ymin, ymax;
{
   unsigned long planes;

   if (onoff) {

      XFDISetClipPlane( GfxDpy, gc0, 1, XFDIClipAfterTransform,
       InvModelMat[0][0], InvModelMat[1][0], InvModelMat[2][0], -xmin, 0.0 );
      XFDISetClipPlane( GfxDpy, gc0, 2, XFDIClipAfterTransform,
       -InvModelMat[0][0], -InvModelMat[1][0], -InvModelMat[2][0], xmax, 0.0 );
      XFDISetClipPlane( GfxDpy, gc0, 3, XFDIClipAfterTransform,
        InvModelMat[0][1], InvModelMat[1][1], InvModelMat[2][1], -ymin, 0.0 );
      XFDISetClipPlane( GfxDpy, gc0, 4, XFDIClipAfterTransform,
       -InvModelMat[0][1], -InvModelMat[1][1], -InvModelMat[2][1], ymax, 0.0 );

      planes = XFDIClipZmin | (XFDIClipClipPlanes ^
	    (XFDIClipPlane0|XFDIClipPlane1|XFDIClipPlane2|
	     XFDIClipPlane3|XFDIClipPlane4));
      XFDISetClipLevel( GfxDpy, gc0, planes );
   }
   else {
      planes = (XFDIClipClipPlanes ^ XFDIClipPlane0) | XFDIClipZmin;
      XFDISetClipLevel( GfxDpy, gc0, planes );
   }
}




/*** set_pretty *******************************************************
   Set pretty rendering mode on or off.
   XFDI:  This function is used to select antialiasing and multi-pass
             transparency.
   SGI:  This function selects line antialiasing.
   Input: onoff - 1 = on, 0 = off.
**********************************************************************/
void set_pretty( onoff )
{
   if (onoff && !pretty_flag) {
      /* turn on */
      /* Create antialiasing buffer */
      aabuff0 = XdbCreateBuffer(GfxDpy, wmain, &att, mask );
      XFDIMakeRenderable(GfxDpy, aabuff0);
      XFDISetBuffer(GfxDpy, gc0, aabuff0, XFDIPmapFilterI );
      /* XFDISetFilter(GfxDpy, gc0, XFDIFilterGeneral ); */
      XFDISetFilter(GfxDpy, gc0, XFDIFilterFilter );

      /* Create multi-pass transparency buffers */
      sorta = XdbCreateBuffer( GfxDpy, wmain, &att, mask );
      sorti = XdbCreateBuffer( GfxDpy, wmain, &att, mask );
      sortz = XdbCreateBuffer( GfxDpy, wmain, &att, mask );
      XFDIMakeRenderable(GfxDpy, sorta);
      XFDIMakeRenderable(GfxDpy, sorti);
      XFDIMakeRenderable(GfxDpy, sortz);
      XFDISetBuffer( GfxDpy, gc0, sorta, XFDIPmapSortA );
      XFDISetBuffer( GfxDpy, gc0, sorti, XFDIPmapSortI );
      XFDISetBuffer( GfxDpy, gc0, sortz, XFDIPmapSortZ );

      XFDISetTransparencyMode( GfxDpy, gc0, XFDITransparencyHQ );
      set_pointer(1);
      pretty_flag = 1;
   }
   else if (!onoff && pretty_flag) {
      /* turn off */
      XdbDestroyBuffer(GfxDpy, aabuff0);
      XdbDestroyBuffer(GfxDpy, sorta);
      XdbDestroyBuffer(GfxDpy, sorti);
      XdbDestroyBuffer(GfxDpy, sortz);
      XFDISetFilter(GfxDpy, gc0, XFDIFilterNone );
      XFDISetTransparencyMode( GfxDpy, gc0, XFDITransparencyFast );
      /* regular cursor */
      set_pointer(0);
      pretty_flag = 0;
   }
}



/*** start_aa_pass ****************************************************
   Start antialiasing pass n, where n is in [0..8].
**********************************************************************/
void start_aa_pass( n )
int n;
{
   if (pretty_flag) {
      XFDISetFilterOffset(GfxDpy, gc0, xoffsets[n],yoffsets[n] );
   }
}




/*** end_aa_pass ******************************************************
   End an antialiasing pass n, where n is in [0..8].
**********************************************************************/
void end_aa_pass( n )
int n;
{
   if (pretty_flag) {
      XFDIFilterBlend( GfxDpy, gc0, blendcoeffs[n], (short) 0, (short) 0,
		       (short) WinWidth, (short) WinHeight );
   }
}



/*** start_hqtrans_pass ***********************************************
   Start a high-quality transparency rendering pass.
**********************************************************************/
void start_hqtrans_pass()
{
   if (pretty_flag) {
      unsigned int width, height;
      width = WinWidth;
      height = WinHeight;

      XFDISetTransparencyMode( GfxDpy, gc0, XFDITransparencyHQ );

      XSetForeground(GfxDpy,xgc,0xffffffff);
      XFillRectangle(GfxDpy,sortz,xgc,0,0,width,height);
      XSetForeground(GfxDpy,xgc,0x0);
      XFillRectangle(GfxDpy,sorti,xgc,0,0,width,height);
      XFillRectangle(GfxDpy,sorta,xgc,0,0,width,height);
   }
}



/*** end_hqtrans_pass *************************************************
   End a high-quality transparency rendering pass.
   Return:  number of unresolved pixels (used to determine when enough
            rendering passes have been made.
**********************************************************************/
int end_hqtrans_pass()
{
   static int prev;
   int pixels = 0;
   if (pretty_flag) {
      short width, height;
      width = WinWidth;
      height = WinHeight;
      XFDITransBlend( GfxDpy, gc0, 0,0, width, height, &pixels );
      XFDISetTransparencyMode( GfxDpy, gc0, XFDITransparencyFast );
   }
   if (pixels==prev)
      pixels = 0;
   prev = pixels;
   return pixels;
}




/*** polytrinorm ******************************************************
   Draw a polytriangle strip with normals.
**********************************************************************/
void polytrinorm( vert, norm, n )
float vert[][3];
float norm[][3];
int n;
{
   if (CurTrans>0.0) {
      XFDIVnormPolyTri( GfxDpy, gc0, vert, norm, n );
   }
}



/*** polytri **********************************************************
   Draw a polytriangle strip without shading.
**********************************************************************/
void polytri( vert, n )
float vert[][3];
int n;
{
   if (CurTrans>0.0) {
      XFDIPolyTri( GfxDpy, gc0, vert, n );
   }
}



/*** polyline *********************************************************
   Draw a 3-D poly line.
**********************************************************************/
void polyline( vert, n )
float vert[][3];
int n;
{
   XFDISetLineWidth( GfxDpy, gc0, linewidth );
   XFDIPolyLine( GfxDpy, gc0, vert, n );
}




/*** disjointpolyline *************************************************
   Draw a series of disjoint 3-D lines.
**********************************************************************/
void disjointpolyline( vert, n )
float vert[][3];
int n;
{
   XFDISetLineWidth( GfxDpy, gc0, linewidth );
   XFDIDisjointPolyLine( GfxDpy, gc0, vert, n );
}





/*** quadmesh *********************************************************
   Draw a quadrilateral mesh with colors at each vertex.
**********************************************************************/
void quadmesh( vert, color, rows, cols )
float vert[][3];
unsigned int color[];
int rows, cols;
{
   if (CurTrans>0.0) {
      /* actually render as a triangle mesh */
      XFDIMeshTriVcolors( GfxDpy, gc0, vert, color, rows, cols );
   }
}



/*** quadmeshnorm *****************************************************
   Draw a quadrilateral mesh with normals and colors at each vertex.
**********************************************************************/
void quadmeshnorm( vert, norm, color, rows, cols )
float vert[][3];
float norm[][3];
unsigned int color[];
int rows, cols;
{
   if (CurTrans>0.0) {
      /* actually render as a triangle mesh */
      XFDIVnormMeshTriVcolors( GfxDpy, gc0, vert, norm, color, rows, cols );
   }
}



void quadmeshnorm_translated( vert, norm, color, rows, cols, tx, ty )
float vert[][3];
float norm[][3];
unsigned int color[];
int rows, cols;
float tx, ty;
{
   quadmeshnorm( vert, norm, color, rows, cols );
}



/*** polyline2d *******************************************************
   Draw a 2-D poly line.  Coordinates are in pixels with the origin
   in the upper-left corner of the window.  NOTE:  vertices are shorts.
**********************************************************************/
void polyline2d( vert, n )
short vert[][2];
int n;
{
   if (pretty_flag)
     XDrawLines( GfxDpy, aabuff0, gc1, vert, n, CoordModeOrigin );
   else
     XDrawLines( GfxDpy, buff0, gc1, vert, n, CoordModeOrigin );
}


/*** draw_rect ********************************************************
   Draw a rectangle in 2-D.
**********************************************************************/
void draw_rect( x1, y1, x2, y2 )
int x1, y1, x2, y2;
{
   short v[5][2];

   v[0][0] = x1;  v[0][1] = y1;
   v[1][0] = x2;  v[1][1] = y1;
   v[2][0] = x2;  v[2][1] = y2;
   v[3][0] = x1;  v[3][1] = y2;
   v[4][0] = x1;  v[4][1] = y1;
   polyline2d( v, 5 );
}



/*** draw_text ********************************************************
   Draw a text string.  Position coordinates are in pixels with the
   origin being in the upper-left corner.
**********************************************************************/
void draw_text( xpos, ypos, str )
int xpos, ypos;
char *str;
{
   if (pretty_flag)
     XDrawString( GfxDpy, aabuff0, gc1, xpos, ypos, str, strlen(str) );
   else
     XDrawString( GfxDpy, buff0, gc1, xpos, ypos, str, strlen(str) );
}



/*** text_width *******************************************************
   Return the width of a text string in pixels.
**********************************************************************/
int text_width( str )
char *str;
{
   int dir, ascent, descent;
   XCharStruct overall;

   XTextExtents(visfont, str, strlen(str), &dir, &ascent, &descent, &overall);
   return overall.width;
}

