/* cursor.c */

/* VIS-5D version 4.0 */

/*
VIS-5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990, 1991, 1992, 1993, 1994  Bill Hibbard, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/



#include <stdio.h>
#include <X11/Xlib.h>
#include <math.h>
#include "api.h"
#include "gui.h"



#define DOT( a, b )       (a[0]*b[0] + a[1]*b[1] + a[2]*b[2])




/*
 * Verify that the cursor is in bounds.  If not, put it in bounds.
 */
void check_cursor_in_bounds( int index)
{
  float row, col, lev;
  float x, y, z;
  int CoordFlag;
  int Nr, Nc, Nl[MAXVARS], LowLev[MAXVARS], MaxNl, MaxNlVar, WindNl, WindLow;

  vis5d_get_cursor(index, &x, &y, &z);

  vis5d_xyz_to_grid(index, -1, -1, x, y, z, &row, &col, &lev);

  CoordFlag = vis5d_graphics_mode(index, VIS5D_GRID_COORDS, VIS5D_GET);

  vis5d_get_size(index, &Nr, &Nc, Nl, LowLev, &MaxNl, &MaxNlVar, &WindNl, &WindLow);

  /* if cursor position is being printed in grid coordinates, snap */
  /* cursor location to discrete grid positions */
  if (CoordFlag==1) {
    row = (float) (int) (row+0.5);
    col = (float) (int) (col+0.5);
    lev = (float) (int) (lev+0.5);
  }

  if (row<0.0)  row = 0.0;
  else if (row>Nr-1)  row = Nr-1;
  if (col<0.0)  col = 0.0;
  else if (col>Nc-1)  col = Nc-1;
  if (lev<0.0)  lev = 0.0;
  else if (lev>MaxNl-1)  lev = MaxNl-1;

  vis5d_grid_to_xyz(index, -1, -1, row, col, lev, &x, &y, &z);

  vis5d_set_cursor(index, x, y, z);
  return;
}




/*** cursor_event ********************************************************
   This function receives an input event and tries to use it to update
   the cursor's position.
   Input: event - a copy of the XEvent.
   Returned:  0 = event not used.
              1 = cursor moved.
	      2 = middle button pressed.
*************************************************************************/
int cursor_event( int index, XEvent event )
{
   static int pressed3 = 0;
   int result;           /* value returned */

   result = 0;

   if (event.type==ButtonPress && event.xbutton.button==Button3) {
      /* Right mouse button has been pressed. */
      pressed3 = 1;
      result = 1;
   }
   else if (event.type==ButtonPress && event.xbutton.button==Button2) {
      /* middle button pressed */
      result = 2;
   }
   else if (event.type==ButtonRelease && event.xbutton.button==Button3) {
      /* Right mouse button has been released. */
      pressed3 = 0;
      result = 1;
   }

   if ( (event.type==MotionNotify || event.type==ButtonPress) && pressed3 ) {
      /* move 3-D cursor */
      float cx, cy, a[3], da[3], dc[3], d;
      float x, y, z;

      cx = (float) event.xbutton.x;
      cy = (float) event.xbutton.y;

      /* unproject (cx,cy) to line in 3-D */
      vis5d_unproject(index, cx, cy, a, da);

      /* vector from a to cursor pos */
      vis5d_get_cursor(index, &x, &y, &z);
      dc[0] = x - a[0];
      dc[1] = y - a[1];
      dc[2] = z - a[2];

      /* find component of dc on da (da already has length 1) */
      d = DOT( dc, da );

      /* calculate position of cursor along 'a' */
      x = a[0] + da[0]*d;
      y = a[1] + da[1]*d;
      z = a[2] + da[2]*d;
      vis5d_set_cursor(index, x, y, z);

      check_cursor_in_bounds(index);

      result = 1;
   }

   return result;
}

