/* rtpRender.c
 * 
 * Background rendering for rtp (real-time plotter) program.  Handles
 * drawing of plot grid, labels, and points.  There are two types of
 * renderings.  The first is the "private" rendering, started with a
 * call to RtpRender::quePrivateRender.  When this is done, the RtpRender
 * starts a rendering into a private pixmap that runs off a QTimer.  
 * When the rendering is done, the buffer is passed on by 
 * signalPrivateRenderDone.
 *
 * The other rendering is "on-line," in the sense that it starts
 * modifying the passed pixmap immediately, and cancels any private
 * rendering in progress.  This is started by RtpRender::newOnlineRender.
 * Because RtpRender draws directly onto the passed pixmap, other code
 * (outside of RtpRender) can draw onto it when new points come in.
 *
 * Copyright (c) 1999, David Watt <wattd@elcsci.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static char rtpRender_c_id[] = "$Id: rtpRender.c,v 1.10 1999/12/05 06:28:04 davidw Exp $";

#include <unistd.h>
#include <sys/time.h>
#include <qpainter.h>
#include <qfont.h>
#include <qfontmetrics.h>
#include <stdio.h>
#include "rtp.h"

/* To initialize RtpRender, just send it a pointer to the points deque
 * to use throughout its life.  The constructor initializes members and
 * connects itself to a timer.
 */
RtpRender::RtpRender(deque<DoubPt> *points)
  : _timer(this), _queuedSize()
{
  _points = points;
  _pti = 0;
  _buf = _privateBuff = NULL;
  _privateRenderQueued = false;

  _map.xscale = _map.yscale = _map.viewPort.minX = _map.viewPort.maxX
  = _map.viewPort.minY = _map.viewPort.maxY
  = _map.xbias = _map.ybias = 0.0;
  _queuedViewPort = _map.viewPort;

  // Connect the timer to the slotWorkAwhile function.
  QObject::connect(&_timer, SIGNAL(timeout()), this, SLOT(slotWorkAwhile()));
} 
  
/* Destructor: the only pointer we worry about is _privateBuff. */
RtpRender::~RtpRender()
{
  if (_privateBuff)
    delete _privateBuff;
}

/* newOnlineRender: Schedule a "preemptive" rendering.  This will delete
 * any private rendering already in progress.  Paint onto up to two
 * devices, as specified.  The online rendering will use the viewport
 * and paintable size passed in.  Generate a repaint signal.
 */
RtpMapping RtpRender::newOnlineRender(BBox& viewPort, QPixmap *buf)
{
  if (_privateBuff) {      // Cancel any private rendering
    delete _privateBuff;
    _privateBuff = NULL;
  }
  _buf = buf;
  startRender(viewPort);   // startRender fills in _map
  signalRepaintWindow();
  return _map;
}

/* quePrivateRender: Schedule another rendering to run after the current
 * one finishes.  This new rendering will be "private" and will return
 * the buffer it writes into via signalPrivateRenderDone.  After calling
 * the signal, RtpRender is not responsible for the pointer any more.
 * If you call quePrivateRender more than once before it gets a chance
 * to start the new rendering, the old mapping will just be replaced,
 * and only one new rendering will occur.
 */
void RtpRender::quePrivateRender(BBox& viewPort, const QSize& bufSize)
{
  // Set to mark that a private rendering is queued.
  _privateRenderQueued = true;
  _queuedViewPort = viewPort;
  _queuedSize = bufSize;

  // If the _timer is inactive, start the private rendering now.
  if (!(_timer.isActive()))
    switchToPrivate();
  
} 

/* switchToPrivate: Called when ready to start with the private buffer
 * rendering.
 */
void RtpRender::switchToPrivate()
{
  _privateBuff = new QPixmap(_queuedSize);
  _privateBuff->fill(white);
  _buf = _privateBuff;
  startRender(_queuedViewPort);
}

/* startRender: Starts a rendering operation, computing the mapping
 * and starting the timer if not already running.  The plot scales
 * and grid will be rendered first.
 */
void RtpRender::startRender(BBox viewPort)
{
  _pti = 0;     // reset iterator
  _privateRenderQueued = false;

  // Get the font and its metrics for writing the plot labels.
  QFont lblFont("helvetica");
  lblFont.setFixedPitch(true);
  QFontMetrics fm(lblFont);
  QSize dataSize(_buf->width(), _buf->height());

  // For now, allow for 15 pixels in addition to character height.
  // If subtracting this from the window size makes the window
  // unreasonably small, don't draw a label.
  int stripHeight = fm.ascent() + 15;

  // must allow for at least 20 pixels of data 
  bool yLabels, xLabels;
  if (dataSize.width() - stripHeight >= 20) {
    yLabels = true;
    dataSize.setWidth(dataSize.width() - stripHeight);
  }
  else
    yLabels = false;
  if (dataSize.height() - stripHeight >= 20) {
    xLabels = true;
    dataSize.setHeight(dataSize.height() - stripHeight);
  }
  else
    xLabels = false;

  // Expand the viewPort for resolution requirement, if necessary.
  rtpResCheck(&viewPort);

  // Complete the scale factors for the map now.
  _map.viewPort = viewPort;
  _map.xscale = double(dataSize.width()) 
              / (_map.viewPort.maxX - _map.viewPort.minX);
  _map.xbias = -(_map.viewPort.minX * _map.xscale) + double(_buf->width()
               - dataSize.width()); 
  _map.yscale = -double(dataSize.height()) 
              / (_map.viewPort.maxY - _map.viewPort.minY);
  _map.ybias = -(_map.viewPort.maxY * _map.yscale);

  // Fill in labels and gridlines for each axis.

  for (int axCode = 0; axCode < 2; ++axCode) {
    double maxVal, minVal;
    int stripLength;                // length of label strip
    int gridLength;
    if (0 == axCode) {              // X
      maxVal = _map.viewPort.maxX;
      minVal = _map.viewPort.minX;
      stripLength = dataSize.width();
      gridLength = dataSize.height();
      if (!xLabels)
        continue;
    } 
    else {                          // Y
      maxVal = _map.viewPort.maxY;
      minVal = _map.viewPort.minY;
      stripLength = dataSize.height();
      gridLength = dataSize.width();
      if (!yLabels)
        continue;
    } 

    // Set up a painter.  Translate and rotate it appropriately depending
    // on which axis we are dealing with. 
    QPainter p; 
    p.begin(_buf);
    p.setFont(lblFont);
    if (0 == axCode)               // X
      p.translate(0, dataSize.height());
    else {                         // Y
      p.translate(stripHeight, 0);
      p.rotate(90.0);
    } 

    // Now we know that we can do some labels.  Compute how many char's
    // of labels we can handle.
    int maxNChars = stripLength / fm.maxWidth();
 
    // Compute desired delta between ticks.  We want about 1 tick per
    // 50 pixels.
    double desiredDeltaTick = 50.0 * (maxVal - minVal) / double(stripLength+1);

    // Generate set of ticks and labels
    deque<RTP_TICK_LABEL> ticks;
    rtpMakeLabels(&ticks, minVal, maxVal, desiredDeltaTick, 2, maxNChars);

    deque<RTP_TICK_LABEL>::iterator tickPtr;

    for (tickPtr = ticks.begin(); tickPtr != ticks.end(); tickPtr++) {
      int tickPoint;
      if (maxVal != minVal) {
        if (axCode == 0)   // X
          tickPoint = int(_map.xscale * (*tickPtr).pos + _map.xbias);
        else               // Y
          tickPoint = int(_map.yscale * (*tickPtr).pos + _map.ybias);
      }
      else
        tickPoint = stripLength / 2;

      // First draw the grid line, as a dashed line.
      p.save();
      p.setPen(DotLine);
      p.drawLine(QPoint(tickPoint, 0), QPoint(tickPoint, -gridLength));
      p.restore();

      // For each label, draw a little 8 pixel long tick, then the text itself
      if (NULL == (*tickPtr).label)
        continue;

      p.drawLine(QPoint(tickPoint, 0), QPoint(tickPoint, 7));
     
      // If the tick point is at least (1/2 its label's strlen) + 1 
      // character away from both boundaries, its label can be drawn.
      int halfLabelWidth = int(double(fm.maxWidth()) 
			 * (double(strlen((*tickPtr).label)) / 2.0 + 1.0));
      if ((0 <= tickPoint - halfLabelWidth) 
            && (stripLength > tickPoint + halfLabelWidth) ) 
          p.drawText(tickPoint - halfLabelWidth, 10, 2*halfLabelWidth, 
                     fm.ascent(), AlignHCenter | AlignTop, (*tickPtr).label);
      if ((*tickPtr).label != NULL) {
        delete [] (*tickPtr).label;       // free up char* pointer
        (*tickPtr).label = NULL;
      }
    }
    p.end();
  }

  if (!(_timer.isActive()))    // Start _timer if necessary.
    _timer.start(0);
}

/* The render slot function is called as a "background" task and must
 * not run for too long.  
 */
void RtpRender::slotWorkAwhile()
{
  DoubPt lastPt;

  if (_pti < _points->size())
    lastPt = (*_points)[_pti++];

  /* Munch for a while on _points, until either we finish it off or run
   * out of time.
   */
  QPainter p;
  p.begin(_buf);

  struct timeval tv;
  struct timezone tz;
  gettimeofday(&tv, &tz);
  double cur_time = double(tv.tv_sec)*1000.0 + double(tv.tv_usec)/1000.0;
  double deadline_time = cur_time + MAX_RENDER_TIME_MSEC;

  bool dirty = false;   // true if _buf changed
  while ((cur_time <= deadline_time) && (_pti < _points->size())) {
    DoubPt newPt = (*_points)[_pti];
    if (rtpClip(_map.viewPort, &lastPt, &newPt)) {
      p.drawLine(rtpMap(_map, lastPt), rtpMap(_map, newPt));
      dirty = true;
    }
    lastPt = (*_points)[_pti++];
    gettimeofday(&tv, &tz);
    cur_time = double(tv.tv_sec)*1000.0 + double(tv.tv_usec)/1000.0;
  }
  p.end();
 
  /* If we are not working on a private buffer, and _buf has been
   * modified, send the repaint signal.
   */
  if (!_privateBuff && dirty)
    signalRepaintWindow();
 
  /* If we finished the job, and _privateBuff, fire off the signal that
   * it's done.  Then, if _privateRenderQueued, switch to the
   * private job.  Otherwise, just turn off the timer when finished.
   */
  if (_pti < _points->size())
    return;

  if (_privateBuff) {
    signalPrivateRenderDone(_privateBuff, _map);
    _privateBuff = NULL;  // lose our reference
  }
  if (_privateRenderQueued)
    switchToPrivate();
  else
    _timer.stop();
}
