/* rtp.h
 * 
 * Header file for rtp (real-time plotter) program.
 *
 * Copyright (c) 1999, David Watt <wattd@elcsci.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _RTP_H_
#define _RTP_H_

static const char rtp_h_id[] = "$Id: rtp.h,v 1.11 1999/12/05 06:27:57 davidw Exp $";

#include <qglobal.h>
#include <qapplication.h>
#include <qpixmap.h>
#include <qtimer.h>
#include <qsize.h>
#include <qpoint.h>
#include <qpen.h>

#include <deque>
using namespace std;

struct DoubPt {
  double x,y;
};

struct BBox {
  double minX, maxX, minY, maxY;
};

struct RtpMapping {
  double xscale, yscale;
  double xbias, ybias;
  BBox viewPort;
};

/* RTP_TICK_LABEL has a number and maybe a string label for plot ticks
 * and labels. */
struct RTP_TICK_LABEL {
  double pos;
  char *label;
};

// Minimum range, and minimum resolution.
#define RTP_MIN_RANGE 1E-100
#define RTP_MIN_RES 1E-12

// Functions from rtp_math.c
bool rtpClip(BBox& box, DoubPt *r1, DoubPt *r2);
void rtpMakeLabels(deque<RTP_TICK_LABEL> *ticks, double minVal, double maxVal,
                   double desiredDeltaTick, int nSpacesPerLabel,
                   int maxCharLabels);
bool rtpResCheck(BBox *b);

// Define the rtpMap function inline for quickness
inline QPoint rtpMap(RtpMapping &map, DoubPt& pt)
{
  return QPoint(int(map.xscale * pt.x + map.xbias), 
	        int(map.yscale * pt.y + map.ybias));
}

#define MAX_RENDER_TIME_MSEC 100.0

class RtpRender : public QObject {
Q_OBJECT

public:
  RtpRender(deque<DoubPt> *points);
  RtpMapping newOnlineRender(BBox& viewPort, QPixmap *buf);
  void quePrivateRender(BBox& viewPort, const QSize& size);
  ~RtpRender();

signals:
  void signalPrivateRenderDone(QPixmap *buf, RtpMapping& map);
  void signalRepaintWindow();

private slots:
  void slotWorkAwhile();

private:  
  RtpRender(RtpRender&);
  RtpRender();
  void switchToPrivate();
  void startRender(BBox viewPort);
 
  deque<DoubPt> *_points;
  QTimer _timer;
  unsigned int _pti;  
  QPixmap *_privateBuff;
  QPixmap *_buf;
  bool _privateRenderQueued;
  RtpMapping _map;
  BBox _queuedViewPort;
  QSize _queuedSize;
};

class PlotWindow : public QWidget
{
Q_OBJECT

public:
  PlotWindow(QWidget * parent = 0, const char * name = 0, WFlags f = 0);

public slots:
  void slotStdinAwake();
  void slotExpand();
  void slotTrack();

private slots:
  void slotPrivateRenderDone(QPixmap *buf, RtpMapping& map);
  void slotRepaintNoErase() { repaint(false); } 

protected:
  virtual void paintEvent(QPaintEvent *);
  virtual void resizeEvent(QResizeEvent *);
  virtual void mousePressEvent(QMouseEvent *);
  virtual void mouseMoveEvent(QMouseEvent *);
  virtual void mouseReleaseEvent(QMouseEvent *);

private:
  PlotWindow();
  PlotWindow(PlotWindow&);
  deque<DoubPt> _points;
  QPixmap *_buffer;
  RtpMapping _map;
  RtpRender _render;
  BBox _allPointsBBox; // Bounding box of all points received
  BBox _trackingPort;
  QRect *_rubberBox;   // rubber box rectangle
  QPoint _rbAnchor;    // anchor (pt of MousePress) for rubber box
  QPen _rbPen;
  enum {AUTO_SCALE, USER_FIXED, TRACKING} _viewMode;

  void expRectForPen(QRect *);
};

#endif
