func pwrite_data(fname,x,y)
/* DOCUMENT pwrite_data(fname,x,y)
  Prints out the arrays of double "x" and "y" to file "fname" in two column 
  format
*/
{ return y_pwrite_data(fname,x,y,numberof(x));
}
extern y_pwrite_data;
/* PROTOTYPE
   int write_data(string fname, double array x, double array y, int n)
*/

func pread_column(fname,line1,line2,xcol)
/* DOCUMENT pread_column(fname, line1, line2, xcol)
  Reads an vector of doubles from file "fname", from column no. "xcol",
  starting at "line1" and ending at "line2". Missing values should be marked
  with a '*'. The result of the function is a vector.
*/
{ local num,xsize,x,y; 
 xsize=(line2-line1+1);
 x=array(double,xsize)
 num=y_pread_column(fname,line1,line2,xcol,x,xsize);
 if (num<xsize)
 { write, "pread_column warning: less values found than expected, check file.", num;
   if (num>=1) y=x(1:num);
   else y=[];
   return y;
 }
 else return x;
}
extern y_pread_column;
/* PROTOTYPE
   int read_data(string fname, int line1, int line2, int xcol, double array x, int n)
*/

func pread_matrix(fname,line1,line2,xcol,xnum)
/* DOCUMENT pread_matrix(fname, line1, line2, xcol,xnum)
  Reads an array of doubles from file "fname", xnum columns starting 
  from column no. "xcol", starting at "line1" and ending at "line2". 
  Missing values should be marked with a '*'. The result of the function is
  an array of doubles.
*/
{ local num,xsize,x,y; 
 xsize=(line2-line1+1);
 x=array(double,xnum,xsize)
 num=y_pread_matrix(fname,line1,line2,xcol,x,xnum);
 if (num<xsize)
 { write, "pread_matrix warning: less values found than expected, check file.", num;
   if (num>=1) y=x(1:xsize,1:num);
   else y=[];
   return y;
 }
 else return x;
}
extern y_pread_matrix;
/* PROTOTYPE
  int my_read_matrix(string fname, int line1, int line2, int xcol, double array x, int n)
*/


func pread_block(fname,blocknum,blockstart,xcol,xsize)
/* DOCUMENT func pread_block(fname,blocnum,blockstart,xcol,xsize)
  Reads an array of doubles from file "fname", from block no. "blocknum".
  A block is defined as starting with the string in "blockstart". Values are
  read from the column "xcol". Missing values should be marked with "*".
  Because the length of the block is unknown, a maximum size "xsize" must be
  specified. The result of the function is a vector.
*/
{ local num,y,x; 
 x=array(double,xsize);
 num=y_pread_block(fname,blocknum,blockstart,xcol,x,xsize);
 if (num<xsize)
 { if (num>=1) y=x(1:num);
   else y=[];
   return y;
 }
 else return x;
}
extern y_pread_block;
/* PROTOTYPE
int read_block(string fname, int bnum, string bstart, int xcol, double array x, int xsize)
*/

func plstart(dev,xpanes,ypanes)
/* DOCUMENT plstart(dev,xpanes,ypanes)
   Starts PLPlot session using device "dev", and dividing the plotting area
   into "xpanes" by "ypanes" subwindows.
*/
{ y_plstart, dev,xpanes,ypanes;
}
extern y_plstart;
/* PROTOTYPE
void c_plstart(string dev,int xpanes, int ypanes)
*/


func plsdev(dev)
/* DOCUMENT plstart(string dev)
   Starts PLPlot session using device "dev".
   Used when you want to set up your own windows.
See also: window, sticky_window, plstart, plgdev
*/
{ y_plsdev, dev;
}
extern y_plsdev;
/* PROTOTYPE
void c_plsdev(string dev)
*/

func plinit
/* DOCUMENT plinit
   Starts PLPlot session. It is your own responsibility to set up the device 
   first, using "plsdev", and the viewport using "plwindow" or "plstick"
   afterwards.
See also plstart, plsdev, plwindow, plstick
*/
{ y_plinit;
}
extern y_plinit;
/* PROTOTYPE
void c_plinit(void)
*/


func plend
/* DOCUMENT plend
   Ends PLPlot session. This is compulsory!
*/
{ y_plend;
}
extern y_plend;
/* PROTOTYPE
void c_plend(void)
*/


func plenv(xmin,xmax,ymin,ymax,just,axis)
/* DOCUMENT func plenv(xmin,xmax,ymin,ymax,just,axis)
 Sets up plotting environment in current window, drawing a box around the
 viewport and labelling axes. The coordinate system is set up using the
 limits "xmin","xmax","ymin","ymax". If "just" is 1 the scales of the x and
 y axes are made equal. The parameter "axis" controls the drawing of the bow
 around the viewport:
  -2: No box or annotation
  -1: Draw box only
   0: Draw box, labeled with coordinate values around edge
   1: Box, labels and two axes at x=0,y=0
   2: Box, labels, axes, grid at major ticks
  10: Log x axis, linear y axis
  11: Log x axis, linear y axis, line at y=0
  20: Log y axis, linear x axis
  21: Log y axis, linear x axis, line at x=0  
  30: Log x and y axes
*/
{ y_plenv, xmin,xmax,ymin,ymax,just,axis;
}
extern y_plenv;
/* PROTOTYPE
void c_plenv(double xmin, double xmax, double ymin, double ymax, int just, int axis)
*/

func plpoin(x,y,code)
/* DOCUMENT func plpoin(x,y,code)
  Draws symbols at the values specified in the arrays "x", "y". The
  symbol type is selected by "code", which accepts ASCII values, as well as
  special characters (0-31). See Appendix A of the Yplot tutorial.
*/
{ y_plpoin,numberof(x),x,y,code;
}
extern y_plpoin;
/* PROTOTYPE
void c_plpoin(int n, double array x,double array y,int code)
*/

func plpoin3(x,y,z,code) 
/* DOCUMENT func plpoin3(x,y,z,code)
  Draws symbols at the values specified in the arrays "x", "y" and "z" in 3D
  space.  The symbol is selected by "code", which accepts ASCII values, as
  well as special characters (0-31).
  See also: plw3, plline3, plbox3, plmesh, plplot3.
*/ 
{ y_plpoin3,numberof(x),x,y,z,code; } 
 extern y_plpoin3; 
/* PROTOTYPE void
  c_plpoin3(int n, double array x,double array y,double array z,int code) */

func plline(x,y)
/* DOCUMENT func plline(x,y)
   Draws a line through the points in the arrays "x" and "y" in the current
   window, in the current line attributes.
See also: pllsty, plcol, plwid.
*/
{ y_plline,numberof(x),x,y;
}
extern y_plline;
/* PROTOTYPE
void c_plline(int n, double array x,double array y)
*/

func plline3(x,y,z)
/* DOCUMENT func plline3(x,y,z)
   Draws a line through the points in the arrays "x", "y" and "z" in the
   current window, in the current line attributes.
  See also: plw3, plline3, plbox3, plmesh, plplot3.
*/
{ y_plline3,numberof(x),x,y,z;
}
extern y_plline3;
/* PROTOTYPE
void c_plline3(int n, double array x,double array y,double array z)
*/


func logxfilter(x,y)
/* DOCUMENT func logfilter(x,y)
   Filters the the vectors "x" and "y" of values that are <=0, so that two
   adjacent values are both rejected if one is unacceptable. Returns the
   modified x vector, so must be called twice with the two unmodified
   vectors to filter out unwated values from both vectors for plotting.
*/
{ local n, n1,xtemp1,ytemp2,ytemp;
  n1=numberof(x);
  if (n1>0)
   { xtemp1=array(double,n1);
     ytemp=array(double,n1);
     n=y_logfilter(n1,x,y,xtemp1,ytemp);
     if (n>0)
       xtemp2=xtemp1(1:n);
     else
       xtemp2=[];
     return xtemp2;
    }
   else
    return [];
}

extern y_logfilter;
/* PROTOTYPE
int logfilter(int n, double array x,double array y, double array a,double array b)
*/


func pljoin(x1,y1,x2,y2)
/* DOCUMENT func pljoin(x1,y1,x2,y2)
   Joins the points (of type double) (x1,y1) and (x2,y2) using the current 
   line attributes.
See also: pllsty, plcol, plwid.
*/
{y_plline,2,[x1,x2],[y1,y2];
}

func pllab(xlabel,ylabel,toplabel)
/* DOCUMENT func pllab(xlabel,ylabel,toplabel)
   Writes labels below, to the left, and above the current viewport.
See also: plptex, plmtex, plschr,plfont,plfontld.
*/
{ y_pllab,xlabel,ylabel,toplabel;
}
extern y_pllab;
/* PROTOTYPE
void c_pllab(string xlabel, string ylabel, string toplabel)
*/

func pllsty(style)
/* DOCUMENT func pllsty(int style)
   Sets the linestyle to "style". Currently defined linestyles include:
    1    Solid
    2	 Long dashes
    3	 Medium dashes
    4	 Short dashes
    5	 Dotted
    6	 Dash-dot-dot
    7	 Dash-dot
See also: plcol, plwid
*/
{ y_pllsty, style;
}
extern y_pllsty;
/* PROTOTYPE
void lstyle(int style)
*/

func plcol(colour)
/* DOCUMENT func plcol(int colour)
   Sets the drawing colour to "colour".
See also: pllsty, plwid
*/
{ y_plcol, colour;
}
extern y_plcol;
/* PROTOTYPE
void c_plcol0(int colour)
*/


func plwid(width)
/* DOCUMENT func plwid(int width)
   Sets the line width to "width".
See also: pllsty, plcol
*/
{ y_plwid, width;
}
extern y_plwid;
/* PROTOTYPE
void c_plwid(int wid)
*/

func plsori(ori)
/* DOCUMENT func plsori(int ori)
   Sets plot orientation, with 0 being the compiled-in default (normally
   landscape). Note that the aspect ratio is preserved when a different
   orientation is selected, so that the page is not neccesarily covered as
   with the compiled-in orientation.
*/
{ y_plsori, ori;
}
extern y_plsori;
/* PROTOTYPE
void c_plsori(int ori)
*/


func pladv(sub)
/* DOCUMENT func pladv(int sub)
   Advances to the next subpage, counting from the top left to the bottom
   right, advancing along a given row.
See also: plpage, plclr
*/
{ y_pladv, sub;
}
extern y_pladv;
/* PROTOTYPE
void c_pladv(int sub)
*/

func plbox(xconfig,xtick,nxsub,yconfig,ytick,nysub)
/* DOCUMENT func plbox(string xconfig,double xtick,int nxsub,
                       string yconfig,double ytick,int nysub)
   Draws a 2D box with annotations if neccesary.
   The two "config" strings consist of a collection of characters which
   control the apperance of the box being drawn:
   Character | Meaning 
   ------------------------------------------------------------------------
   b         | Draw the bottom (left) side of the box
   c         | Draw the top (right) side of the box
   i         | Draw the tick marks to the outside of the box
   l         | Labels are calculated logarithmically 
             | (you should, however calculate the logarithms of the data 
             | to be plotted yourself)
   n         | Draw numerical labels in the conventional position 
             | (left and bottom)
   m         | Draw numerical labels in the unconventional position 
             | (top and right)
   s         | Draw minor tick marks
   t         | Draw major tick marks
   (v)       | (Draw numerical labels vertically in the case of the vertical 
             | sides)

   The interval between major ticks is selected with "xtick" and "ytick" for
   the horizontal and vertical axes respectively. "nxsub" and "nysub" is the
   number of subtick between major ticks. If set to zero, an appropriate
   default is calculated.
See also: plbox3,plenv.
*/
{ y_plbox, xconfig,xtick,nxsub,yconfig,ytick,nysub;
}
extern y_plbox;
/* PROTOTYPE
void my_plbox(string xopt,double xtick,int nxsub,string yopt,double ytick,int nysub)
*/

func plbin(x, y, cen)
/* DOCUMENT func plbin(double array x, double array y, int cen)
   Plots a histogram of the values in "x" and "y". 
 *NOTE* "x" must be a strictly increasing sequence. 
   If cen=0, the lower boundary of the bin will be placed at the "x"-values,
   if cen=1, the "x"-values will be in the middle of the bins.
See also: plbox3,plenv,plhist
*/
{ y_plbin, numberof(x),x,y,cen;
}
extern y_plbin;
/* PROTOTYPE
void c_plbin(int nbin,double array x, double array y,int cen)
*/

func plhist(nbin,data, datmin, datmax,oldwin)
/* DOCUMENT func plhist(int nbin, double array data, double datmin, 
                       double datmax,intoldwin)
   Bins "data" into "nbin" bins between "datmin" and "datmax" and calls
   plbin to draw a histogram. If oldwin=0 plenv is called before plbin, if 
   oldwin=1 the histogram is plotted in the current viewport.
See also: plbin,plenv
*/
{ y_plhist, numberof(data),data,datmin,datmax,nbin,oldwin;
}
extern y_plhist;
/* PROTOTYPE
void c_plhist(int n,double array data, double datmin, double datmax,int nbin,int oldwin)
*/


func plclr
/* DOCUMENT func plclr
   Clears the graphics screen, or ejects the current page from the printer
See also: plpage, pladv
*/
{ y_plclr;
}
extern y_plclr;
/* PROTOTYPE
void c_pleop(void)
*/

func plerrx(xmin,xmax,y)
/* DOCUMENT func plerrx(double array xmin,double array xmax,double array y)
   Draws horizontal error bars , with the i'th bar extending from xmin(i) to
   xmax(i), at a height of y.
See also: plerry
*/
{ y_plerrx,numberof(xmin),xmin,xmax,y;
}
extern y_plerrx;
/* PROTOTYPE
void c_plerrx(int n, double array xmin, double array xmax, double array y)
*/

func plerry(ymin,ymax,x)
/* DOCUMENT func plerry(double array ymin,double array ymax,double array x)
   Draws vertical error bars , with the i'th bar extending from ymin(i) to
   ymax(i), at x.
See also: plerry
*/
{ y_plerry,numberof(ymin),ymin,ymax,x;
}
extern y_plerry;
/* PROTOTYPE
void c_plerry(int n, double array ymin, double array ymax, double array x)
*/

func plfill(x,y)
/* DOCUMENT func plfill(double array x,double array y)
   Fills the polygon with the vertices in x, and y. The last vertex is
   automatically closed. Use the pattern defined by plpat or plpsty.
See also: plpsty,plpat
*/
{ y_plfill,numberof(x),x,y;
}
extern y_plfill;
/* PROTOTYPE
void c_plfill(int n, double array x, double array y)
*/

func plfont(font)
/* DOCUMENT func plfont(int font)
   Selects the text font:
                          1. Normal simplex
                          2. Roman
                          3. Italic
                          4. Script
  For 2-4 to be operational, the extended fonts must be loaded using
  plfontld(1)
See also: plfontld, plmtex
*/
{ y_plfont, font;
}
extern y_plfont;
/* PROTOTYPE
void c_plfont(int font)
*/

func plfontld(set)
/* DOCUMENT func plfontld(int set)
   Loads a font set:      0. Normal simplex
                          1. Extended
See also: plfont
*/
{ y_plfontld, set;
}
extern y_plfontld;
/* PROTOTYPE
void c_plfontld(int set)
*/

func plmtex(side,disp,pos,just,text)
/* DOCUMENT func plmtex(string side,double disp,pos,double just,string text)

   Draws a text string (last argument) relative to some side of the box:
   "t" top,"b" (bottom), "l" (left), and "r"
   (right). To draw the text vertically, add the letter "v" to the string.
   The number "disp" is a floating point number indicating the
   displacement relative to the box drawn in the subpage, with a positive
   number indicating a position outside the box. The number "pos" is the
   position of a reference point along the selected part of the box, for
   which a number between 0 and 1 must be supplied. A value of 0.5 indicates
   the centre of the specified edge of the box, while 0 indicates the
   leftmost point possible along the edge, and 1 indicating the rightmost
   point along the edge (depending on the orientation of the text to be
   written). The number "just" is a floating point number between 0 and
   1 which indicates the position of the reference point along the string,
   with 0.5 indicating that the text is to be centred with respect to the
   reference point, 0 indicating the reference point to be at the left of
   the string, and 1 at the right of the string. In this way it is possible
   to centre, left-justify, and right-justify the string anywhere along one
   of the four edges of the plot.

   Finally, the text to be printed will be drawn in the currently selected
   font. Certain control characters are recognized:

   Code | Meaning
   --------------------------------------------------------------------------
   #d   | Begin subscripting mode, or end superscripting
   #u   | Begin superscripting, or end sunscripting
   #g   | Switch to Greek font
   #r   | Switch to Roman font
   #i   | Use italic font
   #-   | Toggle underline mode
   #+   | Toggle overline mode
   #(14)|  Print Hershey character number 14
        | (See Appendix B for a list of charaters)

   Note however, that in order to use the extra fonts, they must have been
   loaded previously with the command plfontld,1

See also: plfont
*/
{ y_plmtex, side,disp,pos,just,text;
}
extern y_plmtex;
/* PROTOTYPE
void c_plmtex(string side,double disp,double pos,double just,string text)
*/

func plptex(x,y,dx,dy,just,text)
/* DOCUMENT func plptex(double x,y,dx,dy,just,string text)
   Plots text in the viewport at (x,y) at an angle of dx/dy. "just"
   indicates the part of the string to be aligned to (x,y): 0 being the left
   and 1 being the right.
See also: plfont,plmtex
*/
{ y_plptex, x,y,dx,dy,just,text;
}
extern y_plptex;
/* PROTOTYPE
void c_plptex(double x,double y,double dx,double dy,double just,string text)
*/

func plpage
/* DOCUMENT func plpage
   Advances to the next page.
See also: pladv, plclr
*/
{ y_plpage;
}
extern y_plpage;
/* PROTOTYPE
void c_plbop(void)
*/

func pltext
/* DOCUMENT func pltext
   Sets the output device to text mode.
See also: plgr
*/
{ y_pltext;
}
extern y_pltext;
/* PROTOTYPE
void c_pltext(void)
*/

func plprec(set,prec)
/* DOCUMENT func plprec(int set,int prec)
  Sets the number of characters to be drawn after the decimal point by plbox
  and plbox3. If set=0 this is done automatically.
*/
{ y_plprec, set,prec;
}
extern y_plprec;
/* PROTOTYPE
void c_plprec(int set,int prec)
*/

func plschr(height,scale)
/* DOCUMENT func plschr(double height,double scale)
   Sets the height of subsequent characters to be drawn. The actual height is
   the product of the "height" and the "scale" factor. If height=0, the
   previous (or default) height is preserved and only the scale factor is
   changed.
*/
{ y_plschr, height,scale;
}
extern y_plschr;
/* PROTOTYPE
void c_plschr(double height,double scale)
*/

func plsmajor(height,scale)
/* DOCUMENT func plsmaj(double height,double scale)
   Sets the length of subsequent major tick marks. The actual height is
   the product of the "height" and the "scale" factor. If height=0, the
   previous (or default) height is preserved and only the scale factor is
   changed.
*/
{ y_plsmaj, height,scale;
}
extern y_plsmaj;
/* PROTOTYPE
void c_plsmaj(double height,double scale)
*/

func plssym(height,scale)
/* DOCUMENT func plssym(double height,double scale)
   Sets the height of subsequent symbols to be drawn. The actual height is
   the product of the "height" and the "scale" factor. If height=0, the
   previous (or default) height is preserved and only the scale factor is
   changed.
*/
{ y_plssym, height,scale;
}
extern y_plssym;
/* PROTOTYPE
void c_plssym(double height,double scale)
*/

func plsmin(height,scale)
/* DOCUMENT func plsmin(double height,double scale)
   Sets the length of subsequent minor tick marks. The actual height is
   the product of the "height" and the "scale" factor. If height=0, the
   previous (or default) height is preserved and only the scale factor is
   changed.
*/
{ y_plsmin, height,scale;
}
extern y_plsmin;
/* PROTOTYPE
void c_plsmin(double height,double scale)
*/

func plspage(xpixels,ypixels,xlength,ylength,xoffset,yoffset)
/* DOCUMENT func plspage(double xpixels,ypixels,int xlength,ylength,
                         xoffset,yoffset)
   Sets up the page. Used differently by various drivers. The X11 driver
   uses the length and offset parameters to determine window size and
   location.
*/
{ y_plspage, xpixels,ypixels,xlength,ylength,xoffset,yoffset;
}
extern y_plspage;
/* PROTOTYPE
void c_plspage(double xpixels,double ypixels,int xlength,int ylength,int xoffset,int yoffset)
*/

func plsfile(fname)
/* DOCUMENT func plsfile(string fname)
   Sets the current output file, if applicable to the current driver.
*/
{local open;
 open=y_plsfile(fname);
 return open;
}
extern y_plsfile;
/* PROTOTYPE
void plsetfile(string fname)
*/

func plwind(xmin,xmax,ymin,ymax)
/* DOCUMENT func plwind(double xmin,xmax,ymin,ymax)
   Sets up the world coordinates of the edges of the viewport.
   See also plwindow,plstick
*/
{ y_plwind, xmin,xmax,ymin,ymax;
}
extern y_plwind;
/* PROTOTYPE
void c_plwind(double xmin,double xmax,double ymin,double ymax)
*/

func plvpor(xmin,xmax,ymin,ymax)
/* DOCUMENT func plvpor(double xmin,xmax,ymin,ymax)
   Sets up viewport in normalized subpage coordinates, ie between 0 and 1.
*/
{ y_plvpor, xmin,xmax,ymin,ymax;
}
extern y_plvpor;
/* PROTOTYPE
void c_plvpor(double xmin,double xmax,double ymin,double ymax)
*/

func plsvpa(xmin,xmax,ymin,ymax)
/* DOCUMENT func plvpor(double xmin,xmax,ymin,ymax)
   Sets up viewport in millimeters.
*/
{ y_plsvpa, xmin,xmax,ymin,ymax;
}
extern y_plsvps;
/* PROTOTYPE
void c_plsvpa(double xmin,double xmax,double ymin,double ymax)
*/

func plvpas(xmin,xmax,ymin,ymax,aspect)
/* DOCUMENT func plvpas(double xmin,xmax,ymin,ymax,aspect)
   Sets up viewport in normalized subpage coordinates, so that it has the
   given aspect ratio within the coordinates given.
*/
{ y_plvpas, xmin,xmax,ymin,ymax;
}
extern y_plvpas;
/* PROTOTYPE
void c_plvpas(double xmin,double xmax,double ymin,double ymax,double aspect)
*/

func plvasp(aspect)
/* DOCUMENT func plvasp(double aspect)
   Sets up viewport so that it has the given aspect ratio.
*/
{ y_plvasp, xmin,xmax,ymin,ymax;
}
extern y_plvasp;
/* PROTOTYPE
void c_plvasp(double aspect)
*/

func plvsta
/* DOCUMENT func plvsta
   Sets up the standard viewport: seven character heights on the left, and
   three on the other sides.
*/
{ y_plvsta, xmin,xmax,ymin,ymax;
}
extern y_plvsta;
/* PROTOTYPE
void c_plvsta(void)
*/

func plwindow(xnum,ynum,currentx,currenty)
/* DOCUMENT func plwindow(int xnum,ynum,currentx,currenty)
   Sets up a plotting window such that xnum X ynum fits onto a page, and
   make the active window the one at column currentx anr row currenty,
   starting at the top left.
*/
{ y_plwindow, xnum,ynum,currentx,currenty;
}
extern y_plwindow;
/* PROTOTYPE
void window(int xnum,int ynum,int currentx,int currenty)
*/

func plstick(xnum,ynum,currentx,currenty, topspace,rightspace,bottomspace,leftspace)
/* DOCUMENT func plstick(int xnum,ynum,currentx,currenty, 
                         topspace,rightspace,bottomspace,leftspace)
   Sets up a plotting window such that xnum X ynum fits onto a page, and
   make the active window the one at column currentx anr row currenty,
   starting at the top left. The "space" parameters specify whether space is 
   to be left open at the relevant side for annotations etc. If all the
   "space" paramters are set equal to 1, plstick is equivalent to plwindow.
*/
{ y_plstick, xnum,ynum,currentx,currenty,topspace,rightspace,bottomspace,leftspace;
}
extern y_plstick;
/* PROTOTYPE
void sticky_window(int xnum,int ynum,int currentx,int currenty,double
                   topspace,double rightspace,double bottomspace,double leftspace)
*/


func plbox3(xconfig,xlabel,xmajor,xmanor,
            yconfig,ylabel,ymajor,yminor,
            zconfig,zlabel,zmajor,zminor)
/* DOCUMENT func plbox3(xopt,xlabel,xtick,nxsub,yopt,ylabel,ytick,nysub,
            zopt,zlabel,ztick,nzsub,
   Analogous to the the two-dimensional case, the command plbox3 is used
   for drawing a three-dimensional box.
   "xconfig" and "yconfig" are  strings of characters determining the
   way in which the two horizontal sides of the box will be drawn:

   Character | Meaning \\
   --------------------------------------------------------------------------
   b         | Draws an axis at the base of the box at {\tt z=zmin}\\
   i         | Draw the tick marks downward, not upward\\
   l         | Labels are calculated logarithmically\\ 
             | (you should, however calculate the logarithms of the data\\ 
             | to be plotted yourself)\\
   n         | Draw numerical labels at major ticks\\
   s         | Draw minor tick marks\\
   t         | Draw major tick marks\\
   u         | Draw numerical labels vertically under the axis\\

   "xlabel", "ylabel" and "zlabel" are labels to be written parallel
   to the x-, y- and z-axes. "xmajor", "ymajor" and "zmajor" are
   the major tick intervals in world coordinates. "xminor", "yminor"
   and "zminor" are the number of subticks that will be drawn between major
   ticks, with a value of zero causing a suitable number to be chosen.
   "zconfig" is a string of characters determining the way in which the
   vertical sides of the box will be drawn:

   Character | Meaning \\
   ---------------------------------------------------------------------------
   b         | Draws z-axis to the left of the box
   c         | Draws z-axis to the right of the box
   d         | Draws grid lines parallel to the x-y plane at the back
             | of the box.
             | These lines are only drawn once the content of the box
             | has been completed to facilitate hidden line removal
   i         | Draw the tick marks away from the box
   l         | Labels are calculated logarithmically 
             | (you should, however calculate the logarithms of the data 
             | to be plotted yourself)
   m         | Draw numerical labels at major ticks at the right-hand 
             | vertical axis 
   n         | Draw numerical labels at major ticks at the left-hand 
             | vertical axis 
   s         | Draw minor tick marks
   t         | Draw major tick marks
   u         | Write {\tt zlabel} at the left-hand vertical axis
   v         | Write {\tt zlabel} at the right-hand vertical axis

*/
{ y_plbox3, xconfig,xlabel,xmajor,xmanor,
            yconfig,ylabel,ymajor,yminor,
            zconfig,zlabel,zmajor,zminor;
}
extern y_plbox3;
/* PROTOTYPE
void c_plbox3(string xopt,string xlabel,double xtick,int nxsub,
            string yopt,string ylabel,double ytick,int nysub,
            string zopt,string zlabel,double ztick,int nzsub)
*/

func plcont(z,clevel,x,y)
/* DOCUMENT func plcont(z,clevel,x,y)
   Draws contours of the values in z, which is specified at the points in x
   and y. The contours are drawn at the levels specified in the vector
   "clevel".
*/
{ local nx,ny,nlevel,dim; 
  nlevel=numberof(clevel);
  dim=dimsof(z);
  nx=dim(2);
  ny=dim(3);
  y_plcont, z,nx,ny,clevel,nlevel,x,y;
}
extern y_plcont;
/* PROTOTYPE
void pplcont2d(double array z, int nx,int ny,double array clevel, int nlevel, double array x, double array y);
*/

func plcont1(z,nlevel)
/* DOCUMENT func plcont(z,nlevel)
   Draws "nlevel" linearly spaced contours of the values in "z". The values
   in "z" are assumed to be equally spaced eliminating the need to supply
   the arrays "x" and "y".
See also: plcont.
*/
{ local nx,ny,dim; 
  dim=dimsof(z);
  nx=dim(2);
  ny=dim(3);
  clevel=span(min(z),max(z),nlevel);
  y_plcont1, z,nx,ny,clevel,nlevel;
}
extern y_plcont1;
/* PROTOTYPE
void pplcont1(double array z, int nx,int ny,double array clevel,int nlevel);
*/

func plshade(z,clevel,x,y)
/* DOCUMENT func plshade(z,clevel,x,y)
   Draws shaded contours of the values in "z", the intervals must be
   specified in the array "clevel".
*/
{ local nx,ny,dim,zmin,zmax; 
  dim=dimsof(z);
  nlevel=numberof(clevel);
  nx=dim(2);
  ny=dim(3);
  zmin=min(z);
  zmax=max(z);
  nx;ny;zmin;zmax;
  y_plshade0, z,nx,ny,clevel,nlevel,zmin,zmax,x,y;
}
extern y_plshade0;
/* PROTOTYPE
void pplshade0(double array z, int nx,int ny,double array clevel,int nlevel, double zmin,double zmax, double array x, double array y);
*/


func plshade1(z,clevel)
/* DOCUMENT func plshade(z,clevel)
   Draws shaded contours of the values in "z", the intervals must be
   specified in the array clevel. The values of z are assumed to be valid at
   the points specified by the index of the z-value in the 2D matrix. Eg
   z(3,4) specifies the value of z at the position x=3 and y=4.
*/
{ local nx,ny,dim,zmin,zmax; 
  dim=dimsof(z);
  nx=dim(2);
  ny=dim(3);
  zmin=min(z);
  zmax=max(x);
  nlevel=numberof(clevel);
  y_plshade1, z,nx,ny,clevel,nlevel,zmin,zmax;
}
extern y_plshade1;
/* PROTOTYPE
void pplshade1(double array z, int nx,int ny,double array clevel,int nlevel, double zmin,double zmax);
*/


func plw3d(basex,basey,height,xmin,xmax,ymin,ymax,zmin,zmax,alt,az)
/* DOCUMENT func plw3d(double basex,double basey,double height,
            double xmin,xmax,ymin,ymax,zmin,zmax,
            double alt,az)
 Sets up a 3D cuboid with coordinate sizes basex,basey, and height (in world
 coordinates)as set up with plwindow. The actual plotting is then done
 between (x,y,z)min and (x,y,z)max. The box is then viewed at an azimuth of
 "az" in degrees, and an elevation of "alt" in degrees.
*/
{ y_plw3d,basex,basey,height,xmin,xmax,ymin,ymax,zmin,zmax,alt,az;
}
extern y_plw3d;
/* PROTOTYPE	
void c_plw3d(double,double,double,double,double,double,double,double,double,double,double)
*/

func plot3d(x,y,z,opt,side)
/* DOCUMENT func plot3d(double x,y,z,int opt,side)
    Plots a 3D surface plot within the environment set up by plw3d.
   The surface is defined as z[i][j] at the points x[i] and y[j], with nx,nx
   points in total.
   opt: 1. Draw lines showing z as a function of x for each value of y.
        2. Draw lines showing z as a function of y for each value of x.
        3. 1 and 2 combined.
   If side=1, then vertical sides are drawn, so that the surface doesn't
   float in the air.
   See also: plpmesh
*/
{ y_plot3d,x,y,z,numberof(x),numberof(y),opt,side;
}
extern y_plot3d;
/* PROTOTYPE
void my_plot3d(double array x,double array y, double array z, int nx, int ny, int opt, int side)
*/

func plpmesh(x,y,z,opt)
/* DOCUMENT func plot3d(double x,y,z,int opt,side)
   Plots a 3D surface plot within the environment set up by plw3d.
   The surface is defined as z(i,j) at the points x(i) and y(j).

 "opt": 1. Draw lines showing z as a function of x for each value of y.
        2. Draw lines showing z as a function of y for each value of x.
        3. 1 and 2 combined.
   Using plpmesh, the underside of the suface is also visible, as opposed to
   plot3d.
   See also: plot3d.
*/
{ y_plmesh,x,y,z,numberof(x),numberof(y),opt;
}
extern y_plmesh;
/* PROTOTYPE
void my_plmesh(double array x,double array y, double array z, int nx, int ny, int opt)
*/

func pvfield(x,y,xv,yv,size,pchar)
/* DOCUMENT func pvfield(double x,y,xv,yv,double arrowsize,string pointstring)
  Draws a vector field using the base points in x,y. An arrowhead with a
  fraction 'size' is drawn, while a string 'pointstring' is plotted at the
  base points x,y.
*/
{local vnum;

 vnum=numberof(x);
 y_pvfield,vnum,x,y,xv,yv,size,pchar;
}

extern y_pvfield;
/* PROTOTYPE 
void pvfield(int vnum,double array x,double array y,double array xv,double array yv,
             double size,string pchar)
*/

func pvfield1(x,y,size,pchar)
/* DOCUMENT func pvfield1(double x,y,double arrowsize,string pointstring)
  Draws a vector field using the size components in x,y. The base points are
  determimined by taking the position of a particular value in the arrays
  x,y; e.g. x(1,2)=10 and y(1,2)=20, the arrow is drawn from the base
  point(1,2). The first element in the array is taken to have the indices
  (1,1) with the indices assumed to of the order (x,y). An arrowhead with a
  fraction 'size' is drawn, while a string 'pointstring' is plotted at the
  base points.
*/
{local xnum,ynum,n,dim;
  dim=dimsof(x); 
  n=dim(1);
  if (n==2) 
  { xnum=dim(2);
    ynum=dim(3);
  }
  if (n==1)
  { xnum=numberof(x);
    ynum=1;
  }
 y_pvfield1,xnum,ynum,x,y,size,pchar;
}

extern y_pvfield1;
/* PROTOTYPE 
void pvfield1(int xnum,int ynum,double array x,double array y,
             double size,string pchar)
*/

func plsxax(digmax, digits)
/* DOCUMENT func plsxax(int digmax, int digits)
 Sets current values of the digmax and digits flags for the x
 axis.  Currently digits has no effect since this parameter is updated
 to the number of digits actually used after a plot is drawn.  On the other
 hand, digmax can be used to control whether labels are plotted as fixed
 point or in scientific notation.  digmax < 0 is ignored, digmax > 0
 switches to scientific notation if the minimum label size for fixed point
 including sign and decimal point exceeds digmax.  digmax = 0 is treated the
 same as a default value (5) for digmax.
*/
{y_plsxax, digmax, digits;
}

extern y_plsxax;
/* PROTOTYPE 
void c_plsxax(int digmax,int digits)
*/

func plsyax(digmax, digits)
/* DOCUMENT func plsyax(int digmax, int digits)
 Sets current values of the digmax and digits flags for the y
 axis.  Currently digits has no effect since this parameter is updated
 to the number of digits actually used after a plot is drawn.  On the other
 hand, digmax can be used to control whether labels are plotted as fixed
 point or in scientific notation.  digmax < 0 is ignored, digmax > 0
 switches to scientific notation if the minimum label size for fixed point
 including sign and decimal point exceeds digmax.  digmax = 0 is treated the
 same as a default value (5) for digmax.
*/
{y_plsyax, digmax, digits;
}

extern y_plsyax;
/* PROTOTYPE 
void c_plsyax(int digmax,int digits)
*/

func plszax(digmax, digits)
/* DOCUMENT func plszax(int digmax, int digits)
 Sets current values of the digmax and digits flags for the z
 axis.  Currently digits has no effect since this parameter is updated
 to the number of digits actually used after a plot is drawn.  On the other
 hand, digmax can be used to control whether labels are plotted as fixed
 point or in scientific notation.  digmax < 0 is ignored, digmax > 0
 switches to scientific notation if the minimum label size for fixed point
 including sign and decimal point exceeds digmax.  digmax = 0 is treated the
 same as a default value (5) for digmax.
*/
{y_plszax, digmax, digits;
}

extern y_plszax;
/* PROTOTYPE 
void c_plszax(int digmax,int digits)
*/
