open Types

let precision = 0.0001
let max_dist = 10000.
let newton_max_step = truncate (1. /. precision)

exception RootSearchFailed

(* sdichotomy_x is called with y1 < 0 and y2 > 0 *)
let rec sdichotomy_x x1 x2 y1 y2 f = 
  if (y2 -. y1 > max_dist) && (abs_float (x1 -. x2) < precision) then raise RootSearchFailed else
  let x = (x1 +. x2) /. 2. in
  let y = Eval.eval_x x f in
    if ((abs_float y) < precision) 
    then x
    else if y < 0. 
         then sdichotomy_x x x2 y y2 f
         else sdichotomy_x x1 x y1 y f

let dichotomy_x x1 x2 f = 
  let (y1, y2) = Eval.eval_x x1 f, Eval.eval_x x2 f in
    if y1 = 0. then x1
    else if y2 = 0. then x2
    else if y1 < 0. && 0. < y2 then sdichotomy_x x1 x2 y1 y2 f
    else if y2 > 0. && 0. > y1 then sdichotomy_x x2 x1 y2 y1 f
    else raise RootSearchFailed

  
let newton_x x f =
  let f2 = Normalize.normalize (Derive.derive_x f) in
  let rec newton_ n x =
    let y = Eval.eval_x x f 
    and y2 = Eval.eval_x x f2 in
      if ((abs_float y) < precision) then x
      else if (n == 0 || y <> y || y2 <> y2 (*NaN*) || (abs_float y2) < 1e-20) then raise RootSearchFailed
      else newton_ (n - 1) (x -. y /. y2) in
  newton_ newton_max_step x


let symbolic_x f = try Normalize.normalize (Isolate.isolateN "x" f) with Isolate.IsolateFailed -> raise RootSearchFailed

let root_search_x x1 x2 f =
  let verif r e = if (x1 <= r && r <= x2) then e else raise RootSearchFailed in
  try let e = symbolic_x f in verif (Eval.eval_const e) e
  with RootSearchFailed -> try N (dichotomy_x x1 x2 f)
  with RootSearchFailed -> try let r = newton_x x1 f in verif r (N r)
  with RootSearchFailed -> try let r = newton_x x2 f in verif r (N r)
  with RootSearchFailed -> raise RootSearchFailed
