open Types
open Any

let value = function
    N _ -> 1
  | V _ -> 2
  | Sum _ -> 100
  | Prod _ -> 50
  | Pow _ -> 5
  | Func(f, _) -> match f with
      "-" -> 99
    | "inv" -> 49
    | _ -> 10

let rec expr2exprN = function
    N_(n) -> N(n)
  | V_(v) -> V(v)

  | Sum_ (e1, e2) -> Sum ([ expr2exprN e1 ; expr2exprN e2])
  | Diff_(e1, e2) -> Sum ([ expr2exprN e1 ; Func("-", expr2exprN e2)])
  | Prod_(e1, e2) -> Prod([ expr2exprN e1 ; expr2exprN e2])
  | Quot_(e1, e2) -> Prod([ expr2exprN e1 ; Func("inv", expr2exprN e2)])
  | Pow_ (e1, e2) -> Pow ([ expr2exprN e1 ; expr2exprN e2])

  | Func_(f, e) -> Func(f, expr2exprN e)

let rec equation2equationN = function
  | Eq0_ e -> Eq0 (expr2exprN e)
  | Cmp_(cmp, a, b) -> Cmp(cmp, expr2exprN a, expr2exprN b)


(* + **************************************************************************)
let rec rsimplify_sum = function
    [] -> []
  | (N 0. :: l) -> rsimplify_sum l
  | (N n1 :: N n2 :: l) -> rsimplify_sum (N (n1 +. n2) :: l)
  | (e :: l) -> e :: rsimplify_sum l

let simplify_sum l = match rsimplify_sum l with
  [] -> (N 0.)
| [e] -> e
| l2 -> Sum l2

(* * **************************************************************************)
let rec rsimplify_prod = function
    [] -> []
  | (N 0. :: _) -> [ N 0. ]
  | (N 1. :: l) -> rsimplify_prod l
  | (N n1 :: N n2 :: l) -> rsimplify_prod (N (n1 *. n2) :: l)
  | (Func ("-", e) :: l) -> N (-1.) :: rsimplify_prod (e :: l)
  | (e :: l) -> e :: rsimplify_prod l

let simplify_prod l = match rsimplify_prod l with
      [] -> (N 1.)
    | [e] -> e
    | l2 -> Prod l2

(* ^ **************************************************************************)
let rec rsimplify_pow = function
    [] -> []
  | (N 0. :: _) -> [N 0.]
  | (N 1. :: _) -> [N 1.]
  | [ e ; N 0. ] -> [N 1.]
  | [ e ; N 1. ] -> [e]
  | [ Pow (e::l) ] -> e::l
  | (Pow (e::l) :: l2) -> [ e ; Prod [ Pow l ; Pow (rsimplify_pow l2) ] ]
  | (e::l) -> e :: rsimplify_pow l

let simplify_pow l = match rsimplify_pow l with
| [] -> raise (EmptyOperandList "simplify_pow")
| [e] -> e
| l2 -> Pow l2

(* sort ***********************************************************************)
let rec sort l = Sort.list (function a -> function b -> 
  let va, vb = value a, value b in
  match a, b with
  (V sa, V sb) -> sa <= sb
  | _ -> va <= vb
  ) l

(* flattify_sum ******************************************************************)
let rec flattify_sum = function
  | [] -> []
  | (Sum l1 :: l2) -> flattify_sum l1 @ l2
  | (e::l) -> e :: flattify_sum l

(* flattify_prod ******************************************************************)
let rec flattify_prod = function
  | [] -> []
  | (Prod l1 :: l2) -> flattify_prod l1 @ l2
  | (e::l) -> e :: flattify_prod l

(* normalize_step *************************************************************)
let rec normalize_step = function
  | N n -> N n
  | V v -> V v

  | Sum  l -> simplify_sum  (sort (flattify_sum  (List.map normalize_step l)))
  | Prod l -> simplify_prod (sort (flattify_prod (List.map normalize_step l)))
  | Pow  l -> simplify_pow  (List.map normalize_step l)
	
  | Func("-", e) -> normalize_step (Prod [ N (-1.) ; e ])
  | Func("inv", e) -> normalize_step (Pow [ e ; N (-1.) ])

  | Func(f1, Func(f2, e)) -> 
      (try
        if f1 = Inverse.inverse f2 then normalize_step e (* eg: -(-x) = x *)
	else Func(f1, normalize_step (Func(f2, e)))
      with Inverse.InverseImpossible -> Func(f1, normalize_step (Func(f2, e))))
  | Func(f, e) -> Func(f, normalize_step e)



let normalize e = iterate normalize_step e

let normalizeE = function
  | Eq0 e -> Eq0 (normalize e)
  | Cmp(cmp, a, b) -> Cmp(cmp, normalize a, normalize b)

let exprN_from_string s =
  normalize (expr2exprN (Parser.expr Lexer.token (Lexing.from_string (s ^ "\n"))))
let equationN_from_string s =
  normalizeE (equation2equationN (Parser.equation Lexer.token (Lexing.from_string (s ^ "\n"))))

let eval_s_const = compose Eval.eval_const exprN_from_string 
