open Tk
open Any
open Types
open List
open Print
open Graph
open Graph_batch
open Root_search
open Unix


(* common functions *)
let string_of_point(x, y) = "x=" ^ (string_of_exprN x) ^ ", y=" ^ (string_of_exprN y)
let string_of_x_and_e(x, e) = string_of_point(x, N (Eval.eval_x (Eval.eval_const x) (getExprN e))) ^ " of function " ^ (getName e)
let tryit f_msg f arg = try Graph_batch.tryit f_msg f arg with
| Not_found -> f_msg "enter a function first!" (* nearest_function and alike return this *)
| Root_search.RootSearchFailed -> f_msg "search failed"


(* tk functions *)
let toMm_cst = ref 0.
let toMm_init w =
  let f = Frame.create w [Width (Millimeters 100.)] in
  toMm_cst := float (Winfo.reqwidth f) /. 100.;
  destroy f
let toMm x = Millimeters x
let toMm_ x = float x /. !toMm_cst
let toU _ = ()

let focus_next = compose Focus.set Focus.next 
let focus_prev = compose Focus.set Focus.prev

let     bindset  w m e f = Tk.bind     w [m, e] (BindSet ([], fun _ -> f ()))
let     bindsetw w m e f = Tk.bind     w [m, e] (BindSet ([Ev_Widget], fun e -> f e.ev_Widget))
let tag_bindsetw w m e f = Tk.tag_bind w [m, e] (BindSet ([Ev_Widget], fun e -> f e.ev_Widget))
let     bindsetl w m e f = Tk.bind     w [m, e] (BindSet ([Ev_MouseX ; Ev_MouseY], fun e -> f e.ev_MouseX e.ev_MouseY))

let     bind_key  w m k f =     bindset  w m (KeyPressDetail k) f
let     bindw_key w m k f =     bindsetw w m (KeyPressDetail k) f
let tag_bindw_key w m k f = tag_bindsetw w m (KeyPressDetail k) f


let bind_stab w = bindw_key w [] "ISO_Left_Tab" focus_prev

let image t = ImagePhoto (Imagephoto.create [File (Dirs.icondir ^ "/" ^ t ^ ".gif")])

let new_label_entry parent txt action =
  let f = Frame.create parent [] in
  let m = Label.create f [Text txt]
  and e = Entry.create f [Relief Sunken; TextWidth 10] in
  bind_key e [] "Return" (fun () -> action (Entry.get e));
  pack [m][Side Side_Left];
  pack [e][Side Side_Right; Fill Fill_X; Expand true];
  f,e

let create_label_entry parent txt action =
  let (frame, e) = new_label_entry parent txt action in
  pack [frame][Side Side_Top ; Fill Fill_X]; e

let create_dialog parent txt f = 
  let w = Toplevel.create parent [Class "Dialog"] in
  let waitv = Textvariable.create_temporary w in
  let oldfocus = try Some (Focus.get()) with _ -> None in
  Wm.title_set w txt;
  bind_stab w;
  Grab.set w;
  f w waitv;
  Tkwait.variable waitv;
  Grab.release w;
  (match oldfocus with None -> () | Some w -> try Focus.set w with _ -> ());
  destroy w;
  int_of_string (Textvariable.get waitv)

let scroll_link sb tx =
  Text.configure tx [YScrollCommand (Scrollbar.set sb)];
  Scrollbar.configure sb [ScrollCommand (Text.yview tx)]

let ok_cancel_ l f_end w waitv =
  let f(t, i) = Button.create w [Text t ; Command (fun () -> f_end() ;Textvariable.set waitv (string_of_int i))] in
  pack [Frame.create w [Height (Pixels 3)]] [Side Side_Bottom];
  pack (map f l) [Side Side_Left ; Fill Fill_X ; Expand true]
let ok_cancel = ok_cancel_ ["Ok", 1 ; "Cancel", 0]
let ok_only = ok_cancel_ ["Ok", 1] identity


let balloon(w, text) = Balloon.put w 1000 text

let setTextvariable w b = Textvariable.set w (if b then "1" else "0")
let getTextvariable w = ((Textvariable.get w) = "1")
let createTextvariable b = let w = Textvariable.create() in setTextvariable w b; w

class graphTk =
  object(o)

  inherit graph

  val mutable top = openTk()
  val mutable canvas = Widget.dummy
  val mutable actions = default_graphAction()
  val mutable eranges = []
  val mutable tools_buttons = []
  val mutable color = Black
  val mutable point_stack = []
  val mutable tags_stack = []
  val mutable smooth_lines = true
  val mutable lines_or_dots = true
  val mutable statusbar = Textvariable.create()
  val mutable logged_messages = ""
  
  val colors = [| Red ; Blue ; NamedColor "orange" ; NamedColor "skyblue" ; NamedColor "sandybrown" ; NamedColor "salmon" ; NamedColor "peru" ; NamedColor "orchid3" ; NamedColor "DarkOrange" ; NamedColor "Violet" ; NamedColor "bisque" ; NamedColor "greenyellow" ; Black |]


  method line_ = function
    | [] -> ()
    | l -> tags_stack <- append tags_stack
	(if lines_or_dots then 
	  [ Canvas.create_line canvas (map toMm l) [FillColor color ; Smooth smooth_lines] ]
	else
	  map (fun (x, y) -> Canvas.create_rectangle canvas x y x y [Outline color]) (listCombine2 (map toMm l)))

  method flush_lines = o#line_ point_stack ; point_stack <- []

  method line (x1, y1, x2, y2) = o#line_ [x1 ; o#fdj -. y1 ; x2 ; o#fdj -. y2]
      
  method moveto_ x y = o#line_ point_stack ; point_stack <- [ x ; y ]
  method templ s f = f s.i (o#fdj -. (o#y_en_j s.y))
  method moveto s = o#templ s o#moveto_

  method lineto_ x y = point_stack <- x :: y :: point_stack
  method lineto s1 s2 = o#templ s2 o#lineto_

  method draw_text (t, i, j) = tags_stack <- Canvas.create_text canvas (toMm i) (toMm (o#fdj -. j)) [Text t ; Anchor SW] :: tags_stack
  method text_size t =
    let w = Label.create top [Text t] in
    let (l, h) = (Winfo.reqwidth w, Winfo.reqheight w) in
    destroy w; (toMm_ l, toMm_ h)

  (* tested replacing the hand-made arrows by tk ones... Not impressive, leaving back the hand-made ones 
  method arrow_head angle i j =
    tags_stack <- Canvas.create_line canvas (map toMm [i ; o#fdj -. j ; i -. cos(angle) ; o#fdj -. j +. sin(angle)]) [FillColor color ; ArrowStyle Arrow_Last] :: tags_stack
  *)

  method message t = 
    Textvariable.set statusbar t;
    logged_messages <- logged_messages ^ t ^ "\n"

  method bind_buttonPress cursor f = 
    let f = tryit o#message f in
    bindsetl canvas [] (ButtonPressDetail 1) (fun x y -> let x, y = o#ij_en_xy (toMm_ x, o#fdj -. toMm_ y) in f (x, y));
    Canvas.configure canvas [Cursor (XCursor cursor)]

  method bind_buttonRect cursor f1 f2 = 
    let f1 = tryit o#message f1 and f2 = tryit o#message f2 in
    let start x y =
      let rect = ref (Canvas.create_rectangle canvas (Pixels x) (Pixels y) (Pixels x) (Pixels y) []) in
      let del_rect() = Canvas.delete canvas [!rect] in
      let restore() = del_rect(); o#defaultCanvasHandlers; o#bind_buttonRect cursor f1 f2 in

      Tk.bind canvas [[], ButtonPressDetail 1] BindRemove;
      bindset canvas [] ButtonPress restore;
      bindsetl canvas [] Motion (fun x2 y2 -> del_rect(); rect := Canvas.create_rectangle canvas (Pixels x) (Pixels y) (Pixels x2) (Pixels y2) []);
      bindsetl canvas [] (ButtonReleaseDetail 1) (fun x2 y2 -> 
	restore();
	let (a, b) = o#ij_en_xy (toMm_ x , o#fdj -. toMm_ y )
	and (c, d) = o#ij_en_xy (toMm_ x2, o#fdj -. toMm_ y2) in
	if min (abs(x - x2)) (abs(y - y2)) < 4 then f1(a, b) else f2 (min a c, min b d, max a c, max b d)) in
    bindsetl canvas [] (ButtonPressDetail 1) start;
    Canvas.configure canvas [Cursor (XCursor cursor)]

  method add_point p = 
    actions.points <- p :: actions.points;
    o#point p
  method add_fpoint p = 
    actions.fpoints <- p :: actions.fpoints;
    o#fpoint p
  method add_tangent t = 
    actions.tangents <- t :: actions.tangents;
    o#tangent_f t

  method defaultCanvasHandlers = 
    Tk.bind canvas [[], ButtonPress] BindRemove; 
    Tk.bind canvas [[], ButtonPressDetail 1] BindRemove;
    Tk.bind canvas [[], ButtonReleaseDetail 1] BindRemove;
    Tk.bind canvas [[], Motion] BindRemove;
    bindset canvas [] Configure (fun () -> o#draw_all);
    Canvas.configure canvas [Cursor (XCursor "gumby")]

  method tool_tangent_select() = 
    let add x e = (* nearly the same as tool_extremum_select's one *)
      o#add_tangent (name x, e);
      o#message ("added tangent at " ^ string_of_x_and_e(x, e)) in
    o#bind_buttonPress "target" (fun (x, y) -> add (N x) (o#nearest_function actions.plots(x, y)))

  method tool_fpoint_select() = 
    let add x e = (* nearly the same as tool_extremum_select's one *)
      o#add_fpoint ("A", name x, e);
      o#message ("added point at " ^ string_of_x_and_e(x, e)) in
    o#bind_buttonPress "target" (fun (x, y) -> add (N x) (o#nearest_function actions.plots(x, y)))

  method tool_point_select() = 
    let select (x, y) = 
      o#add_point ("A", name (N x), name (N y)) ;
      o#message ("added point at " ^ string_of_point(N x, N y)) in
    o#bind_buttonPress "target" select

  method tool_root_or_extr_select f add = 
    o#bind_buttonRect "crosshair" 
      (fun ((x, y) as l) -> 
	let e = o#nearest_function  actions.plots l in add (N (newton_x x (f (getExprN e)))) e)
      (fun ((x1, y1, x2, y2) as l) -> 
	let e = o#nearest_function_ actions.plots l in add (root_search_x x1 x2 (f (getExprN e))) e)
  method tool_root_select() =
    let add x (s, e) =
      o#add_fpoint ("A", name x, (s, e));
      o#message ("root found at x=" ^ string_of_exprN(x) ^ " of function " ^ s) in
    o#tool_root_or_extr_select identity add
  method tool_extremum_select() = 
    let derive = compose Normalize.normalize Derive.derive_x in
    let add x e =
      o#add_fpoint ("A", name x, e);
      o#message ("extremum found at " ^ string_of_x_and_e(x, e)) in
    o#tool_root_or_extr_select derive add

  method tool_intersection_select() = 
    let sub e1 e2 = Normalize.normalize (Sum [getExprN e1; Func("-", getExprN e2)]) in
    let select (x, y) =
      match o#nearest_functions actions.plots (x, y) with
      | e1 :: e2 :: _ -> 
	  o#add_fpoint ("A", name (N (newton_x x (sub e1 e2))), e1) ;
	  o#message ("intersection found at " ^ string_of_x_and_e(N x, e1) ^ " and " ^ getName e2)
      | _ -> o#message("not enough functions") in
    o#bind_buttonPress "spider" select

  method tool_zoom_select() = o#bind_buttonRect "cross" toU (fun l -> o#range l; o#draw_all)


  method create_canvas =
    canvas <- Canvas.create top [Background White ; Width (Pixels 500)];
    pack [canvas][Side Side_Right ; Fill Fill_Both ; Expand true ; Side Side_Right];
    o#defaultCanvasHandlers

  method draw_one_ f l j i =
    color <- colors.((i + j) mod (Array.length colors));
    f (nth l i);
    o#flush_lines;
    update()

  method draw_one = o#draw_one_ o#draw actions.plots 0
  method xdraw_one = o#draw_one_ o#xdraw actions.xplots (length actions.plots)

  method draw_all =
    Canvas.delete canvas tags_stack; 
    tags_stack <- [];

    imax <- int_of_float (toMm_ (Winfo.width canvas));
    jmax <- int_of_float (toMm_ (Winfo.height canvas));

    o#update_vars;

    iter2 (fun e v ->
      Entry.delete_range e (Number 0) End;
      Entry.insert e Insert (Printf.sprintf "%.4g" v)) eranges [ xmin ; ymin ; xmax ; ymax ];

    let lines_or_dots_ = lines_or_dots in
    lines_or_dots <- true;
    color <- NamedColor "grey"; o#inequations_hatching actions.plots actions.xplots;
    color <- Black; o#axes;
    lines_or_dots <- lines_or_dots_;

    for i = 0 to length actions.plots  - 1 do o#draw_one  i done;
    for i = 0 to length actions.xplots - 1 do o#xdraw_one i done;

    color <- Black;
    iter o#hatching actions.hatchings;  
    iter o#tangent_f actions.tangents;
    iter o#point actions.points;
    iter o#fpoint actions.fpoints;

  method reset() = actions <- default_graphAction()
  method reset_range() = o#rangeR (default_range())

  method open_from_channel channel =
    actions <- getActions channel;
    o#rangeR actions.range;
    o#draw_all;
    try toU (close_process_in channel) with Unix.Unix_error _ -> ()

  method open_string s =
    let in_channel, out_channel = open_process "_mathplot-format-it " in
    output_string out_channel s;
    close_out out_channel;
    o#open_from_channel in_channel
    
  method open_file f = o#open_from_channel (open_process_in ("_mathplot-format-it " ^ f))

  method save_file f =
    let channel = open_out f in
    actions.range <- o#get_range;
    output_string channel (o#string_of_graphAction actions);
    close_out channel

  method print_file f =
    let channel = open_out f in
    actions.range <- o#get_range;
    Graph_postscript.postscript_agraphs actions channel;
    close_out channel

  method print() =
    let channel = open_process_out "lpr" in
    actions.range <- o#get_range;
    Graph_postscript.postscript_agraphs actions channel;
    toU (close_process_out channel)

  method gui_open_file() = Fileselect.f "Open plot file" (function [f] -> o#open_file f | _ -> ()) "*.plot" "" false true
  method gui_save_file() = Fileselect.f "Save plot file" (function [f] -> o#save_file f | _ -> ()) "*.plot" "" false true


  method delete_some_actions() =
    let l = ref (o#strings_of_graphAction actions) in
    let create w waitv =
      let action2button n t =
	let b = Button.create w [Text t ; Anchor W] in
	Button.configure b [Command (fun () -> destroy b; l := delete t !l)]; b in
      pack (mapcount action2button !l) [Fill Fill_X];

      ok_cancel identity w waitv in
    if ((create_dialog top "Delete some actions" create) == 1) then o#open_string (String.concat "\n" !l)
  method modify_some_actions() = 
    let s = ref "" in
    let create w waitv =
      let f = Frame.create w [] in
      let tx = Text.create f [Relief Sunken ; Wrap WrapWord]
      and sb = Scrollbar.create f [] in
      scroll_link sb tx;
      Text.insert tx (TextIndex (End, [])) (o#string_of_graphAction actions) [];

      pack [sb] [Side Side_Right; Fill Fill_Y];
      pack [tx] [Side Side_Left; Fill Fill_Both; Expand true];
      pack [f] [Fill Fill_Both; Expand true];

      ok_cancel (fun _ -> s := Text.get tx (TextIndex (LineChar(0,0), [])) (TextIndex (End, []))) w waitv in
    if ((create_dialog top "Modify some actions" create) == 1) then o#open_string !s

  method logged_messages() =
    let create w waitv =
      let f = Frame.create w [] in
      let tx = Text.create f [Relief Sunken ; Wrap WrapWord]
      and sb = Scrollbar.create f [] in
      scroll_link sb tx;
      Text.insert tx (TextIndex (End, [])) logged_messages [];
      Text.configure tx [State Disabled];

      pack [sb] [Side Side_Right; Fill Fill_Y];
      pack [tx] [Side Side_Left; Fill Fill_Both; Expand true];
      pack [f] [Fill Fill_Both; Expand true];

      ok_only w waitv in
    toU (create_dialog top "Log" create)
    

  method graphs_actions actions_ =
    let ended = ref false in
    let button_was_down = ref false in
    let button_i, button_j = ref 0, ref 0 in
    let redraw = ref true in

    actions <- actions_;
    imin <- 0;
    jmin <- 0;
    minStep <- 1.;
    maxStep <- 7.;
    o#rangeR actions.range;

    toMm_init top;
    Balloon.init();
    Wm.title_set top "mathplot";

    tag_bindw_key "Text" [] "Tab" focus_next;
    tag_bindw_key "Button" [] "Return" Button.invoke;

    let command_ f = Command (function () -> f() ; o#draw_all) in

    let menubar = Frame.create top [Relief Raised; BorderWidth (Pixels 2)] in
    pack [menubar] [Side Side_Top ; Fill Fill_X];
    let file = Menubutton.create menubar [Text "File"; UnderlinedChar 0] in
    pack [file] [Side Side_Left];
    let m = Menu.create file [TearOff false] in
    Menu.add_command m [Label "New"; Accelerator "Ctl+n"; command_ o#reset];
    Menu.add_command m [Label "Open"; Accelerator "Ctl+o"; Command o#gui_open_file];
    Menu.add_command m [Label "Save"; Accelerator "Ctl+s"; Command o#gui_save_file];
    Menu.add_command m [Label "Print"; Accelerator "Ctl+p"; Command o#print];
    Menu.add_command m [Label "Export"; Command (function () -> Fileselect.f "Export to encapsulated postscript" (function [f] -> o#print_file f | _ -> ()) "*.eps" "" false true)];
    Menu.add_separator m;   
    Menu.add_command m [Label "Quit"; Accelerator "Ctl+q"; Command (function () -> exit 0)];
    Menubutton.configure file [Menu m];

    let edit = Menubutton.create menubar [Text "Edit"; UnderlinedChar 0] in
    pack [edit] [Side Side_Left];
    let m = Menu.create edit [TearOff false] in
    Menu.add_command m [Label "Clear"; Accelerator "Ctl+n"; command_ o#reset];
    Menu.add_command m [Label "Delete"; Accelerator "Ctl+D"; Command o#delete_some_actions];
    Menu.add_command m [Label "Modify"; Accelerator "Ctl+m"; Command o#modify_some_actions];
    Menubutton.configure edit [Menu m];

    let view = Menubutton.create menubar [Text "View"; UnderlinedChar 0] in
    pack [view] [Side Side_Left];
    let m = Menu.create view [TearOff false] in
    Menu.add_command m [Label "Clear"; Accelerator "Ctl+n"; command_ o#reset];
    Menu.add_command m [Label "Redraw"; Accelerator "Ctl+l"; command_ (function () -> ())];
    Menu.add_command m [Label "Reset range"; Accelerator "Ctl+r"; command_ o#reset_range];
    Menu.add_separator m;   
    Menu.add_command m [Label "Adjust"; Accelerator "Ctl+a"; command_ (function () -> o#adjust_int_number_of_tick)];   
    Menu.add_command m [Label "Autoscaling"; Accelerator "Ctl+A"; command_ (function () -> o#autoscale actions.plots)];   
    Menu.add_command m [Label "Orthonormal"; Accelerator "Ctl+o"; command_ (function () -> o#orthorme)];   
    Menu.add_separator m;   
    Menu.add_command m [Label "Log"; Accelerator "Ctl+L"; Command o#logged_messages];
    Menubutton.configure view [Menu m];

    let options = Menubutton.create menubar [Text "Options"; UnderlinedChar 0] in
    pack [options] [Side Side_Left];
    let m = Menu.create options [TearOff false] in
    let vlines_or_dots = createTextvariable lines_or_dots in
    Menu.add_checkbutton m [Label "Lines or dots" ; Variable vlines_or_dots ; command_ (fun () -> lines_or_dots <- getTextvariable vlines_or_dots)]; 
    let vsmooth_lines = createTextvariable smooth_lines in
    Menu.add_checkbutton m [Label "Smooth lines" ; Variable vsmooth_lines ; command_ (fun () -> smooth_lines <- getTextvariable vsmooth_lines)]; 
    Menubutton.configure options [Menu m];

    let efunction = create_label_entry top "f: " 
	(tryit o#message (function t -> 
	  actions <- add_graphActions actions (actions_from_lstring ["plot" ; t]) ; 
	  o#message ("added function " ^ t);
	  o#draw_all)) in

    let frame = Frame.create top [] in
    let wstatusbar = Label.create frame [TextVariable statusbar ; Relief Sunken ; Anchor W] in
    let button = Button.create frame [Text "..." ; PadY (Pixels 0) ; PadX (Pixels 0) ; Command o#logged_messages] in
    pack [button] [Side Side_Right];
    pack [wstatusbar] [Side Side_Left ; Fill Fill_X ; Expand true];
    pack [frame] [Side Side_Bottom ; Fill Fill_X ; PadX (Pixels 2) ; PadY (Pixels 2)];

    let toolbar = Frame.create top [] in pack [toolbar] [Side Side_Left];

    let tools = Frame.create toolbar [] in pack [tools] [PadY (Pixels 20)];

    let select_ (w, i1, i2) f = iter (fun (w2, i1, i2) -> Label.configure w2 [if w = w2 then i2 else i1]) tools_buttons;
      let out = ref false in
      Tk.bind w [[], Leave] (BindSet ([], fun _ -> out := true));
      Tk.bind w [[], Enter] (BindSet ([], fun _ -> out := false));
      Tk.bind w [[], ButtonReleaseDetail 1] (BindSet ([], fun _ ->
	iter (fun event -> Tk.bind w [[], event] BindRemove) [Leave ; Enter ; ButtonReleaseDetail 1];
	if !out then Label.configure w [i1] else f()))
    in
    tools_buttons <- map (fun (t, f, row, col) -> 
      let i1 = image t in
      let i2 = image (t ^ "_") in
      let b = Label.create tools [i1 ; BorderWidth (Pixels 0)] in
      (* Balloon.put b 5000 t; <- works awfully :(((( , why??? *)
      grid [b] [Row row; Column col ; PadX (Pixels 1) ; PadY (Pixels 1)];
      Tk.bind b [[], ButtonPressDetail 1] (BindSet ([], fun _ -> select_ (b,i1,i2) f));
      (b, i1, i2))
      [ "zoom", o#tool_zoom_select, 0, 1 ;
	"point", o#tool_point_select, 1, 0 ;
	"funcpoint", o#tool_fpoint_select, 1, 1 ;
	"tangent", o#tool_tangent_select, 1, 2 ;
	"root", o#tool_root_select, 2, 0 ;
	"intersection", o#tool_intersection_select, 2, 1 ;
	"extremum", o#tool_extremum_select, 2, 2 ;
      ];


    let frange = Frame.create toolbar [] in pack [frange] [PadY (Pixels 4)];

    eranges <- map (function t, update ->
      let action = (function t -> update (float_of_string t) ; o#draw_all) in
      let label = Label.create frange [Text t] in
      let entry = Entry.create frange [Relief Sunken; TextWidth 10] in 
      grid [label ; entry] [Sticky "we"];
      bind_key entry [] "Return" (fun () -> action(Entry.get entry));
      entry)
	[ "xmin: ", (function v -> xmin <- v);
	  "ymin: ", (function v -> ymin <- v);
	  "xmax: ", (function v -> xmax <- v);
	  "ymax: ", (function v -> ymax <- v)
	];
    
    let move = Frame.create toolbar [] in pack [move] [PadY (Pixels 4)];

    let button_ w text f = Button.create w [BorderWidth (Pixels 0); image text; Command (function () -> f(); o#draw_all)] in
    let bup     = button_ move "up"     (function () -> o#up 1.   ) in grid [bup    ] [Row 0; Column 1];
    let bleft   = button_ move "left"   (function () -> o#left 1. ) in grid [bleft  ] [Row 1; Column 0];
    let bcenter = button_ move "center" (function () -> o#center  ) in grid [bcenter] [Row 1; Column 1];
    let bright  = button_ move "right"  (function () -> o#right 1.) in grid [bright ] [Row 1; Column 2];
    let bdown   = button_ move "down"   (function () -> o#down 1. ) in grid [bdown  ] [Row 2; Column 1];
    
    let zoom = Frame.create toolbar [] in pack [zoom] [PadY (Pixels 4)];
    let bin  = button_ zoom "zoomin" (function () -> o#zoom_in  2.) in pack [bin ] [Side Side_Left ; Anchor E];
    let bout = button_ zoom "zoomout" (function () -> o#zoom_out 2.) in pack [bout] [Side Side_Right; Anchor W];

    o#create_canvas;
    o#draw_all;

    bind_stab top;

    (* bind is Ctl+key *)
    let bind = bind_key top [Control] in
    (* bind2 is key *)
    let bind2 = bind_key top [] in
    (* bind_ is Ctl+key and draw_all after *)
    let bind_ key f = bind_key top [Control] key (fun _ -> f() ; o#draw_all) in
    (* bindl_ is for each keys, Ctl+key and draw_all after *)
    let bindl_ keys f = iter (function key -> bind_ key f) keys in

    bind  "q"  		(function () -> exit 0);
    bind  "m"  		o#modify_some_actions;
    bind  "D"		o#delete_some_actions;
    bind  "L"		o#logged_messages;
    bind  "p"		o#print;
    bind2 "Print"	o#print;
    bind_ "r"  		o#reset_range;
    bind_ "a"  		(function () -> o#adjust_int_number_of_tick);
    bind_ "A"  		(function () -> o#autoscale actions.plots);
    bind_ "o"  		o#gui_open_file;
    bind_ "s"  		o#gui_save_file;
    bind_ "n"  		o#reset;
    bind_ "l"		(fun () -> ());

    bindl_ [ "c" ; "KP_Begin" ] (function () -> o#center);
    bindl_ [ "Up" ; "KP_Up" ] (function () -> o#up 1.);
    bindl_ [ "Down" ; "KP_Down" ] (function () -> o#down 1.);
    bindl_ [ "Left" ; "KP_Left" ] (function () -> o#left 1.);
    bindl_ [ "Right" ; "KP_Right" ] (function () -> o#right 1.);
    bindl_ [ "plus" ; "KP_Add" ] (function () -> o#zoom_in 2.);
    bindl_ [ "minus" ; "KP_Subtract" ] (function () -> o#zoom_out 2.);
    bindl_ [ "asterisk" ; "KP_Multiply" ] (function () -> o#zoom_in 8.);
    bindl_ [ "slash" ; "KP_Divide" ] (function () -> o#zoom_out 8.);

    iter balloon 
      [ bup, "Scroll up" ; 
	bdown, "Scroll down" ; 
	bleft, "Scroll left" ; 
	bright, "Scroll right" ; 
	bcenter, "Center view" ;
	bin, "Zoom in" ;
	bout, "Zoom out" ;
	efunction, "Enter the function here"
      ];
    (* update is done in draw_one_ *)
    update();
    Pack.propagate_set top false;
    mainLoop()
end

let tk_fgraphs l = new graphTk # graphs_actions (functions2graphAction l)
let tk_graphs c = new graphTk # graphs_actions (getActions c)

let usage progname = output_string Pervasives.stderr ("usage: " ^ progname ^ " [-h] [-f file] [functions...]\n")

let rec go = function
| [] -> default_graphAction()
| "-f" :: file :: l ->
    let f = open_process_in ("_mathplot-format-it " ^ file) in
    let a = getActions f in
    toU(close_process_in f); 
    add_graphActions a (go l)
| l -> actions_from_lstring (concat (map (fun t -> ["plot" ; t]) l))

;;

match Array.to_list Sys.argv with
| [] -> ()
| progname :: "-h" :: _ -> usage progname
| progname :: "--help" :: _ -> usage progname
| progname :: l -> match basename progname with
    "mathplot" -> new graphTk # graphs_actions (go l)
  | "mathplot2ps" -> Graph_postscript.postscript_agraphs (go l) Pervasives.stdout;
  | progname  -> usage progname


(*    print_string "Ici\n"; flush Pervasives.stdout; *)
