open Types
open Any
open List
open Graph
open Graph_batch

let dot2mm = 72. /. 25.4 /. 3.
let mm2dot n = truncate (dot2mm *. float n +. 0.5)

let string_of_point (i, j) = string_of_float (round i 1e3) ^ " " ^ string_of_float (round j 1e3) ^ " "
let templ c (i, j) f = output_string f (string_of_point(i, j) ^ c ^ "\n")
let moveto (i, j) = templ "m" (i, j)
let lineto (i, j) = templ "l" (i, j)


class virtual graphPostscript =
  object(o)
  inherit graph

  val mutable f = stdout

  method line (i1, j1, i2, j2) = moveto(i1, j1) f; lineto(i2, j2) f
  method text_size t = 5. *. float (String.length t), 8.
  method draw_text (t, i, j) = moveto(i, j) f; output_string f ("(" ^ t ^ ") show\n")

  method templ_draw s c = templ c (s.i, o#y_en_j s.y) f
  method moveto s = o#templ_draw s "m"


  method arrow_on_text (i, j) =
    o#line (i -. 1., j +. 8., i +. 4., j +. 8.);
    o#line (i +. 4., j +. 8., i +. 2., j +. 7.6);
    o#line (i +. 4., j +. 8., i +. 2., j +. 8.4)


  method heading_postscript =
    output_string f "%!PS-Adobe-2.0\n";
    Printf.fprintf f "%%%%BoundingBox: %d %d %d %d\n" (mm2dot imin) (mm2dot jmin) (mm2dot imax) (mm2dot jmax);
    output_string f "gsave\n";
    Printf.fprintf f "%f dup scale\n" dot2mm;
    output_string f "0.3 setlinewidth\n";
    output_string f "/Times-Roman findfont 10 scalefont setfont\n";
    output_string f "/m {moveto} bind def\n";
    output_string f "/l {lineto} bind def\n";
    output_string f "/c {curveto} bind def\n";
    output_string f "/p {m 0 0 rlineto gsave 1 setlinecap stroke grestore } bind def\n";
    Printf.fprintf f "newpath %d %d m %d %d l %d %d l %d %d l closepath clip\n" imin jmin imax jmin imax jmax imin jmax;
    output_string f "newpath\n"

  method ending_postscript =
    output_string f "stroke\n";
    output_string f "grestore\n";
    output_string f "showpage\n"

  method graphs_actions actions f_ =
    f <- f_;
    imin <- 100;
    jmin <- 100;
    imax <- 400;
    jmax <- 400;
    o#rangeR actions.range;
    o#update_vars;
    o#heading_postscript;
    o#axes;
    if xmin <= 1. && 1. <= xmax && xtick = 1. then (
      let i, j = (o#x_en_i 0.70, 2.5 +. o#y_en_j 0.) in
      o#draw_text("i", i, j);
      o#arrow_on_text (i, j)
    );

    if ymin <= 1. && 1. <= ymax && ytick = 1. then (
      let i, j = (4. +. o#x_en_i 0., o#y_en_j 0.70) in
      o#draw_text("j", i, j);
      o#arrow_on_text (i, j)
    );

    if ymin < 0. && 0. < ymax then (
      let i, j = (2. +. o#x_en_i 0., o#y_en_j 0. -. 8.) in
      o#draw_text("O", i, j)
    );

    iter o#tangent_f actions.tangents;
    iter o#point actions.points;
    iter o#fpoint actions.fpoints;
    o#inequations_hatching actions.plots actions.xplots;
    iter o#draw actions.plots;
    iter o#xdraw actions.xplots;
    output_string f "stroke 0.1 setlinewidth\n";
    iter o#hatching actions.hatchings;  
    o#ending_postscript

end

class graphPostscriptBezier =
  object(o)
  inherit graphPostscript

  method lineto s1 s2 =
    if s2.i -. s1.i < 1. then o#templ_draw s2 "l" else
    let step = s2.x -. s1.x in
    let stepby3 = step /. 3. in
    let (y1, y2) = (s1.y +. stepby3 *. s1.y2, s2.y -. stepby3 *. s2.y2) in (
      if s1.y < y2 && y2 < y1 or s2.y < y1 && y1 < y2 then
  	let x = (step *. s2.y2 +. s1.y -. s2.y) /. (s2.y2 -. s1.y2) in
  	let y = x *. s1.y2 in
  	let (i, j) = o#xy_en_ij (s1.x +. x, s1.y +. y) in
  	output_string f (string_of_point(i, j) ^ string_of_point(i, j))
      else
  	output_string f (string_of_point(o#xy_en_ij (s1.x +. stepby3, y1)) ^ string_of_point(o#xy_en_ij (s2.x -. stepby3, y2)))
    );
    o#templ_draw s2 "c"
end


class graphPostscriptLines =
  object(o)
  inherit graphPostscript

  method lineto s1 s2 = o#templ_draw s2 "l"
end
      
class graphPostscriptPoints =
  object(o)
  inherit graphPostscript

  method lineto s1 s2 = o#templ_draw s2 "p"
end


let postscript_agraphs a f = new graphPostscriptBezier # graphs_actions a f
let postscript_graphs c = new graphPostscriptBezier # graphs_actions (getActions c) stdout
let postscript_fgraphs l = new graphPostscriptBezier # graphs_actions (functions2graphAction l) stdout
