open Types
open Any
open List
open Normalize
open Eval
open Print

exception NoCurrentFunction
exception NotEnoughFunctions
exception ActionError of string
exception GetExprNError


type range = { mutable xmin : float; mutable xmax : float ; mutable ymin : float ; mutable ymax : float }

let default_range() = { xmin=(-5.); xmax=5.; ymin=(-5.); ymax=5. }



type named_exprN = string * exprN
type named_equationN = string * equationN

type graphActionRaw =
  | Point of string * named_exprN * named_exprN (* Point "A" at (0,1) *)
  | FPoint of string * named_exprN (* FPoint "A" at x=0, on the function *)
  | Tangent of named_exprN (* draw the tangent at x=0 of the current function *)
  | Plot of named_equationN (* plot the function *)
  | XPlot of named_equationN (* plot the function with no y like x=2 *)
  | Hatching (* hatching between the last two functions *)
  | Range of exprN * exprN * exprN * exprN (* plotting range *)

type graphAction = {
    mutable points : (string * named_exprN * named_exprN) list; (* Point "A" at (0,1) *)
    mutable fpoints : (string * named_exprN * named_equationN) list; (* Point "A" at x=0 on the function f *)
    mutable tangents : (named_exprN * named_equationN) list; (* draw the tangent at x=0 of the function f *)
    mutable plots : named_equationN list; (* plot the function *)
    mutable xplots : named_equationN list; (* plot the function with no y like x=2 *)
    mutable hatchings : (named_equationN * named_equationN) list; (* hatching between the functions *)
    mutable range : range
  } 


let default_graphAction() = { points=[]; fpoints=[]; tangents=[]; plots=[]; xplots=[]; hatchings=[]; range=default_range() }
let copy_graphAction a = { points=a.points; fpoints=a.fpoints; tangents=a.tangents; plots=a.plots; xplots=a.xplots; hatchings=a.hatchings; range=a.range }
let add_graphActions a b = { points=append a.points b.points; fpoints=append a.fpoints b.fpoints; tangents=append a.tangents b.tangents; plots=append a.plots b.plots; xplots=append a.xplots b.xplots; hatchings=append a.hatchings b.hatchings; range=a.range }
let functions2graphAction f = let g = default_graphAction() in g.plots <- f; g

let named_exprN_from_string s = s, exprN_from_string s
let named_equationN_from_string s = s, equationN_from_string s

let remap = function
  | Eq0 e -> if Iterateur.exist_in ((=) "y") e then Eq0 e else Cmp(Eq, V "y", e)
  | Cmp(cmp, a, b) -> Cmp(cmp, a, b)

let known_variables = "x" :: "y" :: fst (split Eval.constants)

let equationWithYFirst_from_string f =
  let eq = equationN_from_string f in
  (* next line raises UnknownVariable if there is a variable not in known_variables *)
  let _ = Iterateur.exist_inE (fun v -> if mem v known_variables then false else raise (UnknownVariable v)) eq in  
  try Plot(f, Normalize.normalizeE (Isolate.isolate "y" (remap eq)))
  with Isolate.IsolateFailed -> XPlot (f, Normalize.normalizeE (Isolate.isolate "x" eq))

let getExprN_ = function
  | Cmp(_, V _, e) -> e
  | _ -> raise GetExprNError
let getExprN (_, e) = getExprN_ e
let getName (n, _) = n

let rec parse = function
  | [] -> []
  | ("plot" :: f :: l) -> equationWithYFirst_from_string f :: parse l
  | ("xplot" :: f :: l) -> XPlot (f, Isolate.isolate "x" (equationN_from_string f)) :: parse l
  | ("point" :: name :: x :: y :: l) -> Point(name, named_exprN_from_string x, named_exprN_from_string y) :: parse l
  | ("fpoint" :: name :: x :: l) -> FPoint(name, named_exprN_from_string x) :: parse l
  | ("tangent" :: x :: l) -> Tangent(named_exprN_from_string x) :: parse l
  | ("hatching" :: l) -> Hatching :: parse l
  | ("range" :: x1::x2::y1::y2 :: l) -> Range(exprN_from_string x1, exprN_from_string x2, exprN_from_string y1, exprN_from_string y2) :: parse l
  | e :: l -> raise (ActionError ("don't know what to do with " ^ e ^ " (or missing parameters)"))

let rec transform r = function
  | Point(s, x, y) -> r.points <- (s, x, y) :: r.points
  | FPoint(s, x) -> if r.plots = [] then raise NoCurrentFunction else
                    r.fpoints <- (s, x, hd r.plots) :: r.fpoints
  | Tangent(x) -> if r.plots = [] then raise NoCurrentFunction else
                  r.tangents <- (x, hd r.plots) :: r.tangents
  | Plot(e) -> r.plots <- e :: r.plots
  | XPlot(e) -> r.xplots <- e :: r.xplots
  | Hatching -> if length r.plots < 2 then raise NotEnoughFunctions else
                r.hatchings <- (nth r.plots 0, nth r.plots 1) :: r.hatchings
  | Range(x1, y1, x2, y2) -> 
      let [x1;y1;x2;y2] = map eval_const [x1;y1;x2;y2] in
      r.range <- { xmin=min x1 x2 ; ymin=min y1 y2 ; xmax=max x1 x2 ; ymax=max y1 y2 }

let saveActionsFPoint f (s, x, f2) = (f = f2, FPoint(s, x))
let saveActionsTangent f (x, f2) = (f = f2, Tangent(x))
let saveActionsF a f =
  a.plots <- Any.remove f a.plots;
  concat [[Plot f] ; grepmap (saveActionsFPoint f) a.fpoints ; grepmap (saveActionsTangent f) a.tangents]
  
let rec saveActions2 a = match a.hatchings with
| ((f, g) :: e) -> a.hatchings <- e ; concat [saveActionsF a f ; saveActionsF a g ; [Hatching] ; saveActions2 a]
| [] -> concat (map (saveActionsF a) a.plots)

let saveActionsRange r = Range(N r.xmin, N r.ymin, N r.xmax, N r.ymax)

let saveActions a = let a = copy_graphAction a in
  concat [ [saveActionsRange a.range] ;
	   saveActions2 a ;
	   map (function f -> XPlot f) a.xplots ;
	   map (function (s, x, y) -> Point (s, x, y)) a.points
	 ]

let string_of_graphActionRaw_ = function
  | Point(s, (x, _), (y, _)) -> "point " ^ s ^ " at (" ^ x ^ ", " ^ y ^ ")"
  | FPoint(s, (x, _)) -> "fpoint " ^ s ^ " at " ^ x
  | Tangent((x, _)) -> "tangent " ^ x
  | Plot((e, _)) -> "plot " ^ e
  | XPlot((e, _)) -> "xplot " ^ e
  | Hatching -> "hatching"
  | Range(N x1, N y1, N x2, N y2) -> Printf.sprintf "range: from (%.4g, %.4g) to (%.4g, %.4g)" x1 y1 x2 y2
  | Range(_) -> "" (* for no warning *)
    
let strings_of_graphAction = compose (map string_of_graphActionRaw_) saveActions
let string_of_graphAction = compose (String.concat "\n") strings_of_graphAction

let actions_from_lstring l =
  let a = default_graphAction() in  
    iter (transform a) (parse l); a

let getActions c = actions_from_lstring (readlines c)

let unname (_, x) = x
let unnameE (_, x) = x
let name x = string_of_exprN x, x
let nameE x = string_of_equationN x, x

let tryit f_msg f arg = try f arg with
| Parsing.Parse_error -> f_msg "syntax error in expression"
| Failure("lexing: empty token") -> f_msg "unknown operand"
| Eval.UnknownVariable v -> f_msg ("unknown variable \"" ^ v ^ "\"")
| UnknownFunction f -> f_msg ("unknown function \"" ^ f ^ "\"")
