open Types
open Any
open Graph_batch
open Eval
open List


type state = { mutable pendown: bool ; mutable error: bool; mutable i: float ; mutable x: float ; mutable y: float ; mutable y2 : float }


let new_state i = { pendown = false ; error = true ; i = i ; x=0.; y=0.; y2=0. }
let state_copy s = { pendown=s.pendown; error=s.error; i=s.i ; x=s.x; y=s.y; y2=s.y2 }

class virtual graph =
  object(o)

  val mutable xmin=(default_range()).xmin
  val mutable xmax=(default_range()).xmax
  val mutable ymin=(default_range()).ymin
  val mutable ymax=(default_range()).ymax
  val mutable imin=0
  val mutable imax=400
  val mutable jmin=0
  val mutable jmax=400

  val mutable xtick=1.
  val mutable ytick=1.
  val mutable tickSize=0.005
  val mutable minNbTicks=2.
  val mutable maxNbTicks=20.

  val mutable arrowAngle=pi /. 16.
  val mutable arrowSize=0.02
  val mutable minArrowSize=0.04 (* eg: the markers of (0,i,j) are displayed when there size is better than this *)
  val mutable tangentSize=0.1
  val mutable pointCrossSize=0.005

  val mutable step=5.
  val mutable hatchingStep=0.02
  val mutable autoscaleStep=0.02
  val mutable draw_asymptot=false

  val mutable ratio2start_reducingStep=20.
  val mutable coeff_stepReduction=0.5
  val mutable ratio2start_augmentingStep=40.
  val mutable coeff_stepAugmentation=1.3

  val mutable angleMeaningBreak=pi /. 6.
  val mutable minStep=0.2
  val mutable maxStep=20.

  method virtual line : float * float * float * float -> unit
  method virtual text_size : string -> float * float
  method virtual draw_text : string * float * float -> unit
  method virtual moveto : state -> unit
  method virtual lineto : state -> state -> unit

  method di = imax - imin
  method dj = jmax - jmin
  method dx = xmax -. xmin
  method dy = ymax -. ymin
  method fdi = float o#di
  method fdj = float o#dj

  val mutable xcoeff=0. (* internal *)
  val mutable ycoeff=0. (* internal *)

  method update_vars =
    o#rangeR(o#get_range); (* normalize the range *)
    xcoeff <- o#fdi /. o#dx;
    ycoeff <- o#fdj /. o#dy;

    let rec update_tick t d =
      if d /. t > maxNbTicks then update_tick (t *. 10.) d
      else if d /. t < minNbTicks then update_tick (t /. 10.) d
      else t  in
    xtick <- update_tick xtick o#dx;
    ytick <- update_tick ytick o#dy

  method x_en_i x = (x -. xmin) *. xcoeff +. float imin
  method y_en_j y = (y -. ymin) *. ycoeff +. float jmin
  method i_en_x i = xmin +. (i -. float imin) /. xcoeff
  method j_en_y j = ymin +. (j -. float jmin) /. ycoeff
  method xy_en_ij (x, y) = (o#x_en_i x, o#y_en_j y)
  method ij_en_xy (i, j) = (o#i_en_x i, o#j_en_y j)

  method ijsize = float (o#di + o#dj) /. 2.

  method get_range = { xmin=xmin; ymin=ymin; xmax=xmax; ymax=ymax }
  method strings_of_graphAction actions =
    actions.range <- o#get_range;
    strings_of_graphAction actions
  method string_of_graphAction actions =
    actions.range <- o#get_range;
    string_of_graphAction actions

  method range (nxmin, nymin, nxmax, nymax)  =
    xmin <- min nxmin nxmax;
    ymin <- min nymin nymax;
    xmax <- max nxmin nxmax;
    ymax <- max nymin nymax
  method rangeR r = o#range(r.xmin, r.ymin, r.xmax, r.ymax)

  method mrange (dxmin, dymin, dxmax, dymax) = o#range (xmin +. dxmin, ymin +. dymin, xmax +. dxmax, ymax +. dymax)
  method up    n = let d =   o#dy *. n /. 10. in o#mrange (0., d, 0., d)
  method down  n = let d = -.o#dy *. n /. 10. in o#mrange (0., d, 0., d)
  method right n = let d =   o#dx *. n /. 10. in o#mrange (d, 0., d, 0.)
  method left  n = let d = -.o#dx *. n /. 10. in o#mrange (d, 0., d, 0.)
  method zoom (x, y) = o#mrange (x, y, -.x, -. y)
  method zoom_out factor = o#zoom (o#dx *. (1. -. factor)           /. 2., o#dy *. (1. -. factor)           /. 2.)
  method zoom_in  factor = o#zoom (o#dx *. (factor -. 1.) /. factor /. 2., o#dy *. (factor -. 1.) /. factor /. 2.)
  method center = let x, y = o#dx /. 2., o#dy /. 2. in o#range(-.x, -.y, x, y)    
  method orthorme = 
    if o#dx > o#dy then ymax <- ymin +. o#dx
    else if o#dx < o#dy then xmax <- xmin +. o#dy
  method adjust_int_number_of_tick =
    o#range(round (xmin /. xtick) 2. *. xtick,
	    round (ymin /. ytick) 2. *. ytick,
	    round (xmax /. xtick) 2. *. xtick,
	    round (ymax /. ytick) 2. *. ytick)

  method arrow_head angle i j = 
    let size = o#ijsize *. arrowSize in
    o#line(i, j, i +. size *. cos (angle +. arrowAngle), j +. size *. sin (angle +. arrowAngle));
    o#line(i, j, i +. size *. cos (angle -. arrowAngle), j +. size *. sin (angle -. arrowAngle))

  method tangent slope x y = if y = y then
    let size = o#ijsize *. tangentSize in
    let i, j = o#xy_en_ij (x, y) in
    let slope = slope *. ycoeff /. xcoeff in
    let angle = atan slope in
    let di, dj = size *. cos angle, size *. sin angle in
    o#line(i -. di, j -. dj, i +. di, j +. dj);
    o#arrow_head angle (i -. di) (j -. dj);
    o#arrow_head (angle +. pi) (i +. di) (j +. dj)

  method tangent_f (x, f) =
    let x, f = unname x, getExprN f in
    let x = eval_const x in o#tangent (eval_x x (Normalize.normalize (Derive.derive_x f))) x (eval_x x f)

  method nearest_functions fs (x, y) =
    let l = map (function f -> f, abs_float (eval_x x (getExprN f) -. y)) fs in
    fst (split (Sort.list (fun (_, n1) (_, n2) -> n1 < n2) l))
  method nearest_function = function
  | [] -> raise Not_found
  | fs -> fun l -> hd (o#nearest_functions fs l)

  method nearest_functions_ fs (x1, y1, x2, y2) =
    let l = map (o#nearest_function fs) [x1,y1 ; x2,y1 ; x1,y2 ; x2,y2 ] in
    fst (split (Sort.list (fun (_, n1) (_, n2) -> n1 < n2) (listGroup l)))
  method nearest_function_ = function
  | [] -> raise Not_found
  | fs -> fun l -> hd (o#nearest_functions_ fs l)

  method private point_ (s, x, y) =
    let size = o#ijsize *. pointCrossSize in
    let (i, j) = o#xy_en_ij (x, y) in
    o#line(i, j, i +. size, j +. size);
    o#line(i, j, i +. size, j -. size);
    o#line(i, j, i -. size, j +. size);
    o#line(i, j, i -. size, j -. size);
    o#draw_text(s, i +. size, j +. size)

  method point (s, x, y) = o#point_ (s, eval_const (unname x), eval_const (unname y))
  method fpoint (s, x, f) = 
    let x, f = unname x, getExprN f in
    let x = eval_const x in o#point_ (s, x, eval_x x f)
 

(******************************************************************************)
  method axes =
    (let tickSize = o#fdj *. tickSize in
    let j0 = o#y_en_j(0.) in
      o#line(float imin, j0, float imax, j0);
      if xmin <= 1. && 1. <= xmax  && o#dx *. minArrowSize < 1. then o#arrow_head (-. pi) (o#x_en_i 1.) j0;

    if xtick <> 1. then (
      let t = string_of_float xtick in
  	let (l, h) = o#text_size t in
  	  o#draw_text (t, o#x_en_i xtick -. l /. 2. , j0 -. tickSize -. h)
    );

    (let x = ref ((floor (xmin /. xtick) +. 1.) *. xtick) in
      while !x < xmax do
    	  if round_norm !x <> 1. then (
    	    o#line (o#x_en_i !x, j0 -. tickSize, o#x_en_i !x, j0);
    	    o#line (o#x_en_i !x, j0 +. tickSize, o#x_en_i !x, j0)
          );
    	  x := !x +. xtick
      done)
    );

    (let tickSize = o#fdi *. tickSize in
    let i0 = o#x_en_i(0.) in 
      o#line(i0, float jmin, i0, float jmax);
      if ymin <= 1. && 1. <= ymax && o#dy *. minArrowSize < 1. then o#arrow_head (-. pi /. 2.) i0 (o#y_en_j 1.);
    
      if ytick <> 1. then (
    	let t = string_of_float ytick in
    	  let (l, h) = o#text_size t in
    	    o#draw_text (t, i0 -. l -. 2. *. tickSize, o#y_en_j ytick -. h /. 2.)
      );
    
      (let y = ref ((floor (ymin /. ytick) +. 1.) *. ytick) in
    	while !y < ymax do
    	  if round_norm !y <> 1. then (
    	    o#line(i0 -. tickSize, o#y_en_j !y, i0, o#y_en_j !y);
    	    o#line(i0 +. tickSize, o#y_en_j !y, i0, o#y_en_j !y)
    	  );
    	  y := !y +. ytick
      done)
    )

(******************************************************************************)
  method draw f =
    let f = getExprN f in
    let f2 = Normalize.normalize (Derive.derive_x f) in
    let s = new_state (float imin) in

    let eval s =
      s.x  <- o#i_en_x s.i;
      s.y  <- verify_inf (eval_x s.x f);
      s.y2 <- verify_inf (eval_x s.x f2);
      s.error <- s.y <> s.y; (* case NaN *)
      s.pendown <- (if draw_asymptot then true else s.y < ymax && s.y > ymin) && not s.error in

    eval s;
    s.pendown <- false;

    while (truncate s.i < imax) do
      let olds = state_copy s in
      
        s.i <- s.i +. step;
        
        eval s;

  	(* adjusting the step: too slow *)
        if step < maxStep && abs_float (olds.y -. s.y) < o#dy /. ratio2start_augmentingStep then (
  	  step <- step *. coeff_stepAugmentation;
	  eval s
	);

	(* adjusting the step: too fast *)
	while step > minStep && 
	  (abs_float (olds.y -. s.y) > o#dy /. ratio2start_reducingStep or
	   abs_float (tan (atan ((s.y -. olds.y) /. (s.x -. olds.x)) -. atan olds.y2)) > angleMeaningBreak or
	   s.pendown && olds.error) do
	  step <- step *. coeff_stepReduction;
	  s.i <- s.i -. step;
	  eval s
	done;

        if (s.pendown && not olds.pendown) then
          if olds.error then s.pendown <- false else o#moveto olds;
        if (s.pendown && not olds.error or olds.pendown && not s.error) then o#lineto olds s
    
    done

  method xdraw f =
    let x = o#x_en_i (verify_inf (eval_const (getExprN f))) in
      o#line(x, float jmin, x, float jmax)

  method hatching (* hachures *) (f, g) =
  let (f, g) = (getExprN f, getExprN g) in
  let s = new_state (float imin) in
  let step = o#fdi *. hatchingStep in

    while (truncate s.i < imax) do
        s.i <- s.i +. step;
  	s.x  <- o#i_en_x s.i;
  	s.y  <- verify_inf (eval_x s.x f);
  	s.y2 <- verify_inf (eval_x s.x g);
        if (s.y = s.y && s.y2 = s.y2) then o#line(s.i, o#y_en_j s.y, s.i, o#y_en_j s.y2)    
    done

  method autoscale l =
    let l = map getExprN l in
    let s = new_state (float imin) in
    let step = o#fdi *. autoscaleStep in

    s.x <- o#i_en_x s.i;
    ymin <- verify_inf (eval_x s.x (hd l));
    if ymin = ymin then ymin <- 0.;
    ymax <- ymin;
    
    while (truncate s.i < imax) do
      s.i <- s.i +. step;
      s.x <- o#i_en_x s.i;
      iter (function f -> 
	let y = verify_inf (eval_x s.x f) in if y = y then
        ymin <- min y ymin;
        ymax <- max y ymax) l    
    done

  method inequations_hatching yIneq xIneq =
    let xIneq = grep isInequality (map unnameE xIneq) in
    let yIneq = grep isInequality (map unnameE yIneq) in
    if yIneq <> [] then
    let update_boundaries cmp x (min, max) = if x = x then (
      if cmp = Inf or cmp = InfEq then (if x < max then (min, x) else (min, max))
    				  else (if x > min then (x, max) else (min, max))) else (min, max) in
    let imin_, imax_ = (function (x1, x2) -> (o#x_en_i x1, o#x_en_i x2)) (fold_right (function 
    	    | Eq0 _ -> identity
    	    | Cmp(cmp, _, f) -> let x = verify_inf (eval_const f) in update_boundaries cmp x
    		) xIneq (xmin, xmax)) in
    let step = o#fdi *. hatchingStep in
    let s = new_state (imin_ +. step) in

    while (s.i < imax_) do
  	s.x  <- o#i_en_x s.i;
  	let min, max = fold_right (function 
	  | Eq0 _ -> identity (* for no warning *)
	  | Cmp(cmp, _, f) -> let y = verify_inf (eval_x s.x f) in update_boundaries cmp y
	      ) yIneq (ymin, ymax) in
	if min < max then o#line(s.i, o#y_en_j min, s.i, o#y_en_j max);
        s.i <- s.i +. step
    done

end
