;/*************************************************************************
;** funct-3.1      (command interpreter - funct)                          **
;** graphcmd.c : graphic functions                                        **
;** Copyright (C) 2003  Jean-Marc Drezet                                  **
;**                                                                       **
;**  This library is free software; you can redistribute it and/or        **
;**  modify it under the terms of the GNU Library General Public          **
;**  License as published by the Free Software Foundation; either         **
;**  version 2 of the License, or (at your option) any later version.     **
;**                                                                       **
;**  This library is distributed in the hope that it will be useful,      **
;**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
;**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
;**  Library General Public License for more details.                     **
;**                                                                       **
;**  You should have received a copy of the GNU Library General Public    **
;**  License along with this library; if not, write to the Free           **
;**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
;**                                                                       **
;** Please mail any bug reports/fixes/enhancements to me at:              **
;**      drezet@math.jussieu.fr                                           **
;** or                                                                    **
;**      Jean-Marc Drezet                                                 **
;**      Institut de Mathematiques                                        **
;**      UMR 7586 du CNRS                                                 **
;**      173, rue du Chevaleret                                           **
;**      75013 Paris                                                      **
;**      France                                                           **
;**                                                                       **
;**************************************************************************/

#include "graph.h"



/*---------------------------------------------------------------------------
    Function associated to the command 'defgraph ...'
    This command creates a new Postscript (or PNG, SVG, GIF, FIG) file
---------------------------------------------------------------------------*/
#ifndef __PLOTLIB_X11
int
defgraph_cmd(int argc, char *argv[])
{
    int             i0,
                    cas,
                    height_pix,
                    width_pix;
    char           *k[4];
    float          *xx;
    char            h[200];
    FILE           *s;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    cas = 0;
    height_pix = 0;
    width_pix = 0;
    if (comp(argv[1], "png") == 1) {
        cas = 1;
        if (argc < 6) {
            width_pix = 800;
            height_pix = 600;
        }
        else {
            width_pix = convert_int(argv[4], flow_interp);
            height_pix = convert_int(argv[5], flow_interp);
            if (height_pix < 50 || width_pix < 50) {
                error_mess(flow_interp, _GRAPH_MESS);
                return 1;
            }
        }
    }

    k[0] = (char *) flow_interp;
    k[1] = ch_copy("objdef");
    k[2] = ch_copy_int(_GRAPH_PS - 1);
    k[3] = argv[2];
    i0 = obj_create(3, k + 1);
    if (i0 == -1) {
        error_mess(flow_interp, _GRAPH_MESS + 7);
        return 1;
    }
    free(k[1]);
    free(k[2]);
    sketch_obj_restr(argv[2], &i0, _GRAPH_PS, flow_interp);
    xx = (float *) Obj[_GRAPH_PS - 1][i0].adresse;
    memset(h, 0, 200);
    sprintf(h, "%s/%s", result_rep, argv[3]);
    s = fopen(h, "w");
    if (s == NULL) {
        error_mess(flow_interp, _GRAPH_MESS + 8);
        return 1;
    }
    if (cas == 0) {
        if (comp(argv[4], "landscape") == 1) {
            xx[0] = 600;
            xx[1] = 550;
            xx[5] = 1.;
#ifndef __NULLLIB_X11
            xx[2] = g2_open_PS(h, g2_A4, g2_PS_land);
#endif
        }
        else {
            xx[0] = 600;
            xx[1] = 550;
            xx[5] = 2.;
#ifndef __NULLLIB_X11
            xx[2] = g2_open_PS(h, g2_A4, g2_PS_port);
#endif
        }
        xx[3] = 0.;
        xx[4] = 0.;
    }
    else {
        xx[0] = (float) width_pix;
        xx[1] = (float) height_pix;
#ifndef __NULLLIB_X11
        xx[2] = g2_open_gd(h, width_pix, height_pix, g2_gd_png);
#endif
        xx[3] = 0.;
        xx[4] = 0.;
        xx[5] = 3.;
    }
    g2_set_font_size(xx[2], 12);

    return 0;
}
/*-------------------------------------------------------------------------*/
#else
int
defgraph_cmd(int argc, char *argv[])
{
    int             i,
                    i0,
                    cas,
                    height_pix,
                    width_pix;
    char           *k[4];
    float          *xx;
    char            h[200],
                    hh[200];
    plPlotterParams *plotter_params;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    if (comp(argv[1], "png") == 0 && comp(argv[1], "ps") == 0 &&
        comp(argv[1], "gif") == 0 && comp(argv[1], "fig") == 0) {
        error_mess(flow_interp, _GRAPH_MESS + 6);
        return 1;
    }
    cas = 0;
    height_pix = 0;
    width_pix = 0;
    if (comp(argv[1], "png") == 1 || comp(argv[1], "gif") == 1
        || comp(argv[1], "fig") == 1) {
        cas = 1;
        width_pix = convert_int(argv[4], flow_interp);
        height_pix = convert_int(argv[5], flow_interp);
        if (height_pix < 50 || width_pix < 50) {
            error_mess(flow_interp, _GRAPH_MESS);
            return 1;
        }
    }

    k[0] = (char *) flow_interp;
    k[1] = ch_copy("objdef");
    k[2] = ch_copy_int(_GRAPH_PS - 1);
    k[3] = argv[2];
    i0 = obj_create(3, k + 1);
    if (i0 == -1) {
        error_mess(flow_interp, _GRAPH_MESS + 7);
        return 1;
    }
    free(k[1]);
    free(k[2]);
    sketch_obj_restr(argv[2], &i0, _GRAPH_PS, flow_interp);
    xx = (float *) Obj[_GRAPH_PS - 1][i0].adresse;
    memset(h, 0, 200);
    sprintf(h, "%s/%s", result_rep, argv[3]);
    i0 = -1;

    for (i = 0; i < _nb_plot_max2; i++) {
        if (plotter2[i] == NULL) {
            i0 = i;
            break;
        }
    }

    if (i0 == -1) {
        error_mess(flow_interp, _GRAPH_MESS + 9);
        return 1;
    }
    plot_fic[i0] = fopen(h, "w");
    if (plot_fic[i0] == NULL) {
        error_mess(flow_interp, _GRAPH_MESS + 8);
        return 1;
    }
    xx[2] = (float) i0;
    xx[8] = 1;
    plotter_params = pl_newplparams ();
    if (cas == 0) {
        pl_setplparam(plotter_params, "PAGESIZE", "a4");
        if (comp(argv[4], "landscape") == 1) {
            xx[0] = 600;
            xx[1] = 550;
            xx[5] = 1.;
            pl_setplparam(plotter_params, "ROTATION", "90");
        }
        else {
            xx[0] = 600;
            xx[1] = 550;
            xx[5] = 2.;
        }
        xx[3] = 0.;
        xx[4] = 0.;
        plotter2[i0] = pl_newpl_r(argv[1], NULL, plot_fic[i0], stderr,
            plotter_params);
        pl_openpl_r(plotter2[i0]);
        if (xx[5] < 1.5)
            pl_fspace_r(plotter2[i0], 0.0, 0.0, 715.0, 605.0);
        else
            pl_fspace_r(plotter2[i0], 0.0, 0.0, 605.0, 715.0);
        pl_flinewidth_r(plotter2[i0], 0.25);
        xx[8] = 1.;
    }
    else {
        xx[0] = (float) width_pix;
        xx[1] = (float) height_pix;
        memset(hh, 0, 200);
        sprintf(hh, "%dx%d", width_pix, height_pix);
        pl_setplparam(plotter_params, "BITMAPSIZE", hh);
        plotter2[i0] = pl_newpl_r(argv[1], NULL, plot_fic[i0], stderr,
            plotter_params);
        pl_openpl_r(plotter2[i0]);
        pl_fspace_r(plotter2[i0], 0.0, 0.0, width_pix, height_pix);
        xx[3] = 0.;
        xx[4] = 0.;
        xx[5] = 3.;
    }

    pl_pencolor_r(plotter2[i0], 0, 0, 0);
    pl_erase_r(plotter2[i0]);
    pl_flinewidth_r(plotter2[i0], xx[8] / 4.);
    pl_linemod_r(plotter2[i0], "solid");
    return 0;
}
/*-------------------------------------------------------------------------*/
#endif




/*---------------------------------------------------------------------------
    Function associated to the command 'graphplot ...' used to plot a point
    on a window or a PS-file
---------------------------------------------------------------------------*/
int
graph_plot_cmd(int argc, char *argv[])
{
    int             i0,
                    iw;
    float          *xx;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    _FUNCT_PLOT(iw, (int) xx[2], convert_float(argv[2], flow_interp),
        convert_float(argv[3], flow_interp), xx);
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'graphline ...' used to plot a
    line on a window or a PS-file
---------------------------------------------------------------------------*/
int
graph_lineto_cmd(int argc, char *argv[])
{
    int             i0,
                    iw;
    float          *xx;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
#ifndef __PLOTLIB_X11
    if (xx[5] == 3.)
        _FUNCT_SET_LINE_WIDTH(iw, (int) xx[2], 1., xx);
    else
        _FUNCT_SET_LINE_WIDTH(iw, (int) xx[2], 0.5, xx);
#endif
    _FUNCT_LINE_TO(iw, (int) xx[2], convert_float(argv[2], flow_interp),
        convert_float(argv[3], flow_interp), xx);
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'clear ...', used to "clear" a window
    or a PS-file
---------------------------------------------------------------------------*/
int
clear_graph_cmd(int argc, char *argv[])
{
    int             i0,
                    iw;
    float          *xx;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    _FUNCT_CLEAR(iw, xx[2]);
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'color', used to set the color defined
    by a color object
---------------------------------------------------------------------------*/
int
fixcolor_cmd(int argc, char *argv[])
{
    int             i0,
                    iw;
    double         *Ix;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _GRAPH_COL, flow_interp);
    if (iw != _GRAPH_COL) {
        error_mess(flow_interp, _GRAPH_MESS + 2);
        return 1;
    }
    Ix = (double *) Obj[iw - 1][i0].adresse;
    Ix[0] = convert_float(argv[2], flow_interp);
    Ix[1] = convert_float(argv[3], flow_interp);
    Ix[2] = convert_float(argv[4], flow_interp);
    if (Ix[0] < 0.)
        Ix[0] = 0.;
    if (Ix[0] > 1.)
        Ix[0] = 1.;
    if (Ix[1] < 0.)
        Ix[1] = 0.;
    if (Ix[1] > 1.)
        Ix[1] = 1.;
    if (Ix[2] < 0.)
        Ix[2] = 0.;
    if (Ix[2] > 1.)
        Ix[2] = 1.;
#ifdef __GGILIB_X11
    _nb_ggi_color++;
    ggi_col[_nb_ggi_color].r = (int) (65535. * Ix[0]);
    ggi_col[_nb_ggi_color].g = (int) (65535. * Ix[1]);
    ggi_col[_nb_ggi_color].b = (int) (65535. * Ix[2]);
    Ix[3] = _nb_ggi_color;
#endif
#ifdef __ALLEGROLIB_X11
    _nb_allegro_color++;
    allegro_col[_nb_allegro_color] = makecol32((int) (255. * Ix[0]),
        (int) (255. * Ix[1]), (int) (255. * Ix[2]));
    Ix[3] = _nb_allegro_color;
#endif
#ifdef __PLOTLIB_X11
    _nb_plot_color++;
    plot_color_red[_nb_plot_color] = (int) (65535. * Ix[0]);
    plot_color_green[_nb_plot_color] = (int) (65535. * Ix[1]);
    plot_color_blue[_nb_plot_color] = (int) (65535. * Ix[2]);
    Ix[3] = _nb_plot_color;
#endif
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function used by the command 'setcolor', used to associate color objects
    to windows or PS-files
---------------------------------------------------------------------------*/
int
setcolor_cmd(int argc, char *argv[])
{
    int             i0,
                    iw,
                    i00,
                    iw0;
    double         *Ix;
    float          *xx;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    iw0 = sketch_obj_restr(argv[2], &i00, _GRAPH_COL, flow_interp);
    if (iw0 != _GRAPH_COL) {
        error_mess(flow_interp, _GRAPH_MESS + 2);
        return 1;
    }
    Ix = (double *) Obj[iw0 - 1][i00].adresse;
    i0 = _FUNCT_INK(iw, xx[2], Ix[0], Ix[1], Ix[2], Ix[3]);
    _FUNCT_PEN(iw, xx[2], i0);
    xx[10] = i0;
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'frame', used to set the values of a
    frame object
---------------------------------------------------------------------------*/
int
fixframe_cmd(int argc, char *argv[])
{
    int             i0,
                    iw;
    float          *Ix;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    iw = sketch_obj_restr(argv[1], &i0, _GRAPH_FRAM, flow_interp);
    if (iw != _GRAPH_FRAM) {
        error_mess(flow_interp, _GRAPH_MESS + 3);
        return 1;
    }
    Ix = (float *) Obj[iw - 1][i0].adresse;
    Ix[0] = convert_float(argv[2], flow_interp);
    Ix[1] = convert_float(argv[3], flow_interp);
    Ix[2] = convert_float(argv[4], flow_interp);
    Ix[3] = convert_float(argv[5], flow_interp);
    Ix[4] = convert_float(argv[6], flow_interp);
    Ix[5] = convert_float(argv[7], flow_interp);
    Ix[6] = 1.;
    if (argc >= 10) {
        Ix[7] = convert_float(argv[8], flow_interp);
        Ix[8] = convert_float(argv[9], flow_interp);
    }
    else {
        Ix[7] = 1.;
        Ix[8] = 1.;
    }
    if (Ix[1] <= Ix[0])
        Ix[1] = Ix[0] + 1.;
    if (Ix[3] <= Ix[2])
        Ix[3] = Ix[2] + 1.;
    if (Ix[4] <= 0)
        Ix[4] = 1.;
    if (Ix[5] <= 0)
        Ix[5] = 1.;
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'setframe', used to draw frame
    objects on windows or PS-files
---------------------------------------------------------------------------*/
int
setframe_cmd(int argc, char *argv[])
{
    int             i0,
                    iw,
                    i00,
                    iw0,
                    i,
                    imin,
                    imax,
                    iX,
                    iY,
                    ix,
                    j,
                    jj,
                    jmin,
                    jmax,
                    ikx1,
                    ikx2,
                    iky1,
                    iky2,
                    N;
    float          *xx,
                   *xframe,
                    x0,
                    x1,
                    y0,
                    y1,
                    ax,
                    ay,
                    bx,
                    by,
                    xw,
                    yw,
                    x,
                    y,
                    yy;
    double          z[2],
                    zz;
    char            h[25];
    flow_data      *flow_interp;
    funct_clip     *f_clip;

#ifndef __PLOTLIB_X11
    N = 1;
#else
    N = 2;
#endif
    flow_interp = (flow_data *) argv[-1];
    f_clip = (funct_clip *) flow_interp->extra[2];
    ikx1 = 20;
    iky1 = 15;
    ikx2 = 60;
    iky2 = 5;
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
#ifdef __GGILIB_X11
        ggiSetGCForeground(ggi_vis, color_black);
#endif
#ifdef __ALLEGROLIB_X11
        ikx1 = 10;
        iky1 = 10;
        ikx2 = 50;
        iky2 = 0;
        xx[10] = allegro_col[0];
#endif
#ifdef __G2LIB_X11
        g2_pen(xx[2], g2_ink(xx[2], 0., 0., 0.));
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    iw0 = sketch_obj_restr(argv[2], &i00, _GRAPH_FRAM, flow_interp);
    if (iw0 != _GRAPH_FRAM) {
        error_mess(flow_interp, _GRAPH_MESS + 3);
        return 1;
    }
    xframe = (float *) Obj[iw0 - 1][i00].adresse;
    if (xframe[6] < 1.) {
        error_mess(flow_interp, _GRAPH_MESS + 4);
        return 1;
    }
    xx[10] = 0;
    x0 = xx[0] * f_clip->xinf_clip;
    x1 = xx[0] * f_clip->xsup_clip;
    y0 = xx[1] * f_clip->yinf_clip;
    y1 = xx[1] * f_clip->ysup_clip;
#ifdef __PLOTLIB_X11
    if (iw != -1)
        _FUNCT_SET_LINE_WIDTH(iw, xx[2], 2., xx);  /* !!! */
    else
        _FUNCT_SET_LINE_WIDTH(iw, xx[2], 2., xx);
#else
    _FUNCT_SET_LINE_WIDTH(iw, xx[2], 2., xx);
#endif
    _FUNCT_MOVE(iw, xx[2], x0 - 1, y0 - 1, xx);
    _FUNCT_LINE_TO(iw, xx[2], x1 + 1, y0 - 1, xx);
    _FUNCT_MOVE(iw, xx[2], x1 + 1, y0 - 1, xx);
    if (xx[5] == 3.)
        _FUNCT_SET_LINE_WIDTH(iw, xx[2], 2. * N, xx);
    _FUNCT_LINE_TO(iw, xx[2], x1 + 1, y1 + 1, xx);
    _FUNCT_MOVE(iw, xx[2], x1 + 1, y1 + 1, xx);
    if (xx[5] == 3.)
        _FUNCT_SET_LINE_WIDTH(iw, xx[2], 2. * N, xx);
    _FUNCT_LINE_TO(iw, xx[2], x0 - 1, y1 + 1, xx);
    _FUNCT_MOVE(iw, xx[2], x0 - 1, y1 + 1, xx);
    if (xx[5] == 3.)
        _FUNCT_SET_LINE_WIDTH(iw, xx[2], 2. * N, xx);
    _FUNCT_LINE_TO(iw, xx[2], x0 - 1, y0 - 1, xx);
    _FUNCT_SET_LINE_WIDTH(iw, xx[2], 1., xx);
    z[0] = 5;
    z[1] = 5;
    zz = 0.;
    if (xx[5] < 2.5)
        _FUNCT_SET_DASH(iw, xx[2], N, z, xx);
    imin = (int) (xframe[0] / xframe[4]);
    imax = (int) (xframe[1] / xframe[4]);
    jmin = (int) (xframe[2] / xframe[5]);
    jmax = (int) (xframe[3] / xframe[5]);
    ax = (x0 - x1) / (xframe[0] - xframe[1]);
    bx = x0 - xframe[0] * ax;
    ay = (y0 - y1) / (xframe[2] - xframe[3]);
    by = y0 - xframe[2] * ay;
    jj = (int) ((y1 - y0) / 12.);
    ix = 0;
    if (argc > 3)
        if (comp(argv[3], "noax") == 1)
            ix = 1;
    iX = (int) xframe[7];
    iY = (int) xframe[8];

    if (ix == 0) {
        for (i = imin; i <= imax; i++) {
            x = i * xframe[4];
            if (x < xframe[1] && x > xframe[0]) {
                xw = ax * x + bx;
                _FUNCT_MOVE(iw, xx[2], xw, y0, xx);
                if (i == 0) {
                    _FUNCT_SET_LINE_WIDTH(iw, xx[2], 1., xx);
                    if (xx[5] < 2.5)
                        _FUNCT_SET_DASH(iw, xx[2], 0, NULL, xx);
                   _FUNCT_LINE_TO(iw, xx[2], xw, y1, xx);
                    _FUNCT_SET_LINE_WIDTH(iw, xx[2], 1., xx);
                    if (xx[5] < 2.5)
                        _FUNCT_SET_DASH(iw, xx[2], N, z, xx);
                }
                else {
                    if (xx[5] < 2.5)
                        _FUNCT_LINE_TO(iw, xx[2], xw, y1, xx);
                    else {
                        for (j = 0; j <= jj; j++) {
                            yy = y0 + 12. * j;
                            _FUNCT_MOVE(iw, xx[2], xw, yy, xx);
                            if (yy + 6. < y1)
                                _FUNCT_LINE_TO(iw, xx[2], xw, yy + 6., xx);
                            _FUNCT_SET_LINE_WIDTH(iw, xx[2], 1., xx);
                        }
                    }
                }
                if (iX > 0) {
                    memset(h, 0, 25);
                    sprintf(h, "%1.2e", x * 1.000001);
                    if ((i - imin) % iX == 0)
                        _FUNCT_DRAW_STRING(iw, xx[2], xw - ikx1, y0 - iky1, h,
                            xx);
                }
            }
        }
        jj = (int) ((x1 - x0) / 12.);

        for (i = jmin; i <= jmax; i++) {
            y = i * xframe[5];
            if (y < xframe[3] && y > xframe[2]) {
                yw = ay * y + by;
                _FUNCT_MOVE(iw, xx[2], x0, yw, xx);
                if (i == 0) {
                   _FUNCT_SET_LINE_WIDTH(iw, xx[2], 1., xx);
                    if (xx[5] < 2.5)
                        _FUNCT_SET_DASH(iw, xx[2], 0, NULL, xx);
                    _FUNCT_LINE_TO(iw, xx[2], x1, yw, xx);
                    _FUNCT_SET_LINE_WIDTH(iw, xx[2], 1., xx);
                    if (xx[5] < 2.5)
                        _FUNCT_SET_DASH(iw, xx[2], N, z, xx);
                }
                else {
                    if (xx[5] < 2.5)
                        _FUNCT_LINE_TO(iw, xx[2], x1, yw, xx);
                    else {
                        for (j = 0; j <= jj; j++) {
                            yy = x0 + 12. * j;
                            _FUNCT_MOVE(iw, xx[2], yy, yw, xx);
                            if (yy + 6. < x1)
                                _FUNCT_LINE_TO(iw, xx[2], yy + 6., yw, xx);
                            _FUNCT_SET_LINE_WIDTH(iw, xx[2], 1., xx);
                        }
                    }
                }
                if (iY > 0) {
                    memset(h, 0, 25);
                    sprintf(h, "%1.2e", y * 1.000001);
                    if ((i - jmin) % iY == 0)
                       _FUNCT_DRAW_STRING(iw, xx[2], x0 - ikx2, yw - iky2, h,
                            xx);
                }
            }
        }
    }

    zz = 0.;
    _FUNCT_SET_DASH(iw, xx[2], 0, NULL, xx);
    _FUNCT_SET_LINE_WIDTH(iw, xx[2], 1., xx);
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function used to draw the intersection of a segment of line with the
    square [0,1]X[0,1]. The endpoints of the segment are (x0,y0), (x1,y1).
    The function returns 0 if the intersection is empty, 1 otherwise. In
    this case the endpoints of the intersection are (X[0],Y[0]) and
    (X[1],Y[1]).
---------------------------------------------------------------------------*/
float   min_d = 0.00001;

int
clip_line(float x0, float x1, float y0, float y1, float *X, float *Y)
{
    float           x,
                    y,
                    a,
                    b,
                    r1,
                    r2;

    if (fabs(x0 - x1) < min_d) {
        if (x0 < 0. || x0 > 1.)
            return 0;
        if (y0 > y1) {
            y = y0;
            y0 = y1;
            y1 = y;
        }
        if (y0 < 0.)
            y0 = 0.;
        if (y1 > 1.)
            y1 = 1.;
        if (y1 < y0)
            return 0;
        X[0] = x0;
        X[1] = x0;
        Y[0] = y0;
        Y[1] = y1;
        return 1;
    }
    if (fabs(y0 - y1) < min_d) {
        if (y0 < 0. || y0 > 1.)
            return 0;
        if (x0 > x1) {
            x = x0;
            x0 = x1;
            x1 = x;
        }
        if (x0 < 0.)
            x0 = 0.;
        if (x1 > 1.)
            x1 = 1.;
        if (x1 < x0)
            return 0;
        X[0] = x0;
        X[1] = x1;
        Y[0] = y0;
        Y[1] = y0;
        return 1;
    }
    if (x1 < x0) {
        x = x0;
        x0 = x1;
        x1 = x;
        y = y0;
        y0 = y1;
        y1 = y;
    }
    a = (y1 - y0) / (x1 - x0);
    b = y0 - a * x0;
    if (x0 < 0.)
        x0 = 0.;
    if (x1 > 1.)
        x1 = 1.;
    if (x0 > x1)
        return 0;
    r1 = - b / a;
    r2 = (1. - b) / a;
    if (a < 0.) {
        x = r1;
        r1 = r2;
        r2 = x;
    }
    if (x0 < r1)
        x0 = r1;
    if (x1 > r2)
        x1 = r2;
    if (x0 > x1)
        return 0;
    X[0] = x0;
    X[1] = x1;
    Y[0] = a * x0 + b;
    Y[1] = a * x1 + b;
    return 1;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'graph_plot_c ...' used to plot a
    point on a window or a PS-file with respect to a clipping rectangle
---------------------------------------------------------------------------*/
int
graph_plot_c_cmd(int argc, char *argv[])
{
    int             i0,
                    iw,
                    iw0,
                    i00;
    float          *xx,
                   *Ix,
                    x,
                    y,
                    ax,
                    bx,
                    ay,
                    by;
    flow_data      *flow_interp;
    funct_clip     *f_clip;

    flow_interp = (flow_data *) argv[-1];
    f_clip = (funct_clip *) flow_interp->extra[2];
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    iw0 = sketch_obj_restr(argv[2], &i00, _GRAPH_FRAM, flow_interp);
    if (iw0 != _GRAPH_FRAM) {
        error_mess(flow_interp, _GRAPH_MESS + 2);
        return 1;
    }
    Ix = (float *) Obj[iw0 - 1][i00].adresse;
    ax = (Ix[1] - Ix[0]) / (f_clip->xsup_clip - f_clip->xinf_clip) / xx[0];
    bx = Ix[0] - ax * xx[0] * f_clip->xinf_clip;
    ay = (Ix[3] - Ix[2]) / (f_clip->ysup_clip - f_clip->yinf_clip) / xx[1];
    by = Ix[2] - ay * xx[1] * f_clip->yinf_clip;
    x = convert_float(argv[3], flow_interp);
    y = convert_float(argv[4], flow_interp);
    xx[3] = (x - bx) / ax;
    xx[4] = (y - by) / ay;
    if (xx[3] < f_clip->xsup_clip * xx[0] && xx[3] > f_clip->xinf_clip * xx[0]
        && xx[4] < f_clip->ysup_clip * xx[1] &&
        xx[4] > f_clip->yinf_clip * xx[1])
        _FUNCT_PLOT(iw, xx[2], (double)xx[3], (double)xx[4], xx);
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'graph_line_c ...' used to plot a
    line on a window or a PS-file with respect to a clipping rectangle
---------------------------------------------------------------------------*/
int
graph_lineto_c_cmd(int argc, char *argv[])
{
    int             i0,
                    iw,
                    i00,
                    iw0;
    float          *xx,
                   *Ix,
                    x,
                    y,
                    ax,
                    bx,
                    ay,
                    by,
                    bx2,
                    by2,
                    u,
                    v,
                    xp,
                    yp,
                    x0,
                    y0,
                    x1,
                    y1,
                    X[2],
                    Y[2];
    funct_clip     *f_clip;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    f_clip = (funct_clip *) flow_interp->extra[2];
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    iw0 = sketch_obj_restr(argv[2], &i00, _GRAPH_FRAM, flow_interp);
    if (iw0 != _GRAPH_FRAM) {
        error_mess(flow_interp, _GRAPH_MESS + 2);
        return 1;
    }
    Ix = (float *) Obj[iw0 - 1][i00].adresse;
    u = (f_clip->xsup_clip - f_clip->xinf_clip) * xx[0];
    v = (f_clip->ysup_clip - f_clip->yinf_clip) * xx[1];
    ax = (Ix[1] - Ix[0]) / u;
    bx = Ix[0] - ax * xx[0] * f_clip->xinf_clip;
    ay = (Ix[3] - Ix[2]) / v;
    by = Ix[2] - ay * xx[1] * f_clip->yinf_clip;
    bx2 = - f_clip->xinf_clip * xx[0] / u;
    by2 = - f_clip->yinf_clip * xx[1] / v;
    x = convert_float(argv[3], flow_interp);
    y = convert_float(argv[4], flow_interp);
    xp = (x - bx) / ax;
    yp = (y - by) / ay;
    x0 = xp / u + bx2;
    y0 = yp / v + by2;
    x1 = xx[3] / u + bx2;
    y1 = xx[4] / v + by2;
    xx[3] = xp;
    xx[4] = yp;
    if (clip_line(x0, x1, y0, y1, X, Y) == 1) {
        _FUNCT_PLOT(iw, xx[2], (double) ((X[0] - bx2) * u),
            (double)((Y[0] - by2) * v), xx);
#ifndef __PLOTLIB_X11
        _FUNCT_SET_LINE_WIDTH(iw, xx[2], 0.5, xx);
#endif
        _FUNCT_LINE_TO(iw, xx[2], (double)((X[1] - bx2) * u),
            (double)((Y[1] - by2) * v), xx);
    }
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'setclip ...' used to define the
    "clipping region" in a window
---------------------------------------------------------------------------*/
int
set_clip_cmd(int argc, char *argv[])
{
    flow_data      *flow_interp;
    funct_clip     *f_clip;

    flow_interp = (flow_data *) argv[-1];
    f_clip = (funct_clip *) flow_interp->extra[2];
    f_clip->xinf_clip = convert_float(argv[1], flow_interp);
    f_clip->xsup_clip = convert_float(argv[2], flow_interp);
    f_clip->yinf_clip = convert_float(argv[3], flow_interp);
    f_clip->ysup_clip = convert_float(argv[4], flow_interp);
    if (f_clip->xinf_clip < 0. || f_clip->yinf_clip < 0. ||
        f_clip->xsup_clip > 1. || f_clip->ysup_clip > 1.||
        f_clip->xinf_clip >= f_clip->xsup_clip ||
        f_clip->yinf_clip >= f_clip->ysup_clip) {
        f_clip->xinf_clip = .15;
        f_clip->xsup_clip = .9;
        f_clip->yinf_clip = .1;
        f_clip->ysup_clip = .9;
    }
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'win ...' used to define the
    current X window
---------------------------------------------------------------------------*/
int
set_Xwin_cmd(int argc, char *argv[])
{
#ifdef __PLOTG2LIB_X11
    int             i;
    flow_data      *flow_interp;

        flow_interp = (flow_data *) argv[-1];
        i = convert_int(argv[1], flow_interp);
        if (i < 0 || i > _nb_plot_max) {
            error_mess(flow_interp, _GRAPH_MESS + 5);
            return 1;
        }
        _ind_plot = i;
#endif
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'putstring ...' used to write
    strings
---------------------------------------------------------------------------*/
int
draw_string_cmd(int argc, char *argv[])
{
    int             i,
                    i0,
                    iw,
                    iw0;
    float          *xx,
                    x,
                    y,
                    xh,
                    yh;
    char           *s;
    flow_data      *flow_interp;
    funct_clip     *f_clip;

    s = (char *) malloc((size_t) (__nbargmax + MAX_CARS_COM + 10) *
        sizeof(char));
    memset(s, 0, __nbargmax + MAX_CARS_COM + 10);
    flow_interp = (flow_data *) argv[-1];
    f_clip = (funct_clip *) flow_interp->extra[2];
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            free(s);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    iw0 = sketch_obj_restr(argv[2], &i0, _GRAPH_FRAM, flow_interp);
    if (iw0 != _GRAPH_FRAM) {
        error_mess(flow_interp, _GRAPH_MESS + 2);
        free(s);
        return 1;
    }
    xh = 10;
    yh = 10;
    x = xx[3];
    y = xx[4];
    if (iw == -1) {
#ifdef __ALLEGROLIB_X11
        x = xx[3] - 10;
        y = xx[4] + 5;
#endif
#ifdef __GGILIB_X11
        x = xx[3] - 2;
        y = xx[4] - 4;
#endif
    }
    else {
        y = xx[4] - 5;
#ifndef __PLOTLIB_X11
        x = xx[3] - 13;
#endif
    }
    i0 = 3;
    if (argc < 5)
        strcpy(s, argv[3]);
    else {
        if (argv[3][0] == '-') {
            if (comp(argv[3], "-O") == 1) {
                i0 = 4;
            }
            else {
                if (comp(argv[3], "-E") == 1) {
                    x += xh + xh;
                    i0 = 4;
                }
                else {
                    if (comp(argv[3], "-N") == 1) {
                        x += xh;
                        y += yh;
                        i0 = 4;
                    }
                    else {
                        if (comp(argv[3], "-S") == 1) {
                            x += xh;
                            y -= yh;
                            i0 = 4;
                        }
                        else {
                            if (comp(argv[3], "-SE") == 1) {
                                x += xh + xh;
                                y -= yh;
                                i0 = 4;
                            }
                            else {
                                if (comp(argv[3], "-SO") == 1) {
                                    y -= yh;
                                    i0 = 4;
                                }
                                else {
                                    if (comp(argv[3], "-NO") == 1) {
                                        y += yh;
                                        i0 = 4;
                                    }
                                    else {
                                        if (comp(argv[3], "-NE") == 1) {
                                            y += yh;
                                            x += xh + xh;
                                            i0 = 4;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    for (i = i0; i < argc; i++) {
        if (i > i0)
            s[strlen(s)] = ' ';
        strcat(s, argv[i]);
    }

    if (x < f_clip->xsup_clip * xx[0] && x > f_clip->xinf_clip * xx[0]
        && y < f_clip->ysup_clip * xx[1] && y > f_clip->yinf_clip * xx[1])
        _FUNCT_DRAW_STRING(iw, 0, x, y, s, xx);
    free(s);
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'title ...' used to put a
    title on a graphic
---------------------------------------------------------------------------*/
int
title_cmd(int argc, char *argv[])
{
    int             i,
                    i0,
                    iw,
                    iw0;
    float          *xx,
                    x,
                    y;
    char           *s;
    flow_data      *flow_interp;
    funct_clip     *f_clip;

    s = (char *) malloc((size_t) (__nbargmax + MAX_CARS_COM + 10) *
        sizeof(char));
    memset(s, 0, __nbargmax + MAX_CARS_COM + 10);
    flow_interp = (flow_data *) argv[-1];
    f_clip = (funct_clip *) flow_interp->extra[2];
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            free(s);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    iw0 = sketch_obj_restr(argv[2], &i0, _GRAPH_FRAM, flow_interp);
    if (iw0 != _GRAPH_FRAM) {
        error_mess(flow_interp, _GRAPH_MESS + 2);
        free(s);
        return 1;
    }
    y = f_clip->yinf_clip * xx[1] / 3.;
    if (y < 1.)
        y = 1.;

    for (i = 3; i < argc; i++) {
        if (i > 3)
            s[strlen(s)] = ' ';
        strcat(s, argv[i]);
    }

    x = xx[0] / 2. - 3. * strlen(s);
    if (x < 0.)
        x = 0.;
    _FUNCT_DRAW_STRING(iw, 0, x, y, s, xx);
    free(s);
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'xclose ...' used to delete
    X windows
---------------------------------------------------------------------------*/
int
delete_Xwin_cmd(int argc, char *argv[])
{
#ifdef __PLOTG2LIB_X11
    int             i;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    i = convert_int(argv[1], flow_interp);
    if (i < 0 || i > _nb_plot_max) {
        error_mess(flow_interp, _GRAPH_MESS + 5);
        return 1;
    }
    if (ind0k_gr[i] == 1)
        _FUNCT_CLOSE(-1, i);
    if (_ind_plot == i) {
        _ind_plot = 0;

        for (i = 0; i < _nb_plot_max; i++) {
            if (ind0k_gr[i] == 1) {
                _ind_plot = i;
                break;
            }
        }
    }
#endif
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'dash ...' used to select
    the dash type of lines
---------------------------------------------------------------------------*/
int
line_dash_cmd(int argc, char *argv[])
{
    int             i,
                    i0,
                    iw;
    float         *xx;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    i = convert_int(argv[2], flow_interp);
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    _FUNCT_SET_DASH(iw, xx[2], i, NULL, xx);
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Function associated to the command 'width ...' used to select
    the width of lines
---------------------------------------------------------------------------*/
int
line_width_cmd(int argc, char *argv[])
{
    int             i0,
                    iw;
    float         *xx;
    double         w;
    flow_data      *flow_interp;

    flow_interp = (flow_data *) argv[-1];
    w = convert_float(argv[2], flow_interp);
    if (comp(argv[1], "X") == 1) {
        _FUNCT_OPEN_X11();
#ifdef __PLOTG2LIB_X11
        xx = _GRAPH_xx[_ind_plot];
#else
        xx = _GRAPH_xx;
#endif
#ifdef __G2LIB_X11
        iw = xx[2];
#else
        iw = -1;
#endif
    }
    else {
        iw = sketch_obj_restr(argv[1], &i0, _GRAPH_PS, flow_interp);
        if (iw != _GRAPH_PS) {
            error_mess(flow_interp, _GRAPH_MESS + 1);
            return 1;
        }
        xx = (float *) Obj[iw - 1][i0].adresse;
    }
    _FUNCT_SET_LINE_WIDTH(iw, xx[2], w, xx);
    return 0;
}
/*-------------------------------------------------------------------------*/









