/*************************************************************************
** funct-3.1      (command interpreter - funct)                          **
** geom.c : graphic functions                                            **
** Copyright (C) 2003  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      UMR 7586 du CNRS                                                 **
**      173, rue du Chevaleret                                           **
**      75013 Paris                                                      **
**      France                                                           **
**                                                                       **
**************************************************************************/

#include "geom.h"


/*--------------------------------------------------------------------
    Determines the line spanned by two points
--------------------------------------------------------------------*/
int
Span(Point A, Point B, Line* lin)
{
    double          len;

    len = Dist(B, A);
    if (len < _geom_eps) {
        return 1;
    }
    lin->A = A;
    lin->U.x = (B.x - A.x) / len;
    lin->U.y = (B.y - A.y) / len;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the intersection of two lines
--------------------------------------------------------------------*/
int
Inters_Line_Line(Line L1, Line L2, Point *a)
{
    double          t1,
                    t2;

    if (fabs(L1.U.x * L2.U.y - L1.U.y * L2.U.x) < _geom_eps) {
        return 1;
    }
    Solve_2(L1.U.x, -L2.U.x, L1.U.y, -L2.U.y, L2.A.x - L1.A.x, L2.A.y - L1.A.y,
        &t1, &t2);
    a->x = L1.A.x + L1.U.x * t1;
    a->y = L1.A.y + L1.U.y * t1;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the roots of a linear system of equations in 2 variables
--------------------------------------------------------------------*/
void
Solve_2(double a, double b, double ap, double bp, double c, double cp,
    double *x, double *y)
{
    double          det;

    det = ap * b - a * bp;
    *x = (b * cp - bp * c) / det;
    *y = (ap * c - a * cp) / det;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the symmetric of the point A with respect to the point O
--------------------------------------------------------------------*/
int
Sym_Cent_Point(Point A, Point O, Point *a)
{
    a->x = O.x + O.x - A.x;
    a->y = O.y + O.y - A.y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the symmetric of the line L with respect to the point O
--------------------------------------------------------------------*/
int
Sym_Cent_Line(Point O, Line L, Line *lin)
{
    Point           B,
                    C,
                    D;

    Sym_Cent_Point(L.A, O, &B);
    D.x = L.A.x + L.U.x;
    D.y = L.A.y + L.U.y;
    Sym_Cent_Point(D, O, &C);
    lin->A = B;
    lin->U.x = C.x - B.x;
    lin->U.y = C.y - B.y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the symmetric of the point A with respect to the line L
--------------------------------------------------------------------*/
int
Sym_Line_Point(Line L, Point A, Point *a)
{
    Line            L1;
    Point           P;

    Orthog_Line(L, A, &L1);
    Inters_Line_Line(L, L1, &P);
    a->x = P.x + P.x - A.x;
    a->y = P.y + P.y - A.y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the symmetric of the line L with respect to the line Cen
--------------------------------------------------------------------*/
int
Sym_Line_Line(Line Cen, Line L, Line *lin)
{
    Point           P,
                    Q;

    Sym_Line_Point(Cen, L.A, &lin->A);
    P.x = L.A.x + L.U.x;
    P.y = L.A.y + L.U.y;
    Sym_Line_Point(Cen, P, &Q);
    lin->U.x = Q.x - lin->A.x;
    lin->U.y = Q.y - lin->A.y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the line which contains the point P and is orthogonal to
    the line L
--------------------------------------------------------------------*/
int
Orthog_Line(Line L, Point P, Line *lin)
{
    lin->A = P;
    lin->U.x = -L.U.y;
    lin->U.y = L.U.x;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the orthogonal projection of the point P on the line L
--------------------------------------------------------------------*/
int
Orthog_Proj(Line L, Point A, Point *p)
{
    Line            L1;

    Orthog_Line(L, A, &L1);
    Inters_Line_Line(L, L1, p);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the line which is parallel to the line L and contains
    the point P
--------------------------------------------------------------------*/
int
Parallel_Line(Line L, Point P, Line *lin)
{
    lin->A = P;
    lin->U = L.U;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the middle point of the segment AB
--------------------------------------------------------------------*/
int
Middle_Point(Point A, Point B, Point *m)
{
    m->x = (A.x + B.x) / 2.;
    m->y = (A.y + B.y) / 2.;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the barycenter of the points A, B with coefficient t
--------------------------------------------------------------------*/
int
Bary_Point(Point A, Point B, double t, Point *m)
{
    m->x = t * (A.x - B.x) + B.x;
    m->y = t * (A.y - B.y) + B.y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the barycenter of the points A, B with coefficient t
--------------------------------------------------------------------*/
int
Bary_Many_Points(Point **A, int n)
{
    int             i;
    Point           *P;

    P = A[n - 1];
    P->x = 0.;
    P->y = 0.;

    for (i = 0; i < n - 1; i++) {
        P->x += A[i]->x;
        P->y += A[i]->y;
    }

    if (n > 1) {
        P->x /= (double) (n - 1);
        P->y /= (double) (n - 1);
    }

    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the barycenter of a polygon
--------------------------------------------------------------------*/
int
Bary_Polyg(Polygon P, Point *m)
{
    int             i;

    if (P.nb < 1)
        return 1;
    m->x = 0.;
    m->y = 0.;

    for (i = 0; i < P.nb; i++) {
        m->x += P.vertex[i].x;
        m->y += P.vertex[i].y;
    }

    m->x /= (double) P.nb;
    m->y /= (double) P.nb;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the rotation of the line L with angle Ang around the
    origin of L
--------------------------------------------------------------------*/
int
Rotate_Line(Line L, Line *lin, double Ang)
{
    double          c,
                    s;

    lin->A = L.A;
    c = cos(Ang);
    s = sin(Ang);
    lin->U.x = L.U.x * c - L.U.y * s;
    lin->U.y = L.U.x * s + L.U.y * c;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the bissectrix of the lines L1, L2
--------------------------------------------------------------------*/
int
Bissec_Line(Line L1, Line L2, Line *lin)
{
    return Bissec2_Line(L1, L2, lin, 0.5);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
Bissec2_Line(Line L1, Line L2, Line *lin, double t)
{
    Point           P;
    Line            L;
    double          Ang;

    if (Inters_Line_Line(L1, L2, &P) == 1) {
        lin->A = L1.A;
        lin->U = L1.U;
        return 1;
    }
    L.A = P;
    L.U = L1.U;
    Ang = Angle_Vector(L1.U, L2.U);
    Rotate_Line(L, lin, Ang * t);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the circles spanned by  points
--------------------------------------------------------------------*/
int
Circle_Span(Point A, Point B, Point C, Circle *circ)
{
    double          z,
                    nb,
                    nc;
    Vector          AB,
                    AC;

    AC.x = C.x - A.x;
    AC.y = C.y - A.y;
    AB.x = B.x - A.x;
    AB.y = B.y - A.y;
    nb = Length_Vector(AB);
    nc = Length_Vector(AC);
    if (nb < _geom_eps || nc < _geom_eps) {
        return 1;
    }
    z = Ext_Product_Vector(AB, AC) / nb / nc;
    if (fabs(z) < _geom_eps) {
        return 1;
    }

    z = A.x * A.x + A.y * A.y;
    Solve_2(2. * (B.x - A.x), 2. * (B.y - A.y) , 2. * (C.x - A.x),
        2. * (C.y - A.y), B.x * B.x + B.y * B.y - z, C.x * C.x + C.y * C.y - z,
            &circ->O.x, &circ->O.y);
    circ->R = Dist(circ->O, A);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the distance of two points
--------------------------------------------------------------------*/
double
Dist(Point A, Point B)
{
    double          X,
                    Y;

    X = B.x - A.x;
    Y = B.y - A.y;
    return sqrt(X * X + Y * Y);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the distance from a point to a line
--------------------------------------------------------------------*/
double
Dist_Point_Line(Point A, Line B)
{
    Point           P;

    Orthog_Proj(B, A, &P);
    return Dist(A, P);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Creates a line from an old line, with the same origin
    and inverse vector
--------------------------------------------------------------------*/
int
Inverse_Line(Line L, Line *lin)
{
    lin->A = L.A;
    lin->U.x = - L.U.x;
    lin->U.y = - L.U.y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes (one of) the circle(s) tangent to 3 lines
--------------------------------------------------------------------*/
int
Circle_Span4(Line L1, Line L2, Line L3, Circle *circ)
{
    Line            L,
                    B1,
                    B2;

    Inverse_Line(L1, &L);
    if (Bissec_Line(L2, L, &B1) == 1)
        return 1;;
    Inverse_Line(L2, &L);
    if (Bissec_Line(L3, L, &B2) == 1)
        return 1;;
    if (Inters_Line_Line(B1, B2, &circ->O) == 1)
        return 1;;
    circ->R = Dist_Point_Line(circ->O, L1);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the intersection points of a line and a circle
--------------------------------------------------------------------*/
int
Inters_Line_Circle(Line L, Circle circ, Point *A, Point *B)
{
    double          x,
                    y,
                    delta,
                    a;
    Vector          V;

    V.x = L.A.x - circ.O.x;
    V.y = L.A.y - circ.O.y;
    x = Product_Vector(V, L.U);
    y = Product_Vector(V, V);
    delta = x * x - y + circ.R * circ.R;
    if (delta < 0.)
        return 0;
    delta = sqrt(delta);
    a = - x + delta;
    A->x = L.A.x + a * L.U.x;
    A->y = L.A.y + a * L.U.y;
    if (delta == 0.) {
        B->x = A->x;
        B->y = A->y;
        return 1;
    }
    a = - x - delta;
    B->x = L.A.x + a * L.U.x;
    B->y = L.A.y + a * L.U.y;
    return 2;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the intersection points of 2 circles
--------------------------------------------------------------------*/
int
Inters_Circle_Circle(Circle circ1, Circle circ2, Point *A, Point *B)
{
    double          d,
                    s,
                    t,
                    u;
    Vector          U,
                    V;

    s = circ1.R + circ2.R;
    d = Dist(circ1.O, circ2.O);
    u = d - s;
    if (u > 0.)
        return 0;
    if (u == 0.) {
        t = circ1.R / s;
        A->x = circ1.O.x + t * (circ2.O.x - circ1.O.x);
        A->y = circ1.O.y + t * (circ2.O.y - circ1.O.y);
        return 1;
    }
    t = sqrt((s + d) * (s - d) * (d + circ1.R - circ2.R) *
        (d + circ2.R - circ1.R)) / d / 2.;
    u = (d * d + circ1.R * circ1.R - circ2.R * circ2.R) / d / 2.;
    Def_Vector_Point(circ1.O, circ2.O, &U);
    Norm_Vector(U, &U);
    Orthog_Vector(U, &V);
    A->x = circ1.O.x + u * U.x + t * V.x;
    A->y = circ1.O.y + u * U.y + t * V.y;
    B->x = circ1.O.x + u * U.x - t * V.x;
    B->y = circ1.O.y + u * U.y - t * V.y;
    return 2;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    (not done yet)
--------------------------------------------------------------------*/
int
Tangents_to_Circle_from_Point(Point A, Circle circ, Line *lin1, Line* lin2)
{
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Changes the origin of a line by adding x times its vector
--------------------------------------------------------------------*/
int
Move_Orig_Line(double x, Line *lin)
{
    lin->A.x += x * lin->U.x;
    lin->A.y += x * lin->U.y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Changes the origin of a line by giving a point on it
--------------------------------------------------------------------*/
int
Move_Orig_Line2(Point A, Line *lin)
{
    if (Dist_Point_Line(A, *lin) > _geom_eps)
        return 1;
    lin->A = A;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the vector defined by 2 points
--------------------------------------------------------------------*/
int
Def_Vector_Point(Point A, Point B, Vector *U)
{
    U->x = B.x - A.x;
    U->y = B.y - A.y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the point obtained by adding a vector to a point
--------------------------------------------------------------------*/
int
Def_Point_Vector(Point A, Vector U, Point *B)
{
    B->x = A.x + U.x;
    B->y = A.y + U.y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the polygon defined by 2 functions
--------------------------------------------------------------------*/
int
Polyg_From_Funcs(funct_d X, funct_d Y, Polygon *s)
{
    int             i;

    if (X.nb != Y.nb || X.nb < 3 || X.type != Y.type)
        return 1;
    if (X.type == 1) {
        if (fabs(X.x[1] - Y.x[1])  > _geom_eps ||
            fabs(X.x[2] - Y.x[2])  > _geom_eps) {
            return 1;
        }
    }
    s->nb = X.nb;
    s->calc = 0;
    s->icurv = 0;
    if (s->vertex != NULL)
        free(s->vertex);
    if (s->tgt != NULL)
        free(s->tgt);
    if (s->d_tgt != NULL)
        free(s->d_tgt);
    if (s->def_tgt != NULL)
        free(s->def_tgt);
    s->vertex = (Point *) malloc((size_t) X.nb * sizeof(Point));
    s->tgt = (Vector *) malloc((size_t) X.nb * sizeof(Vector));
    s->d_tgt = (Vector *) malloc((size_t) X.nb * sizeof(Vector));
    s->def_tgt = (int *) malloc((size_t) X.nb * sizeof(int));
    for (i = 0; i < X.nb; i++)
        s->def_tgt[i] = 0;

    for (i = 1; i <= X.nb; i++) {
        if (X.type == 0) {
            if (fabs(X.x[i] - Y.x[i]) > _geom_eps) {
                free(s->vertex);
                free(s->tgt);
                free(s->d_tgt);
                free(s->def_tgt);
                return 1;
            }
        }
        s->vertex[i - 1].x = X.f[i];
        s->vertex[i - 1].y = Y.f[i];
    }

    Compute_Sides(s);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the unit tangent vectors to a polygon
--------------------------------------------------------------------*/
int
Compute_Sides(Polygon *s)
{
    int             i;

    for (i = 1; i < s->nb - 1; i++) {
        Def_Vector_Point(s->vertex[i], s->vertex[i + 1], &s->tgt[i]);
        s->def_tgt[i] = 1 - Norm_Vector(s->tgt[i], &s->tgt[i]);
    }

    Def_Vector_Point(s->vertex[0], s->vertex[1], &s->tgt[0]);
    s->def_tgt[0] = 1 - Norm_Vector(s->tgt[0], &s->tgt[0]);
    Def_Vector_Point(s->vertex[s->nb - 1], s->vertex[s->nb - 2],
        &s->tgt[s->nb - 1]);
    s->def_tgt[s->nb - 1] = 1 - Norm_Vector(s->tgt[s->nb - 1],
        &s->tgt[s->nb - 1]);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the acceleration vectors of a polygon
--------------------------------------------------------------------*/
int
Compute_d_tgt(Polygon *s)
{
    int             i;
    double          x;

    if (s->icurv == 1)
        return 1;

    x = 1. / s->step / s->step;

    for (i = 1; i < s->nb - 1; i++) {
      /*  s->d_tgt[i].x = (s->vertex[i + 1].x - 2. * s->vertex[i].x +
            s->vertex[i - 1].x) * x;
        s->d_tgt[i].y = (s->vertex[i + 1].y - 2. * s->vertex[i].y +
            s->vertex[i - 1].y) * x;*/
        s->d_tgt[i].x = (s->tgt[i + 1].x - s->tgt[i].x) / s->step;
        s->d_tgt[i].y = (s->tgt[i + 1].y - s->tgt[i].y) / s->step;
    }

    s->d_tgt[0].x = s->d_tgt[1].x;
    s->d_tgt[0].y = s->d_tgt[1].y;
    s->d_tgt[s->nb - 1].x = s->d_tgt[s->nb - 2].x;
    s->d_tgt[s->nb - 1].y = s->d_tgt[s->nb - 2].y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the acceleration vectors of a polygon
    (more accurate way, using polynomial approximation)
--------------------------------------------------------------------*/
int
Compute_Sides_dtgt(Polygon *s)
{
    int             i;
    double          x0,
                    x1,
                    x2,
                    x3,
                    x4,
                    u,
                    v;

    u = -1. / 12. / s->step / s->step;
    v = 1. / 12. / s->step;

    for (i = 2; i < s->nb - 2; i++) {
        x0 = s->vertex[i - 2].x;
        x1 = s->vertex[i - 1].x;
        x2 = s->vertex[i].x;
        x3 = s->vertex[i + 1].x;
        x4 = s->vertex[i + 2].x;
        s->tgt[i].x = (x0 - 8.* x1 + 8.* x3 - x4) * v;
        s->d_tgt[i].x = (x0 - 16. * x1 + 30. * x2 - 16. * x3 + x4) * u;
        s->def_tgt[i] = 1;
        x0 = s->vertex[i - 2].y;
        x1 = s->vertex[i - 1].y;
        x2 = s->vertex[i].y;
        x3 = s->vertex[i + 1].y;
        x4 = s->vertex[i + 2].y;
        s->tgt[i].y = (x0 - 8.* x1 + 8.* x3 - x4) * v;
        s->d_tgt[i].y = (x0 - 16. * x1 + 30. * x2 - 16. * x3 + x4) * u;
    }

    x0 = s->vertex[0].x;
    x1 = s->vertex[1].x;
    x2 = s->vertex[2].x;
    s->tgt[1].x = (x2 - x0) / 2. / s->step;
    s->d_tgt[1].x = (x0 - 2.* x1 + x2) / s->step / s->step;
    x0 = s->vertex[0].y;
    x1 = s->vertex[1].y;
    x2 = s->vertex[2].y;
    s->tgt[1].y = (x2 - x0) / 2. / s->step;
    s->d_tgt[1].y = (x0 - 2.* x1 + x2) / s->step / s->step;
    x0 = s->vertex[s->nb - 3].x;
    x1 = s->vertex[s->nb - 2].x;
    x2 = s->vertex[s->nb - 1].x;
    s->tgt[s->nb - 2].x = (x2 - x0) / 2. / s->step;
    s->d_tgt[s->nb - 2].x = (x0 - 2.* x1 + x2) / s->step / s->step;
    x0 = s->vertex[s->nb - 3].y;
    x1 = s->vertex[s->nb - 2].y;
    x2 = s->vertex[s->nb - 1].y;
    s->tgt[s->nb - 2].y = (x2 - x0) / 2. / s->step;
    s->d_tgt[s->nb - 2].y = (x0 - 2.* x1 + x2) / s->step / s->step;
    s->tgt[0].x = (s->vertex[1].x - s->vertex[0].x) / s->step;
    s->d_tgt[0].x = s->d_tgt[1].x;
    s->tgt[0].y = (s->vertex[1].y - s->vertex[0].y) / s->step;
    s->d_tgt[0].y = s->d_tgt[1].y;
    s->tgt[s->nb - 1].x = (s->vertex[s->nb - 1].x - s->vertex[s->nb - 2].x)
        / s->step;
    s->d_tgt[s->nb - 1].x = s->d_tgt[s->nb - 2].x;
    s->tgt[s->nb - 1].y = (s->vertex[s->nb - 1].y - s->vertex[s->nb - 2].y)
        / s->step;
    s->d_tgt[s->nb - 1].y = s->d_tgt[s->nb - 2].y;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes a new polygon from an old one : the new has
    curvilinear coordinates (the distance between consecutive vertices
    is constant)
--------------------------------------------------------------------*/
int
Curvilinear_Polyg(Polygon S, int n, Polygon *s)
{
    int             i,
                    i0,
                    j;
    double         *le,
                    length,
                    dt,
                    t,
                    y[4],
                    h[3];

    length = 0;
    le = (double *) malloc((size_t) S.nb * sizeof(double));
    le[0] = 0;
    for (i = 1; i < S.nb; i++)
        le[i] = le[i - 1] + Dist(S.vertex[i - 1], S.vertex[i]);
    length = le[S.nb - 1];
    dt = length / (double) n;
    i0 = 1;
    if (s->vertex != NULL)
        free(s->vertex);
    s->nb = n + 1;
    s->vertex = (Point *) malloc((size_t) s->nb * sizeof(Point));
    s->vertex[0] = S.vertex[0];
    if (s->tgt != NULL)
        free(s->tgt);
    if (s->d_tgt != NULL)
        free(s->d_tgt);
    if (s->def_tgt != NULL)
        free(s->def_tgt);
    s->tgt = (Vector *) malloc((size_t) s->nb * sizeof(Vector));
    s->d_tgt = (Vector *) malloc((size_t) s->nb * sizeof(Vector));
    s->def_tgt = (int *) malloc((size_t) s->nb * sizeof(int));
    for (i = 0; i < s->nb; i++)
        s->def_tgt[i] = 0;

    for (i = 1; i <= n; i++) {
        t = i * dt;

        for (j = i0; j < S.nb; j++) {
            if (le[j] >= t) {
                i0 = j;
                if (le[j] - le[j - 1] < _geom_eps2)
                    s->vertex[i] = S.vertex[j];
                else {
                    if (j >= 2 && j < S.nb - 1) {
                        h[0] = le[j - 1] - le[j - 2];
                        h[1] = le[j] - le[j - 2];
                        h[2] = le[j + 1] - le[j - 2];
                        y[0] = S.vertex[j - 2].x;
                        y[1] = S.vertex[j - 1].x;
                        y[2] = S.vertex[j].x;
                        y[3] = S.vertex[j + 1].x;
                        s->vertex[i].x =
                            compute_pol_approx(h, y, t - le[j - 2]);
                        y[0] = S.vertex[j - 2].y;
                        y[1] = S.vertex[j - 1].y;
                        y[2] = S.vertex[j].y;
                        y[3] = S.vertex[j + 1].y;
                        s->vertex[i].y =
                            compute_pol_approx(h, y, t - le[j - 2]);
                    }
                    else {
                        s->vertex[i].x = ((le[j] - t) * S.vertex[j - 1].x +
                            (t - le[j - 1]) * S.vertex[j].x) /
                                (le[j] - le[j - 1]);
                        s->vertex[i].y = ((le[j] - t) * S.vertex[j - 1].y +
                            (t - le[j - 1]) * S.vertex[j].y) /
                                (le[j] - le[j - 1]);
                    }
                }
                break;
            }
        }
    }

    s->vertex[0] = S.vertex[0];
    s->vertex[n] = S.vertex[S.nb - 1];
    s->step = dt;
    Compute_Sides_dtgt(s);
    s->calc = 1;
    s->icurv = 1;
    free(le);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the point on a polygon at distance t from its origin
    (the distance beeing measured along the polygon)
    Works only if the polygon has "curvilinear" coordinates
--------------------------------------------------------------------*/
int
Point_on_Polyg(Polygon S, double t, Point *p)
{
    int             i;

    if (S.icurv != 1)
        return 1;
    if (t < 0 || t > (S.nb - 1) * S.step)
        return 2;
    i = t / S.step;
    p->x = ((t - i * S.step) * S.vertex[i + 1].x + ((i + 1) * S.step
        - t) * S.vertex[i].x) / S.step;
    p->y = ((t - i * S.step) * S.vertex[i + 1].y + ((i + 1) * S.step
        - t) * S.vertex[i].y) / S.step;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the point on a polygon at distance t from its origin
    (the distance beeing measured along the polygon), and the tangent
    vector multiplied by t at this point
    Works only if the polygon has "curvilinear" coordinates
--------------------------------------------------------------------*/
int
Point_on_Polyg_v(Polygon S, double t, Point *p, Vector *v)
{
    int             i;
    double          a,
                    u;

    if (S.icurv != 1)
        return 1;
    if (t < 0 || t > (S.nb - 1) * S.step)
        return 2;
    u = t / S.step;
    i = (int) floor(u);
    if (i < S.nb - 1) {
        p->x = ((t - i * S.step) * S.vertex[i + 1].x + ((i + 1) * S.step
            - t) * S.vertex[i].x) / S.step;
        p->y = ((t - i * S.step) * S.vertex[i + 1].y + ((i + 1) * S.step
            - t) * S.vertex[i].y) / S.step;
        if (S.def_tgt[i] * S.def_tgt[i + 1] == 0)
            return 3;
        a = Angle_Vector(S.tgt[i], S.tgt[i + 1]);
        Rotate_Vector(S.tgt[i], (u - i) * a, v);
        return 0;
    }
    else {
        p->x = S.vertex[i].x;
        p->y = S.vertex[i].y;
        if (S.def_tgt[i] == 0)
            return 3;
        v->x = S.tgt[i].x;
        v->y = S.tgt[i].y;
    }
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the point on a polygon at distance t from its origin
    (the distance beeing measured along the polygon), and the
    acceleration vector multiplied by t at this point
    Works only if the polygon has "curvilinear" coordinates
--------------------------------------------------------------------*/
int
Point_on_Polyg_v2(Polygon S, double t, Point *p, Vector *v)
{
    int             i;
    double          u;

    if (S.icurv != 1)
        return 1;
    if (t < 0 || t > (S.nb - 1) * S.step)
        return 2;
    u = t / S.step;
    i = (int) floor(u);
    if (i < S.nb - 1) {
        p->x = ((t - i * S.step) * S.vertex[i + 1].x + ((i + 1) * S.step
            - t) * S.vertex[i].x) / S.step;
        p->y = ((t - i * S.step) * S.vertex[i + 1].y + ((i + 1) * S.step
            - t) * S.vertex[i].y) / S.step;
        if (S.def_tgt[i] * S.def_tgt[i + 1] == 0)
            return 3;
/*        a = Angle_Vector(S.d_tgt[i], S.d_tgt[i + 1]);
        Rotate_Vector(S.d_tgt[i], (u - i) * a, v);*/
        v->x = ((t - i * S.step) * S.d_tgt[i + 1].x + ((i + 1) * S.step
            - t) * S.d_tgt[i].x) / S.step;
        v->y = ((t - i * S.step) * S.d_tgt[i + 1].y + ((i + 1) * S.step
            - t) * S.d_tgt[i].y) / S.step;
        return 0;
    }
    else {
        p->x = S.vertex[i].x;
        p->y = S.vertex[i].y;
        if (S.def_tgt[i] == 0)
            return 3;
        v->x = S.d_tgt[i].x;
        v->y = S.d_tgt[i].y;
    }
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the evolute of a curvilinear polygon
--------------------------------------------------------------------*/
int
Evolute_Polyg(Polygon S, Polygon *s)
{
    int             i;

    if (s->vertex != NULL)
        free(s->vertex);
    s->nb = S.nb;
    s->vertex = (Point *) malloc((size_t) s->nb * sizeof(Point));
    if (s->tgt != NULL)
        free(s->tgt);
    if (s->d_tgt != NULL)
        free(s->d_tgt);
    if (s->def_tgt != NULL)
        free(s->def_tgt);
    s->tgt = (Vector *) malloc((size_t) s->nb * sizeof(Vector));
    s->d_tgt = (Vector *) malloc((size_t) s->nb * sizeof(Vector));
    s->def_tgt = (int *) malloc((size_t) s->nb * sizeof(int));
    for (i = 0; i < s->nb; i++)
        s->def_tgt[i] = 0;

    for (i = 0; i < s->nb; i++) {
        s->vertex[i].x = S.vertex[i].x + S.tgt[i].y * S.tgt[i].y / S.d_tgt[i].x;
        s->vertex[i].y = S.vertex[i].y + S.tgt[i].x * S.tgt[i].x / S.d_tgt[i].y;
    }

    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Polynomial approximation
--------------------------------------------------------------------*/
double
compute_pol_approx(double *h, double *y, double x)
{
    double          a1,
                    a2,
                    a3;

    a3 = -1. / h[0] /h[1] / h[2] * y[0] - 1. / (- h[1] * h[2] + h[0] * h[1]
        + h[0] * h[2] - h[0] * h[0]) / h[0] * y[1] + 1. /(h[0] * h[2] -
            h[0] * h[1] + h[1] * h[1] - h[1] * h[2]) / h[1] * y[2] + 1. /
                (- h[1] * h[2] + h[0] * h[1] + h[2] * h[2] - h[0] * h[2]) /
                    h[2] * y[3];
    a2 = (h[1] + h[0] + h[2]) / h[0] / h[1] / h[2] * y[0] + (h[2] + h[1]) /
        (- h[1] * h[2] + h[0] * h[1] + h[0] * h[2] - h[0] * h[0]) / h[0]
            * y[1] - (h[0] + h[2]) / (h[0] * h[2] - h[0] * h[1] + h[1] * h[1]
                - h[1] * h[2]) / h[1] * y[2] - (h[1] + h[0]) / (- h[1] * h[2]
                    + h[0] * h[1] + h[2] * h[2] - h[0] * h[2]) / h[2] * y[3];
    a1 = -(h[0] * h[1] + h[1] * h[2] + h[0] * h[2]) / h[0] / h[1] / h[2] * y[0]
        - h[1] * h[2] / (- h[1] * h[2] + h[0] * h[1] + h[0] * h[2] - h[0]
            * h[0]) / h[0] * y[1] + h[0] * h[2] / (h[0] * h[2] - h[0] * h[1]
                + h[1] * h[1] - h[1] * h[2]) / h[1] * y[2] + h[0] * h[1] /
                    (- h[1] * h[2] + h[0] * h[1] + h[2] * h[2] - h[0] * h[2])
                        / h[2] * y[3];

    return y[0] + x * (a1 + x * (a2 + a3 * x));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
Add_Vector(Vector U, Vector V, Vector *w)
{
    w->x = U.x + V.x;
    w->y = U.y + V.y;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
Sub_Vector(Vector U, Vector V, Vector *w)
{
    w->x = U.x - V.x;
    w->y = U.y - V.y;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
Mult_Vector(Vector U, double x, Vector *v)
{
    v->x = x * U.x;
    v->y = x * U.y;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
Neg_Vector(Vector U, Vector *v)
{
    v->x = - U.x;
    v->y = - U.y;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
Norm_Vector(Vector U, Vector *v)
{
    double          t;

    if (fabs(U.x) < _geom_eps2 && fabs(U.y) < _geom_eps2)
        return 1;
    t = sqrt(U.x * U.x + U.y * U.y);
    v->x = U.x / t;
    v->y = U.y / t;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
Orthog_Vector(Vector U, Vector *v)
{
    Vector          V;

    if (fabs(U.x) < _geom_eps2 && fabs(U.y) < _geom_eps2)
        return 1;
    V.x = -U.y;
    V.y = U.x;
    Norm_Vector(V, v);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
Rotate_Vector(Vector U, double x, Vector *v)
{
    double          c,
                    s;

    c = cos(x);
    s = sin(x);
    v->x = c * U.x - s * U.y;
    v->y = s * U.x + c * U.y;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
double
Length_Vector(Vector U)
{
    return sqrt(U.x * U.x + U.y * U.y);
}
/*------------------------------------------------------------------*/
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
Transform_Vector(Vector U, LinTransform T, Vector *v)
{
    v->x = T.a * U.x + T.b * U.y;
    v->y = T.c * U.x + T.d * U.y;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
double
Product_Vector(Vector U, Vector V)
{
    return U.x * V.x + U.y * V.y;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
double
Angle_Vector(Vector U, Vector V)
{
    double          lu,
                    l2,
                    c,
                    s,
                    x;

    if ((fabs(U.x) < _geom_eps2 && fabs(U.y) < _geom_eps2) ||
        (fabs(V.x) < _geom_eps2 && fabs(V.y) < _geom_eps2))
        return 0.;
    lu = Length_Vector(U);
    l2 = lu * Length_Vector(V);
    c = Product_Vector(U, V) / l2;
    s = Ext_Product_Vector(U, V) / l2;
    x = acos(c);
    if (s < 0)
        x = - x;
    return x;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
double
Ext_Product_Vector(Vector U, Vector V)
{
    return U.x * V.y - U.y * V.x;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Defines the operations on vectors
--------------------------------------------------------------------*/
void
Oper_Vector(int i0, int i1, int i2, int typ, char *argv[], flow_data
    *flow_interp)
{
    Vector         *u,
                   *v,
                   *w;
    double          x;
    char          **e;

    if (comp(argv[0], "add") == 1) {
        e = (char **) Obj[typ][i0].adresse;
        u = (Vector *) e[0];
        e = (char **) Obj[typ][i1].adresse;
        v = (Vector *) e[0];
        e = (char **) Obj[typ][i2].adresse;
        w = (Vector *) e[0];
        w->x = u->x + v->x;
        w->y = u->y + v->y;
        return;
    }
    if (comp(argv[0], "substract") == 1) {
        e = (char **) Obj[typ][i0].adresse;
        u = (Vector *) e[0];
        e = (char **) Obj[typ][i1].adresse;
        v = (Vector *) e[0];
        e = (char **) Obj[typ][i2].adresse;
        w = (Vector *) e[0];
        w->x = u->x - v->x;
        w->y = u->y - v->y;
        return;
    }
    if (comp(argv[0], "multiply") == 1) {
        e = (char **) Obj[typ][i0].adresse;
        u = (Vector *) e[0];
        e = (char **) Obj[typ][i1].adresse;
        v = (Vector *) e[0];
        x = convert_float(argv[2], flow_interp);
        v->x = u->x * x;
        v->y = u->y * x;
        return;
    }
    if (comp(argv[0], "copy") == 1) {
        e = (char **) Obj[typ][i0].adresse;
        u = (Vector *) e[0];
        e = (char **) Obj[typ][i1].adresse;
        v = (Vector *) e[0];
        v->x = u->x;
        v->y = u->y;
        return;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Defines the operations on linear transforms
--------------------------------------------------------------------*/
void
Oper_LinTransform(int i0, int i1, int i2, int typ, char *argv[], flow_data
    *flow_interp)
{
    LinTransform   *u,
                   *v,
                   *w;
    double          x;
    char          **e;

    if (comp(argv[0], "add") == 1) {
        e = (char **) Obj[typ][i0].adresse;
        u = (LinTransform *) e[0];
        e = (char **) Obj[typ][i1].adresse;
        v = (LinTransform *) e[0];
        e = (char **) Obj[typ][i2].adresse;
        w = (LinTransform *) e[0];
        w->a = u->a + v->a;
        w->b = u->b + v->b;
        w->c = u->c + v->c;
        w->d = u->d + v->d;
        return;
    }
    if (comp(argv[0], "substract") == 1) {
        e = (char **) Obj[typ][i0].adresse;
        u = (LinTransform *) e[0];
        e = (char **) Obj[typ][i1].adresse;
        v = (LinTransform *) e[0];
        e = (char **) Obj[typ][i2].adresse;
        w = (LinTransform *) e[0];
        w->a = u->a - v->a;
        w->b = u->b - v->b;
        w->c = u->c - v->c;
        w->d = u->d - v->d;
        return;
    }
    if (comp(argv[0], "multiply") == 1) {
        e = (char **) Obj[typ][i0].adresse;
        u = (LinTransform *) e[0];
        e = (char **) Obj[typ][i1].adresse;
        v = (LinTransform *) e[0];
        x = convert_float(argv[2], flow_interp);
        v->a = u->a * x;
        v->b = u->b * x;
        v->c = u->c * x;
        v->d = u->d * x;
        return;
    }
    if (comp(argv[0], "mult") == 1) {
        e = (char **) Obj[typ][i0].adresse;
        u = (LinTransform *) e[0];
        e = (char **) Obj[typ][i1].adresse;
        v = (LinTransform *) e[0];
        e = (char **) Obj[typ][i2].adresse;
        w = (LinTransform *) e[0];
        w->a = u->a * v->a + u->b * v->c;
        w->b = u->a * v->b + u->b * v->d;
        w->c = u->c * v->a + u->d * v->c;
        w->d = u->c * v->b + u->d * v->d;
        return;
    }
    if (comp(argv[0], "conjg") == 1) {
        e = (char **) Obj[typ][i0].adresse;
        u = (LinTransform *) e[0];
        e = (char **) Obj[typ][i1].adresse;
        v = (LinTransform *) e[0];
        v->a = u->a;
        v->b = u->c;
        v->c = u->b;
        v->d = u->d;
        return;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Inverse of a linear transform
--------------------------------------------------------------------*/
int
inverse_Lin_Transform(LinTransform T, LinTransform *s)
{
    double          det;

    det = T.a * T.d - T.c * T.b;
    if (fabs(det) < _geom_eps2)
        return 1;
    s->a = T.d / det;
    s->b = -T.b / det;
    s->c = -T.c / det;
    s->d = T.a / det;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the image of a point by an affine transform
--------------------------------------------------------------------*/
void
act_AffTr_Point(AffTransform T, Point P, Point *q)
{
    q->x = T.P.x + T.T.a * P.x + T.T.b * P.y;
    q->y = T.P.y + T.T.c * P.x + T.T.d * P.y;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the image of a line by an affine transform
--------------------------------------------------------------------*/
int
act_AffTr_Line(AffTransform T, Line L, Line *lin)
{
    Vector          u;
    Point           P;
    double          x;

    Transform_Vector(L.U, T.T, &u);
    act_AffTr_Point(T, L.A, &P);
    x = Length_Vector(u);
    if (x < _geom_eps2)
        return 1;
    lin->A = P;
    lin->U.x = u.x / x;
    lin->U.y = u.y / x;
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the image of a polygon by an affine transform
--------------------------------------------------------------------*/
int
act_AffTr_Polyg(AffTransform T, Polygon P, Polygon *q)
{
    int             i;
    Polygon         Q;

    if (Det_Transform(T.T) < _geom_eps2)
        return 1;
    if (P.vertex == NULL || P.tgt == NULL)
        return 2;
    Q.nb = P.nb;
    Q.icurv = 0;
    Q.calc = P.calc;
    Q.step = 0.;
    Q.vertex = (Point *) malloc((size_t) Q.nb * sizeof(Point));
    Q.tgt = (Vector *) malloc((size_t) Q.nb * sizeof(Vector));
    Q.d_tgt = (Vector *) malloc((size_t) Q.nb * sizeof(Vector));
    Q.def_tgt = (int *) malloc((size_t) Q.nb * sizeof(int));

    for (i = 0; i < Q.nb; i++) {
        act_AffTr_Point(T, P.vertex[i], &Q.vertex[i]);
        if (Q.calc == 1) {
            if (P.def_tgt[i] == 1) {
                Transform_Vector(P.d_tgt[i], T.T, &Q.d_tgt[i]);
                Transform_Vector(P.tgt[i], T.T, &Q.tgt[i]);
            }
        }
    }

    copy_Polyg(Q, q);
    destroy_Polyg(Q);
    return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the image of a circle by an affine transform
    It is a polygon
--------------------------------------------------------------------*/
void
act_AffTr_Circle(AffTransform T, Circle C, Polygon *q)
{
    int             i;
    double          dt,
                    t,
                    x,
                    y;

    dt = PID / (double) (q->nb - 1);

    for (i = 0; i < q->nb; i++) {
        t = i * dt;
        x = C.O.x + C.R * cos(t);
        y = C.O.y + C.R * sin(t);
        q->vertex[i].x = T.P.x + T.T.a * x + T.T.b * y;
        q->vertex[i].y = T.P.y + T.T.c * x + T.T.d * y;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the image of a circle by a general transform
    It is a polygon
--------------------------------------------------------------------*/
void
act_GenTr_Circle(GenTransform T, Circle C, Polygon *q, flow_data *flow_interp)
{
    int             i;
    double          dt,
                    t,
                    x,
                    y;

    dt = PID / (double) (q->nb - 1);

    for (i = 0; i < q->nb; i++) {
        t = i * dt;
        x = C.O.x + C.R * cos(t);
        y = C.O.y + C.R * sin(t);
        act_GenTr_XY(T, x, y, &q->vertex[i].x, &q->vertex[i].y, flow_interp);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes a general transform at (x,y), result in (*X,*Y)
--------------------------------------------------------------------*/
void
act_GenTr_XY(GenTransform T, double x, double y, double *X, double *Y,
    flow_data *flow_interp)
{
    SetValue("x", &x, flow_interp);
    SetValue("y", &y, flow_interp);
    *X = convert_float(T.X, flow_interp);
    *Y = convert_float(T.Y, flow_interp);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the image of a point by a general transform
--------------------------------------------------------------------*/
void
act_GenTr_Point(GenTransform T, Point p, Point *q, flow_data *flow_interp)
{
    act_GenTr_XY(T, p.x, p.y, &q->x, &q->y, flow_interp);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the image of a polygon by a general transform
--------------------------------------------------------------------*/
void
act_GenTr_Polyg(GenTransform T, Polygon p, Polygon *q, flow_data *flow_interp)
{
    int             i;

    for (i = 0; i < p.nb; i++) {
        SetValue("x", &p.vertex[i].x, flow_interp);
        SetValue("y", &p.vertex[i].y, flow_interp);
        q->vertex[i].x = convert_float(T.X, flow_interp);
        q->vertex[i].y = convert_float(T.Y, flow_interp);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the image of a line by a general transform
    It is a polygon. It is compute from a to b on the line,
    w.r.t. the origin of the line
--------------------------------------------------------------------*/
void
act_GenTr_Line(GenTransform T, Line L, Polygon *q, double a, double b,
    flow_data *flow_interp)
{
    int             i;
    double          dt,
                    x,
                    y;

    dt = (b - a) / (q->nb - 1);

    for (i = 0; i < q->nb; i++) {
        x = L.A.x + (a + i * dt) * L.U.x;
        y = L.A.y + (a + i * dt) * L.U.y;
        SetValue("x", &x, flow_interp);
        SetValue("y", &y, flow_interp);
        q->vertex[i].x = convert_float(T.X, flow_interp);
        q->vertex[i].y = convert_float(T.Y, flow_interp);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Computes the image of the segment given by the points A,B by a
    general transform. It is a polygon
--------------------------------------------------------------------*/
void
act_GenTr_Segment(GenTransform T, Point A, Point B, Polygon *q,
flow_data *flow_interp)
{
    int             i;
    double          dx,
                    dy,
                    x,
                    y;

    dx = (B.x - A.x) / (q->nb - 1);
    dy = (B.y - A.y) / (q->nb - 1);

    for (i = 0; i < q->nb; i++) {
        x = A.x + i * dx;
        y = A.y + i * dy;
        SetValue("x", &x, flow_interp);
        SetValue("y", &y, flow_interp);
        q->vertex[i].x = convert_float(T.X, flow_interp);
        q->vertex[i].y = convert_float(T.Y, flow_interp);
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Determinant of a linear transform
--------------------------------------------------------------------*/
double
Det_Transform(LinTransform T)
{
    return (T.a * T.d - T.c * T.b);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Copy polygon P to polygon *q
--------------------------------------------------------------------*/
void
copy_Polyg(Polygon P, Polygon *q)
{
    int             i;

    destroy_Polyg(*q);
    q->calc = P.calc;
    q->step = P.step;
    q->icurv = P.icurv;
    q->nb = P.nb;
    if (P.vertex != NULL) {
        q->vertex = (Point *) malloc((size_t) P.nb * sizeof(Point));
        for (i = 0; i < P.nb; i++)
            q->vertex[i] = P.vertex[i];
    }
    if (P.tgt != NULL) {
        q->tgt = (Vector *) malloc((size_t) P.nb * sizeof(Vector));
        q->d_tgt = (Vector *) malloc((size_t) P.nb * sizeof(Vector));
        q->def_tgt = (int *) malloc((size_t) P.nb * sizeof(int));
        if (P.calc == 1) {
            for (i = 1; i < P.nb; i++) {
                q->tgt[i] = P.tgt[i];
                q->def_tgt[i] = P.def_tgt[i];
                q->d_tgt[i] = P.d_tgt[i];
            }
        }
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Destroys a polygon
--------------------------------------------------------------------*/
void
destroy_Polyg(Polygon P)
{
    if (P.vertex != NULL)
        free(P.vertex);
    if (P.tgt != NULL)
        free(P.tgt);
    if (P.d_tgt != NULL)
        free(P.d_tgt);
    if (P.def_tgt != NULL)
        free(P.def_tgt);
}
/*------------------------------------------------------------------*/


















