/*************************************************************************
** funct-3.1      (command interpreter - funct)                          **
** funct.h                                                               **
**                                                                       **
** Copyright (C) 2003  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      UMR 7586 du CNRS                                                 **
**      173, rue du Chevaleret                                           **
**      75013 Paris                                                      **
**      France                                                           **
**                                                                       **
 *************************************************************************/

#ifndef __FUNCT
#define __FUNCT
#include <gsl/gsl_fft_real.h>
#include <gsl/gsl_fft_halfcomplex.h>



/*--------------------------------------------------------------------
    "Function" structures
    Functions of type 0 : pair (x,f(x)) : (a.x[i],a.f[i]),
                          for   1 <= i <= a.nb
    Functions of type 1 : the first x is a.x[1], a.x[2] is the step and
                          a.x[3] is the last x (which is also
                          a.x[1] + (a.nb-1)*a.x[2])).
                          The number of intervals is a.nb-1.
--------------------------------------------------------------------*/

/* Real function in simple precision --------------------------*/
typedef struct FUNCT_f
{
    int             type;     /* Type of the function (0 or 1) */
    int             nb;       /* Number of points              */
    float          *x;        /* Array containing the x        */
    float          *f;        /* Array containing the f(x)     */
    char           *nom;      /* Name of the function          */
} funct_f;


/* Real function in double precision --------------------------*/
typedef struct FUNCT_d
{
    int             type;
    int             nb;
    double         *x;
    double         *f;
    char           *nom;
} funct_d;


/* Complex function in simple precision -----------------------*/
typedef struct FUNCT_C
{
    int             type;
    int             nb;
    float          *x;
    fcomplex       *f;
    char           *nom;
} funct_C;


/* Complex function in double precision -----------------------*/
typedef struct FUNCT_dC
{
    int             type;
    int             nb;
    double         *x;
    dcomplex       *f;
    char           *nom;
} funct_dC;
/*--------------------------------------------------------------------
--------------------------------------------------------------------*/




/*------------------------------------------------------------------*/
typedef void (*f_FUNC)(funct_f *, funct_f *);
typedef void (*d_FUNC)(funct_d *, funct_d *);
typedef void (*C_FUNC)(funct_C *, funct_C *);
typedef void (*dC_FUNC)(funct_dC *, funct_dC *);

typedef struct FUNCT_T {
    f_FUNC          F_f;
    d_FUNC          F_d;
    C_FUNC          F_C;
    dC_FUNC         F_dC;
} funct_t;

typedef void (*f_FUNC2)(funct_f *, funct_f *, funct_f *);
typedef void (*d_FUNC2)(funct_d *, funct_d *, funct_d *);
typedef void (*C_FUNC2)(funct_C *, funct_C *, funct_C *);
typedef void (*dC_FUNC2)(funct_dC *, funct_dC *, funct_dC *);

typedef struct FUNCT2_T {
    f_FUNC2         F_f;
    d_FUNC2         F_d;
    C_FUNC2         F_C;
    dC_FUNC2        F_dC;
} funct2_t;
/*------------------------------------------------------------------*/

#define __FUNC_MAX 1000000
extern  pfi   proc_func[];
int _NBFONC_FUNC;
extern FUNCTION Funcs_func[];

#ifndef PI
#define PI 3.141592653589793238
#endif



/*--------------------------------------------------------------------
    General Fourier transform structures
--------------------------------------------------------------------*/
typedef struct __transf Transf;
struct __transf {
    int             type;         /* 0 = transforms of real functions
                                     1 = transforms of complex functions */
    char           *tr;
};
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Precise Fourier transform structures
--------------------------------------------------------------------*/
typedef struct Four Tfourier;
struct Four {
    double        **a;
    double        **b;
    double          d;
    double          ondeb;
    double          onfin;
    double          ttmax;
    double          ttm;
    double          dt;
    int             npoints_2;
    int             ifft;
    int             isermax;
    double          bmin;
    char           *nom_func;
};

typedef struct CFour CTfourier;
struct CFour {
    Tfourier       *Reel;
    Tfourier       *Imag;
};
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Bessel transform structures
--------------------------------------------------------------------*/
typedef struct Bessel Tbessel;
struct Bessel {
    Tfourier       *Tr;
    int             ind;
    int             nb;
    double         *sin_c;
    double         *tri_c;
};
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
----------------------------------------------------------------------
    Function prototypes
----------------------------------------------------------------------
--------------------------------------------------------------------*/

/*====================================================================
                        bessel.c
====================================================================*/
void            Bessel_tr(Tfourier *, double *, double *, int, int, double *,
                    double *);
int             def_bessel_param(int argc, char *argv[]);
int             trans_bessel(int argc, char *argv[]);


/*====================================================================
                        funct.c
====================================================================*/
float           f_funct_eval(funct_f *, float);
void            f_funct_add(funct_f *, funct_f *, funct_f *);
void            f_funct_mul(funct_f *, funct_f *, funct_f *);
void            f_funct_div(funct_f *, funct_f *, funct_f *);
void            f_fill(funct_f *, float **);
void            f_funct_sub(funct_f *, funct_f *, funct_f *);
void            f_funct_Rmul(float, funct_f *, funct_f *);
void            f_funct_const(float, funct_f *);
void            f_funct_copy(funct_f *, funct_f *);
void            f_funct_init(funct_f *, int, int);
void            f_funct_detruit(funct_f *);
int                     test_f_equal_x(float *, float *, int);
double          d_funct_eval(funct_d *, double);
void            d_funct_add(funct_d *, funct_d *, funct_d *);
void            d_funct_mul(funct_d *, funct_d *, funct_d *);
void            d_funct_div(funct_d *, funct_d *, funct_d *);
void            d_fill(funct_d *, double **);
void            d_funct_sub(funct_d *, funct_d *, funct_d *);
void            d_funct_Rmul(double, funct_d *, funct_d *);
void            d_funct_const(double, funct_d *);
void            d_funct_copy(funct_d *, funct_d *);
void            d_funct_init(funct_d *, int, int);
void            d_funct_detruit(funct_d *);
int             test_d_equal_x(double *, double *, int);
fcomplex        C_funct_eval(funct_C *, float);
void            C_funct_add(funct_C *, funct_C *, funct_C *);
void            C_funct_mul(funct_C *, funct_C *, funct_C *);
void            C_funct_div(funct_C *, funct_C *, funct_C *);
void            C_fill(funct_C *, float **);
void            C_funct_sub(funct_C *, funct_C *, funct_C *);
void            C_funct_Rmul(fcomplex, funct_C *, funct_C *);
void            C_funct_const(fcomplex, funct_C *);
void            C_funct_copy(funct_C *, funct_C *);
void            C_funct_init(funct_C *, int, int);
void            C_funct_detruit(funct_C *);
dcomplex        dC_funct_eval(funct_dC *, double);
void            dC_funct_add(funct_dC *, funct_dC *, funct_dC *);
void            dC_funct_mul(funct_dC *, funct_dC *, funct_dC *);
void            dC_funct_div(funct_dC *, funct_dC *, funct_dC *);
void            dC_fill(funct_dC *, double **);
void            dC_funct_sub(funct_dC *, funct_dC *, funct_dC *);
void            dC_funct_Rmul(dcomplex, funct_dC *, funct_dC *);
void            dC_funct_const(dcomplex, funct_dC *);
void            dC_funct_copy(funct_dC *, funct_dC *);
void            dC_funct_init(funct_dC *, int, int);
void            dC_funct_detruit(funct_dC *);
void            conv_f_Cr(funct_f *, funct_C *);
void            conv_f_Ci(funct_f *, funct_C *);
void            conv_d_dCr(funct_d *, funct_dC *);
void            conv_d_dCi(funct_d *, funct_dC *);
void            reel_f(funct_C *, funct_f *);
void            imag_f(funct_C *, funct_f *);
void            reel_d(funct_dC *, funct_d *);
void            imag_d(funct_dC *, funct_d *);
void            integ_funct_f(funct_f *, funct_f *);
void            integ_funct_d(funct_d *, funct_d *);
void            integ_funct_C(funct_C *, funct_C *);
void            integ_funct_dC(funct_dC *, funct_dC *);
void            diff_funct_f(funct_f *, funct_f *);
void            diff_funct_d(funct_d *, funct_d *);
void            diff_funct_C(funct_C *, funct_C *);
void            diff_funct_dC(funct_dC *, funct_dC *);
void            comp_funct_f(funct_f *, funct_f *, funct_f *);
void            comp_funct_d(funct_d *, funct_d *, funct_d *);
void            comp_functb_f(char *, funct_f *, funct_f *, flow_data *);
void            comp_functb_d(char *, funct_d *, funct_d *, flow_data *);
void            comp_functc_f(funct_f *, char *, funct_f *, flow_data *);
void            comp_functc_d(funct_d *, char *, funct_d *, flow_data *);
void            comp_functd_f(char *, char *, funct_f *, flow_data *);
void            comp_functd_d(char *, char *, funct_d *, flow_data *);
void            inv_sup_funct_f(funct_f *, funct_f *);
void            inv_sup_funct_d(funct_d *, funct_d *);
int             test_correct_x_f(funct_f *);
int             test_correct_x_d(funct_d *);
int             test_correct_x_C(funct_C *);
int             test_correct_x_dC(funct_dC *);
void            Max_funct_f(funct_f *, float *, float *);
void            Max_funct_d(funct_d *, double *, double *);
void            Min_funct_f(funct_f *, float *, float *);
void            Min_funct_d(funct_d *, double *, double *);
void            Min_funct_C(funct_C *, float *, fcomplex *, float *);
void            Min_funct_dC(funct_dC *, double *, dcomplex *, double *);
void            Max_funct_C(funct_C *, float *, fcomplex *, float *);
void            Max_funct_dC(funct_dC *, double *, dcomplex *, double *);
int             load_funct_f(funct_f *, FILE *);
int             load_funct_d(funct_d *, FILE *);
int             load_funct_C(funct_C *, FILE *);
int             load_funct_dC(funct_dC *, FILE *);
void            load_data_f(FILE *, float **, float **, int *);
void            load_data_d(FILE *, double **, double **, int *);
void            load_data_C(FILE *, float **, fcomplex **, int *);
void            load_data_dC(FILE *, double **, dcomplex **, int *);
void            XFFT_func_f(funct_f *, funct_C *, int);
void            XFFT_func_d(funct_d *, funct_dC *, int);
void            XFFT_func_C(funct_C *, funct_C *, int);
void            XFFT_func_dC(funct_dC *, funct_dC *, int);
void            _Xconvol(double *, double *, double *, int, int);
void            Xconvol_f_f(funct_f *, funct_f *, funct_f *, int);
void            Xconvol_d_d(funct_d *, funct_d *, funct_d *, int);
void            Xconvol_f_C(funct_f *, funct_C *, funct_C *, int);
void            Xconvol_C_f(funct_C *, funct_f *, funct_C *, int);
void            Xconvol_C_C(funct_C *, funct_C *, funct_C *, int);
void            Xconvol_d_dC(funct_d *, funct_dC *, funct_dC *, int);
void            Xconvol_dC_d(funct_dC *, funct_d *, funct_dC *, int);
void            Xconvol_dC_dC(funct_dC *, funct_dC *, funct_dC *, int);


/*====================================================================
                        functcmd.c
====================================================================*/
int             def_funct_f(int argc, char *argv[]);
int             def_funct_d(int argc, char *argv[]);
int             def_funct_C(int argc, char *argv[]);
int             def_funct_dC(int argc, char *argv[]);
int             detruit_funct(int, int);
int             def_xrange_f(int argc, char *argv[]);
int             def_xrange_d(int argc, char *argv[]);
int             def_xrange_gen(int argc, char *argv[], int);
int             fix_xrange_f(int argc, char *argv[]);
int             fix_xrange_d(int argc, char *argv[]);
int             fix_func_R(int argc, char *argv[]);
int             fix_func_C(int argc, char *argv[]);
int             sauve_func(int argc, char *argv[]);
int             lit_func(int argc, char *argv[]);
int             exec_func2(int argc, char *argv[], funct2_t);
int             add_func(int argc, char *argv[]);
int             div_func(int argc, char *argv[]);
int             mul_func(int argc, char *argv[]);
int             sub_func(int argc, char *argv[]);
int             fix_func(char ***, char ***, char ***, int *, char *argv[]);
int             desc_func(int argc, char *argv[]);
int             exec_func(int argc, char *argv[], funct_t);
int             copy_func(int argc, char *argv[]);
int             Rmul_func(int argc, char *argv[]);
int             reel_imag_Cfunc(int argc, char *argv[]);
int             reel_imag_fix_func(int argc, char *argv[]);
int             integ_func(int argc, char *argv[]);
int             diff_func(int argc, char *argv[]);
int             comp_func(int argc, char *argv[]);
int             fill_func(int argc, char *argv[]);
int             const_func(int argc, char *argv[]);
void            dest_prop_func(int, int, flow_data *);
int             val_func(int argc, char *argv[]);
int             inv_func_cmd(int argc, char *argv[]);
int             Max_funct_cmd(int argc, char *argv[]);
int             Min_funct_cmd(int argc, char *argv[]);
int             load_funct(int argc, char *argv[]);
int             load_create_f(int argc, char *argv[]);
int             load_create_d(int argc, char *argv[]);
int             load_create_C(int argc, char *argv[]);
int             load_create_dC(int argc, char *argv[]);
int             FFT_func(int argc, char *argv[]);
int             convol_func(int argc, char *argv[]);
int             def_four_prec(int argc, char *argv[]);
int             trans_four_prec(int argc, char *argv[]);
void            detruit_four_prec(int , int);
int             test_xr_f(float *, int);
int             test_xr_d(double *, int);


/*====================================================================
                        fourier.c
====================================================================*/
Tfourier       *Tf(int *, double *, double *, int, int);
Tfourier       *fTf(int *, float *, float *, int, int);
CTfourier      *CTf(int *, float *, fcomplex *, int, int);
CTfourier      *dCTf(int *, double *, dcomplex *, int, int);
dcomplex        Trans(Tfourier *, double);
double          TransR(Tfourier *, double);
double          TransI(Tfourier *, double);
fcomplex        fTrans(Tfourier *, float);
float           fTransR(Tfourier *, float);
float           fTransI(Tfourier *, float);
fcomplex        CTrans(CTfourier *, float);
dcomplex        dCTrans(CTfourier *, double);
void            Detruit(Tfourier *);
void            CDetruit(CTfourier *);


/*--------------------------------------------------------------------
----------------------------------------------------------------------
--------------------------------------------------------------------*/




extern int   _XRANGE_F;   /* 1 */
extern int   _XRANGE_D;   /* 2 */
extern int   _RFUNC_F;    /* 3 */
extern int   _RFUNC_D;    /* 4 */
extern int   _CFUNC_F;    /* 5 */
extern int   _CFUNC_D;    /* 6 */
extern int   _FOUR_TR;    /* 7 */
extern int   _BESS_PAR;   /* 8 */
extern int   _FUNC_MESS;  /* 0 */
#define   _SECURE_FUNC

#ifdef _SECURE_FUNC
#define TX_f(c) test_correct_x_f((funct_f *)((c)))
#define TX_d(c) test_correct_x_d((funct_d *)((c)))
#define TX_C(c) test_correct_x_C((funct_C *)((c)))
#define TX_dC(c) test_correct_x_dC((funct_dC *)((c)))
#else
#define TX_f(c) 1
#define TX_d(c) 1
#define TX_C(c) 1
#define TX_dC(c) 1
#endif

#endif
