/*  
    PFlow 2.5: A potential flow demonstrator for fluid dynamics eduation
    Copyright (C) 1998  Craig P. Earls

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
//STARTPRETTY

#include "pffileparser.hpp"
#include "pffpoint.hpp"
#include "pfsinglist.hpp"
#include "pfstreamlist.hpp"
#include <stdio.h>
#include <qstring.h>
#include <qregexp.h>


PFFileParser::PFFileParser( QObject *parent=0, const char *name ):QObject(parent,name){
  d_PFStatus=PFStatus::Instance();
  d_SingularityList=d_PFStatus->getSingList();
  d_Streamlines=d_PFStatus->getStreamList();
  d_DataPoints=d_PFStatus->getDataPointList();
}

bool PFFileParser::writeFile(QTextStream *tstream){
  Singularity *l_Singularity;
  QString line;

  *tstream << "pflow-2.5\n";
  for(l_Singularity=d_SingularityList->first(); 
      l_Singularity!=NULL; 
      l_Singularity=d_SingularityList->next()){
    switch(l_Singularity->t)
      {
      case Singularity::SOURCE:
	//Source
	line.sprintf("sorc %f %f %f\n",
		     l_Singularity->x,
		     l_Singularity->y,
		     l_Singularity->mag);
	break;
      case Singularity::VORTEX:
	//Vortex
	line.sprintf("vort %f %f %f\n",
		     l_Singularity->x,
		     l_Singularity->y,
		     l_Singularity->mag);
	break;
      case Singularity::DIPOLE:
	//Dipole 
	line.sprintf("dipl %f %f %f %f\n",
		     l_Singularity->x,
		     l_Singularity->y,
		     l_Singularity->mag,
		     l_Singularity->ang*180/3.1415926535);
	break;
      };//switch 
    *tstream << line.data();
  }
  line.sprintf("free %f %f\n", 
	       d_SingularityList->getFreeDirection()*180/3.1415926535, 
	       d_SingularityList->getFreeSpeed() );
  *tstream << line.data();

  DataPoint *l_DataPoint;

  for(l_DataPoint=d_DataPoints->first();
      l_DataPoint!=NULL;
      l_DataPoint=d_DataPoints->next()){
    line.sprintf("data %f %f\n",
		 l_DataPoint->position.x(),l_DataPoint->position.y());
    *tstream << line.data();
  }

  Streamline *l_Streamline;

  for(l_Streamline=d_Streamlines->first();
      l_Streamline!=NULL;
      l_Streamline=d_Streamlines->next()){
    line.sprintf("strm %f %f\n",
		 l_Streamline->first()->x(),l_Streamline->first()->y());
    *tstream << line.data();
  };
  
  return TRUE;
}

bool PFFileParser::parseFile(QFile *inputFile){
  char buffer[100];
  QString line;

  inputFile->readLine(buffer,100);
  line=buffer;

  if(line.contains("pflow-2.5",TRUE)==1){
    while(inputFile->readLine(buffer,100)>0){
      line=buffer;
      if(line.left(4).contains("sorc",TRUE)==1) parseSource(line);
      if(line.left(4).contains("dipl",TRUE)==1) parseDipole(line);
      if(line.left(4).contains("vort",TRUE)==1) parseVortex(line);
      if(line.left(4).contains("free",TRUE)==1) parseFreeStream(line);
      if(line.left(4).contains("data",TRUE)==1) parseDataPoint(line);
      if(line.left(4).contains("strm",TRUE)==1) parseStreamline(line);
    };
    return TRUE;
  };
  return FALSE;
}

bool PFFileParser::parseSource(QString line){
  float floats[3];
  if(floatExtract(line.right(line.length()-4),floats,3)!=3){
    printf("ERROR!!!\n\n");
    return FALSE;
  }
  else{
    d_SingularityList->addNewSingularity(floats[0], 
					 floats[1],
					 Singularity::SOURCE, 
					 floats[2],
					 0,
					 FALSE);
    return TRUE;
  };
};

bool PFFileParser::parseDipole(QString line){

  float floats[4];
  if(floatExtract(line.right(line.length()-5),floats,4)!=4){
    printf("ERROR!!!\n\n");
    return FALSE;
  }
  else{
    d_SingularityList->addNewSingularity(floats[0], 
					 floats[1],
					 Singularity::DIPOLE, 
					 floats[2],
					 floats[3]/180.0*3.1415926535,
					 FALSE);
    return TRUE;
  };
};

bool PFFileParser::parseVortex(QString line){

  float floats[3];
  if(floatExtract(line.right(line.length()-5),floats,3)!=3){
    printf("ERROR!!!\n\n");
    return FALSE;
  }
  else{
    d_SingularityList->addNewSingularity(floats[0], 
					 floats[1],
					 Singularity::VORTEX, 
					 floats[2],
					 0,
					 FALSE);
    return TRUE;
  };
};

bool PFFileParser::parseFreeStream(QString line){

  float floats[2];
  if(floatExtract(line.right(line.length()-5),floats,2)!=2){
    printf("ERROR!!!\n\n");
    return FALSE;
  }
  else{
    d_SingularityList->setFreeStream(floats[1],floats[0]/180.0*3.1415926535);
    return TRUE;
  };
};

bool PFFileParser::parseDataPoint(QString line){

  float floats[2];
  if(floatExtract(line.right(line.length()-5),floats,2)!=2){
    printf("ERROR!!!\n\n");
    return FALSE;
  }
  else{
    d_DataPoints->addNewDataPoint(floats[0], floats[1],FALSE);
    return TRUE;
  };
};

bool PFFileParser::parseStreamline(QString line){

  float floats[2];
  if(floatExtract(line.right(line.length()-5),floats,2)!=2){
    printf("ERROR!!!\n\n");
    return FALSE;
  }
  else{
    d_Streamlines->Add(floats[0], floats[1]);
    return TRUE;
  };
};


int PFFileParser::floatExtract( QString s, float *fa, int max )
{
  // extract character floats (s) into an array (fa) to a maximum of (max).

  int index=0;
  int count=0;

  QString fs = s.simplifyWhiteSpace();

  do {
    float f = fs.toFloat();

    if (count < max) 
      fa[ count++ ] = f;

    index=fs.find((char)32,1);
    fs = fs.right( fs.length()-(index+1));
  } while (  index >= 0 );

  return count;
}   
//ENDPRETTY
  
/*
 * Local variables:
 *  compile-command: "cd ..;make -k"
 * End:
 */

  
