/*  
    PFlow 2.5: A potential flow demonstrator for fluid dynamics eduation
    Copyright (C) 1998  Craig P. Earls

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
//STARTPRETTY
#include "pfdisplay.hpp"

#include <math.h>
#include <qrect.h>
#include <qprogdlg.h>
#include <qpixmap.h>
#include <qpalette.h>
#include <qcolor.h>
#include <qfont.h>
#include <qbrush.h>


#include "pfcalcengine.hpp"

PFDisplay::PFDisplay(QWidget *parent, const char *name ):QFrame(parent,name)
{
  setMinimumSize(300,300);
  setGeometry(100,100,700,730);
 
  setFrameStyle(QFrame::Panel | QFrame::Plain);
  setLineWidth(3);
  setCursor(crossCursor);
  

  d_Gridlines= new Grid(GRIDPACK);

  //Initialize Status Bar
  d_Status=new QLabel(this);
  d_Status->setFrameStyle( QFrame::Panel | QFrame::Sunken );
  d_Status->setPalette(QPalette(QColor("White")));
  d_Status->setLineWidth(3);
  d_Status->setGeometry(0,700,50,50);
  d_Status->show();

  //initialize colors

  resetColors();

  //Initialize Axis Labels

  d_Axes[0] = new QwtScale(QwtScale::Left,this);
  d_Axes[1] = new QwtScale(QwtScale::Right,this);
  d_Axes[2] = new QwtScale(QwtScale::Top,this);
  d_Axes[3] = new QwtScale(QwtScale::Bottom,this);

  for (int i=0;i<axisCnt;i++)
    {
      d_as[i].adjust(-10.0,10.0,TRUE);
      d_sdiv[i] = d_as[i].scaleDiv();
      d_map[i].setDblRange(d_sdiv[i].lBound(), d_sdiv[i].hBound(), d_sdiv[i].logScale());
      d_Axes[i]->setScale(d_sdiv[i]);  
    }
  d_Axes[0]->setBorderDist(4,4);
  d_Axes[1]->setBorderDist(4,4);
  d_Axes[2]->setBorderDist(8,8);
  d_Axes[3]->setBorderDist(8,8);
  d_Axes[0]->setFont(QFont("times",8,QFont::Normal));
  d_Axes[1]->setFont(QFont("times",8,QFont::Normal));
  d_Axes[2]->setFont(QFont("times",8,QFont::Normal));
  d_Axes[3]->setFont(QFont("times",8,QFont::Normal));
  
  d_ShowGridFlag=TRUE;    
  d_ShowStreamFlag=TRUE;  
  d_ShowSingFlag=TRUE;    
  d_ShowPressureFlag=TRUE;
  d_ShowDataPointsFlag=TRUE;
  d_SnapToGrid=TRUE;
  d_ShowDetailsFlag=TRUE;
  d_DrawPressureContours=FALSE;
  d_PressureValid=FALSE;

  d_PFStatus=PFStatus::Instance();
  d_Singularities=d_PFStatus->getSingList();
  d_Streamlines=d_PFStatus->getStreamList();;
  d_DataPoints=d_PFStatus->getDataPointList();
}

void PFDisplay::resetColors(){

  d_FluidColor=new QColor(70,70,100);
  d_GridColor=new QColor(0,200,0);
  d_StreamlineColor=new QColor(0,255,255);
  d_SingColor=new QColor(255,255,0);
  d_DataPointColor=new QColor(255,255,255);
  repaint();
}


void PFDisplay::paintVortex(QPainter *p, int X, int Y, double Mag){
  int Rad=350;
  if(Mag>0){
    p->drawArc(X-Rad,-(Y+Rad),2*Rad,2*Rad,0,-5670);
    p->setBrush(QBrush(p->pen().color(),SolidPattern));
    p->drawPie((int)(X+350-250),
	       (int)(-Y-250),500,500,16*210,16*60);
    p->setBrush(QBrush(NoBrush));
  }
  else{
    p->drawArc(X-Rad,-(Y+Rad),2*Rad,2*Rad,0,5760);
    p->setBrush(QBrush(p->pen().color(),SolidPattern));
    p->drawPie((int)(X+350-250),
	       (int)(-Y-250),500,500,16*90,16*60);
    p->setBrush(QBrush(NoBrush));
  };

}

void PFDisplay::paintSource(QPainter *p,int X, int Y, double Mag){
  int ORad=500;
  int IRad=150;

  if(Mag<0){
      p->moveTo(X+ORad,-Y);
      p->lineTo(X+IRad,-Y);

      p->moveTo(X-ORad,-Y);
      p->lineTo(X-IRad,-Y);

      p->moveTo(X,-(Y+ORad));
      p->lineTo(X,-(Y+IRad));

      p->moveTo(X,-(Y-ORad));
      p->lineTo(X,-(Y-IRad));
      
      p->setBrush(QBrush(p->pen().color()));
      p->drawPie(X-300,-(Y+250),500,500, 160*16, 40*16 ); // -->     
      p->drawPie(X-200,-(Y+250),500,500, 340*16, 40*16 ); // <--
      p->drawPie(X-250,-(Y+200),500,500, 250*16, 40*16 ); // ^
      p->drawPie(X-250,-(Y+300),500,500, 70*16, 40*16 ); // V
      p->setBrush(QBrush(NoBrush));

  }
  else{
    ORad=300;
    p->moveTo(X+ORad,-Y);
    p->lineTo(X-ORad,-Y);

    p->moveTo(X,-(Y+ORad));
    p->lineTo(X,-(Y-ORad));
    
    p->setBrush(QBrush(p->pen().color()));
    p->drawPie(X+250,-(Y+250),500,500, 160*16, 40*16 ); // -->     g
    p->drawPie(X-750,-(Y+250),500,500, 340*16, 40*16 ); // <--
    p->drawPie(X-250,-(Y+750),500,500, 250*16, 40*16 ); // ^
    p->drawPie(X-250,-(Y-250),500,500, 70*16, 40*16 ); // V g
    p->setBrush(QBrush(NoBrush));
  }
}//PaintSource

void PFDisplay::paintDipole(QPainter *p,int X, int Y, double Ang){
 const double Rad=400.0;
 const double RadO2=Rad/2;

 int dir;
 dir=(int)(180.0*Ang/(PI));

 double XOff,YOff,X1,Y1;
 XOff=cos(Ang);
 YOff=sin(Ang);
 X1=Rad*XOff;
 Y1=Rad*YOff;
 p->drawEllipse((int)(X-RadO2*YOff-RadO2),
		-(int)(Y+RadO2*XOff+RadO2),
		(int)Rad,(int)Rad);
 p->drawEllipse((int)(X+RadO2*YOff-RadO2),
		-(int)(Y-RadO2*XOff+RadO2),
		(int)Rad,(int)Rad);
 
 p->moveTo((int)(X-X1),-(int)(Y-Y1));
 p->lineTo((int)(X+X1),-(int)(Y+Y1));

 p->setBrush(QBrush(p->pen().color(),SolidPattern));
 p->drawPie((int)(X+1.3*X1-250),
	    (int)(-(Y+1.3*Y1)-250),
	    500,500,16*(dir+150),16*60);
 p->setBrush(QBrush(NoBrush));
}//PaintDipole

void PFDisplay::paintFreeStream(QPainter *p, double Ang){
  double xoff,yoff,x1,y1,x2,y2,x3,y3;
  double ox,oy,fx,fy,a;
  int dir;
  dir=(int)(180.0*Ang/(PI));
  
  xoff=cos(-Ang);
  yoff=sin(-Ang);

  x1=400*xoff;
  y1=400*yoff;
  x2=x1/3-y1/3; y2=y1/2+x1/3;
  x3=x1/3+y1/3; y3=y1/2-x1/3;
  a=PI/4;
  
  ox=-10.0;oy=-10.0;fx=0.0;fy=0.0;
  if((Ang>7*a)||(Ang<=a)){ox=-9.0;fy=1.0;};//Points Right, on Left
  if((Ang>a)&&(Ang<=3*a)){oy=-9.0;fx=1.0;};//Points Up, on Bottom
  if((Ang>3*a)&&(Ang<=5*a)){ox=9.0;fy=1.0;};//Points Left, on Right
  if((Ang>5*a)&&(Ang<=7*a)){oy=9.0;fx=1.0;};//Points Down, on Top
  for (int j=1;j<20;j++){
    p->moveTo((int)(((ox+fx*j)*1000)-x1),-((int)(((oy+fy*j)*1000)+y1)));
    p->lineTo((int)(((ox+fx*j)*1000)+x1),-((int)(((oy+fy*j)*1000)-y1)));

    p->setBrush(QBrush(p->pen().color(),SolidPattern));

    p->drawPie((int)(((ox+fx*j)*1000)+x1-250),
	       -(int)(((oy+fy*j)*1000)-y1+250),
	       500,500,16*(dir+160),16*40);
    p->setBrush(QBrush(NoBrush));
  }

}//PaintFreeStream

void PFDisplay::plotStreamLine(QPainter *p,Streamline *p_Streamline){
  FPoint *l_Point;
  int X,Y;

  
  X=(int)(1000*p_Streamline->first()->x());
  Y=(int)(1000*p_Streamline->first()->y());
  p->moveTo(X-2,-(Y-2));
  p->lineTo(X+2,-(Y+2));
  p->moveTo(X-2,-(Y+2));
  p->lineTo(X+2,-(Y-2));
  p->moveTo(X,-Y);
  for(l_Point=p_Streamline->first();
      l_Point!=NULL;
      l_Point=p_Streamline->next()){
    X=(int)(1000*l_Point->x());
    Y=(int)(1000*l_Point->y());
    p->lineTo(X,-Y);
  };
};

void PFDisplay::paintStreamLines(QPainter *p_Painter){
  Streamline *l_Streamline;

  for(l_Streamline=d_Streamlines->first();
      l_Streamline!=NULL;
      l_Streamline=d_Streamlines->next()){
    plotStreamLine(p_Painter,l_Streamline);
  };
}

void PFDisplay::paintDataPoints(QPainter *p_Painter){
  DataPoint *l_DataPoint;
  int X,Y;
  

  for(l_DataPoint=d_DataPoints->first();
      l_DataPoint!=NULL;
      l_DataPoint=d_DataPoints->next()){
    X=(int)(1000*l_DataPoint->position.x());
    Y=(int)(1000*l_DataPoint->position.y());
    p_Painter->moveTo(X-100,-(Y-100));
    p_Painter->lineTo(X+100,-(Y+100));
    p_Painter->moveTo(X-100,-(Y+100));
    p_Painter->lineTo(X+100,-(Y-100));
  };
}

void PFDisplay::paintSingularities(QPainter *p){
  Singularity *l_Singularity;
  int X,Y;
  

  if (d_Singularities->getFreeSpeed()>0.0) paintFreeStream(p,d_Singularities->getFreeDirection());
  for(l_Singularity=d_Singularities->first();
      l_Singularity!=NULL;
      l_Singularity=d_Singularities->next()){
      X=(int)(1000*l_Singularity->x);
      Y=(int)(1000*l_Singularity->y);
      switch(l_Singularity->t){
      case Singularity::SOURCE:
	paintSource(p,X,Y,l_Singularity->mag);
	break;
      case Singularity::VORTEX:
	paintVortex(p,X,Y,l_Singularity->mag);
	break;
      case Singularity::DIPOLE:
	paintDipole(p,X,Y,l_Singularity->ang);
	break;
      };//switch
  }
}

void PFDisplay::print(QPainter *p){
  
  if(d_ShowGridFlag){
     d_Gridlines->setGridColor(QColor("grey80"));
     d_Gridlines->plot(p);
  }
  p->setPen(QPen(QColor("black"),30));
  if(d_ShowStreamFlag) paintStreamLines(p);

  p->setPen(QPen(QColor("black"),10));
  if(d_ShowSingFlag) paintSingularities(p);
  if(d_ShowDataPointsFlag) paintDataPoints(p);
}

void PFDisplay::paintEvent( QPaintEvent *e)
{
  QPainter p;


  QPixmap hold(size().width()-80,size().height()-30-80);
  hold.fill(*d_FluidColor);
  p.begin( &hold ); 	//Paint to a pixmap, later BLT it to the screen, 
  			//less flicker

  p.setWindow(-10000,-10000,20000,20000);
  p.setViewport(0,0,hold.width(),hold.height());


  if(d_ShowGridFlag) {
    d_Gridlines->setGridColor(*d_GridColor);
    d_Gridlines->plot(&p);
  }
  if(d_ShowStreamFlag){
      p.setPen(*d_StreamlineColor);
      paintStreamLines(&p);
  }
  if(d_ShowSingFlag){
    //p.setBrush(QBrush(QColor(255,255,0)));
    p.setPen(QPen(*d_SingColor,1));
    paintSingularities(&p);
  }
  if(d_ShowDataPointsFlag){
    p.setPen(QPen(*d_DataPointColor,2));
    paintDataPoints(&p);};
  //this line is a comment
  
  drawContents(&p);

  p.end();
  bitBlt( this, 40, 40, &hold);
}

void PFDisplay::setStatus(const char *s){
  d_Status->setText(s);
}

void PFDisplay::resizeEvent(QResizeEvent *e)
{
  d_XInt=(e->size().width()-80)/(float)(GRIDPACK);
  d_YInt=(e->size().height()-30-80)/(float)(GRIDPACK);
  d_Status->setGeometry(0,height()-30,width(),30);
  d_DrawPressureContours=FALSE;

  d_Axes[0]->setGeometry(3,40-3,37,height()-30-(80-6));//left
  d_Axes[1]->setGeometry(width()-39,40-3,35,height()-30-(80-6));//right
  d_Axes[2]->setGeometry(32,3,width()-(80-16),38);//top
  d_Axes[3]->setGeometry(32,height()-30-39,width()-(80-16),38);//bottom

  repaint(false);
};  

void PFDisplay::mousePressEvent( QMouseEvent * e )
  
{
  float xpos,ypos;
  xpos=((e->pos().x()-40)/d_XInt);
  ypos=((e->pos().y()-40)/d_YInt);
  
  if(d_SnapToGrid){
    xpos=(float)((int)(((e->pos().x()-40)+d_XInt/2)/d_XInt));
    ypos=(float)((int)(((e->pos().y()-40)+d_YInt/2)/d_YInt));
  }
  if(e->button()==LeftButton){ emit LeftClicked(xpos-10.0,10.0-ypos);}//shift coordinates for outside use
  if(e->button()==RightButton){ emit RightClicked(xpos-10.0,10.0-ypos);}
};


void PFDisplay::setDisplayGrid(bool p_Flag){
    d_ShowGridFlag=p_Flag;
    repaint(false);
}

void PFDisplay::setDisplaySingularities(bool p_Flag){
  d_ShowSingFlag=p_Flag;
  repaint(false);
}
void PFDisplay::setDisplayStreamlines(bool p_Flag){
  d_ShowStreamFlag=p_Flag;
  repaint(false);
}
void PFDisplay::setDisplayDataPoints(bool p_Flag){
  d_ShowDataPointsFlag=p_Flag;
  repaint(false);
}
void PFDisplay::setDisplaySnapToGrid(bool p_Flag){
  d_SnapToGrid=p_Flag;
}

void PFDisplay::setFluidColor(QColor *a){
  setStatus("New Fluid Color Chosen");
  delete d_FluidColor;
  d_FluidColor=a;
  repaint(FALSE);
}


void PFDisplay::setGridColor(QColor *a){
  setStatus("New Grid Color Chosen");
  delete d_GridColor;
  d_GridColor=a;
  repaint(FALSE);
};

void PFDisplay::setStreamlineColor(QColor *a){     
  setStatus("New Streamline Color Chosen");
  delete d_StreamlineColor;
  d_StreamlineColor=a;
  repaint(FALSE);
};

void PFDisplay::setSingularityColor(QColor *a){     
  setStatus("New Singularity Color Chosen");
  delete d_SingColor;
  d_SingColor=a;
  repaint(FALSE);
};

void PFDisplay::setDataPointColor(QColor *a){     
  setStatus("New Data PointFluid Chosen");
  delete d_DataPointColor;
  d_DataPointColor=a;
  repaint(FALSE);
};


//ENDPRETTY
  
/*
 * Local variables:
 *  compile-command: "cd ..;make -k"
 * End:
 */
