/*  
    PFlow 2.5: A potential flow demonstrator for fluid dynamics eduation
    Copyright (C) 1998  Craig P. Earls

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
//STARTPRETTY


#include <qlayout.h>
#include <qpixmap.h>
#include <qstring.h>
#include <qfile.h>
#include <qtstream.h>
#include <qpushbt.h>
#include <qfiledlg.h>
#include <qvalidator.h>
#include <qmsgbox.h>
#include <qfont.h>

//#include <math.h>

#include "QdbtTabular-0.31/src/qdbttabcell.h"
#include "QdbtTabular-0.31/src/qdbtsection.h"
#include "pfdatatable.hpp"

DataTabular::DataTabular(QWidget *parent,const char *name,WFlags f) :
  QdbtTabular(parent,name,f)
{
  currentCell=QPoint(-1,-1);
}

DataTabular::~DataTabular()
{
currentCell.setY(0);
}


void DataTabular::mousePressEvent(QMouseEvent *e)
{
  int rowClicked=findRow(e->y());
  int colClicked=findCol(e->x());
  if (rowClicked!=-1){
    if(colClicked<2){
      if(cell(rowClicked,colClicked)->isEditable()){
	editCell(rowClicked,colClicked);
      }
    }
    else{
      setRowSelected(rowClicked,true);
      currentCell.setY(rowClicked);
    }
  }
}


void DataTabular::cellChanged(int row, int col){
  int i=0;
  QDoubleValidator *dval;
  QString *s=new QString(cell(row,col)->text());
  dval=new QDoubleValidator(-10.0,10.0,6,0,"dval");
  if(dval->validate(*s,i)==2){
    emit (NewValue(s->toDouble(),row,col));
  }
  else{
    emit BadValue(row);
  }
}

// constructor
DataTable::DataTable(DataPointList *p_DataPointList, QWidget *parent, const char *name, WFlags f) 
      : QWidget(parent,name,f)
{
  //d_PFStatus=PFStatus::Instance();
  //  d_DataPointList=d_PFStatus->getDataPointList();
  d_DataPointList=p_DataPointList;
  d_NumRows=0;

  setCaption("PFlow 2.5: Data Points");

  QGridLayout *layout  = new QGridLayout(this,2,1,3);
  
  dtabular=new DataTabular(this);
  dtabular->setDimensions(0,5);
  dtabular->setMinimumSize(270,100);
  dtabular->setMaximumWidth(270);
  dtabular->selectByRow(false);
  dtabular->setColumnWidth(0,50);
  dtabular->setColumnWidth(1,50);
  dtabular->setColumnWidth(2,50);
  dtabular->setColumnWidth(3,50);
  dtabular->setColumnWidth(4,50);

  dtabular->setCellFont(QFont("times",10,QFont::Bold));
  connect(dtabular, SIGNAL(cellEdited(int,int)),
	  dtabular, SLOT(cellChanged(int,int)));
  connect(dtabular, SIGNAL(NewValue(double,int,int)),
	  this, SLOT(changeDataPoint(double,int,int)));
  connect(dtabular,SIGNAL(BadValue(int)),
	  this, SLOT(correctBadEntry(int)));

  QdbtSection *section;

  section=dtabular->section(0);
  section->setText("X");
  section->setAlignment(AlignCenter);

  section=dtabular->section(1);
  section->setText("Y");
  section->setAlignment(AlignCenter);

  section=dtabular->section(2);
  section->setText("Ux");
  section->setAlignment(AlignCenter);

  section=dtabular->section(3);
  section->setText("Uy");
  section->setAlignment(AlignCenter);

  section=dtabular->section(4);
  section->setText("P");
  section->setAlignment(AlignCenter);

  connect(d_DataPointList,SIGNAL(NewItemAdded(DataPoint *)),this,SLOT(addDataPoint(DataPoint *)));
  connect(d_DataPointList,SIGNAL(Updated()),this,SLOT(updateall()));


  QPushButton *b=new QPushButton("Dump to File",this,"Dump");
  connect(b,SIGNAL(clicked()),this,SLOT(dumpToFile()));

  b->setFixedHeight(b->sizeHint().height());

  layout->addWidget(dtabular,0,0);
  layout->setColStretch(0,1); // make the table strechable
  layout->setRowStretch(0,1);

  layout->addWidget(b,1,0);
  layout->setColStretch(0,1); // make the table strechable
  layout->setRowStretch(0,0);

  layout->activate();
  
  // Resize the widget to its minimal size
  resize(layout->mainWidget()->sizeHint());
}

// destructor
DataTable::~DataTable()
{
}

void DataTable::dumpToFile(){
  DataPoint *d_DP;
  QString f = QFileDialog::getSaveFileName( 0, "*.pfd", this );
    if ( !f.isEmpty() ) {
      QFile dump(f);
      if ( dump.open(IO_WriteOnly) ) {    // file opened successfully
        QTextStream t( &dump );        // use a text stream
	
	for(uint i=0;i<d_DataPointList->count();i++){
	  d_DP=d_DataPointList->at(i);
	  t<< setfill(' ') 
	   << setw(15) <<d_DP->position.x()\
	   << setw(15) <<d_DP->position.y()\
	   << setw(15) <<d_DP->uX\
	   << setw(15) <<d_DP->uY\
	   << setw(15) <<d_DP->pressure<<"\n";
	}
        dump.close();
      }
    }
    else {
        // the user cancelled the dialog
    }
}
void DataTable::addDataPoint(DataPoint *p_DataPoint){
  int newrow;

  newrow=dtabular->numRows();
  dtabular->setDimensions(newrow+1,5);
  d_NumRows++;
  dataToTable(p_DataPoint,newrow);
 }

void DataTable::clearList(){
  dtabular->setDimensions(0,5);
  d_NumRows=0;
}

void DataTable::dataToTable(DataPoint *p_DataPoint, int row){
  float X,Y,Ux,Uy,P;

  if (row>=d_NumRows){
   
    addDataPoint(p_DataPoint);
  } else{
    X=p_DataPoint->position.x();
    Y=p_DataPoint->position.y();
    Ux=p_DataPoint->uX;
    Uy=p_DataPoint->uY;
    P=p_DataPoint->pressure;
    
    QString contents;
    
    QdbtTableCell out;
    QdbtTableCell coord;
    
    out.setEditable(FALSE);
    out.setText(0);
    out.setSelectable(FALSE);
    out.setBackground(QColor(170,170,170));
    
    coord.setEditable(TRUE);
    coord.setText(0);
    coord.setSelectable(FALSE);
    coord.setBackground(white);
    
    
    contents.sprintf("%+2.3f",X);
    coord.setText(contents);
    dtabular->changeCell(&coord,row,XCOLUMN);
    contents.sprintf("%+2.3f",Y);
    coord.setText(contents);
    dtabular->changeCell(&coord,row,YCOLUMN);
    contents.sprintf("%+2.3f",Ux);
    out.setText(contents);
    dtabular->changeCell(&out,row,UXCOLUMN);
    contents.sprintf("%+2.3f",Uy);
    out.setText(contents);
    dtabular->changeCell(&out,row,UYCOLUMN);
    contents.sprintf("%+2.3f",P);
    out.setText(contents);
    dtabular->changeCell(&out,row,PCOLUMN);
  };
}

void DataTable::updateall(){
  for(uint i=0;i<d_DataPointList->count();i++){
    dataToTable(d_DataPointList->at(i),i);
  }         
}


void DataTable::changeDataPoint(double Val, int row, int col){
  DataPoint *target=d_DataPointList->at(row);
  switch(col){
  case 0:
    d_DataPointList->change(row,Val,target->position.y());
    break;
  case 1:
    d_DataPointList->change(row,target->position.x(),Val);
  }//switch
  dataToTable(d_DataPointList->at(row),row);
}

void DataTable::correctBadEntry(int row){
  dataToTable(d_DataPointList->at(row),row);
  
  QMessageBox *BadEntry=new QMessageBox("PFlow 2.5 Error","\
You have entered an illegal value.\n\
\n\
Legal Values are:\n\
X,Y: -10.0 to 10.0\n",
					QMessageBox::Warning,
					QMessageBox::Ok,
					0,0,this,"badentry",TRUE,0);
  BadEntry->setFont(QFont("helvetica",12,QFont::Bold));
  BadEntry->show();
}


//ENDPRETTY
  
/*
 * Local variables:
 *  compile-command: "cd ..;make -k"
 * End:
 */
