#include <qwt_math.h>
#include <qdrawutl.h>
#include <qwt_wheel.h>

//------------------------------------------------------------
//.H QwtWheel | 3 | 02/04/97 | Qwt Widget Library | Qwt Programmer's Manual
//.I wheel The QwtWheel Widget
//.U NAME
//	QwtWheel - The Wheel Widget
//
//.U SYNOPSIS
//	#include <qwt_wheel.h>
//
//.U INHERITED CLASSES
//     @^QwtSliderBase@
//
//.U PUBLIC MEMBERS
//.R
//	QwtWheel::setTotalAngle --
//		Set the maximum angle by which the wheel can be turned
//	QwtWheel::setTickCnt --
//		set the number of grooves in the wheel's surface 
//	QwtWheel::setOrientation -- Change the orientation
//	QwtWheel::setViewAngle -- Change the visible angle
//	QwtWheel::setInternalBorder -- Change the internal border
//	QwtWheel::setMass -- Set the mass for a flywheel effect
//
//.U DESCRIPTION
//	The wheel widget can be used to change values over a very large range
//	in very small steps. Using the setMass member, it can be configured
//	as a flywheel.
//.U EXAMPLE
//	See the radio example.
//------------------------------------------------------------

//------------------------------------------------------------
//.U MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtWheel::QwtWheel
//	Constructor
//
//.u	Parameters
//.p	QWidget *parent, const char *name
//
//.u	Syntax
//.f	 QwtWheel::QwtWheel(QWidget *parent, const char *name)
//
//------------------------------------------------------------
QwtWheel::QwtWheel(QWidget *parent, const char *name)
: QwtSliderBase(parent,name)
{
    
    //
    // initialize private variables
    //
    d_orient = QwtWheel::Horizontal;
    setTotalAngle(360);
    setViewAngle(175);
    setInternalBorder(2);
    setUpdateTime(50);
    tickCnt = 10;
    d_borderWidth = 2;
    d_colorCnt = 30;
    d_colors = new QColor[d_colorCnt];
    createColors();
    d_pixmap.resize(10,10);
}

//------------------------------------------------------------
//
//.F	QwtWheel::~QwtWheel
//	Destructor
//
//.u	Syntax
//.f	 QwtWheel::~QwtWheel()
//
//------------------------------------------------------------
QwtWheel::~QwtWheel()
{
    delete[] d_colors;
    
}

//------------------------------------------------------------
//.-
//.F	QwtWheel::createColors
//	Set up the color array for the background pixmap.
//
//.u	Syntax
//.f	void QwtWheel::buildColors()
//
//------------------------------------------------------------
void QwtWheel::createColors()
{
    int i;
    int dh, ds, dv, lh, ls, lv;
    double factor;
	static int allocContext = 0;

    if (allocContext)
       QColor::destroyAllocContext(allocContext);

    allocContext = QColor::enterAllocContext();
    
    QColorGroup g = colorGroup();
    d_colors[0] = g.light();
    d_colors[d_colorCnt - 1] = g.dark();

    d_colors[0].rgb(&lh, &ls, &lv);
    d_colors[d_colorCnt - 1].rgb(&dh, &ds, &dv);
    
    for (i=1; i < d_colorCnt - 1; i++)
    {
	factor = double(i) /  double (d_colorCnt);
	
	d_colors[i].setRgb( lh + int (double(dh - lh) * factor ),
			   ls + int ( double(ds - ls) * factor ),
			   lv + int ( double(dv - lv) * factor ));
    }
	QColor::leaveAllocContext();    
}

//------------------------------------------------------------
//
//.F	QwtWheel::setTickCnt
//		Adjust the number of grooves in the wheel's surface.
//
//.u	Syntax
//.f	void QwtWheel::setTickCnt(int cnt)
//
//.u	Parameters
//.p	int cnt -- Number of grooves per 360 degrees
//
//.u	Description
//		The number of grooves is limited to 6 <= cnt <= 50.
//		Values outside this range will be clipped.
//		The default value is 10.
//------------------------------------------------------------
void QwtWheel::setTickCnt(int cnt)
{
    if (cnt < 6)
       tickCnt = 6;
    else if (cnt > 50)
       tickCnt = 50;
    else
       tickCnt = cnt;
}


//------------------------------------------------------------
//
//.F	QwtWheel::setInternalBorder
//      Set the internal border width of the wheel.
//
//.u	Syntax
//.f	void QwtWheel::setInternalBorder(int w)
//
//.u	Parameters
//.p	int w -- border width
//
//.u	Description
//      The internal border must not be smaller than 1
//      and is limited in dependence on the wheel's size.
//      Values outside the allowed range will be clipped.
//      The internal border defaults to 1.
//
//------------------------------------------------------------
void QwtWheel::setInternalBorder(int w) 
{
    if(w < 1) 
       d_intBorder = 1;
    else if ((w > width() / 3) || (w > height() / 3))
       d_intBorder = (width() < height()) ? width() / 3 : height() / 3 ;
    else 
       d_intBorder = w;
}

//------------------------------------------------------------
//.-
//.F	QwtWheel::rebuildPixmap
//		Refill the Wheel's background pixmap
//
//.u	Syntax
//.f	void QwtWheel::rebuildPixmap()
//
//------------------------------------------------------------
void QwtWheel::rebuildPixmap()
{

    QPen darkPen, lightPen;
    int rx, ry, rh, rw;
    int hiPos, nFields;
    int i;
    int x1, x2, y1, y2;


    nFields = d_colorCnt * 13 / 10;
    hiPos = nFields - d_colorCnt + 1;
    
    QColorGroup g = colorGroup();

    const QRect &r = d_pixmap.rect();

    //
    // initialize pens
    //
    lightPen.setColor(g.light());
    lightPen.setWidth(d_intBorder);
    darkPen.setColor(g.dark());
    darkPen.setWidth(d_intBorder);

    //
    // initialize auxiliary variables
    //

    QPainter p;

    if (p.begin(&d_pixmap))
    {
	if (d_orient == QwtWheel::Horizontal)
	{

	    rx = r.x();
	    ry = r.y() + d_intBorder;
	    rh = r.height() - 2* d_intBorder;
	    rw = r.width();
	    //
	    //  draw shaded background
	    //
	    x1 = rx;
	    x2 = rx;
	    for (i=1; i < nFields; i++)
	    {
		x2 = rx + (rw * i) / nFields;
		p.fillRect(x1, ry, x2-x1 + 1 ,rh, d_colors[qwtAbs(i-hiPos)]);
		x1 = x2 + 1;
	    }
	    p.fillRect(x1, ry, rw - (x1 - rx), rh, d_colors[d_colorCnt - 1]);


	    //
	    // draw internal border
	    //
	    p.setPen(lightPen);
	    ry = r.y() + d_intBorder / 2;
	    p.drawLine(r.x(), ry, r.x() + r.width() , ry); 
	    
	    p.setPen(darkPen);
	    ry = r.y() + r.height() - (d_intBorder - d_intBorder / 2);
	    p.drawLine(r.x(), ry , r.x() + r.width(), ry); 

	    
	}
	else
	{

	    rx = r.x() + d_intBorder;
	    ry = r.y();
	    rh = r.height();
	    rw = r.width() - 2 * d_intBorder;

	    //
	    // draw shaded background
	    //
	    y1 = ry;
	    y2 = ry;
	    for (i=1; i < nFields; i++)
	    {
		y2 = ry + (rh * i) / nFields;
		p.fillRect(rx, y1, rw, y2-y1 + 1, d_colors[qwtAbs(i-hiPos)]);
		y1 = y2 + 1;
	    }
	    p.fillRect(rx, y1, rw, rh - (y1 - ry), d_colors[d_colorCnt - 1]);

	    //
	    //	draw internal borders
	    //
	    p.setPen(lightPen);
	    rx = r.x() + d_intBorder / 2;
	    p.drawLine(rx, r.y(), rx, r.y() + r.height()); 
	    
	    p.setPen(darkPen);
	    rx = r.x() + r.width() - (d_intBorder - d_intBorder / 2);
	    p.drawLine(rx, r.y(), rx , r.y() + r.height()); 

	    
	}
    }

    p.end();

}


//------------------------------------------------------------
//
//.F	QwtWheel::setTotalAngle
//		Set the total angle which the wheel can be turned.
//
//.u	Syntax
//.f	void QwtWheel::setTotalAngle(double angle)
//
//.u	Parameters
//.p	double angle -- total angle in degrees
//
//
//.u	Description
//		One full turn of the wheel corresponds to an angle of
//		360 degrees. A total angle of n*360 degrees means
//		that the wheel has to be turned n times around its axis
//		to get from the minimum value to the maximum value.
//		The default setting of the total angle is 360 degrees.
//
//------------------------------------------------------------
void QwtWheel::setTotalAngle(double angle) 
{
    
    // range check for total angle
    if (angle > 3600) totalAngle = 3600;
    else if (angle < 10)  totalAngle = 10;
    else totalAngle = angle;
    
}

//------------------------------------------------------------
//
//.F	QwtWheel::setOrientation
//		Set the wheel's orientation.
//
//.u	Syntax
//.f	void QwtWheel::setOrientation(Orient o)
//
//.u	Parameters
//.p	Orientation o --  Orientation. Allowed values are
//			QwtWheel::Horizontal and QwtWheel::Vertical.
//
//.u	Description
//		The default orientation is QwtWheel::Horizontal.
//
//------------------------------------------------------------
void QwtWheel::setOrientation(Orientation o)
{
    if (d_orient != o)
    {
	d_orient = o;
	resize(size());
	repaint(FALSE);
    }
    
}


//------------------------------------------------------------
//
//.F	QwtWheel::setViewAngle
//		Specify the visible portion of the wheel.
//
//.u	Syntax
//.f	void QwtWheel::setViewAngle(double angle)
//
//.u	Parameters
//.p	double angle --			Visible angle in degrees
//
//
//.u	Description
//	You may use this function for fine-tuning the appearance of
//	the wheel. The default value is 175 degrees. The value is
//	limited from 10 to 175 degrees.
//
//------------------------------------------------------------
void QwtWheel::setViewAngle(double angle) {
	
	if (angle < 10) viewAngle = 10;
	else if (angle > 175) viewAngle = 175;
	else viewAngle = angle;
	
    }

//-------------------------------------------------------------
//.-
//.F	QwtWheel::drawWheel
//	Redraw the wheel
//
//.u Parameters
//	QPainter *p	-- painter
//	const QRect&r -- contents rectangle
//
//
//-------------------------------------------------------------
void QwtWheel::drawWheel( QPainter *p, const	QRect &r)
{

    QColorGroup g = colorGroup();
    double loValue, hiValue, halfIntv, tickWidth, tickValue, halfSize;
    int tickPos;
    double cnvFactor, sinArc;
    double sign;
    int l2 = 0,l1 = 0, maxpos;


    sign = (minValue() < maxValue()) ? 1.0 : -1.0;

    //
    //	calculate some useful values
    //
    cnvFactor = qwtAbs(totalAngle / (maxValue() - minValue()));
    halfIntv = 0.5 * viewAngle / cnvFactor;
    loValue = value() - halfIntv;
    hiValue = value() + halfIntv;
    tickWidth = 360.0 / double(tickCnt) / cnvFactor;
    sinArc = sin(viewAngle * M_PI / 360.0);
    cnvFactor *= M_PI / 180.0;


    //
    // draw background pixmap
    //
    bitBlt(this, r.x(), r.y(), &d_pixmap, 0, 0);
    
    //
    // draw grooves
    //
    if (d_orient == QwtWheel::Horizontal) 
    {
	halfSize = double(r.width()) * 0.5;

	l1 = r.y() + d_intBorder;
	l2 = r.y() + r.height() - d_intBorder - 1;

	// draw one point over the border if border > 1
	if (d_intBorder > 1)
	{
	    l1 --;
	    l2 ++;
	}
	
	maxpos = r.x() + r.width() - 2;
	minPos = r.x() + 2;

	//
	// draw tick marks
	//
	for (tickValue = ceil(loValue / tickWidth) * tickWidth;
	     tickValue < hiValue; tickValue += tickWidth)
	{	
	    //
	    //	calculate position
	    //
	    tickPos = r.x() + r.width()
	       - int( halfSize
		     * (sinArc + sign *  sin((tickValue - value()) * cnvFactor))
		     / sinArc);
	    //
	    // draw vertical line
	    //
	    if ((tickPos <= maxpos) && (tickPos > minPos)) 
	    {
		p->setPen(g.dark());
		p->drawLine(tickPos -1 , l1, tickPos - 1,  l2 );	
		p->setPen(g.light());
		p->drawLine(tickPos, l1, tickPos, l2);	
	    }
	    
	}
    }
    else if (d_orient == QwtWheel::Vertical) 
    {
	halfSize = double(r.height()) * 0.5;

	l1 = r.x() + d_intBorder;
	l2 = r.x() + r.width() - d_intBorder - 1;

	if (d_intBorder > 1) 
	{
	    l1--;
	    l2++;
	}

	maxpos = r.y() + r.height() - 2;
	minPos = r.y() + 2;
	
	//
	// draw tick marks
	//
	for (tickValue = ceil(loValue / tickWidth) * tickWidth;
	     tickValue < hiValue; tickValue += tickWidth)
	{		

	    //
	    // calculate position
	    //
	    tickPos = r.y() + r.height()
	       -  int( halfSize
		      * (sinArc + sign *  sin((tickValue - value()) * cnvFactor))
		      / sinArc);
	    
	    //
	    //	draw horizontal line
	    //
	    if ((tickPos <= maxpos) && (tickPos > minPos))
	    {
		p->setPen(g.dark());
		p->drawLine(l1, tickPos - 1 ,l2, tickPos - 1);	
		p->setPen(g.light());
		p->drawLine(l1, tickPos, l2, tickPos);	
	    }
	    
	}
    }
}


//------------------------------------------------------------
//.-
//.F	QwtWheel::getValue
//	Determine the value corresponding to a specified point
//
//.u	Syntax
//.f	double QwtWheel::getValue(const QPoint &p)
//
//.u	Parameters
//.p	const QPoint &p -- point
//
//.u	Description
//	This function is only called by the QwtSliderBase class.
//
//------------------------------------------------------------
double QwtWheel::getValue( const QPoint &p) 
{  
    double arc;
    int dx,w;

    QRect &r = d_sliderRect;
    
    //
    // return the value corresponding to point p.
    //
    
    if (d_orient == QwtWheel::Vertical)
    {
	w = r.height();
	if (p.y() > r.y() + w)
	   dx = w;
	else if (p.y() < r.y())
	   dx = 0;
	else
	   dx = p.y() - r.y();
    }
    else
    {
	w = r.width();
	if (p.x() > r.x() + w)
	   dx =  w;
	else if (p.x() < r.x())
	   dx = 0;
	else
	   dx = p.x() - r.x();
    }
    
    arc = asin( double( 2 * dx - w ) / double(w)
	       * sin(viewAngle * M_PI / 360.0 ))
       /  M_PI * 180.0;
    
    //
    // If this seems wrong to you, you're right...
    // But the QwtSliderBase class only needs the
    // correct offset here, not the real value.
    //
    return arc / totalAngle       
       * (maxValue() - minValue());   
    
    
}

//------------------------------------------------------------
//.-
//.F	QwtWheel::resizeEvent
//		Qt Resize Event
//
//.u	Syntax
//.f	void QwtWheel::resizeEvent(QResizeEvent *e )
//
//.u	Parameters
//.p	QResizeEvent *e 
//
//------------------------------------------------------------
void QwtWheel::resizeEvent(QResizeEvent *e )
{
    const QSize &s = e->size();
    d_sliderRect.setRect(rect().x() + d_borderWidth, rect().y() + d_borderWidth,
			 s.width() - 2*d_borderWidth, s.height() - 2*d_borderWidth);
    
    d_pixmap.resize(d_sliderRect.width(), d_sliderRect.height());
    
    rebuildPixmap();
}

//------------------------------------------------------------
//.-
//.F	QwtWheel::paintEvent
//	Qt paint event
//
//.u	Syntax
//.f	void QwtWheel::paintEvent(QPaintEvent *e)
//
//.u	Parameters
//.p	QPaintEvent *e
//
//------------------------------------------------------------
void QwtWheel::paintEvent(QPaintEvent *e)
{
    QPainter p;
    if(p.begin(this))
    {
	qDrawShadePanel(&p,
			rect().x(), rect().y(),
			rect().width(), rect().height(),
			colorGroup(), TRUE, d_borderWidth);
	
	drawWheel(&p, d_sliderRect);
    }
    p.end();
    
}

//------------------------------------------------------------
//.-
//.F	QwtWheel::valueChange
//	Notify value change	
//
//.u	Syntax
//.f	void QwtWheel::valueChange()
//
//.u	Description
//	Called by QwtSliderBase
//
//------------------------------------------------------------
void QwtWheel::valueChange()
{
    repaint(FALSE);
    QwtSliderBase::valueChange();
}


//------------------------------------------------------------
//.-
//.F	QwtWheel::getScrollMode
//	Determine the scrolling mode and direction corresponding
//	to a specified point
//
//.u	Syntax
//.f	 void QwtWheel::getScrollMode( const QPoint &p, int &scrollMode, int &direction)
//
//.u	Parameters
//.p	 const QPoint &p -- point
//       int &scrollMode	-- scrolling mode
//	 int &direction		-- direction
//
//.u Description
//	Called by QwtSliderbase
//
//------------------------------------------------------------
void QwtWheel::getScrollMode( const QPoint &p, int &scrollMode, int &direction) 
{
    //
    // set scrollMode and direction
    //
    if (d_sliderRect.contains(p))
       scrollMode = ScrMouse;
    else
       scrollMode = ScrNone;
    
    direction = 0;
    
}



//------------------------------------------------------------
//
//.F	QwtWheel::setMass
//	Set the mass of the wheel
//
//.u	Syntax
//.f	void QwtWheel::setMass(double val)
//
//.u	Parameters
//.p	double val	-	the wheel's mass
//
//.u	Description
//	Assigning a mass turns the wheel into a flywheel.
//
//------------------------------------------------------------
void QwtWheel::setMass(double val)
{
    QwtSliderBase::setMass(val);
}































