#include <qpainter.h>
#include <qdrawutl.h>
#include <qwt_math.h>
#include "qwt_thermo.h"
//------------------------------------------------------------
//.H QwtThermo|3|27/04/97|Qwt Widget Library|Qwt Programmer's manual
//.I thermo QwtThermo widgets
//.U NAME
//	QwtThermo - The Thermometer Widget
//
//.U SYNOPSIS
//	#include <qwt_thermo.h>
//
//.U INHERITED CLASSES 
//	@QwtScaleIf@, QWidget
//
//.U PUBLIC MEMBERS
//.R
//	QwtThermo -- Constructor
//	~QwtThermo -- Destructor
//	QwtThermo::setOrientation  -- Change orientation
//	QwtThermo::setBorderWidth -- change the border width of the pipe
//	 QwtThermo::setFillColor -- set fill color
//	 QwtThermo::setAlarmColor -- set color for the range exceeding the alarm threshold
//	 QwtThermo::setAlarmLevel -- set alarm threshold
//	 QwtThermo::setAlarmEnabled  -- enable alarm threshold
//	 QwtThermo::setPipeWidth -- set the width of the pipe
//	 QwtThermo::setRange -- set the range to be displayed
//	 QwtThermo::setMargin -- change margin between the pipe and the widget border
//	 QwtThermo::maxValue  -- return the upper limit of the range
//	 QwtThermo::minValue -- return the lower limit of the range
//	  QwtThermo::value -- return current value
//
//------------------------------------------------------------

//------------------------------------------------------------
//.C MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F QwtThermo::QwtThermo
//  	Constructor
//
//.u Syntax
//.f QwtThermo::QwtThermo(QWidget *parent, const char *name)
//
//.u Parameters
//.p	QWidget *parent -- parent widget
//	const char *name -- name
//------------------------------------------------------------
QwtThermo::QwtThermo(QWidget *parent, const char *name)
: QWidget(parent, name)
{
    init();
}

//------------------------------------------------------------
//
//.F	QwtThermo::~QwtThermo
//	Destructor
//
//.u	Syntax
//.f	QwtThermo::~QwtThermo() 
//------------------------------------------------------------
QwtThermo::~QwtThermo()
{
    
}


//------------------------------------------------------------
//.-
//.F    QwtThermo::init
//	initialize data members
//------------------------------------------------------------
void QwtThermo::init()
{
    // initialize data members
    d_orient = Vertical;
    d_scalePos = Left;
    d_vertPad = 10;
    d_horPad = 20;
    d_scaleDist = 3;
    d_thermoWidth = 10;
    d_borderWidth = 2;
    d_maxValue = 1.0;
    d_minValue = 0.0;
    d_value = 0.0;
    d_alarmLevel = 0.0;
    d_alarmEnabled = 0;

    // initialize colors;
    d_fillColor = black;
    d_alarmColor = white;
    
    // initialize scales
    d_map.setDblRange(d_minValue, d_maxValue);
    d_scale.setScale(d_minValue, d_maxValue, d_maxMajor, d_maxMinor);

    setMinimumSize(5,5);
}


//------------------------------------------------------------
//
//.F	QwtThermo::setValue
//	Set the current value
//
//.u	Syntax
//.f	void QwtThermo::setValue(double v)
//------------------------------------------------------------
void QwtThermo::setValue(double v)
{
    if (d_value != v)
    {
	d_value = v;
	repaint(d_thermoRect, FALSE);
    }
}

//------------------------------------------------------------
//.-
//.F	QwtThermo::paintEvent
//	Qt paint event handler
//
//.u	Parameters
//.p	QPaintEvent *e -- paint event structure
//
//------------------------------------------------------------
void QwtThermo::paintEvent(QPaintEvent *e)
{

    QPainter p;
    QColorGroup g = colorGroup();
    QRect r = e->rect();
    
    p.begin(this);
    
    if (d_thermoRect.contains(r))
    {
	drawThermo(&p);
    }
    else
    {
	p.fillRect(rect(), this->backgroundColor());

	if (d_scalePos != None)
	   d_scale.draw(&p);

	qDrawShadePanel(&p,
			d_thermoRect.x() - d_borderWidth,
			d_thermoRect.y() - d_borderWidth,
			d_thermoRect.width() + 2*d_borderWidth,
			d_thermoRect.height() + 2*d_borderWidth,
			g, TRUE, d_borderWidth,0);
	drawThermo(&p);
    }
    p.end();
    
}

//------------------------------------------------------------
//.-
//.F QwtThermo::resizeEvent
//	Qt resize event handler
//
//.u	Parameters
//.p	QResizeEvent *e
//
//.u Description
//
//------------------------------------------------------------
void QwtThermo::resizeEvent(QResizeEvent *e)
{
    switch(d_orient)
    {
    case Horizontal:
	
	switch(d_scalePos)
	{
	case None:

	    d_thermoRect.setRect(this->rect().x() + d_borderWidth,
				 this->rect().y() + d_borderWidth,
				 e->size().width() - 2*d_borderWidth,
				 e->size().height() - 2*d_borderWidth);
	    break;
	    
	case Bottom:

	    d_thermoRect.setRect(this->rect().x() + d_horPad + d_borderWidth,
				 this->rect().y() + d_borderWidth,
				 e->size().width() - 2*(d_borderWidth + d_horPad),
				 d_thermoWidth);
	    d_scale.setGeometry(d_thermoRect.x(),
				d_thermoRect.y() + d_thermoRect.height()
				+ d_borderWidth + d_scaleDist,
				d_thermoRect.width(),
				QwtScaleDraw::Bottom);
	    break;

	case Top:
	default:
	    d_thermoRect.setRect(this->rect().x() + d_horPad + d_borderWidth,
				 this->rect().y() + e->size().height()
				 - d_thermoWidth - 2*d_borderWidth,
				 e->size().width() - 2*(d_borderWidth + d_horPad),
				 d_thermoWidth);
	    d_scale.setGeometry(d_thermoRect.x(),
				d_thermoRect.y() - d_borderWidth - d_scaleDist,
				d_thermoRect.width(),
				QwtScaleDraw::Top);
	    break;
	    
	}
	d_map.setIntRange(d_thermoRect.x(), d_thermoRect.x() + d_thermoRect.width() - 1);
	break;
	
    case Vertical:
    default:

	switch(d_scalePos)
	{
	case None:

	    d_thermoRect.setRect(this->rect().x() + d_borderWidth,
				 this->rect().y() + d_borderWidth,
				 e->size().width() - 2*d_borderWidth,
				 e->size().height() - 2*d_borderWidth);
	    break;
	    
	case Left:

	    d_thermoRect.setRect(this->rect().x() +  e->size().width()
				 - 2*d_borderWidth - d_thermoWidth,
				 this->rect().y() + d_vertPad + d_borderWidth,
				 d_thermoWidth,
				 e->size().height() - 2*(d_borderWidth + d_vertPad));
	    d_scale.setGeometry(d_thermoRect.x() - d_scaleDist - d_borderWidth,
				d_thermoRect.y(),
				d_thermoRect.height(),
				QwtScaleDraw::Left);
	    break;

	case Right:
	default:

	    d_thermoRect.setRect(this->rect().x() + d_borderWidth,
				 this->rect().y() + d_vertPad + d_borderWidth,
				 d_thermoWidth,
				 e->size().height() - 2*(d_borderWidth + d_vertPad));
	    d_scale.setGeometry(d_thermoRect.x() + d_thermoRect.width()
				+ d_borderWidth + d_scaleDist,
				d_thermoRect.y(),
				d_thermoRect.height(),
				QwtScaleDraw::Right);
	    break;
	}
	d_map.setIntRange( d_thermoRect.y() + d_thermoRect.height() - 1,d_thermoRect.y());
	break;
	
    }
    
}

//------------------------------------------------------------
//
//.F	QwtThermo::setOrientation
//	Change the thermometer's orientation
//
//.u	Syntax
//.f	void QwtThermo::setOrientation(Orientation o, ScalePos s)
//
//.u	Parameters
//.p	Orientation o	-- orientation. Possible values are
//				Horizontal and Vertical.
//				The default value is Vertical.
//	ScalePos s	-- Position of the scale. For a horizontal
//				orientation, the scale position can
//				be Top, Bottom or None. A vertical
//				thermometer may have the scale positions
//				Left, Right or None. The default is None.
//
//.u Description
//	The scale position None disables the scale.
//
//------------------------------------------------------------
void QwtThermo::setOrientation(Orientation o, ScalePos s)
{
    switch(o)
    {
    case Horizontal:
	
	d_orient = Horizontal;
	if ((s == None) || (s == Bottom) || (s == Top))
	{
	    d_scalePos = s;
	}
	else
	   d_scalePos = None;
	break;

    case Vertical:

	d_orient = Vertical;
	if ((s == None) || (s == Left) || (s == Right))
	{
	    d_scalePos = s;
	}
	else
	   d_scalePos = None;
	break;

    }
    resize(size());
    
}

//------------------------------------------------------------
//.-
//.F	QwtThermo::fontChange
//	notify changed font
//
//.u	Parameters
//.p	const QFont &oldFont
//
//.u	Syntax
//.f	void QwtThermo::fontChange(const QFont &oldFont)
//
//------------------------------------------------------------
void QwtThermo::fontChange(const QFont &oldFont)
{
    repaint(FALSE);
}

//------------------------------------------------------------
//.-
//.F	QwtThermo::scaleChange
//	notify changed scale
//
//.u	Syntax
//.f	void QwtThermo::scaleChange()
//
//------------------------------------------------------------
void QwtThermo::scaleChange()
{
    repaint(FALSE);
}

//------------------------------------------------------------
//.-
//.F QwtThermo::drawThermo
//	Redraw the thermometer
//
//.u	Parameters
//.p	QPainter *p
//
//------------------------------------------------------------
void QwtThermo::drawThermo(QPainter *p)
{
    int tval;
    int alarm  = 0, taval = 0;

    QRect fRect;
    QRect aRect;
    QRect bRect;
    
    int inverted = ( d_maxValue < d_minValue );

    //
    //	Determine if value exceeds alarm threshold.
    //  Note: The alarm value is allowed to lie
    //        outside the interval (minValue, maxValue).
    //
    if (d_alarmEnabled)
    {
	if (inverted)
	{
	    alarm = ((d_alarmLevel >= d_maxValue)
		     && (d_alarmLevel <= d_minValue)
		     && (d_value >= d_alarmLevel));
	    
	}
	else
	{
	    alarm = (( d_alarmLevel >= d_minValue)
		     && (d_alarmLevel <= d_maxValue)
		     && (d_value >= d_alarmLevel));
	}
    }
    
    //
    //  transform values
    //
    tval = d_map.limTransform(d_value);
    
    if (alarm)
       taval = d_map.limTransform(d_alarmLevel);
    
    //
    //  calculate recangles
    //
    switch(d_orient)
    {
    case Horizontal:
	
	if (inverted)
	{
	    bRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
			  tval - d_thermoRect.x(),
			  d_thermoRect.height());
	    
	    if (alarm)
	    {
		aRect.setRect(tval, d_thermoRect.y(),
			      taval - tval + 1,
			      d_thermoRect.height());
		fRect.setRect(taval + 1, d_thermoRect.y(),
			      d_thermoRect.x() + d_thermoRect.width() - (taval + 1),
			      d_thermoRect.height());
	    }
	    else
	    {
		fRect.setRect(tval, d_thermoRect.y(),
			      d_thermoRect.x() + d_thermoRect.width() - tval,
			      d_thermoRect.height());
	    }
	}
	else
	{
	    bRect.setRect(tval + 1, d_thermoRect.y(),
			  d_thermoRect.width() - (tval + 1 - d_thermoRect.x()),
			  d_thermoRect.height());
	    
	    if (alarm)
	    {
		aRect.setRect(taval, d_thermoRect.y(),
			      tval - taval + 1,
			      d_thermoRect.height());
		fRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
			      taval - d_thermoRect.x(),
			      d_thermoRect.height());
	    }
	    else
	    {
		fRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
			      tval - d_thermoRect.x() + 1,
			      d_thermoRect.height());
	    }
	    
	}
	break;
	
    case Vertical:
    default:
	
	if (tval < d_thermoRect.y())
	   tval = d_thermoRect.y();
	else if (tval > d_thermoRect.y() + d_thermoRect.height())
	   tval = d_thermoRect.y() + d_thermoRect.height();
	
	if (inverted)
	{
	    bRect.setRect(d_thermoRect.x(), tval + 1,
			  d_thermoRect.width(),
			  d_thermoRect.height() - (tval + 1 - d_thermoRect.y()));
	    
	    if (alarm)
	    {
		aRect.setRect(d_thermoRect.x(), taval,
			      d_thermoRect.width(),
			      tval - taval + 1);
		fRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
			      d_thermoRect.width(),
			      taval - d_thermoRect.y());
	    }
	    else
	    {
		fRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
			      d_thermoRect.width(),
			      tval - d_thermoRect.y() + 1);
	    }
	    
	}
	else
	{
	    bRect.setRect(d_thermoRect.x(), d_thermoRect.y(),
			  d_thermoRect.width(),
			  tval - d_thermoRect.y());
	    if (alarm)
	    {
		aRect.setRect(d_thermoRect.x(),tval,
			      d_thermoRect.width(),
			      taval - tval + 1);
		fRect.setRect(d_thermoRect.x(),taval + 1, 
			      d_thermoRect.width(),
			      d_thermoRect.y() + d_thermoRect.height() - (taval + 1));
	    }
	    else
	    {
		fRect.setRect(d_thermoRect.x(),tval, 
			      d_thermoRect.width(),
			      d_thermoRect.y() + d_thermoRect.height() - tval);
	    }
	    
	}
	break;
	
	p->fillRect(d_thermoRect.x(),
		    d_thermoRect.y(),
		    d_thermoRect.width(),
		    tval + 1 - d_thermoRect.y(),
		    backgroundColor());
	
	p->fillRect(d_thermoRect.x(),
		    tval,
		    d_thermoRect.width(),
		    d_thermoRect.height() - (tval - d_thermoRect.y()),
		    d_fillColor);
	break;
	
    }
    
    //
    // paint thermometer
    //
    p->fillRect(bRect, backgroundColor());
    
    if (alarm)
       p->fillRect(aRect, d_alarmColor);
    
    p->fillRect(fRect, d_fillColor);
    
    
}

//------------------------------------------------------------
//
//.F	QwtThermo::setBorderWidth
//	Set the border width of the pipe.
//
//.u	Syntax
//	void QwtThermo::setBorderWidth(int w)
//
//.u	Parameters
//.p	int w	--	border width
//
//------------------------------------------------------------
void QwtThermo::setBorderWidth(int w)
{
    if ((w >= 0) && (w < (qwtMin(d_thermoRect.width(),
				 d_thermoRect.height())
			  + d_borderWidth) / 2  - 1))
       d_borderWidth = w;
    resize(size());
}

//------------------------------------------------------------
//
//.F	QwtThermo::setRange
//	Set the range
//  
//.u	Syntax
//.f	void QwtThermo::setRange(double vmin, double vmax)
//
//.u	Parameters
//.p
//	double vmin	-- Value corresponding lower or left end of the thermometer
//	double vmax     -- Value corresponding to the upper or right end of the
//
//------------------------------------------------------------
void QwtThermo::setRange(double vmin, double vmax)
{
    QwtScaleDiv oldscl(d_scale.scaleDiv());
    d_minValue = vmin;
    d_maxValue = vmax;
    
    d_map.setDblRange(d_minValue, d_maxValue);
    if (!hasUserScale())
    {
	d_scale.setScale(d_minValue, d_maxValue, d_maxMajor, d_maxMinor);
	if (oldscl != d_scale.scaleDiv())
	   scaleChange();
    }
}

//------------------------------------------------------------
//
//.F	QwtThermo::setFillColor
//	Change the color of the liquid. 
//
//.u	Syntax
//.f	void QwtThermo::setFillColor(const QColor &c)
//
//.u	Parameters
//.p	const QColor &c -- New color. The default color is black.
//
//.u	Note
//	The default color is black.
//
//------------------------------------------------------------
void QwtThermo::setFillColor(const QColor &c)
{
    d_fillColor = c;
    repaint(d_thermoRect,FALSE);
}

//------------------------------------------------------------
//
//.F	QwtThermo::setAlarmColor
//	Specify liquid color above the alarm threshold
//  
//.u	Syntax
//.f	void QwtThermo::setAlarmColor(const QColor &c)
//
//.u	Parameters
//.p	const QColor &c	--	New color. The default is white.
//
//.u	Note
//	The default color is white.
//
//------------------------------------------------------------
void QwtThermo::setAlarmColor(const QColor &c)
{
    d_alarmColor = c;
    repaint(d_thermoRect,FALSE);
}

//------------------------------------------------------------
//
//.F	QThermo::setAlarmLevel
//	Specify the alarm threshold
//  
//.u	Syntax
//.f	void QwtThermo::setAlarmLevel(double v)
//
//.u	Parameters
//.p	double v   --	threshold value
//
//------------------------------------------------------------
void QwtThermo::setAlarmLevel(double v)
{
    d_alarmLevel = v;
    d_alarmEnabled = 1;
    repaint(d_thermoRect,FALSE);
}

//------------------------------------------------------------
//
//.F	QwtThermo::setPipeWidth
//	Change the width of the pipe
//
//.u	Syntax
//.f	void QwtThermo::setPipeWidth(int w)
//
//.u	Parameters
//.p	int w -- width
//
//------------------------------------------------------------
void QwtThermo::setPipeWidth(int w)
{
    if(w > 0)
       d_thermoWidth = w;
    resize(size());
}


//------------------------------------------------------------
//
//.F	QwtThermo::setMargin
//	Specify the distance between the pipe's endpoints
//	and the widget's border
//
//.u	Syntax
//.f	void QwtThermo::setMargin(int m)
//
//.u	Parameters
//.p	int m	-- New Margin. The default values are 10 for
//		   horizontal orientation and 20 for vertical
//		   orientation.
//
//.u	Description
//	The margin is used to leave some space for the scale
//	labels. If a large font is used, it is advisable to
//	adjust the margins.
//	
//.u	Note
//	The margin has no effect if the scale is disabled.
//
//------------------------------------------------------------
void QwtThermo::setMargin(int m)
{
    if (m >= 0)
    {
	d_horPad = m;
	d_vertPad = m;
	resize(size());
    }
}


//------------------------------------------------------------
//
//.F QwtThermo::setAlarmEnabled
//	Enable or disable alarm threshold
//
//.u	Syntax
//.f	void QwtThermo::setAlarmEnabled(int tf)
//
//.u	Parameters
//.p	int tf	--	TRUE (enabled) or FALSE(disabled)
//
//------------------------------------------------------------
void QwtThermo::setAlarmEnabled(int tf)
{
    d_alarmEnabled = tf;
    repaint(d_thermoRect,FALSE);
}















