#include "qwt_symbol.h"
#include <qrect.h>
#include <qpainter.h>

QwtSymbol NoSymbol();

//------------------------------------------------------------
//.H QwtSymbol | 3 | 27/04/96 | Qwt Widget Library | Qwt Programmer's Manual
//
//.U NAME
//	QwtSymbol - A class for drawing symbols
//
//.U SYNOPSIS
//	#include <qwt_symbol.h>
//
//.U PUBLIC MEMBERS 
//.R
//	QwtSymbol::QwtSymbol (1) -- Constructor
//	QwtSymbol::QwtSymbol (2) -- Constructor
//	QwtSymbol::~QwtSymbol (3) -- Destructor
//	QwtSymbol::draw (1) -- Draw the symbol
//	QwtSymbol::draw (2) -- Draw the symbol
//	QwtSymbol::draw (3) -- Draw the symbol
//	QwtSymbol::setSize (1) --  Change the size 
//	QwtSymbol::setSize (2) -- Change the size
//	QwtSymbol::setBrush -- Assign a brush
//	QwtSymbol::setPen -- Assign a pen
//	QwtSymbol::setStyle -- Change the style
//	QwtSymbol::brush -- Return the brush
//	QwtSymbol::pen -- Return the pen
//	QwtSymbol::size -- Return the size
//	QwtSymbol::style -- Return the style
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtSymbol::QwtSymbol (1)
//	Default Constructor
//
//.u	Syntax
//.f	 QwtSymbol::QwtSymbol()
//
//.u	Description
//	The symbol is constructed with gray interior,
//	black outline with zero width, no size and style 'None'.
//
//------------------------------------------------------------
QwtSymbol::QwtSymbol()
: d_brush(gray), d_pen(black), d_size(0,0)
{
    d_style = QwtSymbol::None;
}



//------------------------------------------------------------
//
//.F	QwtSymbol::QwtSymbol (2)
//	Constructor
//
//.u	Syntax
//.f	 QwtSymbol::QwtSymbol(QwtSymbol::Style st, const QBrush &bd, const QPen &pn, const QSize &s)
//
//.u	Parameters
//.p	QwtSymbol::Style st -- Symbol Style
//	QBrush &bd -- brush to fill the interior
//	QPen &pn -- outline pen 
//	QSize &size --size 
//
//------------------------------------------------------------
QwtSymbol::QwtSymbol(QwtSymbol::Style st, const QBrush &br, const QPen &pn, const QSize &s)
: d_brush(br), d_pen(pn), d_size(s)
{
    d_style = st;
}

//------------------------------------------------------------
//
//.F	QwtSymbol::~QwtSymbol
//	Destructor
//
//.u	Syntax
//.f	 QwtSymbol::~QwtSymbol()
//
//------------------------------------------------------------
QwtSymbol::~QwtSymbol()
{
    
}


//------------------------------------------------------------
//
//.F	QwtSymbol::setSize (1)
//	Specify the symbol's size
//
//.u	Syntax
//.f	void QwtSymbol::setSize(int w, int h)
//
//.u	Parameters
//.p	int w -- width
//	int h = -1 -- height
//
//.u	Description
//	If the 'h' parameter is left out or less than 0,
//	and the 'w' parameter is greater than or equal to 0,
//	the symbol size will be set to (w,w).
//
//------------------------------------------------------------
void QwtSymbol::setSize(int w, int h)
{
    if ((w >= 0) && (h < 0)) h = w;
    d_size = QSize(w,h);
}

//------------------------------------------------------------
//
//.F	QwtSymbol::setSize (2)
//	Set the symbol's size
//
//.u	Syntax
//.f	void QwtSymbol::setSize(const QSize &s)
//
//.u	Parameters
//.p	const QSize &s	-- new size
//
//------------------------------------------------------------
void QwtSymbol::setSize(const QSize &s)
{
    if (s.isValid()) d_size = s;
}

//------------------------------------------------------------
//
//.F	QwtSymbol::setBrush
//	Assign a brush
//
//.u	Syntax
//.f	void QwtSymbol::setBrush(const QBrush &br)
//
//.u	Parameters
//.p	const QBrush &br	--	brush
//
//.u	Description
//	The brush is used to draw the interior of the symbol.
//
//------------------------------------------------------------
void QwtSymbol::setBrush(const QBrush &br)
{
    d_brush = br;
}

//------------------------------------------------------------
//
//.F	QwtSymbol::setPen
//	Assign a pen
//
//.u	Syntax
//.f	void QwtSymbol::setPen(const QPen &pn)
//
//.u	Parameters
//.p	const QPen &pn -- pen
//
//.u	Description
//	The pen is used to draw the symbol's outline.
//
//------------------------------------------------------------
void QwtSymbol::setPen(const QPen &pn)
{
    d_pen = pn;
}

//------------------------------------------------------------
//
//.F	QwtSymbol::draw (1)
//	Draw the symbol at a point (x,y).
//
//.u	Syntax
//.f	void QwtSymbol::draw(QPainter *p, int x, int y)
//
//.u	Parameters
//.p	QPainter *p, int x, int y
//
//.u	Description
//	The point (x,y) is the upper left corner of a rectangle with
//	the symbol's size.
//
//------------------------------------------------------------
void QwtSymbol::draw(QPainter *p, int x, int y)
{
    QRect r (x,y,d_size.width(), d_size.height());
    draw(p, r);
}


//------------------------------------------------------------
//
//.F	QwtSymbol::draw (2)
//	Draw the symbol into a bounding rectangle.
//
//.u	Syntax
//.f	void QwtSymbol::draw(QPainter *p, const QRect& r)
//
//.u	Parameters
//.p	QPainter *p	-- Painter
//	const QRect& r  -- Bounding rectangle
//
//.u	Description
//	This function overrides the symbol's size settings,
//	but it doesn't modify them.
//
//------------------------------------------------------------
void QwtSymbol::draw(QPainter *p, const QRect& r)
{

    int w2 = r.width() / 2;
    int h2 = r.height() / 2;

    p->setBrush(d_brush);
    p->setPen(d_pen);
    
    switch(d_style)
    {
    case QwtSymbol::Ellipse:
	p->drawEllipse(r);
	break;
    case QwtSymbol::Rect:
  	p->drawRect(r);
    break;
    case QwtSymbol::Diamond:
	d_pa.resize(4);
	d_pa.setPoint(0, r.x() + w2, r.y());
	d_pa.setPoint(1, r.right(), r.y() + h2);
	d_pa.setPoint(2, r.x() + w2, r.bottom());
	d_pa.setPoint(3, r.x(), r.y() + h2);
	p->drawPolygon(d_pa);
	break;
    case QwtSymbol::Cross:
	p->drawLine(r.x() + w2, r.y(), r.x() + w2, r.bottom());
	p->drawLine(r.x(), r.y() + h2, r.right(), r.y() + h2);
	break;
    case QwtSymbol::XCross:
	p->drawLine(r.x(), r.y(), r.right(), r.bottom());
	p->drawLine(r.x(), r.bottom(), r.right(), r.top());
	break;
    case QwtSymbol::Triangle:
    case QwtSymbol::UTriangle:
	d_pa.resize(3);
	d_pa.setPoint(0, r.x() + w2, r.y());
	d_pa.setPoint(1, r.right(), r.bottom());
	d_pa.setPoint(2, r.x(), r.bottom());
	p->drawPolygon(d_pa);
	break;
    case QwtSymbol::DTriangle:
	d_pa.resize(3);
	d_pa.setPoint(0, r.x(), r.y());
	d_pa.setPoint(1, r.right(), r.y());
	d_pa.setPoint(2, r.x() +  w2, r.bottom());
	p->drawPolygon(d_pa);
	break;
    case QwtSymbol::LTriangle:
	d_pa.resize(3);
	d_pa.setPoint(0, r.x(), r.y());
	d_pa.setPoint(1, r.right(), r.y() + h2);
	d_pa.setPoint(2, r.x(), r.bottom());
	p->drawPolygon(d_pa);
	break;
    case QwtSymbol::RTriangle:
	d_pa.resize(3);
	d_pa.setPoint(0, r.right(), r.y());
	d_pa.setPoint(1, r.x(), r.y() + h2);
	d_pa.setPoint(2, r.right(), r.bottom());
	p->drawPolygon(d_pa);
	break;
    default:
	break;
    }

}

//------------------------------------------------------------
//
//.F	QwtSymbol::draw (3)
//	Draw the symbol at a specified point
//
//.u	Syntax
//.f	void QwtSymbol::draw(QPainter *p, const QPoint &p)
//
//.u	Parameters
//.p	QPainter *p	-- painter
//	const QPoint &pt -- point
//
//.u	Description
//	The point specifies the upper left corner of a
//	rectangle with the symbol's size.
//
//------------------------------------------------------------
void QwtSymbol::draw(QPainter *p, const QPoint &pt)
{
    draw(p, pt.x(), pt.y());
}

//------------------------------------------------------------
//
//.F	QwtSymbol::setStyle
//	Specify the symbol style
//
//.u	Syntax
//.f	QwtSymbol::Style QwtSymbol::setStyle(QwtSymbol::Style s)
//
//.u	Parameters
//.p	QwtSymbol::Style s -- style
//
//.u	Description
//	The following styles are defined:
//.t	QwtSymbol::None	--	No Style. The symbol cannot be drawn.
//      QwtSymbol::Ellipse --	Ellipse or circle
//      QwtSymbol::Rect --	Rectangle
//      QwtSymbol::Diamond --	Diamond
//      QwtSymbol::Triangle --	Triangle pointing upwards
//      QwtSymbol::DTriangle -- Triangle pointing downwards
//      QwtSymbol::UTriangle -- Triangle pointing upwards
//      QwtSymbol::LTriangle -- Triangle pointing left
//      QwtSymbol::RTriangle -- Triangle pointing right
//      QwtSymbol::Cross --	Cross
//      QwtSymbol::XCross --	Diagonal cross
//
//------------------------------------------------------------
void QwtSymbol::setStyle(QwtSymbol::Style s)
{
    d_style = s;
}

























