#include <qwt_spline.h>
#include <qwt_math.h>
#include <qwt.h>

//-----------------------------------------------------------
//.H QwtSpline|3|02/04/98|Qwt Widget Library|Qwt Programmer's Reference
//
//.U NAME
//	QwtSpline - A class for spline interpolation
//
//.U SYNOPSIS
//	#include<qwt_spline.h>
//
//.U PUBLIC MEMBERS 
//.R
//	QwtSpline::QwtSpline --	Constructor
//	QwtSpline::~QwtSpline -- Destructor
//	QwtSpline::value -- Return interpolated value
//	QwtSpline::recalc -- Build the spline
//	QwtSpline::copyValues -- Set policy: copy  or attach arrays 
//
//.U DESCRIPTION
//
//	The QwtSpline class is used for cubical spline interpolation.
//	Two types of splines, natural and periodic, are supported.
//
//.P	Usage:
//.e	1.	First call
//		@QwtSpline::recalc@ to
//		determine the spline coefficients for a tabulated function
//		y(x).
//	2.	After the coefficients have been set up, the interpolated
//		function value for an argument x can be determined by calling 
//		@QwtSpline::value@.
//.P
//	In order to save storage space, QwtSpline can be advised
//	not to buffer the contents of x and y.
//	This means that the arrays have to remain valid and unchanged
//	for the interpolation to work properly. This can be achieved
//	by calling @QwtSpline::copyValues@.
//
//.U EXAMPLE
//
//.c
//
//      #include<qwt_spline.h>
//      #include<iostream.h>
//
//      QwtSpline s;
//      double x[30], y[30], xInter[300], yInter[300];
//      int i;
//
//      for(i=0;i<30;i++)               // fill up x[] and y[]
//          cin >> x[i] >> y[i];
//
//      if (s.recalc(x,y,30,0) == 0)    // build natural spline
//      {
//          for(i=0;i<300;i++)          // interpolate
//          {
//              xInter[i] = x[0] + double(i) * (x[29] - x[0]) / 299.0;
//              yInter[i] = s.value( xInter[i] );
//          }
//
//          do_something(xInter, yInter, 300);
//      }
//      else
//          cerr << "Uhhh...\n";
//.-
//.U AUTHOR
//	Josef Wilgen (jwlg@mail.desy.de)
//------------------------------------------------------------

//------------------------------------------------------------
//.C MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtSpline::value
//		Calculate the interpolated function value corresponding 
//		to a given argument x.
//
//.u	Syntax
//.f	double QwtSpline::value(double x) const
//
//------------------------------------------------------------
double QwtSpline::value(double x) const
{
    int i;
    double delta;
    
    if (!d_a)
    {
	return 0.0;
    }
    else
    {
	i = lookup(x);

	delta = x - d_x[i];
	return( ( ( ( d_a[i] * delta) + d_b[i] ) * delta + d_c[i] ) * delta + d_y[i] );
    }
    
}

//------------------------------------------------------------
//
//.F	QwtSpline::QwtSpline
//		Construct a QwtSpline instance
//
//.u	Syntax
//.f	 QwtSpline::QwtSpline()
//
//------------------------------------------------------------
QwtSpline::QwtSpline()
{
    d_a = d_b = d_c = 0;
    d_xbuffer = d_ybuffer = d_x = d_y = 0;
    d_size = 0;
    d_buffered = 0;
}

//------------------------------------------------------------
//
//.F	QwtSpline::copyValues
//		Advise	recalc() to buffer the tabulated function
//		or switch off internal buffering
//
//.u	Syntax
//.f	void QwtSpline::copyValues()
//
//.u	Parameters
//.p		int tf	--		if nonzero, the function table will be buffered
//
//.u	Description
//	By default, QwtSpline maintains an internal copy of the
//      tabulated function given as *x and *y arguments
//      of recalc(). 
//      If copyValues() is called with zero argument,
//      subsequent calls to recalc() will not buffer these values
//      anymore and just store the pointers instead. The value()
//      function will then silently assume that these pointers remained valid
//      and that the contents of the arrays have not been changed since
//      the last recalc().
//
//      If called with no or nonzero argument,
//      any following recalc() calls will use the internal buffer. 
//
//.u	Note
//      copyValues() resets all the contents of QwtSpline.
//      A subsequent recalc() will be necessary.
//
//------------------------------------------------------------
void QwtSpline::copyValues(int tf)
{
    cleanup();
    d_buffered = tf;
}

//------------------------------------------------------------
//
//.F	QwtSpline::~QwtSpline
//		Destroy a QwtSpline instance
//
//.u	Syntax
//.f	 QwtSpline::~QwtSpline()
//
//------------------------------------------------------------
QwtSpline::~QwtSpline()
{
    cleanup();
}

//------------------------------------------------------------
//.-
//.F	QwtSpline::lookup
//      Determine the function table index 
//		corresponding to a value x
//
//.u	Syntax
//.f	int QwtSpline::lookup(double x) const
//
//------------------------------------------------------------
int QwtSpline::lookup(double x) const
{

    int i1, i2, i3;
    
    if (x <= d_x[0])
       i1 = 0;
    else if (x >= d_x[d_size - 2])
       i1 = d_size -2;
    else
    {
	i1 = 0;
	i2 = d_size -2;
	i3 = 0;

	while ( i2 - i1 > 1 )
	{
	    i3 = i1 + ((i2 - i1) >> 1);

	    if (d_x[i3] > x)
	       i2 = i3;
	    else
	       i1 = i3;

	}
    }
    return i1;
}


//------------------------------------------------------------
//
//.F	QwtSpline::recalc
//		re-calculate the spline coefficients
//
//.u	Parameters
//.p	double *x, double *y	--		points
//     int n					--		number of points
//     int periodic			--		if non-zero, calculate periodic spline
//
//.u	Return Type
//		int
//
//.u   Return Value
//.t      0 -- successful
//  	   Qwt::ErrMono -- Sequence of x values is not strictly monotone increasing
//  	   Qwt::ErrNoMem -- Couldn't allocate memory
//
//
//.u	Description
//
//		Depending on the value of <periodic>, this function
//      will determine the coefficients for a natural or a periodic
//      spline and store them internally. By default, it also buffers the
//      values of x and y, which are needed for the
//      interpolation (See value()). In order to save memory,
//		this last behaviour may be changed with the copyValues() function.
//
//.u	Note
//		The sequence of x (but not y) values has to be strictly monotone
//		increasing, which means
//.n         x[0] < x[1] < .... < x[n-1].
//.n    If this is not the case, the function will return an error.
//
//.u	Syntax
//.f	void QwtSpline::recalc(double *x, double *y, int n)
//
//------------------------------------------------------------
int QwtSpline::recalc(double *x, double *y, int n, int periodic)
{

    int i, rv = 0;

    cleanup();

    if (n > 2)
    {

	d_size = n;

	if (d_buffered)
	{
	    d_xbuffer = new double[n-1];
	    d_ybuffer = new double[n-1];

	    if ((!d_xbuffer) || (!d_ybuffer))
	    {
		cleanup();
		return Qwt::ErrNoMem;
	    }
	    else
	    {
		for (i=0;i<n;i++)
		{
		    d_xbuffer[i] = x[i];
		    d_ybuffer[i] = y[i];
		}
		d_x = d_xbuffer;
		d_y = d_ybuffer;
	    }
	}
	else
	{
	    d_x = x;
	    d_y = y;
	}
	
	
	d_a = new double[n-1];
	d_b = new double[n-1];
	d_c = new double[n-1];

	if ( (!d_a) || (!d_b) || (!d_c) )
	{
	    cleanup();
	    return Qwt::ErrMono;
	}

	if(periodic)
	   rv =  buildPerSpline();
	else
	   rv =  buildNatSpline();

	if (rv) cleanup();
	
    }

    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtSpline::buildNatSpline
//		Determines the coefficients for a natural spline
//
//.u	Return Type
//		void
//
//.u   Return Value
//.t      0 -- successful
//  	   Qwt::ErrMono -- Sequence of x values is not strictly monotone increasing
//  	   Qwt::ErrNoMem -- Couldn't allocate memory
//
//.u	Description
//
//.u	Syntax
//.f	void QwtSpline::buildNatSpline()
//
//------------------------------------------------------------
int QwtSpline::buildNatSpline()
{
    int i;
    double dy1, dy2;
    
    double *d = new double[d_size-1];
    double *h = new double[d_size-1];
    double *s = new double[d_size];

    if ( (!d) || (!h) || (!s) )
    {
	cleanup();
	if (h) delete[] h;
	if (s) delete[] s;
	if (d) delete[] d;
	return Qwt::ErrNoMem;
    }

    //
    //  set up tridiagonal equation system; use coefficient
    //  vectors as temporary buffers
    for (i=0; i<d_size - 1; i++) 
    {
	h[i] = d_x[i+1] - d_x[i];
	if (h[i] <= 0)
	{
	    delete[] h;
	    delete[] s;
	    delete[] d;
	    return Qwt::ErrMono;
	}
    }
    

    dy1 = (d_y[1] - d_y[0]) / h[0];
    for (i=1; i < d_size - 1; i++)
    {

	d_b[i] = d_c[i] = h[i];
	d_a[i] = 2.0 * (h[i-1] + h[i]);

	dy2 = (d_y[i+1] - d_y[i]) / h[i];
	d[i] = 6.0 * ( dy1 - dy2);
	dy1 = dy2;

    }

    //
    // solve it
    //
    
    // L-U Factorization
    for(i=1;i<d_size -2;i++)
    {
	d_c[i] /= d_a[i];
	d_a[i+1] -= d_b[i] * d_c[i]; 
    }

    // forward elimination
    s[1] = d[1];
    for(i=2;i<d_size - 1;i++)
       s[i] = d[i] - d_c[i-1] * s[i-1];
    
    // backward elimination
    s[d_size - 2] = - s[d_size - 2] / d_a[d_size - 2];
    for (i= d_size -3; i > 0; i--)
       s[i] = - (s[i] + d_b[i] * s[i+1]) / d_a[i];

    //
    // Finally, determine the spline coefficients
    //
    s[d_size - 1] = s[0] = 0.0;
    for (i=0;i<d_size-1;i++)
    {
	d_a[i] = ( s[i+1] - s[i] ) / ( 6.0 * h[i]);
	d_b[i] = 0.5 * s[i];
	d_c[i] = ( d_y[i+1] - d_y[i] ) / h[i] - (s[i+1] + 2.0 * s[i] ) * h[i] / 6.0; 
    }

    delete[] d;
    delete[] s;
    delete[] h;

    return 0;
    
}

//------------------------------------------------------------
//
//.F	QwtSpline::buildPerSpline
//		Determines the coefficients for a periodic spline
//
//.u	Return Type
//		int
//
//.u	Return Values
//.t      0 -- successful
//  	   Qwt::ErrMono -- Sequence of x values is not strictly monotone increasing
//  	   Qwt::ErrNoMem -- Couldn't allocate memory
//
//.u	Description
//
//.u	Syntax
//.f	void QwtSpline::buildPerSpline()
//
//------------------------------------------------------------
int QwtSpline::buildPerSpline()
{

    int i,imax;
    double sum;
    double dy1, dy2,htmp;
    
    double *d = new double[d_size-1];
    double *h = new double[d_size-1];
    double *s = new double[d_size];

    
    if ( (!d) || (!h) || (!s) )
    {
	cleanup();
	if (h) delete[] h;
	if (s) delete[] s;
	if (d) delete[] d;
	return Qwt::ErrNoMem;
    }

    //
    //  setup equation system; use coefficient
    //  vectors as temporary buffers
    //
    for (i=0; i<d_size - 1; i++)
    {
	h[i] = d_x[i+1] - d_x[i];
	if (h[i] <= 0.0)
	{
	    delete[] h;
	    delete[] s;
	    delete[] d;
	    return Qwt::ErrMono;
	}
    }
    
    imax = d_size - 2;
    htmp = h[imax];
    dy1 = (d_y[0] - d_y[imax]) / htmp;
    for (i=0; i <= imax; i++)
    {
	d_b[i] = d_c[i] = h[i];
	d_a[i] = 2.0 * (htmp + h[i]);
	dy2 = (d_y[i+1] - d_y[i]) / h[i];
	d[i] = 6.0 * ( dy1 - dy2);
	dy1 = dy2;
	htmp = h[i];
    }

    //
    // solve it
    //
    
    // L-U Factorization
    d_a[0] = sqrt(d_a[0]);
    d_c[0] = h[imax] / d_a[0];
    sum = 0;

    for(i=0;i<imax-1;i++)
    {
	d_b[i] /= d_a[i];
	if (i > 0)
	   d_c[i] = - d_c[i-1] * d_b[i-1] / d_a[i];
	d_a[i+1] = sqrt( d_a[i+1] - qwtSqr(d_b[i]));
	sum += qwtSqr(d_c[i]);
    }
    d_b[imax-1] = (d_b[imax-1] - d_c[imax-2] * d_b[imax-2]) / d_a[imax-1];
    d_a[imax] = sqrt(d_a[imax] - qwtSqr(d_b[imax-1]) - sum);
    

    // forward elimination
    s[0] = d[0] / d_a[0];
    sum = 0;
    for(i=1;i<imax;i++)
    {
	s[i] = (d[i] - d_b[i-1] * s[i-1]) / d_a[i];
	sum += d_c[i-1] * s[i-1];
    }
    s[imax] = (d[imax] - d_b[imax-1]*s[imax-1] - sum) / d_a[imax];
    
    
    // backward elimination
    s[imax] = - s[imax] / d_a[imax];
    s[imax-1] = -(s[imax-1] + d_b[imax-1] * s[imax]) / d_a[imax-1];
    for (i= imax - 2; i >= 0; i--)
       s[i] = - (s[i] + d_b[i] * s[i+1] + d_c[i] * s[imax]) / d_a[i];

    //
    // Finally, determine the spline coefficients
    //
    s[d_size-1] = s[0];
    for (i=0;i<d_size-1;i++)
    {
	d_a[i] = ( s[i+1] - s[i] ) / ( 6.0 * h[i]);
	d_b[i] = 0.5 * s[i];
	d_c[i] = ( d_y[i+1] - d_y[i] ) / h[i] - (s[i+1] + 2.0 * s[i] ) * h[i] / 6.0; 
    }

    delete[] d;
    delete[] s;
    delete[] h;

    return 0;
    
}


//------------------------------------------------------------
//.-
//.F	QwtSpline::cleanup
//		Free allocated memory and set size to 0
//
//.u	Syntax
//.f	void QwtSpline::cleanup()
//
//------------------------------------------------------------
void QwtSpline::cleanup()
{
    if (d_a) delete[] d_a;
    if (d_b) delete[] d_b;
    if (d_c) delete[] d_c;
    if (d_xbuffer) delete[] d_xbuffer;
    if (d_ybuffer) delete[] d_ybuffer;
    d_a = d_b = d_c = 0;
    d_xbuffer = d_ybuffer = d_x = d_y = 0;
    d_size = 0;
}







