#include <math.h>
#include <qdrawutl.h>
#include "qwt_sldbase.h"
#include <qwt_math.h>

//---------------------------------------------------------
//.H QwtSliderBase|3|02/04/98|Qwt Widget Library|Qwt Programmer's manual
//
//.U NAME
//	QwtSliderBase - A Base class for sliders
//
//.U	SYNOPSIS
//	#include <qwt_sldbase.h>
//
//.U INHERITED CLASSES
//	QWidget, @^QwtDblRange@
//
//.U PUBLIC MEMBERS
//.R
//    QwtSliderBase::QwtSliderBase -- constructor
//    QwtSliderBase::QwtSliderBase -- destructor
//    QwtSliderBase::stopMoving -- stop moving 
//    QwtSliderBase::setTracking  -- enable or disable tracking
//
//.U SIGNALS
//.R
//   QwtSliderbase::valueChanged -- Emitted when the value has been changed
//   QwtSliderbase::sliderPressed -- Emitted when the button has been pressed
//   QwtSliderbase::sliderReleased() - Emitted when the button has been released
//   QwtSliderbase::sliderMoved -- Emitted when the button has been moved
//
//.U	PUBLIC SLOTS
//.R
//	QwtSliderBase::setValue -- move to value
//	QwtSliderBase::fitValue -- move to nearest integer multiple of the step size
//
//.U PROTECTED MEMBERS
//.R
//	QwtSliderBase::setUpdateTime --
//			Change the update time for automatic scrolling
//	QwtSliderBase::setMass --
//		set the mass for flywheel effect
//	QwtSliderBase::setPosition -- adjust the position
//	QwtSliderBase::valueChange --  notify change of value
//	QwtSliderBase::mass -- return the mass
//
//	QwtSliderBase::getValue  --
//		Determine the value corresponding to a point p
//	QwtSliderBase::getScrollMode  --
//			     Determine the scrolling mode corresponding to a point p
//.U	DESCRIPTION
//	QwtSliderBase is a base class for
//	slider widgets. QwtSliderBase handles the mouse events
//	and updates the slider's value accordingly. Derived classes
//	only have to implement the @QwtSliderBase::getValue@ and 
//	@QwtSliderBase::getScrollMode@ members, and should react to a
//	@QwtSliderbase::valueChange@, which normally requires repainting. 
//-----------------------------------------------------------

//-----------------------------------------------------------
//.C MEMBER FUNCTION DESCRIPTION
//-----------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtSliderBase::QwtSliderBase
//	Constructor
//
//.u	Syntax
//.f	 QwtSliderBase::QwtSliderBase(QWidget *parent, const char *name)
//
//.u	Parameters
//.p	QWidget *parent, const char *name
//
//------------------------------------------------------------
QwtSliderBase::QwtSliderBase(QWidget *parent, const char *name)
: QWidget(parent,name)
{
    d_tmrID = 0;
    d_updTime = 150;
    d_mass = 0.0;
    d_tracking = TRUE;
    d_mouseOffset = 0.0;
    setRange(0.0,100.0, 1.0);
}

//------------------------------------------------------------
//
//.F	QwtSliderBase::~QwtSliderBase
//	Destructor
//
//.u	Syntax
//.f	 QwtSliderBase::~QwtSliderBase()
//
//------------------------------------------------------------
QwtSliderBase::~QwtSliderBase()
{
    if(d_tmrID) killTimer(d_tmrID);
}

//------------------------------------------------------------
//
//.F	QwtSliderBase::stopMoving
//	Stop updating if automatic scrolling is active
//
//.u	Syntax
//.f	void QwtSliderBase::stopMoving()
//
//------------------------------------------------------------
void QwtSliderBase::stopMoving() 
{
    if(d_tmrID)
    {
	killTimer(d_tmrID);
	d_tmrID = 0;
    }
}


//------------------------------------------------------------
//
//.F	QwtSliderBase::setUpdateTime
//	Specify the update interval for automatic scrolling
//
//.u	Syntax
//.f	void QwtSliderBase::setUpdateTime(int t)
//
//.u	Parameters
//.p	int t	-- update interval in milliseconds
//
//.u	See also
//	@QwtSliderBase::getScrollMode@
//
//------------------------------------------------------------
void QwtSliderBase::setUpdateTime(int t) 
{
    if (t < 50) t = 50;
    d_updTime = t;
}


//------------------------------------------------------------
//.-
//.F	QwtSliderBase::mousePressEvent
//	Mouse press event handler
//
//.u	Syntax
//.f	void QwtSliderBase::mousePressEvent(QMouseEvent *e)
//
//.u	Parameters
//.p	QMouseEvent *e -- Qt Mouse press event
//
//------------------------------------------------------------
void QwtSliderBase::mousePressEvent(QMouseEvent *e) 
{

    const QPoint &p = e->pos();

    d_timerTick = 0;

    getScrollMode(p, d_scrollMode, d_direction);
    stopMoving();
    
    switch(d_scrollMode)
    {
    case ScrPage:
    case ScrTimer:
	d_mouseOffset = 0;
	d_tmrID = startTimer(qwtMax(250, 2 * d_updTime));
	break;
	
    case ScrMouse:
	d_time.start();
	d_speed = 0;
	d_mouseOffset = getValue(p) - value();
	emit sliderPressed();
	break;
	
    default:
	d_mouseOffset = 0;
	d_direction = 0;
	break;
	
    }
    
}


//------------------------------------------------------------
//.-
//.F	QwtSliderBase::buttonRelease
//	Emit a valueChanged() signal if necessary
//
//.u	Syntax
//.f	void QwtSliderBase::buttonReleased()
//
//------------------------------------------------------------
void QwtSliderBase::buttonReleased()
{
    if ((!d_tracking) || (value() != prevValue()))
       emit valueChanged(value());
}


//------------------------------------------------------------
//
//.F	QwtSliderBase::mouseReleaseEvent
//	Mouse Release Event handler
//
//.u	Syntax
//.f	void QwtSliderBase::mouseReleaseEvent(QMouseEvent *e)
//
//.u	Parameters
//.p	QMouseEvent *e -- Qt Mouse Event
//
//------------------------------------------------------------
void QwtSliderBase::mouseReleaseEvent(QMouseEvent *e)
{
    int ms = 0;
    double inc = step();
    
    switch(d_scrollMode) 
    {

    case ScrMouse:

	setPosition(e->pos());
	d_direction = 0;
	d_mouseOffset = 0;
	if (d_mass > 0.0) 
	{
	    ms = d_time.elapsed();
	    if ((fabs(d_speed) >  0.0) && (ms < 50))
	       d_tmrID = startTimer(d_updTime);
	}
	else
	{
	    d_scrollMode = ScrNone;
	    buttonReleased();
	}
	emit sliderReleased();
	
	break;

    case ScrDirect:
	
	setPosition(e->pos());
	d_direction = 0;
	d_mouseOffset = 0;
	d_scrollMode = ScrNone;
	buttonReleased();
	break;

    case ScrPage:
	stopMoving();
	if (!d_timerTick)
	   QwtDblRange::incPages(d_direction);
	d_timerTick = 0;
	buttonReleased();
	d_scrollMode = ScrNone;
	break;

    case ScrTimer:
	stopMoving();
	if (!d_timerTick)
	   QwtDblRange::fitValue(value() + double(d_direction) * inc);
	d_timerTick = 0;
	buttonReleased();
	d_scrollMode = ScrNone;
	break;

    default:
	d_scrollMode = ScrNone;
	buttonReleased();
    }
}


//------------------------------------------------------------
//
//.F	QwtSliderBase::setPosition
//	Move the slider to a specified point, adjust the value
//	and emit signals if necessary
//
//.u	Syntax
//.f	void QwtSliderBase::setPosition(const QPoint &p)
//
//.u	Parameters
//.p	const QPoint &p
//
//------------------------------------------------------------
void QwtSliderBase::setPosition(const QPoint &p) 
{
    QwtDblRange::fitValue(getValue(p) - d_mouseOffset);
}


//------------------------------------------------------------
//
//.F QwtSliderBase::setTracking
//
//	Enables or disables tracking.
//
//.u Syntax
//.f	void QwtSliderBase::setTracking(bool enable)
//
//.u Parameters
//.p  bool enable -- enable (TRUE) or disable (FALSE) tracking
//
//.u Description
//
//	If tracking is enabled, the slider emits a
//	valueChanged() signal whenever its value
//	changes (the default behaviour). If tracking
//	is disabled, the value changed() signal will only
//	be emitted if
//.i	-- the user releases the mouse
//	button and the value has changed or
//	-- at the end of automatic scrolling. 
//.P
//	Tracking is enabled by default.
//------------------------------------------------------------
void QwtSliderBase::setTracking(bool enable)
{
    d_tracking = enable;
}

//------------------------------------------------------------
//.-
//.F	QwtSliderBase::mouseMoveEvent
//	Mouse Move Event handler
//
//.u	Syntax
//.f	void QwtSliderBase::mouseMoveEvent(QMouseEvent *e)
//
//.u	Parameters
//.p	QMouseEvent *e	-- Qt Mouse Move Event
//
//------------------------------------------------------------
void QwtSliderBase::mouseMoveEvent(QMouseEvent *e)
{
    double ms = 0.0;
    if (d_scrollMode == ScrMouse )
    {
	setPosition(e->pos());
	if (d_mass > 0.0) 
	{
	    ms = double(d_time.elapsed());
	    if (ms < 1.0) ms = 1.0;
	    d_speed = (exactValue() - exactPrevValue()) / ms;
	    d_time.start();
	}
	if (value() != prevValue())
	   emit sliderMoved(value());
    }
    
}



//------------------------------------------------------------
//
//.F	QwtSliderBase::timerEvent
//	Timer event handler
//
//.u	Syntax
//.f	void QwtSliderBase::timerEvent(QTimerEvent *e)
//
//.u	Parameters
//.p	QTimerEvent *e	--	Qt timer event
//
//------------------------------------------------------------
void QwtSliderBase::timerEvent(QTimerEvent *e)
{
    double newval;
    double inc = step();

    switch (d_scrollMode)
    {
    case ScrMouse:
	if (d_mass > 0.0)
	{
	    d_speed *= exp( - double(d_updTime) * 0.001 / d_mass );
	    newval = exactValue() + d_speed * double(d_updTime);
	    QwtDblRange::fitValue(newval);
	    // stop if d_speed < one step per second
	    if (fabs(d_speed) < 0.001 * fabs(step()))
	    {
		d_speed = 0;
		stopMoving();
		buttonReleased();
	    }

	}
	else
	   stopMoving();

	break;

    case ScrPage:
	QwtDblRange::incPages(d_direction);
	if (!d_timerTick) 
	{
	    killTimer(d_tmrID);
	    d_tmrID = startTimer(d_updTime);
	}
	break;
    case ScrTimer:
	QwtDblRange::fitValue(value() +  double(d_direction) * inc);
	if (!d_timerTick) 
	{
	    killTimer(d_tmrID);
	    d_tmrID = startTimer(d_updTime);
	}
	break;
    default:
	stopMoving();
	break;
    }

    d_timerTick = 1;
}


//------------------------------------------------------------
//
//.F	QwtSliderBase::valueChange
//	Notify change of value
//
//.u	Syntax
//.f	void QwtSliderBase::valueChange()
//
//.u	Parameters
//.p	double x  --    new value
//
//.u	Description
//	This function can be reimplemented by derived classes
//	in order to keep track of changes, i.e. repaint the widget.
//	The default implementation emits a valueChanged() signal
//	if tracking is enabled.
//
//------------------------------------------------------------
void QwtSliderBase::valueChange() 
{
    if (d_tracking)
       emit valueChanged(value());	
}

//------------------------------------------------------------
//
//.F QwtSliderBase::setMass
//	Set the slider's mass for flywheel effect.
//
//.u Syntax
//.f	void QwtSliderBase::setMass(double val)
//
//.u Parameters
//.p	    double val    --      new mass in kg
//
//.u Description
//
//     If the slider's mass is greater then 0, it will continue
//     to move after the mouse button has been released. Its speed
//     decreases with time at a rate depending on the slider's mass.
//     A large mass means that it will continue to move for a
//     long time.
//
//     Limits: If the mass is smaller than 1g, it is set to zero.
//     The maximal mass is limited to 100kg.
//
//     Derived widgets may overload this function to make it public.
//
//------------------------------------------------------------
void QwtSliderBase::setMass(double val)
{
    if (val < 0.001)
       d_mass = 0.0;
    else if (val > 100.0)
       d_mass = 100.0;
    else
       d_mass = val;
}


//------------------------------------------------------------
//
//.F	QwtSliderBase::setValue
//	Move the slider to a specified value
//
//.u	Syntax
//.f	void QwtSliderBase::setValue(double val)
//
//.u	Parameters
//.p	double val	--	new value
//
//.u	Description
//	This function can be used to move the slider to a value
//	which is not an integer multiple of the step size.
//
//.u	See also
//	@QwtSliderBase::fitValue@
//------------------------------------------------------------
void QwtSliderBase::setValue(double val)
{
    if (d_scrollMode == ScrMouse) stopMoving();
    QwtDblRange::setValue(val);
}


//------------------------------------------------------------
//
//.F	QSlider::fitValue
//	Set the slider's value to the nearest integer multiple
//      of the step size.
//
//.u	Syntax
//.f	void QwtSliderBase::fitValue(double val)
//
//.u	See also:
//	@QwtSliderBase::setValue@
//------------------------------------------------------------
void QwtSliderBase::fitValue(double val)
{
    if (d_scrollMode == ScrMouse) stopMoving();
    QwtDblRange::fitValue(val);
}


//------------------------------------------------------------
//
//.F	QwtSliderBase::incValue
//	Increment the value by a specified number of steps
//
//.u	Syntax
//.f	void QwtSliderBase::incValue(int steps)
//
//.u	Parameters
//.p	int steps	--   number of steps
//
//------------------------------------------------------------
void QwtSliderBase::incValue(int steps)
{
    if (d_scrollMode == ScrMouse) stopMoving();
    QwtDblRange::incValue(steps);
}



//------------------------------------------------------------
//
//.F	QwtSliderBase::getValue
//	Determine the value corresponding to a specified poind
//
//.u	Syntax
//.f	void QwtSliderBase::getValue(const QPoint &p)
//
//.u	Parameters
//.p	const QPoint &p	-- point 
//
//.u	Description
//	This is an abstract virtual function which is called when
//	the user presses or releases a mouse button or moves the
//	mouse. It has to be implemented by the derived class.
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtSliderBase::getScrollMode
//	Determine what to do when the user presses a mouse button.
//
//.u	Syntax
//.f	void QwtSliderBase::getScrollMode(const QPoint &p, int &scrollMode, int &direction)
//
//.u	Input Parameters
//.p	const QPoint &p	--	point where the mouse was pressed
//
//.u	Output parameters
//	int &scrollMode --	The scrolling mode
//	int &direction  --	direction: 1, 0, or -1.
//
//.u	Description
//	This function is abstract and has to be implemented by derived classes.
//	It is called on a mousePress event. The derived class can determine
//	what should happen next in dependence of the position where the mouse
//	was pressed by returning scrolling mode and direction. QwtSliderBase
//	knows the following modes:
//.t
//	QwtSliderBase::ScrNone -- Scrolling switched off. Don't change the value.
//	QwtSliderBase::ScrMouse -- Change the value while the user keeps the
//					button pressed and moves the mouse.
//	QwtSliderBase::ScrTimer -- Automatic scrolling. Increment the value
//					in the specified direction as long as
//					the user keeps the button pressed.
//	QwtSliderBase::ScrPage -- Automatic scrolling. Same as ScrTimer, but
//					increment by page size.
//
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtSliderBase::valueChanged
//	Notify a change of value.
//
//.u	Syntax
//.f	void QwtSliderBase::valueChanged(double value)
//
//.u	Parameters
//.p	double value -- new value
//
//.u	Description
//      In the default setting 
//	(tracking enabled), this signal will be emitted every 
//	time the value changes ( see setTracking() ). 
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtSliderBase::sliderPressed
//	This signal is emitted when the user presses the 
//	movable part of the slider (start ScrMouse Mode).
//
//.u	Syntax
//.f	void QwtSliderBase::sliderPressed()
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtSliderBase::SliderReleased
//	This signal is emitted when the user releases the
//	movable part of the slider.
//
//.u	Syntax
//.f	 void QwtSliderbase::SliderReleased()
//
//------------------------------------------------------------


//------------------------------------------------------------
//
//.F	QwtSliderBase::sliderMoved
//	This signal is emitted when the user moves the
//	slider with the mouse.
//
//.u	Syntax
//.f	void QwtSliderBase::sliderMoved(double value)
//
//.u	Parameters
//.p	double value	-- new value
//
//------------------------------------------------------------









