#include "qwt_scale.h"
#include <qwt_math.h>

//------------------------------------------------------------
//
//.H QwtScale | 3 | 02/05/98 | Qwt Widget Library | Qwt Programmer's Manual
//
//
//.U NAME
//	QwtScale - A Widget which contains a scale
//
//.U SYNOPSIS 
//	#include <qwt_scale.h>
//
//.U INHERITED CLASSES
//	QWidget
//
//.U PUBLIC MEMBERS
//.R
//	QwtScale::QwtScale --	Constructor
//	QwtScale::~QwtScale --	Destructor
//	QwtScale::enableTitle --  Enable or disable the title
//	QwtScale::setBaselineDist -- Specify the border distance of the base line 
//	QwtScale::setBorderDist -- Set the border distances of the scale's endpoints
//	QwtScale::setScale --	Assign a scale
//	QwtScale::setTitle --	Assign a title
//	QwtScale::setTitleFont -- Set the title font
//	QwtScale::setTitleColor -- Set the title color
//	QwtScale::title -- Return the title
//	QwtScale::titleFont -- Return the title font
//	QwtScale::titleColor -- Return the title color
//	QwtScale::minBorderDist -- Return a recommended minimum offset
//	QwtScale::sizeHint -- Return a size hint
//
//.U PROTECTED MEMBERS
//.R
//	QwtScale::scaleChange -- Notify a change of the scale
//	QwtScale::fontChange --	Notify a change of the font
//
//.U DESCRIPTION
//	This Widget can be used to decorate composite widgets with
//	a scale.
//------------------------------------------------------------

//------------------------------------------------------------
//.U MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtScale::QwtScale
//	Construct a QwtScale object
//
//.u	Syntax
//.f	 QwtScale::QwtScale(QwtScale::Orientation ori, QWidget *parent, const char *name = 0)
//
//.u	Parameters
//.p	QwtScale::Orientation ori	-- Orientation. Must be one of
//					   QwtScale::Left, QwtScale::Right,
//					   QwtScale::Bottom, or QwtScale::Top
//	QWidget *parent		-- Parent widget
//	const char *name	-- name
//
//------------------------------------------------------------
QwtScale::QwtScale( QwtScale::Orientation ori, QWidget *parent, const char *name)
: QWidget(parent,name)
{
    QwtScaleDraw::Orientation sdo;
    d_borderDist[0] = 0;
    d_borderDist[1] = 0;
    d_baseDist = 4;
    d_titleOffset = 0;
    d_titleDist = 8;

    switch(ori)
    {
    case QwtScale::Left:
	sdo = QwtScaleDraw::Left;
	break;
    case QwtScale::Right:
	sdo = QwtScaleDraw::Right;
	break;
    case QwtScale::Bottom:
	sdo = QwtScaleDraw::Bottom;
	break;
    case QwtScale::Top:
	sdo = QwtScaleDraw::Top;
	break;
    default:
	sdo = QwtScaleDraw::Right;
    }
    
    d_scale.setGeometry(0,0,10,sdo);
    setMinimumSize(30,30);
}

//------------------------------------------------------------
//
//.F	QwtScale::~QwtScale
//	Destroy a QwtScale object
//
//.u	Syntax
//.f	 QwtScale::~QwtScale()
//------------------------------------------------------------
QwtScale::~QwtScale()
{
}


//------------------------------------------------------------
//
//.F	QwtScale::setTitle
//	Assign a title
//
//.u	Syntax
//.f	void QwtScale::setTitle(const char *title)
//
//.u	Parameters
//.p	const char *title -- new title
//
//------------------------------------------------------------
void QwtScale::setTitle(const char *title)
{
    d_title = QString(title);
    repaint();
}


//------------------------------------------------------------
//
//.F	QwtScale::setBorderDist
//	Specify distances of the scale's endpoints from the
//	widget's borders.
//
//
//.u	Syntax
//.f	void QwtScale::setBorderDist(int dist1, int dist2)
//
//.u	Parameters
//.p	int dist1  --   Left or top Distance
//	int dist2 --	Right or bottom distance
//
//
//------------------------------------------------------------
void QwtScale::setBorderDist(int dist1, int dist2)
{
    d_borderDist[0] = dist1;
    d_borderDist[1] = dist2;
    resizeScale();
    repaint();
}


//------------------------------------------------------------
//
//.F	QwtScale::setBaselineDist
//      Specify the border distance of the scale's base line.
//
//.u	Syntax
//.f	void QwtScale::setBaselineDist(int bd)
//
//.u	Parameters
//.p	int bd	-- border distance of the baseline
//
//------------------------------------------------------------
void QwtScale::setBaselineDist(int bd)
{
    d_baseDist = qwtMax(0,bd);
    repaint();
}

//------------------------------------------------------------
//
//.F	QwtScale::setTitleFont
//	Change the title font
//
//.u	Syntax
//.f	void QwtScale::setTitleFont(const QFont &f)
//
//.u	Parameters
//.p	const QFont &f	-- new font
//
//------------------------------------------------------------
void QwtScale::setTitleFont(const QFont &f)
{
    d_titleFont = f;
    repaint();
}

//------------------------------------------------------------
//
//.F	QwtScale::setTitleColor
//	Change the title color
//
//.u	Syntax
//.f	void QwtScale::setTitleColor(const QColor &c)
//
//.u	Parameters
//.p	const QColor &c  --    new title color
//
//------------------------------------------------------------
void QwtScale::setTitleColor(const QColor &c)
{
    d_titleColor = c;
    repaint();
}


//------------------------------------------------------------
//.-
//.F	QwtScale::paintEvent
//	Paint Event
//
//.u	Syntax
//.f	void QwtScale::paintEvent(QPaintEvent *e)
//
//.u	Parameters
//.p	QPaintEvent *e
//
//------------------------------------------------------------
void QwtScale::paintEvent(QPaintEvent *e)
{
    int th, ta, ttop;
   	 
    QRect r = this->rect();
    
    QPainter p;
    p.begin(this);

    ta = p.fontMetrics().ascent();
    th = p.fontMetrics().height();
    ttop = ta + d_titleOffset;

    // draw scale
    d_scale.draw(&p);

    // draw title
    p.setFont(d_titleFont);
    switch(d_scale.orientation())
    {
    case QwtScaleDraw::Left:
	p.rotate(-90.0);
	ttop = qwtMin(ttop + (r.width() - ttop) / 4, r.width());
	p.drawText(-r.bottom(), r.right() - ttop,
	   r.height(), 2*th,AlignHCenter|AlignTop, d_title);
	break;
    case QwtScaleDraw::Right:
    
	p.rotate(90.0);
	ttop = qwtMin(ttop + (r.width() - ttop) / 4, r.width());
	p.drawText(r.top(), -r.left() - ttop,
	   r.height(), 2*th, AlignHCenter|AlignTop, d_title);
	break;
    case QwtScaleDraw::Top:
	p.drawText(r.x(), r.bottom() - ttop,
		    r.width(), 2*th,
		    AlignHCenter|AlignTop ,d_title);
	break;
    case QwtScaleDraw::Bottom:
	p.drawText(r.x(), r.top() + d_titleOffset,
		    r.width(), 2*th,
		    AlignHCenter|AlignTop,d_title);
	break;
    default:
	break;
    }

    p.end();
    
}

//------------------------------------------------------------
//.-
//.F	QwtScale::resizeEvent
//	Resize Event
//
//.u	Syntax
//.f	void QwtScale::resizeEvent(QResizeEvent *e)
//
//.u	Parameters
//.p	QResizeEvent *e
//
//------------------------------------------------------------
void QwtScale::resizeEvent(QResizeEvent *e)
{
    resizeScale();
}


//------------------------------------------------------------
//.-
//.F	QwtScale::resizeScale
//	Resize the scale
//
//.u	Syntax
//.f	void QwtScale::resizeScale()
//
//------------------------------------------------------------
void QwtScale::resizeScale()
{
    
    QRect r = this->rect();
    QPainter p;

    p.begin(this);

    switch(d_scale.orientation())
    {
    case QwtScaleDraw::Bottom:
	d_scale.setGeometry(r.x() + d_borderDist[0],
			    r.y() + d_baseDist,
			    r.width() - d_borderDist[1] - d_borderDist[0],
			    QwtScaleDraw::Bottom);

	d_titleOffset = d_titleDist + d_baseDist + d_scale.maxHeight(&p);
	break;
	
    case QwtScaleDraw::Top:
	d_scale.setGeometry(r.x() + d_borderDist[0],
			    r.bottom() - d_baseDist,
			    r.width() - d_borderDist[1] - d_borderDist[0],
			    QwtScaleDraw::Top);
	d_titleOffset =  d_titleDist + d_baseDist + d_scale.maxHeight(&p);
	break;
	
    case QwtScaleDraw::Left:
	d_scale.setGeometry(r.right() - d_baseDist,
			    r.y() + d_borderDist[0],
			    r.height() - d_borderDist[0] - d_borderDist[1],
			    QwtScaleDraw::Left);
	d_titleOffset = d_baseDist + d_titleDist + d_scale.maxWidth(&p,FALSE);
	break;
	
    case QwtScaleDraw::Right:
	d_scale.setGeometry(r.x() + d_baseDist,
			    r.y() + d_borderDist[0],
			    r.height() - d_borderDist[0] - d_borderDist[1],
			    QwtScaleDraw::Right);
	d_titleOffset = d_baseDist + d_titleDist + d_scale.maxWidth(&p,FALSE);
	break;
    default:
	break;
    }
    p.end();
    
}

//------------------------------------------------------------
//
//.F	QwtScale::scaleChange			
//	Notify a change of the scale
//
//.u	Syntax
//.f	void QwtScale::scaleChange()
//
//.u	Description
//	This virtual function can be overloaded by derived
//	classes. The default implementation repaints the widget.
//
//------------------------------------------------------------
void QwtScale::scaleChange()
{
    repaint();
}



//------------------------------------------------------------
//
//.F	QwtScale::sizeHint
//	Return a size hint
//
//.u	Syntax
//.f	QSize QwtScale::sizeHint() const
//
//.u	Note
//	This function uses a QPainter internally. Don't call
//	it while another QPainter is active.
//
//------------------------------------------------------------
QSize QwtScale::sizeHint() const
{
    int th,ta, tw;
    
    QPainter p;
    p.begin(this);

    p.setFont(d_titleFont);
    QFontMetrics fm = p.fontMetrics();
   
	th = fm.height();
    ta = fm.ascent();
    tw = fm.width(d_title);
    
    QSize rv;
    switch(d_scale.orientation())
    {
    case QwtScaleDraw::Left:
    case QwtScaleDraw::Right:
	rv.setWidth(d_scale.maxWidth(&p,TRUE) + d_baseDist + ta);
	rv.setHeight(d_scale.scaleDiv().majCnt() * 2 * d_scale.maxHeight(&p));
	break;
    case QwtScaleDraw::Top:
    case QwtScaleDraw::Bottom:
	rv.setHeight(d_scale.maxHeight(&p) + d_baseDist + th + d_titleDist);
	rv.setWidth(d_scale.scaleDiv().majCnt()  * d_scale.maxWidth(&p,TRUE));
	break;
    default:
	break;
    }
    p.end();
    
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtScale::minBorderDist
//	Determine the minimum Border distance
//
//.u	Syntax
//.f	int QwtScale::minBorderDist()
//
//.u	Description
//	This member function returns the minimum distance
//	of the scale's endpoints from the widget borders which
//	is required for the mark labels to fit into the widget.
//
//.u	Notes
//.i
//	-- This function uses a QPainter internally. Don't call it
//	while another QPainter is active.
//	-- The minimum border distance depends on the font.
//
//.u	See Also
//	@QwtScale::setBorderDist@
//
//------------------------------------------------------------
int QwtScale::minBorderDist() const
{
    int rv = 0;
    
    QPainter p;
    p.begin(this);
    QFontMetrics fm = p.fontMetrics();
    

    switch(d_scale.orientation())
    {
    case QwtScaleDraw::Left:
    case QwtScaleDraw::Right:
	rv = (fm.ascent() + fm.descent() + 1) / 2;
	break;
    case QwtScaleDraw::Top:
    case QwtScaleDraw::Bottom:
	rv = (d_scale.maxLabelWidth(&p,TRUE) + 1) / 2;
	break;
    default:
	break;
    }
    p.end();
    
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtScale::fontChange
//	Notify a change of the font
//
//.u	Syntax
//.f	void QwtScale::fontChange(const QFont &oldfont)
//
//.u	Parameters
//.p	const QFont &oldfont -- old font
//
//.u	Description
//	This virtual function may be overloaded by derived widgets.
//	The default implementation resizes the scale and repaints
//	the widget.
//------------------------------------------------------------
void QwtScale::fontChange(const QFont &oldfont)
{
    resizeScale();
    repaint();
}



//------------------------------------------------------------
//
//.F	QwtScale::setScale
//	Assign a scale division
//
//.u	Syntax
//.f	void QwtScale::setScale(const QwtScaleDiv &sd)
//
//.u	Parameters
//.p	const QwtScaleDiv &sd -- Scale Division
//
//.u	Description
//	The scale division determines where to set the tick
//	marks. For more information about scale divisions, see
//	@^QwtScaleDiv@.
//
//------------------------------------------------------------
void QwtScale::setScale(const QwtScaleDiv &sd)
{
    if (d_scale.scaleDiv() != sd)
    {
    	d_scale.setScale(sd);
    	resize(this->size());
    	repaint();
    }
}









