#include "qwt_plot.h"
#include "qwt_plot_dict.h"
#include <qwt_math.h>

//------------------------------------------------------------
//.C	Accessing and Manipulating Markers
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::closestMarker
//	Find the marker which is closest to
//	a given point.
//
//.u	Syntax
//.f	long QwtPlot::closestMarker(int xpos, int ypos, int &dist)
//
//.u	Input Parameters
//.p	int xpos, int ypos -- coordinates of a point in the plotting region
//
//.u	Output Parameters
//.p	int &dist -- Distance in points between the marker and the specified point.
//
//.u	Return Value
//	Key of the closest marker or 0 if no marker was found
//
//------------------------------------------------------------
long QwtPlot::closestMarker(int xpos, int ypos, int &dist) const
{
    long rv = 0;
    QwtPlotMarker *m;
    QIntDictIterator<QwtPlotMarker> itm(*d_markers);

    double dmin = 1.0e10;
    double cx, cy;
    double d;
    
    itm.toFirst();
    while((m = itm.current()))
    {
	cx = d_map[m->xAxis()].xTransform(m->xValue());
	cy = d_map[m->yAxis()].xTransform(m->yValue());

	if (m->lineStyle() == QwtMarker::HLine)
	{
	    if (m->symbol().style() == QwtSymbol::None)
	       cx = double(xpos);
	}
	else if (m->lineStyle() == QwtMarker::VLine)
	{
	    if (m->symbol().style() == QwtSymbol::None)
	       cy = double(ypos);
	}
	
	d = qwtSqr(cx - double(xpos)) + qwtSqr(cy - double(ypos));
	if (d < dmin)
	{
	    dmin = d;
	    rv = itm.currentKey();
	}
	
	++itm;
    }
    dist = int(sqrt(dmin));
    return rv;
}




//------------------------------------------------------------
//.-
//.F	QwtPlot::newMarkerKey
//	Generate a key for a new marker
//
//.u	Syntax
//.f	long QwtPlot::newMarkerKey()
//
//
//------------------------------------------------------------
long QwtPlot::newMarkerKey()
{
    long newkey = d_markers->count() + 1;

    if (newkey > 1)			// count > 0
    {
	if (d_markers->find(newkey))	// key count+1 exists => there must be a
					// free key <= count
	{
	    // find the first available key <= count
	    newkey = 1;
	    while (newkey <= long(d_markers->count()))
	    {
		if (d_markers->find(newkey))
		   newkey++;
		else
		   break;
	    }

	    // This can't happen. Just paranoia.
	    if (newkey > long(d_markers->count()))
	    {
		while (!d_markers->find(newkey))
		{
		    newkey++;
		    if (newkey > 10000) // prevent infinite loop
		    {
			newkey = 0;
			break;
		    }
		}
	    }
	}
    }
    return newkey;
    
}


//------------------------------------------------------------
//.-
//.F	QwtPlot::insertLineMarker
//
//.u	Syntax
//.f	bool QwtPlot::insertLineMarker(long key, char *label, int axis)
//
//.u	Parameters
//.p	long key, char *label, int axis
//
//.u	Return Value
//		TRUE if the marker has been inserted successfully.
//------------------------------------------------------------
bool QwtPlot::insertLineMarker(long key, const char *label, int axis)
{
    int rv = FALSE;
    QwtPlotMarker *m;
    
    switch(axis)
    {
    case xTop:
    case xBottom:
	rv = insertMarker(key, label, axis, DefaultYAxis);
	m = d_markers->find(key);
	if (rv && m)
	{
	    m->setLineStyle(QwtMarker::VLine);
	    m->setLabelAlignment(AlignRight|AlignTop);
	}
	break;

    case yLeft:
    case yRight:
	rv = insertMarker(key, label, DefaultXAxis, axis);
	m = d_markers->find(key);
	if (rv && m)
	{
	    m->setLineStyle(QwtMarker::HLine);
	    m->setLabelAlignment(AlignRight|AlignTop);
	}
	break;
	
    }
    autoRefresh();
    return rv;
    
}


//------------------------------------------------------------
//
//.F	QwtPlot::insertLineMarker
//
//.u	Syntax
//.f	long QwtPlot::insertLineMarker(const char *label, int axis)
//
//.u	Parameters
//.p	const char *label	-- Label
//	int axis	-- Axis to be attached
//
//.u	Return Value
//		New key if the marker could be inserted, 0 if not.
//
//.u	Description
//	This function is a shortcut to insert a horizontal or vertical
//	line marker, dependent on the specified axis.
//------------------------------------------------------------
long QwtPlot::insertLineMarker(const char *label, int axis)
{
    long rv = 0;
    if ((rv = newMarkerKey()))
    {
	insertLineMarker(rv, label, axis);
    }
    return rv;
}

//------------------------------------------------------------
//.-
//.F	QwtPlot::insertMarker
//
//.u	Syntax
//.f	bool QwtPlot::insertMarker(long key, char *label, int xAxis, int yAxis)
//
//.u	Parameters
//.p	long key, char *label, int xAxis, int yAxis
//
//------------------------------------------------------------
bool QwtPlot::insertMarker(long key, const char *label, int xAxis, int yAxis)
{

    int rv = FALSE;
    if (d_markers->find(key) == 0)
    {
	QwtPlotMarker *nm = new QwtPlotMarker(this);
	if (nm) 
	{
	    nm->setXAxis(verifyXAxis(xAxis));
	    nm->setYAxis(verifyYAxis(yAxis));
	    nm->setXValue(0.0);
	    nm->setYValue(0.0);
	    nm->setLabel(label);
	    d_markers->insert(key, nm);
	    rv = TRUE;
	}
    }
    autoRefresh();
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::insertMarker
//	Insert a new marker
//
//.u	Syntax
//.f	long QwtPlot::insertMarker(char *label, int xAxis, int yAxis)
//
//.u	Parameters
//.p	char *label	-- Label
//	int xAxis	-- X axis to be attached
//	int yAxis	-- Y axis to be attached
//
//.u	Syntax
//.f	long QwtPlot::insertMarker(char *label, int xAxis, int yAxis)
//
//------------------------------------------------------------
long QwtPlot::insertMarker(const char *label, int xAxis, int yAxis)
{
    long newkey = newMarkerKey();
    if (newkey > 0)
	if (!insertMarker(newkey, label, xAxis, yAxis))
	   newkey = 0;
    return newkey;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::markerKeys
//	Return an array containing the keys of all markers
//
//.u	Syntax
//.f	QArray<long> QwtPlot::markerKeys() const
//
//------------------------------------------------------------
QArray<long> QwtPlot::markerKeys() const
{

    int i;
    QIntDictIterator<QwtPlotMarker> im(*d_markers);
    QArray<long> rv(d_markers->count());

    im.toFirst();
    i = 0;
    while (im.current() && (uint(i) < rv.size()))
    {
	rv[i] = im.currentKey();
	++im;
	++i;
    }	
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::markerFont
//	Return the font of a marker
//
//.u	Syntax
//.f	const QFont& QwtPlot::markerFont(long key) const
//
//.u	Parameters
//.p	long key
//
//------------------------------------------------------------
const QFont& QwtPlot::markerFont(long key) const
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
	return m->font();
    else
	return DummyFont;
}

//------------------------------------------------------------
//
//.F	QwtPlot::markerLabel
//	Return a marker's label
//
//.u	Syntax
//.f	const QString& QwtPlot::markerLabel(long key) const
//
//.u	Parameters
//.p	long key	-- Marker key
//
//------------------------------------------------------------
const QString& QwtPlot::markerLabel(long key) const
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
	return m->label();
    else
	return QwtPlot::DummyString;
}

//------------------------------------------------------------
//
//.F	QwtPlot::markerLabelAlign
//	Return a marker's label alignment
//
//.u	Syntax
//.f	int QwtPlot::markerLabelAlign(long key) const
//
//.u	Parameters
//.p	long key -- Marker key
//
//------------------------------------------------------------
int QwtPlot::markerLabelAlign(long key) const
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
	return m->labelAlignment();
    else
	return 0;
}

//------------------------------------------------------------
//
//.F	QwtPlot::markerLabelPen
//	Return the pen of a marker's label
//
//.u	Syntax
//.f	const QPen& QwtPlot::markerLabelPen(long key) const
//
//.u	Parameters
//.p	long key	-- Marker key
//
//------------------------------------------------------------
const QPen& QwtPlot::markerLabelPen(long key) const
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
	return m->labelPen();
    else
	return DummyPen;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::markerLinePen
//	Return a marker's line pen
//
//.u	Syntax
//.f	const QPen& QwtPlot::markerLinePen(long key) const
//
//.u	Parameters
//.p	long key	-- Marker key
//
//------------------------------------------------------------
const QPen& QwtPlot::markerLinePen(long key) const 
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
	return m->linePen();
    else
	return DummyPen;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::markerLineStyle
//	Return a marker's line style
//
//.u	Syntax
//.f	QwtMarker::lineStyle QwtPlot::markerLineStyle(long key) const
//
//.u	Parameters
//.p	long key	-- Marker key
//
//------------------------------------------------------------
QwtMarker::LineStyle QwtPlot::markerLineStyle(long key) const
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
	return m->lineStyle();
    else
	return QwtMarker::NoLine;
}

//------------------------------------------------------------
//
//.F	QwtPlot::markerPos
//	Get the position of a marker
//
//.u	Syntax
//.f	void QwtPlot::markerPos(long key, double &mx, double &my ) const
//
//
//.u	Input Parameters
//.p	long key	-- Marker key
//
//.u	Output Parameters
//.p	double &mx, double &my	-- Marker position 
//
//------------------------------------------------------------
void QwtPlot::markerPos(long key, double &mx, double &my ) const
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
    {
	mx = m->xValue();
	my = m->yValue();
    }
    else
    {
	mx = 0;
	my = 0;
    }
}

//------------------------------------------------------------
//
//.F	QwtPlot::markerSymbol
//	Return a marker's symbol
//
//.u	Syntax
//.f	const QwtSymbol& QwtPlot::markerSymbol(long key) const
//
//.u	Parameters
//.p	long key -- Marker key
//
//------------------------------------------------------------
const QwtSymbol& QwtPlot::markerSymbol(long key) const
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
	return m->symbol();
    else
	return DummySymbol;
}


//------------------------------------------------------------
//
//.F	QwtPlot::markerXAxis
//	Return the x axis to which a marker is attached
//
//.u	Syntax
//.f	int QwtPlot::markerXAxis(long key) const
//
//.u	Parameters
//.p	long key -- Marker key
//
//------------------------------------------------------------
int QwtPlot::markerXAxis(long key) const
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
	return m->xAxis();
    else
	return -1;
    
}


//------------------------------------------------------------
//
//.F	QwtPlot::markerYAxis
//	Return the Y axis to which a marker is attached
//
//.u	Syntax
//.f	int QwtPlot::markerYAxis(long key) const
//
//.u	Parameters
//.p	long key	-- Marker key
//
//------------------------------------------------------------
int QwtPlot::markerYAxis(long key) const
{
    QwtPlotMarker *m = d_markers->find(key);
    if (m)
	return m->yAxis();
    else
	return -1;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::removeMarker
//	Remove the marker indexed by key
//
//.u	Syntax
//.f	bool QwtPlot::removeMarker(long key)
//
//.u	Parameters
//.p	long key	--	unique key
//
//------------------------------------------------------------
bool QwtPlot::removeMarker(long key)
{
    if (d_markers->remove(key))
    {
	autoRefresh();
	return TRUE;
    }
    else
       return FALSE;
}


//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerXAxis
//	Attach the marker to an x axis
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerXAxis(long key, int axis)
//
//.u	Return Value
//		True if the specified marker exists.
//------------------------------------------------------------
bool QwtPlot::setMarkerXAxis(long key, int axis)
{
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setXAxis(verifyXAxis(axis));
	return TRUE;
    }
    else
       return FALSE;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerYAxis
//	Attach the marker to a Y axis
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerYAxis(long key, int axis)
//
//.u	Parameters
//.p	long key	-- Marker key
//	int axis	-- Axis to be attached
//
//.u	Return Value
//		True if the specified marker exists
//------------------------------------------------------------
bool QwtPlot::setMarkerYAxis(long key, int axis)
{
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setYAxis(verifyYAxis(axis));
	return TRUE;
    }
    else
       return FALSE;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerFont
//	Specify a font for a marker's label
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerFont(long key, const QFont &f)
//
//.u	Parameters
//.p	long key	-- Marker key
//	const QFont &f	-- New font
//
//.u	Return Value
//		True if the specified marker exists
//------------------------------------------------------------
bool QwtPlot::setMarkerFont(long key, const QFont &f)
{
    int rv = FALSE;
    
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setFont(f);
	rv = TRUE;
    }
    return rv;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerLinePen
//	Specify a pen for a marker's line
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerLinePen(long key, const QPen &p)
//
//.u	Parameters
//.p	long key	-- Marker key
//	const QPen &p	-- New pen
//
//.u	Return Value
//	True if the marker exists
//------------------------------------------------------------
bool QwtPlot::setMarkerLinePen(long key, const QPen &p)
{
    int rv = FALSE;
    
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setLinePen(p);
	rv = TRUE;
    }
    return rv;

}


//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerLineStyle
//	Specify the line style for a marker
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerLineStyle(long key, QwtMarker::LineStyle st)
//
//.u	Parameters
//.p	long key	-- Marker key
//	QwtMarker::LineStyle st	-- Line style: QwtMarker::HLine, QwtMarker::VLine,
//			QwtMarker::NoLine or a combination of them.
//
//.u	Return Value
//	True if the specified marker exists.
//------------------------------------------------------------
bool QwtPlot::setMarkerLineStyle(long key, QwtMarker::LineStyle st)
{
    int rv = FALSE;
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setLineStyle(st);
	rv = TRUE;
    }
    return rv;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerPen
//	Specify a pen for a marker's label.
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerPen(long key, const QPen &p)
//
//.u	Parameters
//.p	long key	-- Marker key
//	const QPen &p	-- New pen
//
//.u	Return Value
//	True if the marker exists
//------------------------------------------------------------
bool QwtPlot::setMarkerPen(long key, const QPen &p)
{
    int rv = FALSE;
    
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setLinePen(p);
	m->setLabelPen(p);
	rv = TRUE;
    }
    return rv;
    
}


//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerPos
//	Change the position of a marker
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerPos(long key, double xval, double yval)
//
//.u	Parameters
//.p	long key	-- Marker key
//	double xval, double yval	-- Position of the marker in axis coordinates.
//
//------------------------------------------------------------
bool QwtPlot::setMarkerPos(long key, double xval, double yval)
{
    int rv = FALSE;
    
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setXValue(xval);
	m->setYValue(yval);
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerXPos
//	Specify the X position of a marker
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerXPos(long key, double val)
//
//.u	Parameters
//.p	long key	-- Marker key
//	double val	-- X position of the marker
//
//.u	Return Value
//	True if the specified marker exists.
//------------------------------------------------------------
bool QwtPlot::setMarkerXPos(long key, double val)
{
    int rv = FALSE;
    
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setXValue(val);
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerYPos
//	Specify the Y position of the marker
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerYPos(long key, double val)
//
//
//.u	Parameters
//.p	long key	-- Marker key
//	double val	-- Y position of the marker
//
//.u	Return Value
//	 True if the specified marker exists
//
//------------------------------------------------------------
bool QwtPlot::setMarkerYPos(long key, double val)
{
    int rv = FALSE;
    
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setYValue(val);
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerSymbol
//	Assign a symbol to a specified marker
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerSymbol(long key, const QwtSymbol &s)
//
//
//.u	Parameters
//.p	long key	-- Marker key
//	const QwtSymbol &s	-- new symbol
//
//.u	Return Value
//		True if the specified marker exists
//------------------------------------------------------------
bool QwtPlot::setMarkerSymbol(long key, const QwtSymbol &s)
{
    int rv = FALSE;
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setSymbol(s);
	rv = TRUE;
    }
    return rv;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerLabel
//	Assign a label to a marker
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerLabel(long key, const char *txt)
//
//.u	Parameters
//.p	long key	-- Marker key
//	const char *txt	-- Label text
//
//.u	Return Value
//	True if the specified marker exists
//
//------------------------------------------------------------
bool QwtPlot::setMarkerLabel(long key, const char *txt)
{
    int rv = FALSE;
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setLabel(txt);
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerLabelAlign
//	Specify the alignment of a marker's label
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerLabelAlign(long key, int align)
//
//.u	Parameters
//.p	long key	-- Marker key
//	int align	-- Alignment: AlignLeft, AlignRight, AlignTop, AlignBottom,
//			  AlignHCenter, AlignVCenter, AlignCenter or a combination
//				of them.
//
//.u	Return Value
//	True if the specified marker exists.
//
//.u	Description
//	The alignment determines the position of the label relative to the
//	marker's position. The default setting is AlignCenter.
//
//------------------------------------------------------------
bool QwtPlot::setMarkerLabelAlign(long key, int align)
{
    int rv = FALSE;
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setLabelAlignment(align);
	rv = TRUE;
    }
    return rv;

}

//------------------------------------------------------------
//
//.F	QwtPlot::setMarkerLabelPen
//	Specify a pen for a marker's label
//
//.u	Syntax
//.f	bool QwtPlot::setMarkerLabelPen(long key, const QPen &p)
//
//.u	Parameters
//.p	long key	-- New key
//	const QPen &p	-- Label pen
//
//.u	Return Value
//	True if the specified marker exists.
//
//------------------------------------------------------------
bool QwtPlot::setMarkerLabelPen(long key, const QPen &p)
{
    int rv = FALSE;
    QwtPlotMarker *m;
    if ((m = d_markers->find(key)))
    {
	m->setLabelPen(p);
	rv = TRUE;
    }
    return rv;
    
}




