#include "qwt_plot.h"
#include "qwt_scale.h"
#include "qwt_math.h"

//------------------------------------------------------------
//.C Members for Accessing and Manipulating Axes
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::axisAutoScale
//	Return TRUE if autoscaling is enabled
//
//.u	Syntax
//.f	bool QwtPlot::axisAutoScale(int axis) const
//
//.u	Parameters
//.p	int axis	-- axis index
//
//------------------------------------------------------------
bool QwtPlot::axisAutoScale(int axis) const
{
    if (axisValid(axis))
	return d_as[axis].autoScale();
    else
	return FALSE;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisEnabled
//	Return TRUE if a specified axis is enabled
//
//.u	Syntax
//.f	bool QwtPlot::axisEnabled(int axis) const
//
//.u	Parameters
//.p	int axis	 -- axis index
//
//------------------------------------------------------------
bool QwtPlot::axisEnabled(int axis) const
{
    if (axisValid(axis))
	return d_axisEnabled[axis];
    else
	return FALSE;
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisFont
//	Return the font of the scale labels for a specified axis
//
//.u	Syntax
//.f	const QFont& QwtPlot::axisFont(int axis) const
//
//.u	Parameters
//.p	int axis	-- axis index
//
//------------------------------------------------------------
const QFont& QwtPlot::axisFont(int axis)
{
    if (axisValid(axis))
	return d_scale[axis]->font();
    else
	return DummyFont;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisMargins
//	Determine the scale margins for a specified axis
//
//.u	Syntax
//.f	bool QwtPlot::axisMargins(int axis, double &mlo, double &mhi)
//
//.u	Input Parameters
//.p	int axis
//.u	Output Parameters
//.p	double &mlo	-- Margin at the lower limit
//	double &mhi	-- Margin at the upper limit
//
//.u	See also:
//	@^QwtAutoScale#QwtAutoScale::margins@
//
//------------------------------------------------------------
bool QwtPlot::axisMargins(int axis, double &mlo, double &mhi) const
{
    int rv = FALSE;
    
    if (axisValid(axis))
    {
	mlo = d_as[axis].loMargin();
	mhi = d_as[axis].hiMargin();
	rv = TRUE;
    }
    
    return rv;
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisMaxMajor
//	Return the maximum number of major ticks for a specified axis
//
//.u	Syntax
//.f	int QwtPlot::axisMaxMajor(int axis) const
//
//.u	Parameters
//.p	int axis	-- axis index
//
//------------------------------------------------------------
int QwtPlot::axisMaxMajor(int axis) const
{
    if (axisValid(axis))
	return d_as[axis].maxMajor();
    else
	return 0;
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisMaxMinor
//	Return the maximum number of minor ticks for a specified axis
//
//.u	Syntax
//.f	int QwtPlot::axisMaxMinor(int axis)
//
//.u	Parameters
//.p	int axis	-- axis index
//
//------------------------------------------------------------
int QwtPlot::axisMaxMinor(int axis) const
{
    if (axisValid(axis))
	return d_as[axis].maxMinor();
    else
	return 0;
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisOptions
//	Return the autoscaling options of a specified axis
//
//.u	Syntax
//.f	int QwtPlot::axisOptions(int axis)
//
//.u	Parameters
//.p	int axis	-- axis index
//
//.u	See also
//	@^QwtAutoScale#QwtAutoScale::options@
//
//------------------------------------------------------------
int QwtPlot::axisOptions(int axis) const
{
    if (axisValid(axis))
	return d_as[axis].options();
    else
	return 0;
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisReference
//	Return the reference value of an axis
//
//.u	Syntax
//.f	double QwtPlot::axisReference(int axis) const
//
//.u	Parameters
//.p	int axis	--	axis index
//
//.u	Description
//	The reference value is needed if the autoscaling options
//	QwtAutoScale::Symmetric or QwtAutoScale::IncludeRef are
//	set.
//
//.u    See also
//	@QwtAutoScale@, @QwtPlot::setAxisOptions@
//
//------------------------------------------------------------
double QwtPlot::axisReference(int axis) const
{
    if (axisValid(axis))
	return d_as[axis].reference();
    else
	return 0.0;
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisScale
//	Returns the scale division of a specified axis
//
//.u	Syntax
//.f	const QwtScaleDiv& QwtPlot::axisScale(int axis) const
//
//.u	Parameters
//.p	int axis	--	axis index
//
//.u	See also
//	@QwtScaleDiv@
//------------------------------------------------------------
const QwtScaleDiv& QwtPlot::axisScale(int axis) const
{
    if (axisValid(axis))
	return d_as[axis].scaleDiv();
    else
	return DummyScale;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisTitle
//	Return the title of a specified axis
//
//.u	Syntax
//.f	const QString& QwtPlot::axisTitle(int axis) const
//
//.u	Parameters
//.p	int axis	-- axis index
//
//
//------------------------------------------------------------
const QString& QwtPlot::axisTitle(int axis) const
{
    if (axisValid(axis))
	return d_scale[axis]->title();
    else
	return QwtPlot::DummyString;
}

//------------------------------------------------------------
//
//.F	QwtPlot::axisTitleFont
//	Return the title font of a specified axis
//
//.u	Syntax
//.f	const QFont& QwtPlot::axisTitleFont(int axis) const
//
//.u	Parameters
//.p	int axis	-- axis index
//
//------------------------------------------------------------
const QFont& QwtPlot::axisTitleFont(int axis) const
{
    if (axisValid(axis))
	return d_scale[axis]->titleFont();
    else
	return DummyFont;
}

//------------------------------------------------------------
//
//.F	QwtPlot::changeAxisOptions
//	Change specified autoscaling options of an axis
//
//.u	Syntax
//.f	void QwtPlot::changeAxisOptions(int axis, int opt, bool value)
//
//.u	Parameters
//.p	int axis -- axis index
//	int opt	-- Set of or'ed options
//	bool value -- Value to be assigned to all the specified options (TRUE or FALSE)
//
//.u	See also
//	@^QwtAutoScale#QwtAutoScale::changeAxisOptions@
//------------------------------------------------------------
void QwtPlot::changeAxisOptions(int axis, int opt, bool value)
{
    if (axisValid(axis))
    {
	d_as[axis].changeOptions(opt, value);
	autoRefresh();
    }
}


//------------------------------------------------------------
//
//.u	QwtPlot::enableAxis
//	Enable or disable a specified axis
//
//.u	Syntax
//.f	void QwtPlot::enableAxis(int axis, bool tf)
//
//.u	Parameters
//.p	int axis	--	 axis index
//	bool tf		--	TRUE (enabled) or FALSE disabled)
//.u
//	Description
//	When an axis is disabled, this only means that it is not
//	visible on the screen. Curves, markers and can be attached
//	to disabled axes, and transformation of screen coordinates
//	into values works as normal.
//	Only xBottom and yLeft are enabled by default.
//
//------------------------------------------------------------
void QwtPlot::enableAxis(int axis, bool tf)
{
    if (axisValid(axis))
    {
	d_axisEnabled[axis] = tf;
	autoRefresh();
    }
}

//------------------------------------------------------------
//
//.F	QwtPlot::invTransform
//	Transform the x or y coordinate of a position in the
//	drawing region into a value.
//
//.u	Syntax
//.f	double QwtPlot::invTransform(int axis, int pos)
//
//.u	Parameters
//.p	int axis	-- axis index
//	int pos		-- position
//
//.u	Note
//	The position can be an x or a y coordinate,
//	depending on the specified axis.
//
//------------------------------------------------------------
double QwtPlot::invTransform(int axis, int pos) const
{
    if (axisValid(axis))
       return(d_map[axis].invTransform(pos));
    else
       return 0.0;
}


//------------------------------------------------------------
//
//.F	QwtPlot::transform
//	Transform a value into a coordinate in the plotting region
//
//.u	Syntax
//.f	int QwtPlot::transform(int axis, double value) const
//
//.u	Parameters
//.p	int axis	--	axis index
//	double value	--	value
//
//.u	Return value
//	X or y coordinate in the plotting region corresponding
//	to the value.
//------------------------------------------------------------
int QwtPlot::transform(int axis, double value) const
{
    if (axisValid(axis))
       return(d_map[axis].transform(value));
    else
       return 0;
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::setAxisFont
//	Change the font of an axis
//
//.u	Syntax
//.f	void QwtPlot::setAxisFont(int axis, QFont &f)
//
//
//.u	Parameters
//.p	int axis	--	axis index
//	QFont &f	--	font
//
//.u	Note
//	This function changes the font of the tick labels,
//	not of the axis title.
//------------------------------------------------------------
void QwtPlot::setAxisFont(int axis, const QFont &f)
{
    if (axisValid(axis))
	d_scale[axis]->setFont(f);
}

//------------------------------------------------------------
//
//.F	QwtPlot::setAxisMargins
//	Assign margins to a specified axis
//
//.u	Syntax
//.f	void QwtPlot::setAxisMargins(int axis, double mlo, double mhi)
//
//
//.u	Parameters
//.p	int axis	--	index of the axis
//	double mlo	--	margin at the lower end of the scale
//	double mhi	--	margin at the upper end of the scale
//
//.u	Note
//     If the scale is logarithmic, the margins are measured in
//		decades
//
//.u    See also
//	@^QwtAutoScale#QwtAutoScale::setMargins@
//------------------------------------------------------------
void QwtPlot::setAxisMargins(int axis, double mlo, double mhi)
{
    if (axisValid(axis))
    {
	d_as[axis].setMargins(mlo, mhi);
	autoRefresh();
    }
}

//------------------------------------------------------------
//
//.F	QwtPlot::setAxisOptions
//	Reset scale options and set specified options
//	for a specified axis
//
//.u	Syntax
//.f	void QwtPlot::setAxisOptions(int axis, int opt)
//
//.u	Parameters
//.p	int axis	--	axis index
//	int opt		--	set of options
//
//
//.u	See also
//       @^QwtAutoScale#QwtAutoScale::setOptions@
//------------------------------------------------------------
void QwtPlot::setAxisOptions(int axis, int opt)
{
    if (axisValid(axis))
    {
	d_as[axis].setOptions(opt);
	autoRefresh();
    }
}


//------------------------------------------------------------
//
//.F	QwtPlot::setAxisAutoScale
//	Enable autoscaling for a specified axis
//
//.u	Syntax
//.f	void QwtPlot::setAxisAutoScale(int axis)
//
//.u	Parameters
//.p	int axis	-- axis index
//
//.u	Description
//	This member function is used to switch back to autoscaling mode
//	after a fixed scale has been set. Autoscaling is enabled by default.
//.u	See also
//	@QwtPlot::setAxisScale@
//------------------------------------------------------------
void QwtPlot::setAxisAutoScale(int axis)
{
    if (axisValid(axis))
    {
	d_as[axis].setAutoScale();
	autoRefresh();
    }
}

//------------------------------------------------------------
//
//.F	QwtPlot::setAxisScale
//	Disable autoscaling and specify a fixed scale for
//	a selected axis.
//
//.u	Syntax
//.f	void QwtPlot::setAxisScale(int axis, double min, double max, double step = 0)
//
//.u	Parameters
//.p	int axis	--	axis index
//	double min, double max -- minimum and maximum of the scale
//	double step	-- Major step size. If step == 0, the step size is
//				calculated automatically using the maxMajor
//				setting.
//.u	See also
//	@QwtPlot::setAxisMaxMajor@
//
//------------------------------------------------------------
void QwtPlot::setAxisScale(int axis, double min, double max, double step)
{
    if (axisValid(axis))
    {
	d_as[axis].setScale(min,max,step);
	autoRefresh();
    }
    
}


//------------------------------------------------------------
//
//.F	QwtPlot::setAxisTitleFont
//	Change the title font of a selected axis
//
//.u	Syntax
//.f	void QwtPlot::setAxisTitleFont(int axis, const QFont &f)
//
//.u	Parameters
//.p	int axis, const QFont &f
//
//------------------------------------------------------------
void QwtPlot::setAxisTitleFont(int axis, const QFont &f)
{
    if (axisValid(axis))
	d_scale[axis]->setTitleFont(f);
}



//------------------------------------------------------------
//
//.F	QwtPlot::setAxisMaxMinor
//	Set the maximum number of minor scale intervals
//	for a specified axis
//
//.u	Syntax
//.f	void QwtPlot::setAxisMaxMinor(int axis, int maxMinor)
//
//.u	Parameters
//.p	int axis	-- axis index
//	int maxMinor	-- maximum number of minor steps
//
//.u	See also
//	@^QwtAutoScale#QwtAutoScale::setMaxMinor@
//------------------------------------------------------------
void QwtPlot::setAxisMaxMinor(int axis, int maxMinor)
{
    if (axisValid(axis))
    {
	d_as[axis].setMaxMinor(maxMinor);
	autoRefresh();
    }
}

//------------------------------------------------------------
//
//.F	QwtPlot::setAxisMaxMajor
//	Set the maximum number of major scale intervals
//	for a specified axis
//
//.u	Syntax
//.f	void QwtPlot::setAxisMaxMajor(int axis, int maxMajor)
//
//.u	Parameters
//.p	int axis	-- axis index
//	int maxMajor	-- maximum number of major steps
//
//.u	See also
//	@QwtAutoScale::setMaxMajor@
//------------------------------------------------------------
void QwtPlot::setAxisMaxMajor(int axis, int maxMajor)
{
    if (axisValid(axis))
    {
	d_as[axis].setMaxMajor(maxMajor);
	autoRefresh();
    }
}

//------------------------------------------------------------
//
//.F	QwtPlot::setAxisReference
//	Set a reference value for a specified axis
//
//.u	Syntax
//.f	void QwtPlot::setAxisReference(int axis, double value)
//
//.u	Parameters
//.p	int axis	-- axis index
//	double value	-- reference value
//
//.u	Description
//	The reference value is used by some autoscaling modes.
//
//.u	See also
//	@^QwtAutoScale#QwtAutoScale::setReference@, @QwtPlot:.setAxisOptions@
//
//------------------------------------------------------------
void QwtPlot::setAxisReference(int axis, double value)
{
    if (axisValid(axis))
    {
	d_as[axis].setReference(value);
	autoRefresh();
    }
}

//------------------------------------------------------------
//.F	QwtPlot::setAxisTitle
//	Change the title of a specified axis
//
//.u	Syntax
//.f	void QwtPlot::setAxisTitle(int axis, const char *t)
//
//.u	Parameters
//.p	int axis	-- axis index
//	const char *t	-- axis title
//
//------------------------------------------------------------
void QwtPlot::setAxisTitle(int axis, const char *t)
{
    if (axisValid(axis))
    {
	d_scale[axis]->setTitle(t);
	autoRefresh();
    }
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::verifyXAxis
//	Check if an index is a valid  x axis
//
//.u	Access
//	protected
//
//.u	Syntax
//.f	int QwtPlot::verifyXAxis(int axis) const
//
//.u	Parameters
//.p	int axis	-- axis index
//
//.u	Return Value
//	  Returns its input value if the specified axis is an x axis.
//	  Returns the default x axis if not.
//
//------------------------------------------------------------
int QwtPlot::verifyXAxis(int axis) const
{
    if ((axis == QwtPlot::xBottom) || (axis == QwtPlot::xTop))
	return axis;
    else
       return DefaultXAxis;
}


//------------------------------------------------------------
//
//.F	QwtPlot::verifyYAxis
//	Checks if an integer is a valid index for a y axis
//
//.u	Syntax
//.f	int QwtPlot::verifyYAxis(int axis) const
//
//.u	Access
//	protected
//
//.u	Parameters
//.p	int axis
//
//.u	Return Value
//	  Returns its input value if the specified axis is an x axis.
//	  Returns the default x axis if not.
//
//------------------------------------------------------------
int QwtPlot::verifyYAxis(int axis) const
{
    if ((axis == QwtPlot::yLeft) || (axis == QwtPlot::yRight))
	return axis;
    else
       return DefaultYAxis;
}










