//------------------------------------------------------------
//.H QwtPlot | 3 | 01/02/98 | Qwt Widget Library | Qwt Programmer's Manual
//.I plot The QwtPlot widget
//.U NAME
//	Qwtplot - A 2-D plotting widget
//
//.U SYNOPSIS
//	#include<qwt_plot.h>
//
//.U INHERITED CLASSES 
//	QFrame
//
//.U DESCRIPTION
//
//	QwtPlot is a widget for plotting two-dimensional graphs.
//	An unlimited number of data pairs can be displayed as
//	curves in different styles and colors. A plot can have
//	up to four axes, with each curve attached to an x- and
//	a y axis. The scales at the axes are dimensioned automatically
//	using an algorithm which can be configured separately for each
//	axis. Linear and logarithmic scaling is supported. Markers of
//	different styles can be added to the plot. 
//.P
//	Curves and markers are identified by unique keys which are generated
//	automatically when a curve or a marker is created. These keys are used to
//	access the properties of the corresponding curves and markers.
//.P
//	A QwtPlot widget can have up to four axes which are indexed by
//	the constants QwtPlot::yLeft, QwtPlot::yRight, QwtPlot::xTop,
//	and QwtPlot::xBottom. Curves, markers, and the grid must be
//	attached to an x axis and a y axis (Default: yLeft and xBottom). 
//
//.U USAGE
//	The following example shows (schematically) the most simple
//	way to use QwtPlot. By default, only the left and bottom axes are
//	visible and their scales are computed automatically.
//.c
//      #include <qwt_plot.h>
//
//      QwtPlot *myPlot;
//      long curve1, curve2;                    // keys
//      double x[100], y1[100], y2[100];        // x and y values
//
//      myPlot = new QwtPlot("Two Graphs", parent, name);
//
//      // add curves
//      curve1 = myPlot->insertCurve("Graph 1");
//      curve2 = myPlot->insertCurve("Graph 2");
//
//      getSomeValues(x, y1, y2);
//
//      // copy the data into the curves
//      myPlot->setCurveData(curve1, x, y1, 100);       
//      myPlot->setCurveData(curve2, x, y2, 100);
//
//      // finally, refresh the plot
//      myPlot->replot();               
//
//
//.U PUBLIC MEMBERS
//.u General
//.R
//
//	QwtPlot::QwtPlot (1) -- Constructor
//	QwtPlot::QwtPlot (2) -- Constructor
//	QwtPlot::~QwtPlot -- Destructor
//	QwtPlot::setTitle -- Change the title
//	QwtPlot::setTitleFont -- Specify the title font
//	QwtPlot::enableOutline -- Enable/disable outline drawing
//	QwtPlot::setPlotBackground -- Set a background color for the plotting region
//	QwtPlot::setPlotBorder -- Set border width for the plotting region
//	QwtPlot::setOutlineStyle -- Specify the outline style
//	QwtPlot::setOutlinePen -- Assign a pen for outline drawing
//	QwtPlot::setAutoReplot -- Enable automatic replot
//	QwtPlot::outlineEnabled -- Enable outline drawing
//	QwtPlot::autoReplot -- Return TRUE if automatic replot enabled
//	QwtPlot::plotBackground -- Return background color of the plotting region
//	QwtPlot::plotBorder -- Return border width of the plotting region 
//	QwtPlot::outlinePen -- Return pen for outline drawing
//	QwtPlot::outlineStyle -- Return outline style
//	QwtPlot::print -- Print to file or printer
//	QwtPlot::title -- Return the title
//	QwtPlot::titleFont -- Return the title font
//	QwtPlot::axisValid -- Return TRUE if a number is a valid axis key
//	QwtPlot::clear -- Remove all curves and markers
//
//.u Curves
//.R
//	QwtPlot::insertCurve -- Insert a new curve
//	QwtPlot::removeCurve -- Remove a curve
//	QwtPlot::removeCurves -- Remove all curves
//	QwtPlot::setCurveData -- Assign data to a curve
//	QwtPlot::setCurveOptions -- Specify curve
//	QwtPlot::setCurvePen -- Set Style options
//	QwtPlot::setCurveRawData -- Attach data
//	QwtPlot::setCurveSplineSize -- Set spline size
//	QwtPlot::setCurveStyle -- Set curve style
//	QwtPlot::setCurveSymbol -- Assign a symbol
//	QwtPlot::setCurveTitle -- Assign a title
//	QwtPlot::setCurveXAxis -- Attach an x axis
//	QwtPlot::setCurveYAxis -- Attach a y axis
//	QwtPlot::setCurveBaseline -- Set baseline value
//	QwtPlot::curveKeys -- Return all valid curve keys
//	QwtPlot::curveOptions -- Return style options
//	QwtPlot::curvePen -- Return pen
//	QwtPlot::curveSplineSize -- Return spline size
//	QwtPlot::curveStyle -- Return style
//	QwtPlot::curveSymbol -- Return symbol
//	QwtPlot::curveTitle -- Return title
//	QwtPlot::curveXAxis -- Return attached x axis
//	QwtPlot::curveYAxis -- Return attached y axis
//	QwtPlot::curveBaseline -- Return baseline value
//	QwtPlot::closestCurve (1) -- Find closest curve to a given location
//	QwtPlot::closestCurve (2) -- Find closest data point to a given location
//
//.u Grid
//.R
//	QwtPlot::enableGridX -- Enable or disable vertical gridlines
//	QwtPlot::enableGridXMin -- Enable or disable minor vertical gridlines
//	QwtPlot::enableGridY -- Enable or disable horizontal gridlines
//	QwtPlot::enableGridYMin -- Enable or disable minor horizontal gridlines
//	QwtPlot::setGridXAxis -- Attach an x axis
//	QwtPlot::setGridYAxis -- Attach a y axis
//	QwtPlot::setGridPen -- Specify pen for all gridlines
//	QwtPlot::setGridMajPen -- Specify pen for major gridlines
//	QwtPlot::setGridMinPen -- Specify pen for minor gridlines
//	QwtPlot::gridXAxis -- Return attached x axis
//	QwtPlot::gridYAxis -- Return attached y axis
//	QwtPlot::gridMinPen -- Return pen for minor gridlines
//	QwtPlot::gridMajPen -- Return pen for major gridlines
//
//.u Axes
//.R
//	QwtPlot::changeAxisOptions -- Change options
//	QwtPlot::enableAxis -- Enable/disable axis
//	QwtPlot::setAxisFont -- Specify font for tick labels
//	QwtPlot::setAxisMargins -- Specify margins
//	QwtPlot::setAxisOptions -- Specify options
//	QwtPlot::setAxisAutoScale -- Enable autoscaling
//	QwtPlot::setAxisScale -- Set fixed scale
//	QwtPlot::setAxisTitleFont -- Set Title font
//	QwtPlot::setAxisMaxMinor -- Set max. number of major tickmarks
//	QwtPlot::setAxisMaxMajor -- Set max. number of minor tickmarks
//	QwtPlot::setAxisReference -- Set a reference value
//	QwtPlot::setAxisTitle -- Assign a title
//	QwtPlot::invTransform -- Transform position to value
//	QwtPlot::transform -- Transform value to position
//	QwtPlot::axisAutoScale -- Return TRUE if autoscaling enabled
//	QwtPlot::axisEnabled -- Return TRUE if axis enabled
//	QwtPlot::axisFont -- Return font
//	QwtPlot::axisMargins -- Return margins
//	QwtPlot::axisMaxMajor -- Return max. number of major tickmarks
//	QwtPlot::axisMaxMinor -- Return max. number of minor tickmarks
//	QwtPlot::axisOptions -- Return options
//	QwtPlot::axisReference -- Return reference value
//	QwtPlot::axisScale -- Return scale division
//	QwtPlot::axisTitle -- Return title
//	QwtPlot::axisTitleFont -- Return the title's font
//
//.u Markers
//.R
//	QwtPlot::insertMarker -- Insert a new marker
//	QwtPlot::insertLineMarker -- Insert horizontal or vertical line
//	QwtPlot::removeMarker -- Remove a marker
//	QwtPlot::removeMarkers -- Remove all markers
//	QwtPlot::setMarkerFont -- Specify font for the label
//	QwtPlot::setMarkerLinePen -- Specify pen for line drawing
//	QwtPlot::setMarkerLineStyle -- Specify line style
//	QwtPlot::setMarkerPos -- Set (x,y) position
//	QwtPlot::setMarkerSymbol -- Assign a symbol
//	QwtPlot::setMarkerLabel -- Assign a label
//	QwtPlot::setMarkerLabelAlign -- Specify label alignment
//	QwtPlot::setMarkerLabelPen -- Specify pen for label 
//	QwtPlot::setMarkerXPos -- Change x position
//	QwtPlot::setMarkerYPos -- Change y position
//	QwtPlot::setMarkerPen -- Specify a pen for label and lines
//	QwtPlot::setMarkerXAxis -- Attach an x axis
//	QwtPlot::setMarkerYAxis -- Attach a y axis
//	QwtPlot::markerKeys -- Return all valid marker keys
//	QwtPlot::markerFont -- Return font
//	QwtPlot::markerLinePen -- Return line pen
//	QwtPlot::markerLineStyle -- Return [line style
//	QwtPlot::markerPos -- Return (x,y) position
//	QwtPlot::markerSymbol -- Return symbol
//	QwtPlot::markerLabel -- Return label text
//	QwtPlot::markerLabelAlign -- Return label alignment
//	QwtPlot::markerLabelPen -- Return label pen
//	QwtPlot::markerXAxis -- Return attached x axis
//	QwtPlot::markerYAxis -- Return attached y axis
//	QwtPlot::closestMarker -- Find closest marker to a given position
//
//.u Legend
//.R
//	QwtPlot::enableLegend -- Enable legend
//	QwtPlot::setLegendPos -- Set relative position of the legend
//	QwtPlot::setLegendFont -- Specify font for the legend items
//	QwtPlot::setLegendFrameStyle -- Specify frame style of the legend
//	QwtPlot::legendEnabled -- Return TRUE if legend enabled
//	QwtPlot::legendPos -- Return legend position
//	QwtPlot::legendFrameStyle -- Return frame style
//	QwtPlot::legendFont -- Return font
//
//	
//
//.U SIGNALS
//.R
//	QwtPlot::plotMousePressed -- Mouse pressed in plotting region
//	QwtPlot::plotMouseReleased -- Mouse released in plotting region
//	QwtPlot::plotMouseMoved -- Mouse moved in plotting region
//	QwtPlot::legendClicked -- Legend item clicked
//
//.U SLOTS
//.R
//	QwtPlot::replot -- Update the plot
//
//.U EXAMPLES
//	see examples/simple_plot, examples/rate_plot and examples/bode
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
// 	 
//------------------------------------------------------------

//------------------------------------------------------------
//.C MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

#include "qwt_plot.h"
#include "qwt_plot_dict.h"
#include "qwt_rect.h"
#include "qwt_scale.h"
#include "qwt_legend.h"
#include "qwt_plot_pixframe.h"
#include <qlabel.h>
#include <qwt_math.h>
#include <qwmatrix.h>
#include <qpdevmet.h>
#include <qbitmap.h>

const int QwtPlot::DefaultXAxis = QwtPlot::xBottom;
const int QwtPlot::DefaultYAxis = QwtPlot::yLeft;
const QwtSymbol QwtPlot::DummySymbol;
const QPen QwtPlot::DummyPen(NoPen);
const QFont QwtPlot::DummyFont;
const QString QwtPlot::DummyString("");
const QwtScaleDiv QwtPlot::DummyScale;
static const int ScrBarWidth = 20;

//------------------------------------------------------------
//.C Constructors / Destructors
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::QwtPlot (1)
//      Constructor
//
//.u	Syntax
//.f	 QwtPlot::QwtPlot(QWidget *p, const char * name)
//
//.u	Parameters
//.p	QWidget *p	--	parent
//	const char *name --	name
//
//
//------------------------------------------------------------
QwtPlot::QwtPlot(QWidget *p, const char *name)
: QFrame(p,name), d_grid(this)
{
    initPlot();
}


//------------------------------------------------------------
//
//.F	QwtPlot::QwtPlot (2)
//	Constructor
//
//.u	Syntax
//.f	 QwtPlot::QwtPlot(const char *title, QWidget *parent, const char *name)
//
//.u	Parameters
//.p	const char *title -- title text
//	QWidget *p	-- parent widget
//	const char *name -- widget name
//
//------------------------------------------------------------
QwtPlot::QwtPlot(const char *title, QWidget *p, const char *name)
: QFrame(p,name), d_grid(this)
{
    initPlot(title);
}    

//------------------------------------------------------------
//.-
//.F	QwtPlot::initPlot
//	Initializes a QwtPlot instance
//
//.u	Syntax
//.f	void QwtPlot::initPlot(const char *title)
//
//.u	Parameters
//.p	const char *title
//
//.u	Return Value
//
//.u	Description
//
//------------------------------------------------------------
void QwtPlot::initPlot(const char *title)
{

    QString txt(title);
    
    QFont fscl("Helvetica", 10);
    QFont fttl("Helvetica", 12, QFont::Bold);
    
    d_scale[yLeft] = new QwtScale(QwtScale::Left,this);
    d_scale[yRight] = new QwtScale(QwtScale::Right,this);
    d_scale[xTop] = new QwtScale(QwtScale::Top,this);
    d_scale[xBottom] = new QwtScale(QwtScale::Bottom,this);
    d_frmPlot = new QwtPlotPixFrame(this);
    d_lblTitle = new QLabel(title, this);
    d_legend = new QwtLegend(this);

    d_plotBorder = 2;
    d_titleDist = 20;
    d_titleEnabled = (!txt.isEmpty());
    d_autoReplot = FALSE;

    d_legendPos = Qwt::Bottom;
    d_legendEnabled = FALSE;
    
    initAxes();
    
    d_lblTitle->setFont(QFont("Helvetica", 14,QFont::Bold));
    d_lblTitle->setAlignment(AlignHCenter|AlignBottom);

    d_frmPlot->setFrameStyle(QFrame::Panel|QFrame::Sunken);
    d_frmPlot->setLineWidth(2);
    d_frmPlot->setMidLineWidth(0);

    d_margin = 10;
    setLineWidth(2);
    setFrameStyle(QFrame::Panel|QFrame::Raised);
    setMinimumSize(300, 200);

    d_grid.setPen(QPen(black, 0, DotLine));
    
    d_grid.setXDiv(d_sdiv[xBottom]);
    d_grid.setYDiv(d_sdiv[yLeft]);
    d_grid.enableXMin(FALSE);
    d_grid.enableYMin(FALSE);
    d_grid.setXAxis(xBottom);
    d_grid.setYAxis(yLeft);
    
    d_curves = new QwtCurveDict;
    d_markers = new QwtMarkerDict;

    d_frmPlot->setCursor(crossCursor);
    connect(d_frmPlot, SIGNAL(pixmapResized()), SLOT(updatePixmap()));
    connect(d_legend, SIGNAL(clicked(int)), SLOT(lgdClicked(int)));
    
}

//------------------------------------------------------------
//
//.F	QwtPlot::~QwtPlot
//	Destructor
//
//.u	Syntax
//.f	 QwtPlot::~QwtPlot()
//
//------------------------------------------------------------
QwtPlot::~QwtPlot()
{
    int i;

    delete d_curves;
    delete d_markers;

    delete d_frmPlot;
    for (i=0;i<axisCnt;i++)
       delete d_scale[i];
    delete d_lblTitle;
    delete d_legend;
}


//------------------------------------------------------------
//.C	Signals
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::plotMousePressed
//	A signal which is emitted when the mouse is pressed in the
//	plotting area.
//
//.u	Syntax
//.f	void QwtPlot::plotMousePressed(const QMouseEvent &e)
//
//.u	Parameters
//.p	const QMouseEvent &e  -- Mouse event object, event coordinates
//				referring to the plotting area
//.u	See also
//	QMouseEvent in the Qt manual 
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::plotMouseMoved
//
//	A signal which is emitted when the mouse is pressed in the
//	plotting area.
//
//.u	Syntax
//.f	void QwtPlot::plotMouseMoved(const QMouseEvent &e)
//
//.u	Parameters
//.p	const QMouseEvent &e  -- Mouse event object, event coordinates
//				referring to the plotting area
//.u	See also
//	QMouseEvent in the Qt manual 
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::plotMouseReleased
//
//	A signal which is emitted when a mouse button has been
//	released in the	plotting area.
//
//.u	Syntax
//.f	void QwtPlot::plotMouseReleased(const QMouseEvent &e)
//
//.u	Parameters
//.p	const QMouseEvent &e  -- Mouse event object, event coordinates
//				referring to the plotting area
//
//.u	See also
//	QMouseEvent in the Qt manual 
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::legendClicked
//	A signal which is emitted when the user has clicked
//	on a legend item.
//
//.u	Syntax
//.f	void QwtPlot::legendClicked(long key)
//
//.u	Parameters
//.p	long key	-- Key of the curve corresponding to the
//			   selected legend item
//------------------------------------------------------------



//------------------------------------------------------------
//.C	General Properties
//------------------------------------------------------------

//------------------------------------------------------------
//.-
//.F	QwtPlot::autoRefresh
//     
//.u	Syntax
//.f	void QwtPlot::autoRefresh()
//
//------------------------------------------------------------
void QwtPlot::autoRefresh()
{
     if (d_autoReplot) replot();
}

//------------------------------------------------------------
//
//.F	QwtPlot::autoReplot
//	Returns TRUE if the autoReplot option is set.
//
//.u	Syntax
//.f	bool QwtPlot::autoReplot()
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::enableOutline
//	Enables or disables outline drawing.
//
//.u	Syntax
//.f	void QwtPlot::enableOutline(bool tf)
//
//.u	Parameters
//.p	bool tf	--	TRUE(enabled) or FALSE (disabled)
//
//.u	Description
//	When the outline feature is enabled, a shape will be drawn
//	in the plotting region  when the user presses
//	or drags the mouse. It can be used to implement crosshairs,
//	mark a selected region, etc.
//
//.u    Note
//	An outline style has to be specified. See @QwtPlot::setOutlineStyle@. 
//
//.u	See also
//	@QwtPlot::setOutlineStyle@
//
//.u	Example
//	See examples/bode
//
//------------------------------------------------------------
void QwtPlot::enableOutline(bool tf)
{
    d_frmPlot->enableOutline(tf);
}

//------------------------------------------------------------
//.-
//.F	QwtPlot::initAxes
//	initialize axes
//
//.u	Syntax
//.f	void QwtPlot::initAxes()
//
//------------------------------------------------------------
void QwtPlot::initAxes()
{
    int i;

    QFont fscl("Helvetica", 10);
    QFont fttl("Helvetica", 12, QFont::Bold);
 
    
    for(i=0;i<axisCnt;i++)
    {
	d_scale[i]->setFont(fscl);
	d_scale[i]->setTitleFont(fttl);
    }

    d_axisEnabled[yLeft] = TRUE;
    d_axisEnabled[yRight] = FALSE;
    d_axisEnabled[xBottom] = TRUE;
    d_axisEnabled[xTop] = FALSE;

    for (i=0;i<axisCnt;i++)
    {
	d_as[i].adjust(0.0,1000.0,TRUE);
	d_sdiv[i] = d_as[i].scaleDiv();
	d_map[i].setDblRange(d_sdiv[i].lBound(), d_sdiv[i].hBound(), d_sdiv[i].logScale());
	d_scale[i]->setScale(d_sdiv[i]);
    }

}

//------------------------------------------------------------
//
//.F	QwtPlot::plotBorder
//	Returns the border width of the plotting area
//
//.u	Syntax
//.f	int QwtPlot::plotBorder()
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::setAutoReplot
//	Set or reset the autoReplot option
//
//.u	Parameters
//.p	bool tf		--	TRUE or FALSE. Defaults to TRUE.
//
//.u	Return Type
//		void
//
//.u	Return Value
//
//.u	Description
//	If the autoReplot option is set, the plot will be
//	updated implicitly by manipulating member functions.
//	Since this may be time-consuming, it is recommended
//	to leave this option switched off and call replot()
//	explicitly if necessary.
//	The autoReplot option is set to FALSE by default, which
//	means that the user has to call replot() in order to make
//	changes visible.
//
//.u	See also
//	@QwtPlot::replot@
//.u	Syntax
//.f	void QwtPlot::setAutoReplot(bool tf)
//
//------------------------------------------------------------
void QwtPlot::setAutoReplot(bool tf)
{
    d_autoReplot = tf;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setTitle
//	Change the plot's title
//
//.u	Syntax
//.f	void QwtPlot::setTitle(const char *t)
//
//.u	Parameters
//.p	const char *t -- new title
//
//------------------------------------------------------------
void QwtPlot::setTitle(const char *t)
{
    QString txt(d_lblTitle->text());
    QString ntxt(t);
    
    d_lblTitle->setText(t);

    if (txt.isEmpty() && (!ntxt.isEmpty()))
    {
	d_titleEnabled = TRUE;
	resize(size());
    }
    else if (ntxt.isEmpty() && (!txt.isEmpty()))
    {
	d_titleEnabled = FALSE;
	resize(size());
    }
	
}


//------------------------------------------------------------
//
//.F	QwtPlot::setTitleFont
//	Change the title font
//
//.u	Syntax
//.f	void QwtPlot::setTitleFont(const QFont &f)
//
//.u	Parameters
//.p	const QFont &f - new title font
//
//------------------------------------------------------------
void QwtPlot::setTitleFont(const QFont &f)
{
    d_lblTitle->setFont(f);
    this->resize(this->size());
}


//------------------------------------------------------------
//
//.F	QwtPlot::replot
//	Redraw the plot
//
//.u	Syntax
//.f	void QwtPlot::replot()
//
//
//.u	Description
//	If the autoReplot option is not set (which is the default)
//	or if any curves are attached to raw data, the plot has to
//	be refreshed explicitly in order to make changes visible.
//
//.u	See also
//	@Qwtplot::setAutoReplot@
//
//
//------------------------------------------------------------
void QwtPlot::replot()
{
    int ar = d_autoReplot;
    d_autoReplot = FALSE;
    updateAxes();
    d_frmPlot->clearPixmap();
    updatePixmap();
    d_frmPlot->updatePixmap();
    d_autoReplot = ar;
}

//------------------------------------------------------------
//.-
//.F	QwtPlot::resizeEvent
//	Qt resize event
//
//.u	Syntax
//.f	void QwtPlot::resizeEvent(QResizeEvent *e)
//
//.u	Parameters
//.p	QResizeEvent *e
//
//------------------------------------------------------------
void QwtPlot::resizeEvent(QResizeEvent *e)
{

    QRect r, rTitle, rTop, rBottom, rLeft, rRight;
    QwtRect rPlot, rLegend;
    
    int hDist, vDist, h;
    
    QFrame::resizeEvent(e);

    //
    //  Determine the minimum distances of the
    //	scale origins from the border of frmPlot.
    //
    vDist = hDist = d_frmPlot->frameWidth() + 2;

    rPlot = this->contentsRect();
    
    if (d_axisEnabled[xTop]) 
        hDist = qwtMax(hDist, d_scale[xTop]->minBorderDist());
    if (d_axisEnabled[xBottom]) 
	hDist = qwtMax(hDist, d_scale[xBottom]->minBorderDist());
    
    if (d_axisEnabled[yLeft])
       vDist = qwtMax(vDist, d_scale[yLeft]->minBorderDist());
    if (d_axisEnabled[yRight])
       vDist = qwtMax(vDist, d_scale[yRight]->minBorderDist());


    //
    // Determine the rectangles of the visible elements
    //
    if (d_legendEnabled)
    {
	switch(d_legendPos)
	{
	case Qwt::Top:
	    d_legend->setMaxCols((rPlot.width() - 2 * d_margin - ScrBarWidth)
				 / d_legend->colWidth());
	    h = qwtMin(d_legend->sizeHint().height(), rPlot.height() / 5);
	    rLegend = rPlot.cutTop(h, d_margin);
	    rLegend.cutMargin(d_margin, d_margin,0,0);
	    break;
        case Qwt::Right:
	    d_legend->setMaxCols(1);
	    h = qwtMin(d_legend->sizeHint().width(), rPlot.width() / 5);
	    rLegend = rPlot.cutRight(h, 0, d_margin);
	    break;
	case Qwt::Left:
	    d_legend->setMaxCols(1);
	    h = qwtMin(d_legend->sizeHint().width(), rPlot.width() / 5);
	    rLegend = rPlot.cutLeft(h, d_margin);
	    rLegend.cutMargin(0,0,d_margin, d_margin);
	    break;
	default:
	    d_legend->setMaxCols( ( rPlot.width() - 2 * d_margin - ScrBarWidth )
				 / d_legend->colWidth() );
	    h = qwtMin(d_legend->sizeHint().height(), rPlot.height() / 5);
	    rLegend = rPlot.cutBottom(h, 0, d_margin);
	    rLegend.cutMargin(d_margin, d_margin,0,0);
	    break;
	}
    }
    
    if (d_titleEnabled)
       rTitle = rPlot.cutTop(d_lblTitle->sizeHint().height(), d_margin,
			     d_margin);
    else
       rTitle = rPlot.cutTop(d_margin);

    if (d_axisEnabled[xTop])
       rTop = rPlot.cutTop(d_scale[xTop]->sizeHint().height());

    if (d_axisEnabled[xBottom])
       rBottom = rPlot.cutBottom(d_scale[xBottom]->sizeHint().height(), 0, d_margin);
    else
       rPlot.cutBottom(qwtMax(d_margin, vDist));
    
    if (d_axisEnabled[yLeft])
       rLeft = rPlot.cutLeft(d_scale[yLeft]->sizeHint().width());
    else
       rLeft = rPlot.cutLeft(qwtMax(d_margin, hDist));
    
    if (d_axisEnabled[yRight])
       rRight = rPlot.cutRight(d_scale[yRight]->sizeHint().width());
    else
       rRight = rPlot.cutRight(qwtMax(d_margin, hDist)) ;


    //
    // Center scales and title
    //
    rTop.setLeft(rPlot.x() + vDist - hDist);
    rTop.setRight(rPlot.right() + hDist - vDist);
    rBottom.setLeft(rTop.left());
    rBottom.setRight(rTop.right());

    rTitle.setWidth(rPlot.width() + 2 * qwtMin(rLeft.width(), rRight.width()));
    rTitle.moveBy( rPlot.center().x() - rTitle.center().x(), 0 );

    //
    // Correct legend position
    //
    if (d_legendEnabled)
    {
	switch(d_legendPos)
	{

	case Qwt::Right:
	case Qwt::Left:

	    if (rLegend.height() > d_legend->sizeHint().height() + ScrBarWidth)
	    {
		// adjust the height; prevent scrollbars where possible
		if (rLegend.width() >= d_legend->sizeHint().width()) // has no hor. scrollbar
		   rLegend.setHeight(d_legend->sizeHint().height());
		else
		   rLegend.setHeight(d_legend->sizeHint().height() + ScrBarWidth);
		
		// put it at same the y position as rPlot if useful; center otherwise 
		if (rLegend.height() <= rPlot.height())
		    rLegend.moveBy(0, rPlot.y() - rLegend.y());
		else
		    rLegend.moveBy(0, contentsRect().center().y() - rLegend.center().y());
	    }
	    break;

	default:

	    if (rLegend.width() > (d_legend->sizeHint().width() + ScrBarWidth))
	    {
		// adjust the width
		if (rLegend.height() >= d_legend->sizeHint().height()) // has no vert. scrollbar
		   rLegend.setWidth(d_legend->sizeHint().width());
		else
		   rLegend.setWidth(d_legend->sizeHint().width() + ScrBarWidth);

		// put it in the center 
		if (rLegend.width() <= (rPlot.width() + 2 * qwtMin(rLeft.width(), rRight.width())))
		    rLegend.moveBy(rPlot.center().x() - rLegend.center().x(), 0);
		else
		    rLegend.moveBy(contentsRect().center().x() - rLegend.center().x(), 0);
	    }
	    break;
	}
    }
    
    
    //
    // resize and show the visible widgets 
    //
    if (d_titleEnabled)
    {
	d_lblTitle->setGeometry(rTitle);
	if (!d_lblTitle->isVisible()) d_lblTitle->show();
    }
    else
       d_lblTitle->hide();
    
    if (d_axisEnabled[yLeft])
    {
	d_scale[yLeft]->setBorderDist(vDist, vDist);
	d_scale[yLeft]->setGeometry(rLeft);
	if (!d_scale[yLeft]->isVisible()) d_scale[yLeft]->show();
    }
    else
       d_scale[yLeft]->hide();
 
    if (d_axisEnabled[yRight])
    {
	d_scale[yRight]->setBorderDist(vDist, vDist);
	d_scale[yRight]->setGeometry(rRight);
	if (!d_scale[yRight]->isVisible()) d_scale[yRight]->show();
    }
    else
       d_scale[yRight]->hide();
    
    if (d_axisEnabled[xTop])
    {
	d_scale[xTop]->setBorderDist(hDist, hDist);
	d_scale[xTop]->setGeometry(rTop);
	if (!d_scale[xTop]->isVisible()) d_scale[xTop]->show();
    }
    else
       d_scale[xTop]->hide();

    if (d_axisEnabled[xBottom])
    {
	d_scale[xBottom]->setBorderDist(hDist, hDist);
	d_scale[xBottom]->setGeometry(rBottom);
	if (!d_scale[xBottom]->isVisible()) d_scale[xBottom]->show();
    }
    else
       d_scale[xBottom]->hide();


    if (d_legendEnabled)
    {
	d_legend->setGeometry(rLegend);
	d_legend->show();
    }
    else
       d_legend->hide();
    
    
    //
    // update the axes.
    //
    updateAxes();
    
    //
    // adjust the maps in order to fit the pixmap rectangle
    //
    r.setRect(0,0,rPlot.width() - 2*d_frmPlot->frameWidth(),
	      rPlot.height() - 2*d_frmPlot->frameWidth());
    vDist = qwtMax(0,vDist - d_frmPlot->frameWidth());
    d_map[xBottom].setIntRange(vDist, r.right() - vDist);
    d_map[xTop].setIntRange(vDist, r.right() - vDist);
    d_map[yLeft].setIntRange(r.bottom() - vDist, vDist);
    d_map[yRight].setIntRange(r.bottom() - vDist, vDist);

    d_frmPlot->setGeometry(rPlot);

}



//------------------------------------------------------------
//.-
//.F	QwtPlot::updateAxes
//	Rebuild the scales and maps
//
//.u	Syntax
//.f	void QwtPlot::updateAxes()
//
//------------------------------------------------------------
void QwtPlot::updateAxes()
{
    int i;
    QwtPlotCurve *c;

    QIntDictIterator<QwtPlotCurve> itc(*d_curves);

    //
    //  Adjust autoscalers
    //
    
    for (i=0; i<axisCnt; i++)
       d_as[i].reset();

    itc.toFirst();
    while((c = itc.current()))
    {
	if (c->dataSize() > 0)	// don't scale curves with no data
	{			//  (minXValue() et al. would return 0.0)
	    d_as[c->xAxis()].adjust(c->minXValue(), c->maxXValue());
	    d_as[c->yAxis()].adjust(c->minYValue(), c->maxYValue());
	}
	++itc;
    }

    //
    // Adjust scales
    //
    for (i=0; i<axisCnt; i++)
    {
	d_sdiv[i] = d_as[i].scaleDiv();
	d_map[i].setDblRange(d_sdiv[i].lBound(), d_sdiv[i].hBound(), d_sdiv[i].logScale());
	d_scale[i]->setScale(d_sdiv[i]);
    }

    d_grid.setXDiv(d_sdiv[d_grid.xAxis()]);
    d_grid.setYDiv(d_sdiv[d_grid.yAxis()]);
    
}


//------------------------------------------------------------
//.-
//.u	QwtPlot::updatePixmap
//	Redraw grid, curves, and markers
//
//.u	Syntax
//.f	void QwtPlot::updatePixmap()
//
//------------------------------------------------------------
void QwtPlot::updatePixmap()
{

    QPainter p;
    QwtPlotCurve *curCurve;
    QwtPlotMarker *curMarker;
    QRect pixrect = d_frmPlot->pixmap()->rect();

    if (pixrect.isEmpty()) return;
    
    p.begin(d_frmPlot->pixmap());
    p.setClipping(TRUE);
    p.setClipRect(d_frmPlot->pixmap()->rect());

    //
    // draw grid
    //
    
    d_grid.draw(&p, pixrect, d_map[d_grid.xAxis()], d_map[d_grid.yAxis()]); 

     
    //
    //  draw curves
    //
    QIntDictIterator<QwtPlotCurve> itc(*d_curves);
    itc.toFirst();
    while((curCurve = itc.current()))
    {
	curCurve->draw(&p, d_map[curCurve->xAxis()], d_map[curCurve->yAxis()]);
	++itc;
    }
    
    //
    // draw markers
    //
    QIntDictIterator<QwtPlotMarker> itm(*d_markers);
    itm.toFirst();
    while((curMarker = itm.current()))
    {
    	curMarker->draw(&p,
			d_map[curMarker->xAxis()].transform(curMarker->xValue()),
			d_map[curMarker->yAxis()].transform(curMarker->yValue()),
			pixrect);
	++itm;
    }

   p.end();

}

//------------------------------------------------------------
//
//.F	QwtPlot::setPlotBackground
//	Change the background of the plotting area
//
//.u	Syntax
//.f	void QwtPlot::setPlotBackground(const QColor &c)
//
//.u	Parameters
//.p	const QColor &c	-- new background color
//
//------------------------------------------------------------
void QwtPlot::setPlotBackground(const QColor &c)
{
    d_frmPlot->setPixmapBackground(c);
    autoRefresh();
}

//------------------------------------------------------------
//
//.F	QwtPlot::setPlotBorder
//	Change the border width of the plotting area
//
//.u	Syntax
//.f	void QwtPlot::setPlotBorder(int w)
//
//.u	Parameters
//.p	int w -- new border width
//
//------------------------------------------------------------
void QwtPlot::setPlotBorder(int w)
{
    w = qwtLim(w, 0, 10);
    d_plotBorder = w;
    d_frmPlot->setLineWidth(w);
    this->resize(this->size());
}


//------------------------------------------------------------
//
//.F	QwtPlot::setOutlineStyle
//	Specify the style of the outline
//
//.u	Syntax
//.f	void QwtPlot::setOutlineStyle(Qwt::Shape os)
//
//.u	Parameters
//.p	Qwt::Shape os -- Outline Style. Valid values are: Qwt::HLine,
//			 Qwt::VLine, Qwt::Cross, Qwt::Rectangle, Qwt::Ellipse
//
//.u	Description
//	The outline style determines which kind of shape is drawn
//	in the plotting region when the user presses a mouse button
//	or drags the mouse. Valid Styles are:
//.t
//	Qwt::Cross      --	Cross hairs are drawn across the plotting area
//				when the user presses a mouse button. The lines
//				intersect at the point where the mouse was pressed
//				and move with the mouse pointer.
//	Qwt::HLine, Qwt::VLine -- A horizontal or vertical line appears when
//				the user presses a mouse button. This is useful
//				for moving line markers.
//	Qwt::Rectangle  --	A rectangle is displayed when the user drags
//				the mouse. One corner is fixed at the point where
//				the mouse was pressed, and the opposite corner moves
//				with the mouse pointer. This can be used for selecting
//				regions.
//	Qwt::Ellipse  --	Similar to Qwt::Rectangle, but with an ellipse inside
//				a bounding rectangle.
//
//.u	See also
//	@QwtPlot::enableOutline@
//
//------------------------------------------------------------
void QwtPlot::setOutlineStyle(Qwt::Shape os)
{
    d_frmPlot->setOutlineStyle(os);
}



//------------------------------------------------------------
//
//.F	QwtPlot::setOutlinePen
//	Specify a pen for the outline
//
//.u	Syntax
//.f	void QwtPlot::setOutlinePen(const QPen &pn)
//
//.u	Parameters
//.p	const QPen &pn	-- new pen
//
//------------------------------------------------------------
void QwtPlot::setOutlinePen(const QPen &pn)
{
    d_frmPlot->setOutlinePen(pn);
}




//------------------------------------------------------------
//
//.F	QwtPlot::print
//	Print the plot to a QPrinter
//
//.u	Syntax
//.f	void QwtPlot::print(QPrinter &pr,  const QwtColorFilter &cfilter = NoColorFilter)
//
//.u	Parameters
//.p	const QPrinter pr	--	The printer
//	const QwtColorFilter &cfilter -- color filter for curves and markers
//
//.u	Description
//	This function prints the contents of a QwtPlot instance to
//	QPrinter object. The printed output is
//	independent of the widget displayed on the screen.
//
//.u	Note
//.i
//	--  The background of the plot is not printed.
//	--  Scales and titles are always printed in black color.
//	--  major gridlines are printed in dark gray; minor gridlines are
//	    printed in gray.
//      --  Curves, markers, symbols, etc. are printed in their screen colors
//	    if no color filter is specified. A color filter may be useful if
//	    curves have bright colors (which would be hardly visible), or
//	    if a grayscale output is desired.
//	--  Font sizes and line widths are not scaled.
//	--  The size of the printed plot is determined automatically,
//	    dependent on the page size.
//
//
//.u	Example calls
//.c
//
//	myPlot->print(printer);			 // screen colors
//	myPlot->print(printer, QwtFltrDim(200)); // colors darkened
//	myPlot->print(printer, QwtFltrInv());	 // colors inverted
//
//.u	Bugs
//.i
//	-- The legend is not printed.
//	-- QPrinter has some bugs in Qt 1.2 (see Qt manual on QPrinter).
//
//.u	See also
//	@^QwtColorFilter@
//------------------------------------------------------------
void QwtPlot::print(QPrinter &pr, const QwtColorFilter &cfilter)
{

    
    QRect r, rPlot;
    QPen p1, p2,p3;			// buffers for some pens
    
    int th,w,i;

    int arf = d_autoReplot;
    d_autoReplot = FALSE;
    
    QwtPlotCurve *curCurve;
    QwtPlotMarker *curMarker;
    
    QPaintDeviceMetrics mpr(&pr);
    QwtScaleDraw sd[axisCnt];
    QwtDiMap map[axisCnt];
    QwtCurve cpCurve;
    QwtMarker cpMarker;
    QwtSymbol cpSym;
    QwtGrid cpGrid;
    QPen cpPen;
    QBrush cpBrush;
    pr.setDocName(d_lblTitle->text());
    pr.setCreator("QwtPlot");
    int printMargin;

    int ttpos = 0, btpos = 0;
     
    QPainter p;

    printMargin = qwtMin(mpr.width(), mpr.height()) / 10;
    

    r = QRect(printMargin, printMargin,
		 mpr.width() - 2 * printMargin,
		 mpr.height()- 2 * printMargin );

    if (r.isNull()) return;

    
    rPlot = r;

    rPlot.setRect(r.left(),
		 r.top(),
		 r.width(), qwtMin(r.height(), int(0.66 * double(r.width()))));

    sd[yLeft].setGeometry(rPlot.left(), rPlot.top(),
			  rPlot.height(),QwtScaleDraw::Left);
    sd[yRight].setGeometry(rPlot.left(), rPlot.top(),
			   rPlot.height(),QwtScaleDraw::Right);
    sd[xTop].setGeometry(rPlot.left(), rPlot.top(),
			 rPlot.height(),QwtScaleDraw::Top);
    sd[xBottom].setGeometry(rPlot.left(), rPlot.top(),
			    rPlot.height(),QwtScaleDraw::Bottom);
    sd[yLeft].setScale(d_as[yLeft].scaleDiv());
    sd[yRight].setScale(d_as[yRight].scaleDiv());
    sd[xTop].setScale(d_as[xTop].scaleDiv());
    sd[xBottom].setScale(d_as[xBottom].scaleDiv());

    if ((mpr.width() == 0) || (mpr.height() == 0) || this->size().isNull())
       return;
    
    p.begin(&pr);

    p.setFont(d_lblTitle->font());
    p.setPen(black);


    if (d_titleEnabled)
    {
	p.drawText(rPlot.x() + ( rPlot.width() -
			    p.fontMetrics().width(d_lblTitle->text())) / 2,
		   r.y() + p.fontMetrics().ascent() + 1,
		   d_lblTitle->text());
	rPlot.setTop(rPlot.top() + p.fontMetrics().height() * 2);
    }
    th = rPlot.top();

    if (d_axisEnabled[yLeft])
    {
	p.setFont(d_scale[yLeft]->titleFont());
	w = p.fontMetrics().height() * 2;
	p.setFont(d_scale[yLeft]->font());
	w += sd[yLeft].maxWidth(&p, FALSE);
	rPlot.setLeft(rPlot.left() + w);
    }
    
    if (d_axisEnabled[yRight])
    {
	p.setFont(d_scale[yRight]->titleFont());
	w = p.fontMetrics().height() * 2;
	p.setFont(d_scale[yRight]->font());
	w += sd[yRight].maxWidth(&p, FALSE);
	rPlot.setRight(rPlot.right() - w);
    }
    
    if (d_axisEnabled[xTop])
    {
	p.setFont(d_scale[xTop]->titleFont());
	w = p.fontMetrics().height() * 2;
	ttpos = rPlot.top() + p.fontMetrics().height();
	p.setFont(d_scale[xTop]->font());
	w += sd[xTop].maxHeight(&p);
	rPlot.setTop(rPlot.top() + w);
    }

    if (d_axisEnabled[xBottom])
    {
	p.setFont(d_scale[xBottom]->titleFont());
	w = p.fontMetrics().height() * 2;
	btpos = rPlot.bottom() - p.fontMetrics().descent() - 1;
	p.setFont(d_scale[xBottom]->font());
	w += sd[xBottom].maxHeight(&p);
	rPlot.setBottom(rPlot.bottom() - w);
    }

    //
    // build maps
    //
    for (i=0;i<axisCnt;i++)
	map[i].setDblRange(d_sdiv[i].lBound(), d_sdiv[i].hBound(), d_sdiv[i].logScale());

    map[yLeft].setIntRange(rPlot.bottom(), rPlot.top());
    map[yRight].setIntRange(rPlot.bottom(), rPlot.top());
    map[xTop].setIntRange(rPlot.left(), rPlot.right());
    map[xBottom].setIntRange(rPlot.left(), rPlot.right());

    p.setPen(black);
    p.setBrush(NoBrush);
//    p.save();

    //
    // draw scales
    //
    if (d_axisEnabled[yLeft])
    {
	p.setFont(d_scale[yLeft]->font());
	sd[yLeft].setGeometry(rPlot.left(), rPlot.top(), rPlot.height(),QwtScaleDraw::Left);
	sd[yLeft].draw(&p);
	p.setFont(d_scale[yLeft]->titleFont());
	p.translate(r.left(), rPlot.bottom());
	p.rotate(-90.0);
	p.drawText(  ( rPlot.height()
		      - p.fontMetrics().width(d_scale[yLeft]->title()) )  / 2,
		   p.fontMetrics().ascent() + 1,
		   d_scale[yLeft]->title());
	p.resetXForm();
	
    }
    
    if (d_axisEnabled[yRight])
    {
	p.setFont(d_scale[yRight]->font());
	sd[yRight].setGeometry(rPlot.right(), rPlot.top(), rPlot.height(), QwtScaleDraw::Right);
	sd[yRight].draw(&p);
	
	p.setFont(d_scale[yRight]->titleFont());
	p.translate(r.right(), rPlot.top());
	p.rotate(90.0);
	p.drawText(  ( rPlot.height()
		      - p.fontMetrics().width(d_scale[yRight]->title()) ) / 2,
		   p.fontMetrics().ascent() + 1,
		   d_scale[yRight]->title());
	p.resetXForm();
    }

    if (d_axisEnabled[xTop])
    {
	p.setFont(d_scale[xTop]->font());
	sd[xTop].setGeometry(rPlot.left(), rPlot.top(), rPlot.width(),
			     QwtScaleDraw::Top);
	sd[xTop].draw(&p);

	p.setFont(d_scale[xTop]->titleFont());
	p.drawText( rPlot.left()
		   + ( rPlot.width() 
		      - p.fontMetrics().width(d_scale[xTop]->title()) ) / 2,
		   ttpos,
		   d_scale[xTop]->title());
    }

    if (d_axisEnabled[xBottom])
    {
	p.setFont(d_scale[xBottom]->font());
	sd[xBottom].setGeometry(rPlot.left(), rPlot.bottom(),
				rPlot.width(), QwtScaleDraw::Bottom);
	sd[xBottom].draw(&p);
	p.setFont(d_scale[xTop]->titleFont());
	p.drawText( rPlot.left()
		   + ( rPlot.width()
		      - p.fontMetrics().width(d_scale[xBottom]->title())) / 2,
		   btpos,
		   d_scale[xBottom]->title());
    }

//    p.restore();
//    p.save();

    //
    // draw grid
    //
    cpGrid = d_grid;
    cpPen = cpGrid.majPen();
    cpPen.setColor(darkGray);
    cpGrid.setMajPen(cpPen);
    cpPen = cpGrid.minPen();
    cpPen.setColor(gray);
    cpGrid.setMinPen(cpPen);
    
    cpGrid.draw(&p, rPlot, map[d_grid.xAxis()], map[d_grid.yAxis()]); 
    
//    p.restore();

    //
    //  draw curves
    //
    
    QIntDictIterator<QwtPlotCurve> itc(*d_curves);
    itc.toFirst();
    while((curCurve = itc.current()))
    {
	p.save();
	cpCurve = *curCurve;
	cpPen = cpCurve.pen();
	cpPen.setColor(cfilter(cpPen.color()));
	cpCurve.setPen(cpPen);
	cpCurve.draw(&p, map[curCurve->xAxis()], map[curCurve->yAxis()]);
	++itc;
	p.restore();
    }
    
    //
    // draw markers
    //
    QIntDictIterator<QwtPlotMarker> itm(*d_markers);
    itm.toFirst();
    while((curMarker = itm.current()))
    {
	p.save();
	cpMarker = *curMarker;

	cpPen = cpMarker.linePen();
	cpPen.setColor(cfilter(cpPen.color()));
	cpMarker.setLinePen(cpPen);

	cpPen = cpMarker.labelPen();
	cpPen.setColor(cfilter(cpPen.color()));
	cpMarker.setLabelPen(cpPen);
	
	cpSym = cpMarker.symbol();
	cpPen = cpSym.pen();
	cpPen.setColor(cfilter(cpPen.color()));
	cpSym.setPen(cpPen);

	cpBrush = cpSym.brush();
	cpBrush.setColor(cfilter(cpBrush.color()));
	cpSym.setBrush(cpBrush);
	cpMarker.setSymbol(cpSym);
	
    	cpMarker.draw(&p,
			map[curMarker->xAxis()].transform(curMarker->xValue()),
			map[curMarker->yAxis()].transform(curMarker->yValue()),
			rPlot);
	++itm;
	p.restore();
    }

    
    //draw bounding rectangle
    p.drawRect(rPlot.x(), rPlot.y(), rPlot.width() - 1, rPlot.height() - 1);
    
    p.end();

    d_autoReplot = arf;
}


//------------------------------------------------------------
//
//.F	QwtPlot::axisValid
//	Return TRUE if the specified axis exists.
//	Static member function. 
//
//.u	Syntax
//.f	static bool QwtPlot::axisValid(int axis)
//
//.u	Access
//	protected
//
//.u	Parameters
//.p	int axis	-- axis index
//
//------------------------------------------------------------
bool QwtPlot::axisValid(int axis)
{
    return ((axis >= QwtPlot::yLeft) && (axis < QwtPlot::axisCnt));
}


//------------------------------------------------------------
//
//.F	QwtPlot::plotBackground
//	Return the background color of the plotting area
//
//.u	Syntax
//.f	const QColor & QwtPlot::plotBackground() const
//
//------------------------------------------------------------
const QColor & QwtPlot::plotBackground() const
{
    return d_frmPlot->pixmapBackground();   
}

//------------------------------------------------------------
//
//.F	QwtPlot::outlineEnabled
//	Return TRUE if the outline feature is enabled
//
//.u	Syntax
//.f	bool QwtPlot::outlineEnabled() const
//
//------------------------------------------------------------
bool QwtPlot::outlineEnabled() const
{
     return d_frmPlot->outlineEnabled();
}

//------------------------------------------------------------
//
//.F	QwtPlot::outlinePen
//	Return the pen used to draw outlines
//
//.u	Syntax
//.f	const QPen & QwtPlot::outlinePen() const
//
//------------------------------------------------------------
const QPen & QwtPlot::outlinePen() const
{
    return d_frmPlot->outlinePen();   
}

//------------------------------------------------------------
//
//.F	QwtPlot::outlineStyle
//	Return the outline style
//
//.u	Syntax
//.f	Qwt::Shape QwtPlot::outlineStyle()
//
//.u	See also
//	@QwtPlot::setOutlineStyle@
//
//------------------------------------------------------------
Qwt::Shape QwtPlot::outlineStyle() const
{
    return d_frmPlot->outlineStyle();
}

//------------------------------------------------------------
//.-
//.F	QwtPlot::plotPressed
//	Called internally if a mouse button has been pressed.
//	Emits a plotMousePressed signal.
//
//.u	Syntax
//.f	void QwtPlot::plotPressed(QMouseEvent *e)
//
//------------------------------------------------------------
void QwtPlot::plotPressed(QMouseEvent *e)
{
    emit plotMousePressed(*e);  
}


//------------------------------------------------------------
//.-
//.F	QwtPlot::plotReleased
//
//	Called internally if a mouse button has been released.
//	Emits a plotMouseReleased signal.
//
//.u	Syntax
//.f	void QwtPlot::plotReleased(QMouseEvent *e)
//
//------------------------------------------------------------
void QwtPlot::plotReleased(QMouseEvent *e)
{
    emit plotMouseReleased(*e);
}


//------------------------------------------------------------
//.-
//.F	QwtPlot::plotMoved
//
//	Called internally if a mouse button has been released.
//	Emits a plotMouseMoved signal.
//
//.u	Syntax
//.f	void QwtPlot::plotMoved(QMouseEvent *e)
//
//------------------------------------------------------------
void QwtPlot::plotMoved(QMouseEvent *e)
{
    emit plotMouseMoved(*e);
}

//------------------------------------------------------------
//.-
//.F	QwtPlot::lgdClicked
//	Called internally when the legend has been clicked on.
//	Emits a legendClicked() signal.
//
//.u	Syntax
//.f	void QwtPlot::lgdClicked(int index)
//
//------------------------------------------------------------
void QwtPlot::lgdClicked(int index)
{
    emit legendClicked(d_legend->key(index));
}

//------------------------------------------------------------
//
//.F	QwtPlot::clear
//	Remove all curves and markers
//
//.u	Syntax
//.f	void QwtPlot::clear()
//
//------------------------------------------------------------
void QwtPlot::clear()
{
    d_legend->clear();
    d_curves->clear();
    d_markers->clear();
}


//------------------------------------------------------------
//
//.F	QwtPlot::removeCurves
//	Remove all curves
//
//.u	Syntax
//.f	void QwtPlot::removeCurves()
//
//------------------------------------------------------------
void QwtPlot::removeCurves()
{
    d_curves->clear();
    d_legend->clear();
    replot();
}

//------------------------------------------------------------
//
//.F	QwtPlot::removeMarkers
//	Remove all markers
//
//.u	Syntax
//.f	void QwtPlot::removeMarkers()
//
//------------------------------------------------------------
void QwtPlot::removeMarkers()
{
    d_markers->clear();
    replot();
}

//------------------------------------------------------------
//.C	Legend Properties
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPlot::enableLegend
//	Enable or disable the legend
//
//.u	Syntax
//.f	void QwtPlot::enableLegend(bool tf)
//
//.u	Parameters
//.p	bool tf	-- TRUE(enabled) or FALSE(disabled)
//
//.u	See also
//	@QwtPlot::setLegendPos@
//
//------------------------------------------------------------
void QwtPlot::enableLegend(bool tf)
{
    QwtPlotCurve *curCurve;
    
    if (d_legendEnabled != tf)
    {
	if (tf)
	{
	    QIntDictIterator<QwtPlotCurve> itc(*d_curves);
	    itc.toFirst();
	    while ((curCurve = itc.current()))
	    {
		d_legend->appendItem(curCurve->title(), curCurve->symbol(),
				     curCurve->pen(), itc.currentKey());
		autoRefresh();
		++itc;
	    }
	    
	}
	else
	{
	    
	    d_legend->clear();
	    autoRefresh();
	}
	
    }	
    d_legendEnabled = tf;
}

//------------------------------------------------------------
//
//.F	QwtPlot::setLegendPos
//	Specify the position of the legend within the widget
//
//.u	Syntax
//.f	void QwtPlot::setLegendPos(int pos)
//
//.u	Parameters
//.p	int pos		--	The legend's position. Valid values are
//				Qwt::Left, Qwt::Right, Qwt::Top, QwtBottom.
//------------------------------------------------------------
void QwtPlot::setLegendPos(int pos)
{
    if (pos != d_legendPos)
    {
	switch(pos)
	{
	case Qwt::Top:
	case Qwt::Bottom:
	    break;
	    
	case Qwt::Left:
	case Qwt::Right:
	    d_legend->setMaxCols(1);
	    break;
	default:
	    pos = Qwt::Bottom;
	    break;
	}
	d_legendPos = pos;
	this->resize(this->size());
    }
}

//------------------------------------------------------------
//
//.F	QwtPlot::setLegendFont
//	Change the font of the legend items
//
//.u	Syntax
//.f	void QwtPlot::setLegendFont(const QFont &f)
//
//.u	Parameters
//.p	const QFont &f -- new font
//
//------------------------------------------------------------
void QwtPlot::setLegendFont(const QFont &f)
{
    d_legend->setFont(f);
    if (d_legendEnabled) resize(this->size());
}

//------------------------------------------------------------
//
//.F	QwtPlot::setLegendFrameStyle
//	Change the legend's frame style
//
//.u	Syntax
//.f	void QwtPlot::setLegendFrameStyle(int st)
//
//.u	Parameters
//.p	int st	--	Frame Style. See Qt manual on QFrame.
//
//------------------------------------------------------------
void QwtPlot::setLegendFrameStyle(int st)
{
    d_legend->setFrameStyle(st);
}



//------------------------------------------------------------
//
//.F	QwtPlot::legendFrameStyle
//	Return the frame style of the legend
//
//.u	Syntax
//.f	int QwtPlot::legendFrameStyle() const
//------------------------------------------------------------
int QwtPlot::legendFrameStyle() const
{
    return d_legend->frameStyle(); 
}

//------------------------------------------------------------
//
//.F	QwtPlot::legendFont
//	Return the font of the legend items
//
//.u	Syntax
//.f	const QFont & QwtPlot::legendFont() const
//
//------------------------------------------------------------
const QFont& QwtPlot::legendFont() const
{
    return d_legend->font(); 
}







