#include "qwt_pixframe.h"
#include "qwt_math.h"
#include <qpainter.h>

//------------------------------------------------------------
//.H QwtPixFrame|3|02/02/1998|Qwt Widget Library|Qwt Programmer's manual
//.U NAME
//	QwtPixFrame -- A frame containing a pixmap into which
//	can be drawn.
//
//.U INHERITED CLASSES
//	QFrame
//
//.U PUBLIC MEMBERS
//.R
//	QwtPixFrame::clearPixmap -- Clear the pixmap
//	QwtPixFrame::fillPixmap -- Fill the pixmap with a specified color
//	QwtPixFrame::setPixmapBackground -- Assign a background color
//	QwtPixFrame::updatePixmap -- Repaint the pixmap
//	QwtPixFrame::updatePixmapRect -- Repaint a part of the pixmap
//	QwtPixFrame::pixmap -- Return a pointer to the embedded pixmap
//	QwtPixFrame::pixmapBackground -- Return the pixmap's background color
//
//.U SIGNALS
//.R
//	QwtPixFrame::pixmapResized -- Notify a change of the pixmap's size
//
//.U DESCRIPTION
//	QwtPixFrame is a frame which contains a pixmap. The size of the
//	pixmap is dynamically adjusted in order to fit the frame's
//	contents.
//
//.U USAGE
//.c
//      #include <qwt_pixframe.h>
//
//      class ParentClass : ...
//      {
//              ...
//              QwtPixFrame *pixframe;
//              ...
//              void newPicture();
//              ..
//      private slots:
//              void repaintPixmap();
//      }
//
//
//      //
//      //     Connect pixmapResized() to a slot
//      //
//      ParentClass::ParentClass(...)
//      {
//              ...
//              connect(pixframe, SIGNAL(pixmapResized()), SLOT(repaintPixmap()));
//      }
//
//
//      //
//      // Repaint when the pixmap has been resized
//      //
//      ParentClass::repaintPixmap()
//      {
//              QPainter p;
//              p.begin(pixframe->pixmap());
//
//              redraw_picture(&p, pixframe->pixmap().rect());
//
//              p.end();
//
//              // Note: No updatePixmap() required here
//      }
//
//      //
//      // Renew the pixmap's contents independently
//      //
//      ParentClass::newPicture()
//      {
//              // Change the pixmap's contents
//              draw_new_picture(pixframe->pixmap());
//
//              // Generate paint event
//              pixframe->updatePixmap();
//      }
//
//
//.U EXAMPLES
//	see examples/curvdemo/curvdemo2
//
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
// 	 
//------------------------------------------------------------


//------------------------------------------------------------
//
//.F	QwtPixFrame::QwtPixFrame
//	Constructs a QwtPixFrame object.
//
//.u	Syntax
//.f	 QwtPixFrame::QwtPixFrame(QWidget *parent, const char *name)
//
//.u	Parameters
//.p	QWidget *parent  -- parent widget (default = 0)
//	const char *name -- name (default = 0)
//
//------------------------------------------------------------
QwtPixFrame::QwtPixFrame(QWidget *parent, const char *name)
    : QFrame(parent, name)
{
    d_bg = backgroundColor();
    d_adjustPixmap = TRUE;
}


//------------------------------------------------------------
//
//.F	QwtPixFrame::~QwtPixFrame
//	Destroys a QwtPixFrame object.
//
//.u	Syntax
//.f	 QwtPixFrame::~QwtPixFrame()
//
//------------------------------------------------------------
QwtPixFrame::~QwtPixFrame()
{
    
}


//------------------------------------------------------------
//.-
//.F	QwtPixFrame::paintEvent
//
//.u	Syntax
//.f	void QwtPixFrame::paintEvent(QPaintEvetn *e)
//
//.u	Parameters
//.p	QPaintEvent *e
//
//------------------------------------------------------------
void QwtPixFrame::paintEvent(QPaintEvent *e)
{
    QRect r = contentsRect();
    QRect ri;
    QPainter p;

    p.begin(this);

    // redraw frame if necessary
    if (!r.contains(e->rect(), TRUE))
       drawFrame(&p);
    
    // redraw pixmap portion
    if (r.intersects(e->rect()))
    {
	ri = r.intersect(e->rect());

	//
	// repaint pixmap portion
	//
	p.drawPixmap(ri.x(), ri.y(), d_pix,
		     ri.x() - r.x(), ri.y() - r.y(),
		     qwtMin(ri.width(), d_pix.width()),
		     qwtMin(ri.height(), d_pix.height()));

    }

    p.end();
    
	
}

//------------------------------------------------------------
//.-
//.F	QwtPixFrame::setPixmap
//	Assign a pixmap.
//
//.u	Syntax
//.f	void QwtPixFrame::setPixmap(const QPixmap &pix)
//
//.u	Parameters
//.p	const QPixmap &pix -- Reference to the new pixmap
//
//.u	Description
//	If the variable size feature is enabled (disabled by default),
//	the size of the frame will not change when a new pixmap
//	has been assigned. 
//
//------------------------------------------------------------
void QwtPixFrame::setPixmap(const QPixmap &pix)
{
    d_pix = pix;
    if (!d_adjustPixmap)
       adjustFrame();
}

//------------------------------------------------------------
//.-
//.F	QwtPixFrame::adjustPixmap
//	Resize the pixmap according to the frame size.
//
//.u	Syntax
//.f	void QwtPixFrame::adjustPixmap()
//
//------------------------------------------------------------
void QwtPixFrame::adjustPixmap()
{
    if (contentsRect().size() != d_pix.size())
    {
	d_pix.resize(contentsRect().size());
	clearPixmap();
    }
}

//------------------------------------------------------------
//.-
//.F	QwtPixFrame::adjustFrame
//	Resize the frame according to the pixmap size.
//
//.u	Syntax
//.f	void QwtPixFrame::adjustFrame()
//
//------------------------------------------------------------
void QwtPixFrame::adjustFrame()
{
    setFixedSize(d_pix.width() + 2 * frameWidth(), d_pix.height() + 2 * frameWidth());
}


//------------------------------------------------------------
//
//.F	QwtPixFrame::fillPixmap
//	Fill the pixmap with a specified color
//
//.u	Syntax
//.f	void QwtPixFrame::fillPixmap(const QColor &c)
//
//.u	Parameters
//.p	const QColor &c -- fill color
//
//------------------------------------------------------------
void QwtPixFrame::fillPixmap(const QColor &c)
{
    d_pix.fill(c);
}


//------------------------------------------------------------
//.-
//.F	QwtPixFrame::resizeEvent
//
//.u	Syntax
//.f	void QwtPixFrame::resizeEvent(QResizeEvent *e)
//
//.u	Parameters
//.p	QResizeEvent *e
//
//------------------------------------------------------------
void QwtPixFrame::resizeEvent(QResizeEvent *e)
{

    QFrame::resizeEvent(e);
    if (d_adjustPixmap)
    {
	adjustPixmap();
	emit pixmapResized();
    }
}

//------------------------------------------------------------
//.-
//.F	QwtPixFrame::frameChanged
//	Notify a change of the frame
//
//.u	Syntax
//.f	void QwtPixFrame::frameChanged()
//
//------------------------------------------------------------
void QwtPixFrame::frameChanged()
{
    if (d_adjustPixmap)
    {
	this->resize(this->size());
	this->repaint(FALSE);
    }
    else
       adjustFrame();
}

//------------------------------------------------------------
//
//.F	QwtPixFrame::updatePixmap
//	Maps the pixmap to the screen
//
//.u	Syntax
//.f	void QwtPixFrame::updatePixmap()
//
//.u	Description
//	This function synchronizes the screen contents with the
//	current contents of the pixmap. It generates a paint event
//	which updates the widget.
//
//.u	Note
//	When the widget gets resized, a paint event is generated
//	automatically after the pixmapResized() signal has been
//	emitted. It is not necessary to call updatePixmap()
//	within a slot which is connected to pixmapResized().
//------------------------------------------------------------
void QwtPixFrame::updatePixmap()
{
    repaint(contentsRect(), FALSE);
}

//------------------------------------------------------------
//
//.F	QwtPixFrame::updatePixmapRect
//	Repaint a part of the pixmap
//
//.u	Syntax
//.f	void QwtPixFrame::updatePixmapRect(const QRect& r)
//
//.u	Parameters
//.p	const QRect& r   - part of the pixmap to be redrawn
//			   in pixmap coordinates
//
//------------------------------------------------------------
void QwtPixFrame::updatePixmapRect(const QRect &r)
{
    QRect pr = d_pix.rect().intersect(r);
    if (pr.isValid() && (!pr.isNull()))
    {
	pr.moveTopLeft(pr.topLeft() + contentsRect().topLeft());
	repaint(pr, FALSE);
    }
}

//------------------------------------------------------------
//.-
//.F	QwtPixFrame::setVariableSize
//      Set resizing policy.
//
//.u	Parameters
//.p	bool tf -- adjust pixmap size (TRUE) or frame size (FALSE).
//
//.u	Return Type
//		void
//
//.u	Return Value
//
//.u	Description
//      If tf is TRUE, the pixmap will be adjusted when the
//	frame is resized or when the frame style is changed.
//      If set to FALSE, the size of the pixmap will remain
//	constant and the frame size will be adjusted.
//	The default value is TRUE. 
//
//.u	Syntax
//.f	void QwtPixFrame::setVariableSize(bool tf)
//
//------------------------------------------------------------
void QwtPixFrame::setVariableSize(bool tf)
{
    if (d_adjustPixmap != tf) 
    {
	d_adjustPixmap = tf;
	if (tf) 
	    adjustFrame();
	else
	   adjustPixmap();
    }
}


//------------------------------------------------------------
//
//.F	QwtPixFrame::setPixmapBackground
//	Assign a background color the the pixmap
//
//.u	Syntax
//.f	void QwtPixFrame::setPixmapBackground(const QColor &c)
//
//.u	Parameters
//.p	const QColor &c -- new background color
//
//.u    Description
//------------------------------------------------------------
void QwtPixFrame::setPixmapBackground(const QColor &c)
{
    d_bg = c;
}


//------------------------------------------------------------
//
//.F	QwtPixFrame::clearPixmap
//	Clear the contents of the pixmap and fill it with
//	its background color
//
//.u	Syntax
//.f	void QwtPixFrame::clearPixmap()
//
//
//.u	See also
//	@QwtPixFrame::setPixmapBackground@
//------------------------------------------------------------
void QwtPixFrame::clearPixmap()
{
    d_pix.fill(d_bg);
}

//------------------------------------------------------------
//
//.F	QwtPixFrame::pixmapResized
//	A signal which is emitted when the size of the
//	pixmap has been changed.
//
//.u	Syntax
//.f	void QwtPixFrame::pixmapResized()
//
//.u	Description
//	When resized, the pixmap is cleared and filled with
//	its background color.
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPixFrame::pixmapBackground
//	Return the pixmap's background color
//
//.u	Syntax
//.f	const QColor& QwtPixFrame::pixmapBackground()
//
//.u	See Also
//	@QwtPixFrame::setPixmapBackground@
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtPixFrame::pixmap
//	return a pointer to the enclosed pixmap
//
//.u	Syntax
//.f	const QPixmap* QwtPixFrame::pixmap()
//
//------------------------------------------------------------









