//------------------------------------------------------------
//.H QwtMarker | 3 | 01/01/98 | QwtWidget Library | Qwt Programmer's Manual
//
//.U NAME
//	QwtMarker - A class for drawing markers
//
//.U SYNOPSIS
//	#include <qwt_marker.h>
//
//.U PUBLIC MEMBERS
//.R
//	QwtMarker::QwtMarker -- Constructor
//	QwtMarker::~QwtMarker -- Destructor
//	QwtMarker::operator= -- Assignment operator
//	QwtMarker::setLineStyle -- Set the line style
//	QwtMarker::setSymbol -- Assign a symbol
//	QwtMarker::setLabel -- Assign a label
//	QwtMarker::setLabelAlignment -- Set the label's alignment
//	QwtMarker::setLabelPen -- Assign a pen for drawing the label
//	QwtMarker::setLinePen -- Assign a pen for drawing the line
//	QwtMarker::setFont -- Assign a font for the label
//	QwtMarker::lineStyle -- Return the line style
//	QwtMarker::symbol -- Return the symbol
//	QwtMarker::labelPen -- Return the label's pen
//	QwtMarker::linePen -- Return the line pen
//	QwtMarker::font -- Return the font
//	QwtMarker::labelAlignment -- Return the label's alignment
//	QwtMarker::label -- Return the label
//
//.U PROTECTED MEMBERS 
//	QwtMarker::markerChanged -- Notify a change of parameters
//
//.U DESCRIPTION
//
//	A marker can be a horizontal line, a vertical line,
//	a symbol, a label or any combination of them, which can
//	be drawn around a center point inside a bounding rectangle.
//
//	The setSymbol member assigns a symbol to the marker.
//	The symbol is drawn at the specified point.
//
//	With setLabel, a label can be assigned to the marker.
//	The setAlignment member specifies where the label is
//	drawn. All the Align*-constants in qwindefs.h (see Qt documentation)
//	are valid. The interpretation of the alignment depends on the marker's
//	line style. The alignment refers to the center point of
//	the marker, which means, for example, that the label would be printed
//	left above the center point if the alignment was set to AlignLeft|AlignTop.
//
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//.+ 	 
//
//------------------------------------------------------------

#include "qwt_marker.h"
#include "qwt_math.h"
#include <qpainter.h>
static const int LabelDist = 2;
//------------------------------------------------------------
//.U	MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtMarker::QwtMarker
//		Constructor
//.u	Syntax
//.f		QwtMarker::QwtMarker()
//------------------------------------------------------------
QwtMarker::QwtMarker()
{
    d_align = AlignCenter;
    d_style = NoLine;
}

//------------------------------------------------------------
//
//.F	QwtMarker::~QwtMarker
//	Destructor
//
//.u	Syntax
//.f	 QwtMarker::~QwtMarker()
//
//------------------------------------------------------------
QwtMarker::~QwtMarker()
{
    
}

//------------------------------------------------------------
//
//.F	QwtMarker::draw
//	Draw the marker
//
//.u	Syntax
//.f	void QwtMarker::draw(QPainter *p, int x, int y, const QRect &r)
//
//.u	Parameters
//.p	QPainter *p	-- Painter
//	int x, int y	-- Position
//	const QRect &r  -- Bounding rectangle
//
//------------------------------------------------------------
void QwtMarker::draw(QPainter *p, int x, int y, const QRect &r)
{
    QRect tr;
    QSize sSym;
    int tw, th, lw, lw1;
    
    
    // draw lines
    if (d_style != NoLine)
    {
	p->setPen(d_pen);
	if ((d_style == HLine) || (d_style == Cross))
	   p->drawLine(r.left(), y, r.right(), y);
	if ((d_style == VLine)||(d_style == Cross))
	   p->drawLine(x, r.top(), x, r.bottom());
    }

    // draw symbol
    if (d_sym.style() != QwtSymbol::None)
    {
	sSym = d_sym.size();
	d_sym.draw(p, x - (sSym.width() - 1) / 2 ,y - (sSym.width() - 1) / 2);
    }
    else
    {
	sSym.setWidth(0);
	sSym.setHeight(0);
    }
    

    // draw label
    if (d_label != "")
    {
	p->setPen(d_tPen);
	p->setFont(d_font);
	th = p->fontMetrics().height();
	tw = p->fontMetrics().width(d_label);
	lw = qwtMax(int(d_pen.width()), 1);

	if ((d_style == VLine) || (d_style == HLine))
	{
	    lw1 = (lw + 1) / 2 + LabelDist;
	    lw = lw / 2 + LabelDist;
	}
	else 
	{
	    lw1 = qwtMax((lw + 1) / 2, (sSym.width() + 1) / 2) + LabelDist;
	    lw = qwtMax(lw / 2, (sSym.width() + 1) / 2) + LabelDist;
	}

	
	if (d_style == VLine)
	{
	    if (d_align & (int)AlignTop)
	       tr.setY(r.top() + LabelDist);
	    else if (d_align & (int)AlignBottom)
	       tr.setY(r.bottom() - LabelDist - th);
	    else
	       tr.setY(r.top() + (r.bottom() - r.top()) / 2);
	}
	else 
	{
	    if (d_align & (int)AlignTop)
	       tr.setY(y - lw - th);
	    else if (d_align & (int)AlignBottom)
	       tr.setY(y + lw1);
	    else
	       tr.setY(y - th/2);
	}


	if (d_style == HLine)
	{
	    if (d_align & (int)AlignLeft)
	       tr.setX(r.left() + LabelDist);
	    else if (d_align & (int)AlignRight)
	       tr.setX(r.right() - tw - LabelDist);
	    else
	       tr.setX(r.left() + (r.right() - r.left()) / 2);
	}
	else
	{
	    if (d_align & (int)AlignLeft)
	       tr.setX(x - tw - lw);
	    else if (d_align & (int)AlignRight)
	       tr.setX(x + lw1);
	    else
	       tr.setX(x - tw/ 2);
	}

	tr.setHeight(th);
	tr.setWidth(tw);	

	p->drawText(tr, AlignTop|AlignHCenter, d_label); 
	
    }
    
    
}

//------------------------------------------------------------
//
//.F	QwtMarker::setFont
//	Specify the font for the label
//
//.u	Syntax
//.f	void QwtMarker::setFont(const QFont &f)
//
//
//.u	Parameters
//.p	const QFont &f	 -- new font
//------------------------------------------------------------
void QwtMarker::setFont(const QFont &f)
{
    d_font = f;
    markerChanged();
}



//------------------------------------------------------------
//
//.F	QwtMarker::setLineStyle
//	
//.u	Syntax
//.f	void QwtMarker::setLineStyle(QwtMarker::LineStyle st)
//
//
//.u	Parameters
//.p	QwtMarker::LineStyle st	-- Line style. Can be one of QwtMarker::NoLine,
//		   QwtMarker::HLine, QwtMarker::VLine or QwtMarker::Cross
//
//------------------------------------------------------------
void QwtMarker::setLineStyle(QwtMarker::LineStyle st)
{
    d_style = st;
    markerChanged();
}



//------------------------------------------------------------
//
//.F	QwtMarker::setSymbol
//	Specify a symbol
//
//.u	Syntax
//.f	void QwtMarker::setSymbol(const QwtSymbol &s)
//
//.u	Parameters
//.p	const QwtSymbol &s	-- New symbol
//
//------------------------------------------------------------
void QwtMarker::setSymbol(const QwtSymbol &s)
{
    d_sym = s;
    markerChanged();
}

//------------------------------------------------------------
//
//.F	QwtMarker::setLabel
//	Set the label
//
//.u	Syntax
//.f	void QwtMarker::setLabel(const char *txt)
//
//.u	Parameters
//.p	const char *txt	-- label text	
//------------------------------------------------------------
void QwtMarker::setLabel(const char *txt)
{
    d_label = txt;
    markerChanged();
}

//------------------------------------------------------------
//
//.F	QwtMarker::setLabelAlignment
//	Set the alignment of the label
//
//.u	Syntax
//.f	void QwtMarker::setLabelAlignment(int align)
//
//.u	Parameters
//.p	int align -- Alignment. A combination of AlignTop, AlignBottom,
//		    AlignLeft, AlignRight, AlignCenter, AlgnHCenter,
//		AlignVCenter.
//
//.u	Description
//	The alignment determines where the label is drawn relative to
//	the marker's position.
//
//------------------------------------------------------------
void QwtMarker::setLabelAlignment(int align)
{
    d_align = align;
    markerChanged();
}

//------------------------------------------------------------
//
//.F	QwtMarker::setLinePen
//	Specify a pen for the line.
//
//.u	Syntax
//.f	void QwtMarker::setLinePen(const QPen &p)
//
//.u	Parameters
//.p	const QPen &p	-- New pen
//
//------------------------------------------------------------
void QwtMarker::setLinePen(const QPen &p)
{
    d_pen = p;
    markerChanged();
}

//------------------------------------------------------------
//
//.F	QwtMarker::setLabelPen
//	Specify a pen for the label
//
//.u	Syntax
//.f	void QwtMarker::setLabelPen(const QPen &p)
//
//.u	Parameters
//.p	const QPen &p	-- New pen
//
//------------------------------------------------------------
void QwtMarker::setLabelPen(const QPen &p)
{
    d_tPen = p;
    markerChanged();
}

//------------------------------------------------------------
//
//.F	QwtMarker::operator=
//	Assignment operator
//
//.u	Syntax
//.f	const QwtMarker& QwtMarker::operator=(const QwtMarker &m)
//
//.u	Parameters
//.p	const QwtMarker &m -- Marker
//
//------------------------------------------------------------
const QwtMarker& QwtMarker::operator=(const QwtMarker &m)
{
    d_label = m.d_label;
    d_pen = m.d_pen;
    d_tPen = m.d_tPen;
    d_font = m.d_font;
    d_sym = m.d_sym;
    d_xMap = m.d_xMap;
    d_yMap = m.d_yMap;
    d_align = m.d_align;
    d_style = m.d_style;

    return *this;
}

//------------------------------------------------------------
//
//.F	QwtMarker::lineStyle
//	Return the line style
//
//.u	Syntax
//.f	QwtMarker::LineStyle QwtMarker::lineStyle()
//
//.u    Description
//	For a description of line styles, see @QwtMarker::setLineStyle@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtMarker::symbol
//	Return the symbol
//
//.u	Syntax
//.f	const QwtSymbol & QwtMarker::symbol()
//
//.u	See also
//	@QwtMarker::setSymbol@, @^QwtSymbol@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtMarker::labelPen
//	Return the label's pen
//
//.u	Syntax
//.f	const Qpen & QwtMarker::labelPen()
//
//.u	See also
//	@QwtMarker::setLabelPen@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtMarker::line pen
//	Return the line pen
//
//.u	Syntax
//.f	const QPen & QwtMarker::line pen()
//
//.u	See Also
//	@QwtMarker::setLinePen@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtMarker::font
//	Return the font used for the label
//
//.u	Syntax
//.f	const QFont & QwtMarker::font()
//
//.u	See Also
//	@QwtMarker::setFont@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtMarker::labelAlignment
//	Return the label alignment
//
//.u	Syntax
//.f	int QwtMarker::labelAlignment()
//
//.u	See Also
//	@QwtMarker::setLabelAlignment@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtMarker::label
//	Return the label
//
//.u	Syntax
//.f	const QString & QwtMarker::label()
//
//.u	See Also
//	@QwtMarker::setLabel@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtMarker::markerChanged
//	Notify a change of parameters
//
//.u	Syntax
//.f	void QwtMarker::markerChanged()
//
//.u	Description
//	This virtual function is called when a marker property
//	has been changed and redrawing is advisable. It can be
//	reimplemented by derived classes. The default implementation
//	does nothing.
//------------------------------------------------------------







