#include "qwt_legend.h"
#include <qpainter.h>
#include <qwt_math.h>

static const int MinSymWidth = 5;
static const int MinLblWidth = 50;
static const int MinHBdrDist = 6;
static const int MinVBdrDist = 4;
//------------------------------------------------------------
//.H QwtLegend|3|01/01/98|Qwt Widget Library|Qwt Programmer's manual
//
//.U NAME
//	QwtLegend - The Legend Widget
//
//.U SYNOPSIS
//	#include<qwt_legend.h>
//
//.U INHERITED CLASSES
//	QTableView
//
//.U PUBLIC MEMBERS
//.R
//	QwtLegend::appendItem -- Append an item to the legend
//	QwtLegend::insertItem -- Insert an item at a specified position
//	QwtLegend::setText --	Change the label text of an item
//	QwtLegend::setSymbol --	Change the symbol of an item
//	QwtLegend::setPen --   Change the line pen of an item
//	QwtLegend::setKey --	Change the key of an item
//	QwtLegend::key --	Return the key of an item
//	QwtLegend::removeItem -- Remove an item
//	QwtLegend::setMaxCols -- Set the maximum number of columns
//	QwtLegend::setMaxRows -- Set the maximum number of rows
//	QwtLegend::colWidth -- Return the column width
//	QwtLegend::rowHeight -- Return the row height
//	QwtLegend::clear -- Remove all items
//	QwtLegend::sizeHint -- Return a size hint
//	QwtLegend::itemCnt -- Return the number of items
//	QwtLegend::findFirstKey -- Find the index of an item identified by a key
//
//
//.U PROTECTED MEMBERS
//.R
//	QwtLegend::fontChange -- Notify font change
//
//.U SIGNALS
//.R
//	QwtLegend::pressed -- Emitted when the user presses the mouse button
//				down on an item
//	QwtLegend::clicked -- Emitted when the user clicks an item
//
//.U DESCRIPTION
//
//	The QwtLegend widget is a tabular arrangement of cells,
//	called legend items, which consist of a label, a line
//	a symbol, and a key of type long as an 
//	identifier. The symbol is drawn on top of
//	the line at the left side of the label. An item's
//	property can be accessed using the item's index,
//	which is independent of the key. The cells can
//	be arranged in row or column order with @QwtLegend::setMaxRows@
//	or @QwtLegend::setMaxCols@. Scrollbars are displayed
//	automatically if	the cells don't fit into the widget area.
//
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//.+ 	 
//------------------------------------------------------------

//------------------------------------------------------------
//.U	MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------


//------------------------------------------------------------
//.-
//.F	QwtLegendItem::QwtLegendItem
//	Constructor
//
//.u	Parameters
//.p	const QString &txt, const QwtSymbol &sym, const QPen &pn, long key
//
//.u	Syntax
//.f	 QwtLegendItem::QwtLegendItem(const QString &txt, const QwtSymbol &sym, const QPen &pn, long key)
//
//------------------------------------------------------------
QwtLegendItem::QwtLegendItem(const QString &txt, const QwtSymbol &sym, const QPen &pn, long key)
: d_pen(pn), d_sym(sym), d_text(txt),  d_key(key) 
{
    
}

//------------------------------------------------------------
//.-
//.F	QwtLegendItem::QwtLegendItem
//	Default Constructor
//
//.u	Syntax
//.f	 QwtLegendItem::QwtLegendItem()
//
//------------------------------------------------------------
QwtLegendItem::QwtLegendItem()
: d_text(""), d_key(0)
{
}


//------------------------------------------------------------
//.-
//.F	QwtLegendItem::draw
//
//
//.u	Parameters
//.p	QPainter *p
//
//.u	Syntax
//.f	void QwtLegendItem::draw(QPainter *p)
//
//------------------------------------------------------------
void QwtLegendItem::draw(QPainter *p, const QRect& symRect, const QRect& lblRect, int align)
{
    int yl = symRect.top() + symRect.height() / 2;
    p->save();
    
    p->drawText(lblRect, align, d_text);
    p->setPen(d_pen);
    p->drawLine(symRect.x(), yl, symRect.right(), yl);
    d_sym.draw(p, symRect.x() + symRect.width() / 2 - d_sym.size().width() / 2,
	       yl - d_sym.size().height() / 2);
    p->restore();
    
}


//------------------------------------------------------------
//
//.F	QwtLegend::QwtLegend
//	Construct a QwtLegend instance
//
//.u	Syntax
//.f	 QwtLegend::QwtLegend(QWidget *p, const char *name)
//
//.u	Parameters
//.p	QWidget *p, const char *name
//
//
//------------------------------------------------------------
QwtLegend::QwtLegend(QWidget *p, const char *name)
: QTableView(p,name)
{
    d_maxCols = 5;
    d_maxRows = 0;
    d_sel.active = FALSE;
    d_sel.r = 0;
    d_sel.c = 0;
    d_item.setAutoDelete(TRUE);
    d_align = int(AlignLeft|AlignVCenter);
    setTableFlags(Tbl_autoScrollBars|Tbl_clipCellPainting);
    
}


//------------------------------------------------------------
//
//.F	QwtLegend::~QwtLegend
//	Destroy a QwtLegend instance
//
//.u	Syntax
//.f	void QwtLegend::~QwtLegend()
//
//------------------------------------------------------------
QwtLegend::~QwtLegend()
{
    d_item.clear();
}


//------------------------------------------------------------
//
//.F	QwtLegend::clear
//	Remove all items from a legend
//
//.u	Syntax
//.f	void QwtLegend::clear()
//
//------------------------------------------------------------
void QwtLegend::clear()
{
    d_item.clear();
}

//------------------------------------------------------------
//
//.F	QwtLegend::insertItem
//	Insert a new item at a specified position
//
//.u	Syntax
//.f	bool QwtLegend::insertItem(const QString &txt, const QwtSymbol &s, const QPen &p, long key, uint index)
//
//.u	Parameters
//.p	const QString &txt -- Label 
//	const QwtSymbol &s -- Symbol
//	const QPen &p	-- line pen
//      long key -- key
//	uint index -- position where to insert the item
//
//------------------------------------------------------------
bool QwtLegend::insertItem(const QString &txt, const QwtSymbol &s, const QPen &p,
			   long key,  uint index)
{
    QwtLegendItem *newItem;
    newItem = new QwtLegendItem(txt, s, p, key);
    if (newItem) 
    {
	d_item.insert(index, newItem);
	updateTable();
	return TRUE;
    }
    else
       return FALSE;
}

//------------------------------------------------------------
//
//.F	QwtLegend::appendItem
//	Append a new item to the legend
//
//.u	Parameters
//.p	const QString &txt -- Label
//	const QwtSymbol &s -- Symbol
//	const QPen &p -- Line pen
//	long key -- key
//
//.u	Return Value
//		The new size of the list
//.u	Description
//
//.u	Syntax
//.f	uint QwtLegend::appendItem(const QString &txt, const QwtSymbol &s, const QPen &p, long key)
//
//------------------------------------------------------------
uint QwtLegend::appendItem(const QString &txt, const QwtSymbol &s,
			   const QPen &p, long key)
{
    QwtLegendItem *newItem;
    newItem = new QwtLegendItem(txt, s, p, key);
    if (newItem)
    {
	d_item.append(newItem);
	updateTable();
    }
    return d_item.count();
}

//------------------------------------------------------------
//.-
//.F	QwtLegend::updateTable
//	Update the cell dimendions and the table dimensions
//
//.u	Syntax
//.f	void QwtLegend::updateTable()
//
//.u	Description
//		This function combines @QwtLegend::resizeCells@
//		and @QwtLegend::resizeTable@
//------------------------------------------------------------
void QwtLegend::updateTable()
{
    setAutoUpdate(FALSE);
    resizeCells();
    resizeTable();
    setAutoUpdate(TRUE);
    repaint();
}

//------------------------------------------------------------
//.-
//.F	Qwtlegend::paintCell
//	paint the contents of a cell
//
//.u	Syntax
//.f	void Qwtlegend::paintCell(QPainter *p, int row, int col)
//
//.u	Parameters
//.p	QPainter *p, int row, int col
//
//------------------------------------------------------------
void QwtLegend::paintCell(QPainter *p, int row, int col)
{
    QwtLegendItem *curItem;
    uint index;

    index = findIndex(row, col);
    
    if (index < d_item.count())
    {
	if (d_sel.active && (d_sel.r == row) && (d_sel.c == col) )
	   p->fillRect(cellUpdateRect(), colorGroup().light());
	curItem = d_item.at(index);
	if (curItem)
	   curItem->draw(p, d_symRect, d_lblRect, d_align);
    }
}

//------------------------------------------------------------
//
//.F	QwtLegend::findIndex
//	Return the index of an item with specified row and column
//
//.u	Syntax
//.f	uint QwtLegend::findIndex(int row, int col)
//
//.u	Parameters
//.p	int row, int col -- row and column
//
//------------------------------------------------------------
uint QwtLegend::findIndex(int row, int col)
{
    uint rv;
    if (d_maxCols)
       rv = uint( row * numCols() + col);
    else
       rv = uint( col * numRows() + row);
    
    return qwtMin(d_item.count(), rv);
}

//------------------------------------------------------------
//
//.F	QwtLegend::setMaxCols
//	Set the maximum number of columns and determine the layout policy.
//
//.u	Syntax
//.f	void QwtLegend::setMaxCols(int n)
//
//.u	Parameters
//.p	int n	-- max. number of columns
//
//.u	Description
//	The setMaxCols and setMaxRows members determine in which order 
//	the cells are arranged in the table. If a maximum number of
//	columns is set, the legend items will be arranged in rows from
//	the left to the right,
//	starting at the first row, and beginning a new row when the
//	actual number of columns has reached that maximum. The number of
//	rows is thereby unlimited.
//.P
//	The default setting is a maximum of 5 columns with an unlimited
//	number of rows.
//
//.u	Note
//	setMaxRows and setMaxCols are mutually exclusive. The last one
//	wins.
//
//------------------------------------------------------------
void QwtLegend::setMaxCols(int n)
{
    d_maxRows = 0;
    d_maxCols = qwtMax(n,1);
    resizeTable();
}

//------------------------------------------------------------
//
//.F	QwtLegend::setMaxRows
//	Specify the maximum number of rows determine the layout policy.
//
//.u	Syntax
//.f	void QwtLegend::setMaxRows(int n)
//
//.u	Parameters
//.p	int n		-- max. number of rows
//
//.u	Description
//	The setMaxRows and setMaxCols members determine in which order 
//	the cells are arranged in the table. If a maximum number of
//	rows is set, the legend items will be arranged in columns from
//	the top to the bottom,
//	starting at the first column, and beginning a new column when the
//	actual number or rows has reached that maximum. The number of
//	columns is thereby unlimited.
//.P
//	The default setting is a maximum of 5 columns with an unlimited
//	number of rows.
//
//.u	Note
//	setMaxRows and setMaxCols are mutually exclusive. The last one
//	wins.
//
//------------------------------------------------------------
void QwtLegend::setMaxRows(int n)
{
    d_maxCols = 0;
    d_maxRows = qwtMax(n,1);
    resizeTable();
}

//------------------------------------------------------------
//
//.F	QwtLegend::resizeTable
//      Update the table dimensions (rows x cols)
//
//.u	Syntax
//.f	void QwtLegend::resizeTable()
//
//.u	Access
//	protected
//
//.u	Description
//	   This function is called when an item has been
//	    appended or deleted or when the max. number of
//	    rows/columns has been changed.
//
//------------------------------------------------------------
void QwtLegend::resizeTable()
{

    int rows, cols;
    
    bool au = autoUpdate();
    
    setAutoUpdate(FALSE);
    
    //
    // adjust table dimensions 
    // 
    if (d_maxCols)
    {
	cols = qwtMin(d_maxCols, int(d_item.count()));
	cols = qwtMax(cols, 1);
	rows = qwtMax( (int(d_item.count()) - 1) / cols + 1, 1);
    }
    if (d_maxRows)
    {
	rows = qwtMin(d_maxRows, int(d_item.count()));
	rows = qwtMax(rows, 1);
	cols = qwtMax( (int(d_item.count()) - 1) / rows + 1, 1);
    }

    if (rows != numRows()) setNumRows(rows);
    if (cols != numCols()) setNumCols(cols);

    setTopLeftCell(0,0);
    
    if (au)
    {
	setAutoUpdate(TRUE);
	repaint(TRUE);
    }

}


//------------------------------------------------------------
//
//.F	QwtLegend::resizeCells
//	Resize the cells
//
//.u	Syntax
//.f	void QwtLegend::resizeCells()
//
//.u	Description
//	This function is called when an item has been inserted, removed
//	or changed. It adjusts the dimensions of the table according to
//	the layout policy and sets the cell dimensions such that all
//	items fit into the cells.
//
//.u	See also
//	@Qwtplot::setMaxRows@, @QwtPlot::setMaxCols@
//------------------------------------------------------------
void QwtLegend::resizeCells()
{
    QwtLegendItem *curItem;
    
    int h = 0, wl = 0, ws = 0, wd = 0;

    bool au = autoUpdate();
    
    setAutoUpdate(FALSE);

    //
    // adjust cell dimensions
    //
    QListIterator<QwtLegendItem> iti(d_item);
    QPainter p;
    p.begin(this);
    h = p.fontMetrics().height();

    ws = MinSymWidth;
    wl = MinLblWidth;
    
    iti.toFirst();
    while ((curItem = iti.current()))
    {
	h = qwtMax(h, curItem->symbol().size().height());
	ws = qwtMax(ws, curItem->symbol().size().width());
	wl = qwtMax(wl, p.fontMetrics().width(curItem->text()));
	++iti;
    }
    ws *= 3;
    wd = qwtMax(MinHBdrDist, h / 2);
    wl += wd;
    h = h * 3 / 2;

    p.end();

    d_symRect = QRect(wd, 0, ws, h);
    d_lblRect = QRect(d_symRect.right() + wd, 0, wl, h); 

    setCellHeight(h);
    setCellWidth(ws + 2*wd + wl);

    
    if (au)
    {
	setAutoUpdate(TRUE);
	repaint(TRUE);
    }
    
}

//------------------------------------------------------------
//
//.F	QLegend::sizeHint
//	Return a size hint
//
//.u	Parameters
//.p	
//
//.u	Syntax
//.f	QSize QLegend::sizeHint()
//
//------------------------------------------------------------
QSize QwtLegend::sizeHint() const
{
    QSize s(numCols() * cellWidth() + 2 * frameWidth(),
	    numRows() * cellHeight() + 2 * frameWidth());
    return s;
}


//------------------------------------------------------------
//
//.F	QwtLegend::findFirstKey
//	Find the index of an item with a specified key
//
//.u	Syntax
//.f	uint QwtLegend::findFirstKey(long key)
//
//.u	Parameters
//.p	long key
//
//.u	Return Value
//	   The index of the first item which has the specified key
//	   or the total number of items if the key was not found.
//------------------------------------------------------------
uint QwtLegend::findFirstKey(long key)
{
    uint rv = 0;
    QwtLegendItem *curItem;
    QListIterator<QwtLegendItem> iti(d_item);

    iti.toFirst();
    while ((curItem = iti.current()))
    {
	if (curItem->key() == key) break;
	++rv;
	++iti;
    }
	   
    return rv;
    
}

//------------------------------------------------------------
//.-
//.F	QwtLegend::mousePressEvent
//
//
//.u	Syntax
//.f	void QwtLegend::mousePressEvent(QMouseEvent *e)
//
//.u	Parameters
//.p	QMouseEvent *e
//
//------------------------------------------------------------
void QwtLegend::mousePressEvent(QMouseEvent *e)
{
    uint index;
    int r, c;
    r = findRow(e->pos().y());
    c = findCol(e->pos().x());
    if ((c >= 0) && (r >= 0))
    {
	index = findIndex(r, c);
	if (index < d_item.count())
	{
	    d_sel.active = TRUE;
	    d_sel.r = r;
	    d_sel.c = c;
	    updateCell(r,c);
	    emit pressed(index);
	}
	
    }
}


//------------------------------------------------------------
//.-
//.F	QwtLegend::mouseReleaseEvent
//
//.u	Syntax
//.f	void QwtLegend::mouseReleaseEvent(QMouseEvent *e)
//
//
//.u	Parameters
//.p	QMouseEvent *e
//
//------------------------------------------------------------
void QwtLegend::mouseReleaseEvent(QMouseEvent *e)
{
    uint index;
    int r, c;
    r = findRow(e->pos().y());
    c = findCol(e->pos().x());

    // undo selection
    if (d_sel.active)
    {
	d_sel.active = FALSE;
	updateCell(d_sel.r, d_sel.c);
    }

    // emit clicked() signal
    if ((c >= 0) && (r >= 0))
    {
	index = findIndex(r, c);
	if (index < d_item.count())
	{
	    emit clicked(index);
	}
	
    }
    
}
//------------------------------------------------------------
//
//.F	QwtLegend::setText
//	Change the label of an item
//
//.u	Syntax
//.f	bool QwtLegend::setText(uint index, const QString &s)
//
//.u	Parameters
//.p	uint index	-- The item's index
//	const QString &s -- New label
//
//.u	Return Value
//		TRUE if the index is valid
//
//------------------------------------------------------------
bool QwtLegend::setText(uint index, const QString &s)
{
    bool rv = FALSE;
    QwtLegendItem *curItem = d_item.at(index);
    if (curItem)
    {
	curItem->setText(s);
	resizeCells();
	rv = TRUE;
    }
    return rv;
}

//------------------------------------------------------------
//
//.F	QwtLegend::setSymbol
//	Change the symbol of an item
//
//.u	Syntax
//.f	bool QwtLegend::setSymbol(uint index, const QwtSymbol &s)
//
//.u	Parameters
//.p	uint index	 -- The item's index
//	const QwtSymbol &s  -- New symbol
//
//.u	Return Value
//		TRUE if the index is valid
//
//------------------------------------------------------------
bool QwtLegend::setSymbol(uint index, const QwtSymbol &s)
{
    bool rv = FALSE;
    QwtLegendItem *curItem = d_item.at(index);
    if (curItem)
    {
	curItem->setSymbol(s);
	resizeCells();
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtLegend::setPen
//	Change the line pen of an item
//
//.u	Syntax
//.f	bool QwtLegend::setPen(uint index, const QPen &pn)
//
//.u	Parameters
//.p	uint index	-- The item's index
//	const QPen &pn	-- New pen
//
//.u	Return Value
//		TRUE if the index is valid
//
//------------------------------------------------------------
bool QwtLegend::setPen(uint index, const QPen &pn)
{
    bool rv = FALSE;
    int row, col;
    QwtLegendItem *curItem = d_item.at(index);
    if (curItem)
    {
	curItem->setPen(pn);
	findRowCol(index, row, col);
	updateCell(row, col);   
	rv = TRUE;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.F	QwtLegend::setKey
//	Change the key of an item
//
//.u	Syntax
//.f	bool QwtLegend::setKey(uint index, long key)
//
//.u	Parameters
//.p	uint index	--	The item's index
//	long key	--	new key
//
//.u	Return Value
//		TRUE if the index is valid
//
//------------------------------------------------------------
bool QwtLegend::setKey(uint index, long key)
{
    bool rv = FALSE;
    QwtLegendItem *curItem = d_item.at(index);
    if (curItem)
    {
	curItem->setKey(key);
	rv = TRUE;
    }
    return rv;
}

//------------------------------------------------------------
//
//.F	QwtLegend::removeItem
//	Remove an item
//
//.u	Syntax
//.f	void QwtLegend::removeItem (uint index)
//
//.u	Parameters
//.p	uint index -- The item's index
//
//------------------------------------------------------------
bool QwtLegend::removeItem(uint index)
{
    return d_item.remove(index);
    updateTable();
}

//------------------------------------------------------------
//.-
//.F	QwtLegend::findRowCol
//	Get row and column of an item with a specified index
//
//.u	Syntax
//.f	void QwtLegend::findRowCol(uint index, int &row, int &col)
//
//.u	Input Parameters
//.p	uint index	-- The item's index
//
//.u	Output Parameters
//.p	int &row, int &col -- row and column
//
//------------------------------------------------------------
void QwtLegend::findRowCol(uint index, int &row, int &col)
{
	if (d_maxCols)
	{
	    row = int(index) / numCols();
	    col = int(index) - row * numCols();
	}
	else
	{
	    col = int(index) / numRows();
	    row = int(index) - row * numRows();
	}
}

//------------------------------------------------------------
//
//.u	QwtLegend::fontChange
//	Notify a font change
//
//.u	Syntax
//.f	void QwtLegend::fontChange(const QFont &oldFont)
//
//.u	Parameters
//.p	const QFont &oldFont -- Old font
//
//------------------------------------------------------------
void QwtLegend::fontChange(const QFont &oldFont)
{
    resizeCells();
}

//------------------------------------------------------------
//
//.F	QwtLegend::key
//	Return an item's key,
//	or 0 if the specified index is invalid
//
//.u	Syntax
//.f	long QwtLegend::key(uint index)
//
//.u	Parameters
//.p	uint index -- The item's index
//
//------------------------------------------------------------
long QwtLegend::key(uint index)
{
    QwtLegendItem *curItem = d_item.at(index);
    if (curItem)
	return curItem->key();
    else
       return 0;
}

//------------------------------------------------------------
//
//.F	QwtLegend::colWidth
//	Return the column width
//
//.u	Syntax
//.f	int QwtLegend::colWidth()
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtLegend::rowHeight
//	Return the row height
//
//.u	Syntax
//.f	int QwtLegend::rowHeight()
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtLegend::itemCnt
//	Return the number of items
//
//.u	Syntax
//.f	uint QwtLegend::itemCnt()
//
//------------------------------------------------------------


//------------------------------------------------------------
//
//.F	QwtLegend::pressed
//	A signal which is emitted when the user presses a
//	mouse button down on an item
//
//.u	Syntax
//.f	void QwtLegend::pressed(int index)
//
//.u	Parameters
//.p	int index	-- The item's index
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtLegend::clicked
//	A signal which is emitted when the user clicks on an item.
//
//.u	Syntax
//.f	void QwtLegend::clicked(int index)
//
//.u	Parameters
//.p	int index	-- The item's index
//
//------------------------------------------------------------
