#include "qwt_dimap.h"
#include "qwt_math.h"

const double QwtDiMap::LogMin = 1.0e-150;
const double QwtDiMap::LogMax = 1.0e150;
//------------------------------------------------------------
//.H QwtDiMap | 3 | 01/01/98 | Qwt Widget Library | Qwt Programmer's Manual
//------------------------------------------------------------
//------------------------------------------------------------
//
//.U NAME
//	QwtDiMap - Map a double interval into an integer interval
//
//.U SYNOPSIS
//	#include <qwt_dimap.h>
//
//.U DESCRIPTION
//	The QwtDiMap class maps an interval of type double into an interval of
//	type integer. It consists
//	of two intervals D = [d1, d2] (double) and I = [i1, i2] (int), which are
//	specified with the @QwtDiMap::setDblRange@ and @QwtDiMap::setIntRange@
//	members. The point d1 is mapped to the point i1, and d2 is mapped to i2. 
//	Any point inside or outside D can be mapped to a point inside or outside
//	I using @QwtDiMap::transform@ or @QwtDiMap::limTransform@ or vice versa
//	using @QwtPlot::invTransform@. D can be scaled linearly or
//	logarithmically, as specified with @QwtDiMap::setDblRange@.
//
//.U INHERITED CLASSES
//	@QwtScaleDraw@
//
//.U PUBLIC MEMBERS
//.R
//
//    QwtDiMap::QwtDiMap (1) --	Constructor
//    QwtDiMap::QwtDiMap (2) -- Constructor
//    QwtDiMap::~QwtDiMap -- Destructor
//    QwtDiMap::contains (1) -- value contained in double interval?
//    QwtDiMap::contains (2) -- value contained in int interval?
//    QwtDiMap::setIntRange -- Specify integer interval
//    QwtDiMap::setDblRange -- Specify double interval
//    QwtDiMap::transform   -- transform double -> int
//    QwtDiMap::invTransform -- transform int -> double
//    QwtDiMap::xTransform -- transform double -> double(int)
//    QwtDiMap::limTransform(double x) -- transform double -> int
//    QwtDiMap::d1 -- first border of double interval 
//    QwtDiMap::d2 -- second border of double interval
//    QwtDiMap::i1 -- first border of integer interval
//    QwtDiMap::i2 -- second border of int interval
//    QwtDiMap::logarithmic -- double interval scaled logarithmically?
//
//.U  USAGE
//.c
//      #include <qwt_dimap.h>
//
//      QwtDiMap map;
//      int ival;
//      double dval;
//
//      map.setDblRange(0.0, 3.1415);   // Assign an interval of type double with
//                                      // linear mapping
//      map.setIntRange(0,100);         // Assign an integer interval
//
//      ival = map.transform(1.0);      // obtain integer value corresponding to 1.0
//      dval = map.invTransform(77);    // obtain double value corresponding to 77
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//.+ 	 
//------------------------------------------------------------

//------------------------------------------------------------
//.U MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtDiMap::QwtDiMap (1)
//	Construct a QwtDiMap instance.
//
//.u	Syntax
//.f	 QwtDiMap::QwtDiMap()
//
//.u	Description
//	The double and integer intervals are both set to [0,1].
//
//------------------------------------------------------------
QwtDiMap::QwtDiMap()
{
    d_x1 = 0.0;
    d_x2 = 1.0;
    d_y1 = 0;
    d_y2 = 1;
    d_cnv = 1.0;
}


//------------------------------------------------------------
//
//.F	QwtDiMap::QwtDiMap (2)
//	Construct a QwtDiMap instance with initial integer
//	and double intervals
//
//.u	Syntax
//.f	 QwtDiMap::QwtDiMap(int i1, int i2, double d1, double d2, bool logarithmic)
//
//.u	Parameters
//.p	int i1		--	first border of integer interval
//	int i2		--	second border of integer interval
//	double d1	--	first border of double interval
//	double d2	--	second border of double interval
//	bool logarithmic  -- logarithmic mapping, TRUE or FALSE. Defaults
//					to FALSE.
//
//------------------------------------------------------------
QwtDiMap::QwtDiMap(int i1, int i2, double d1, double d2, bool logarithmic)
{
    d_log = logarithmic;
    setIntRange(i1,i2);
    setDblRange(d1, d2);
}

//------------------------------------------------------------
//
//.F	QwtDiMap::~QwtDiMap
//	Destroy a QwtDiMap instance.
//
//.u	Syntax
//.f	 QwtDiMap::~QwtDiMap()
//
//------------------------------------------------------------
QwtDiMap::~QwtDiMap()
{
}

//------------------------------------------------------------
//
//.F	QwtDiMap::contains (1)
//	Returns TRUE if a value x lies inside or at the border of the
//	map's double range.
//
//.u	Syntax
//.f	bool QwtDiMap::contains(double x)
//
//.u	Parameters
//.p	double x -- value 
//
//------------------------------------------------------------
bool QwtDiMap::contains(double x) const
{
    return ( (x >= qwtMin(d_x1, d_x1)) && (x <= qwtMax(d_x1, d_x2)));
}

//------------------------------------------------------------
//
//.F	QwtDiMap::contains (2)
//	Returns TRUE if a value x lies inside or at the border of the
//	map's integer range
//
//.u	Syntax
//.f	bool QwtDiMap::contains(int x)
//
//.u	Parameters
//.p	int x -- value 
//
//------------------------------------------------------------
bool QwtDiMap::contains(int x) const
{
    return ( (x >= qwtMin(d_y1, d_y1)) && (x <= qwtMax(d_y1, d_y2)));
}

//------------------------------------------------------------
//
//.F	QwtDiMap::setDblRange
//	Specify the borders of the double interval 
//
//.u	Syntax
//.f	void QwtDiMap::setDblRange(double d1, double d2, bool lg = FALSE)
//
//.u	Parameters
//.p	double d1	--	first border
//	double d2	--	second border
//	bool lg		--	logarithmic (TRUE) or linear (FALSE)
//				scaling. Defaults to FALSE.
//
//------------------------------------------------------------
void QwtDiMap::setDblRange(double d1, double d2, bool lg)
{
    if (lg)
    {
	d_log = TRUE;
	if (d1 < LogMin) 
	   d1 = LogMin;
	else if (d1 > LogMax) 
	   d1 = LogMax;
	
	if (d2 < LogMin) 
	   d2 = LogMin;
	else if (d2 > LogMax) 
	   d2 = LogMax;
	
	d_x1 = log(d1);
	d_x2 = log(d2);
    }
    else
    {
	d_log = FALSE;
	d_x1 = d1;
	d_x2 = d2;
    }
    newFactor();

}

//------------------------------------------------------------
//
//.F	QwtDiMap::setIntRange
//	Specify the borders of the integer interval
//
//.u	Syntax
//.f	void QwtDiMap::setIntRange(int i1, int i2)
//
//.u	Parameters
//.p	int i1	--	first border
//	int i2  --	second border
//
//------------------------------------------------------------
void QwtDiMap::setIntRange(int i1, int i2)
{
    d_y1 = i1;
    d_y2 = i2;
    newFactor();
}



//------------------------------------------------------------
//
//.F	QwtDiMap::transform
//	Transform a point in double interval into an point in the
//	integer interval
//
//.u	Syntax
//.f	int QwtDiMap::transform(double x)
//
//.u	Parameters
//.p	double x
//
//.u	Return Value
//.t
//	linear mapping:	-- rint(i1 + (i2 - i1) / (d2 - d1) * (x - d1))
//	logarithmic mapping: -- rint(i1 + (i2 - i1) / log(d2 / d1) * log(x / d1))
//
//.u    Note
//	The specified point is allowed to lie outside the intervals. If you
//	want to limit the returned value, use @QwtDiMap::limTransform@.
//------------------------------------------------------------
int QwtDiMap::transform(double x) const
{
    if (d_log)
       return (d_y1 + int(rint( (log(x) - d_x1) * d_cnv )));    
    else
       return (d_y1 + int(rint( (x - d_x1) * d_cnv )));
}

//------------------------------------------------------------
//
//.F	QwtDiMap::invTransform
//	Transform an integer value into a double value
//
//.u	Syntax
//.f	double QwtDiMap::invTransform(int y)
//
//.u	Parameters
//.p	int y	--	integer value to be transformed
//
//.u	Return Value
//.t
//	linear mapping:	-- d1 + (d2 - d1) / (i2 - i1) * (y - i1)
//	logarithmic mapping: -- d1 + (d2 - d1) / log(i2 / i1) * log(y / i1)
//
//
//------------------------------------------------------------
double QwtDiMap::invTransform(int y) const 
{
    if (d_cnv == 0.0)
       return 0.0;
    else
    {
	if(d_log) 
	   return exp(d_x1 + double(y - d_y1) / d_cnv );
	else
	   return ( d_x1 + double(y - d_y1) / d_cnv );	
    }
}


//------------------------------------------------------------
//
//.F	QwtDiMap::limTransform
//	Transform and limit
//
//.u	Syntax
//.f	int QwtDiMap::limTransform(double x)
//
//.u	Parameters
//.p	double x
//
//.u	Return Value
//		transformed value
//
//.u	Description
//	The function is similar to @QwtDiMap::transform@, but limits the input value
//	to the nearest border of the map's double interval if it lies outside
//	that interval.
//
//------------------------------------------------------------
int QwtDiMap::limTransform(double x) const
{
    if ( x > qwtMax(d_x1, d_x2) )
       x = qwtMax(d_x1, d_x2);
    else if ( x < qwtMin(d_x1, d_x2))
       x = qwtMin(d_x1, d_x2);
    return transform(x);
}

//------------------------------------------------------------
//
//.F	QwtDiMap::xTransform
//	Exact transformation
//
//.u	Syntax
//.f	double QwtDiMap::dTransform(double x)
//
//.u	Parameters
//.p		double x	-- value to be transformed
//
//.u	Return Value
//.t
//	linear mapping:	-- i1 + (i2 - i1) / (d2 - d1) * (x - d1)
//	logarithmic mapping: -- i1 + (i2 - i1) / log(d2 / d1) * log(x / d1)
//
//.u	Description
//	This function is similar to @QwtDiMap::transform@, but
//	makes the integer interval appear to be double. 
//
//------------------------------------------------------------
double QwtDiMap::xTransform(double x) const
{
    double rv;
    
    if (d_log)
       rv = double(d_y1) + (log(x) - d_x1) * d_cnv;    
    else
       rv = double(d_y1) + (x - d_x1) * d_cnv;

    return rv;
}


//------------------------------------------------------------
//.-
//.F	QwtDiMap::newFactor
//	Re-calculate the conversion factor.
//------------------------------------------------------------
void QwtDiMap::newFactor()
{
    if (d_x2 != d_x1)
       d_cnv = double(d_y2 - d_y1) / (d_x2 - d_x1); 
    else 
       d_cnv = 0.0;
}

//------------------------------------------------------------
//
//.F	QwtDiMap::d1
//	Returns the first border of the double interval
//
//.u	Syntax
//.f	double QwtDiMap::d1()
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtDiMap::d2
//	Returns the second border of the double interval
//
//.u	Syntax
//.f	double QwtDiMap::d2()
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtDiMap::i1
//	Returns the first border of the integer interval
//
//.u	Syntax
//.f	int QwtDiMap::i1()
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtDiMap::i2
//	Returns the second border of the integer interval
//
//.u	Syntax
//.f	int QwtDiMap::i2()
//
//------------------------------------------------------------


//------------------------------------------------------------
//
//.F	QwtDiMap::logarithmic
//	Returns TRUE if the double interval is scaled logarithmically
//
//.u	Syntax
//.f	bool QwtDiMap::logarithmic()
//
//------------------------------------------------------------




