#include "qwt_curve.h"
#include "qwt_math.h"
#include <qpntarry.h>
#include <qpainter.h>

//--------------------------------------------------
//.H  QwtCurve | 3 | 01/01/98 | Qwt Widget Library | Qwt Programmer's Manual
//.I curves Different curve styles
//.U  NAME
//	QwtCurve - A class which draws curves
//
//.U DESCRIPTION
//	This class can be used to display data as a curve in the  x-y plane.
//	It supports different display styles, spline interpolation and symbols.
//
//.U USAGE
//.u	A. Assign y and y data
//	Data can be assigned in two ways:
//.i
//	-- @QwtCurve::setData@ copies the x and y data from the specified
//	    arrays into its internal buffer.
//	-- @QwtCurve::setRawData@ does not make a copy of the data, but
//	   only stores the pointers and size information instead. This
//	   function is less safe (you must not delete the data while they are attached),
//	   but more efficient, and more
//	   convenient if the data change dynamically.
//
//.u	B. Assign Properties
//	When a curve is created, it is configured to draw black solid lines
//	with no symbols. You can change this by calling the @QwtCurve::setPen@
//	and @QwtCurve::setSymbolY@ members.
//
//.u	C. Assign y and y maps
//	Before a curve can be drawn, it has be mapped into a rectangle.
//	This can either be done implicitly with the
//	@QwtCurve::draw (2)@ member, or explicitly using @QwtCurve::setMap (1)@
//	or @QwtCurve::setMap (2)@ before drawing. The map can also be changed with the
//	@QwtCurve::setRect@ and @QwtCurve::setRange@ members.
//
//.u    D. Draw
//	The @QwtCurve::draw (1)@ member assumes that a curve has already been mapped.
//	The  @QwtCurve::draw (2)@ member draws the curve with specified y and y
//	maps. The @QwtCurve::draw (3)@ member draws the curve into a rectangle,
//	thereby adjusting the curve's mapping.
//
//.U PUBLIC MEMBERS
//.R
//	QwtCurve::draw (1) -- Draw the curve 
//	QwtCurve::draw (2) -- Assign maps and raw the curve
//	QwtCurve::draw (3) -- Assign a bounding rectangle and draw the curve
//	QwtCurve::drawIntv -- Draw a specified part of the curve
//	QwtCurve::QwtCurve (1) -- Constructor
//	QwtCurve::QwtCurve (2) -- Constructor
//	QwtCurve::~QwtCurve -- Destructor
//	QwtCurve::operator= -- Assignment operator
//	QwtCurve::setData -- Copy data from arrays
//	QwtCurve::setMap (1) -- Assign maps
//	QwtCurve::setMap (2) -- Assign maps
//	QwtCurve::setOptions -- Specify style options
//	QwtCurve::setPen --	Assign a pen
//	QwtCurve::setRange --	Specify a range
//	QwtCurve::setRawData -- Attach arrays
//	QwtCurve::setRect --	Specify drawing region
//	QwtCurve::setBaseline --  Set  baseline to value
//	QwtCurve::setStyle --	Set style
//	QwtCurve::setSymbol --	Assign a symbol
//	QwtCurve::setSplineSize -- Change Spline size	
//	QwtCurve::setTitle -- Assign a title
//
//	QwtCurve::dataSize -- Return data size
//	QwtCurve::minXValue -- Return smallest x value
//	QwtCurve::minYValue -- Return smallest y value
//	QwtCurve::maxXValue -- Return greatest x value
//	QwtCurve::maxYValue -- Return greatest y value
//	QwtCurve::options --	Return style options
//	QwtCurve::pen --	Return pen
//	QwtCurve::baseline --	Return value of the baseline
//	QwtCurve::splineSize --	Return spline size
//	QwtCurve::style --	Return style
//	QwtCurve::symbol --	Return symbol
//	QwtCurve::title --	Return title
//	QwtCurve::x --		Return x value with specified index
//	QwtCurve::y --		Return y value with specified index	
//
//.U	Protected members
//.R
//	QwtCurve::curveChanged -- Called when properties have been changed (virtual)
//	QwtCurve::verifyRange -- Correct a specified range if invalid
//
//.U	See also
//      @^QwtSymbol@, @^QwtDiMap@
//
//.U EXAMPLE
//	see examples/curvdemo
//
//.-
//.U COPYING
//
//	Copyright (C) 1997  Josef Wilgen
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License, version 2,
//	as published by	the Free Software Foundation.
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//.+ 	 
//--------------------------------------------------

//------------------------------------------------------------
//.U MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

//------------------------------------------------------------
//.-
//.F	QwtCurve::init
//	Initialize data members
//
//.u	Syntax
//.f	void QwtCurve::init()
//
//------------------------------------------------------------
void QwtCurve::init(const char *title)
{

    d_pen = QPen(QColor(255,255,255), 1);
    d_ref = 0.0;
    d_splineSize = 250;
    d_options = Auto;
    d_xraw = FALSE;
    d_yraw = FALSE;
    
    if (title)
       d_title = title;
    else
       d_title = "";
	
    d_style = Lines;

}

//------------------------------------------------------------
//.-
//.F	QwtCurve::copy
//	Copy the contents of a curve into another curve
//
//.u	Syntax
//.f	void QwtCurve::copy(const QwtCurve &c)
//
//.u	Parameters
//.p	const QwtCurve &c
//
//------------------------------------------------------------
void QwtCurve::copy(const QwtCurve &c)
{

    d_ref = c.d_ref;
    d_xMap = c.d_xMap;
    d_yMap = c.d_yMap;
    d_sym = c.d_sym;
    d_pen = c.d_pen;
    d_title = c.d_title;
    d_xraw = FALSE;
    d_yraw = FALSE;
    d_style = c.d_style;

    d_splineSize = c.d_splineSize;
    d_options = c.d_options;
    
    if (c.d_xraw)
       d_x.duplicate(c.d_x);
    else
       d_x = c.d_x;

    if (c.d_yraw)
       d_y.duplicate(c.d_y);
    else
       d_y = c.d_y;
}

//------------------------------------------------------------
//
//.F	QwtCurve::~QwtCurve
//	Destructor
//
//.u	Syntax
//.f	 QwtCurve::~QwtCurve()
//
//------------------------------------------------------------
QwtCurve::~QwtCurve()
{
    if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
    if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());

}

//------------------------------------------------------------
//
//.F	QwtCurve::QwtCurve
//	Copy Constructor
//
//.u	Syntax
//.f	 QwtCurve::QwtCurve(const QwtCurve &c)
//
//.u	Note
//	If the c has attached its data in raw mode
//	(see @QwtCurve::setRawData@), the copy constructor
//	creates a deep copy of the data. The data are implicitly
//	shared otherwise.
//	
//------------------------------------------------------------
QwtCurve::QwtCurve(const QwtCurve &c)
{
    init(c.d_title);
    copy(c);
}

//------------------------------------------------------------
//
//.F	QwtCurve::operator=
//	Assignment operator
//
//.u	Syntax
//.f	const QwtCurve & QwtCurve::operator=(const QwtCurve &)
//
//.u	Note
//	If the c has attached its data in raw mode
//	(see @QwtCurve::setRawData@), the assignment operator
//	creates a deep copy of the data. The data are implicitly
//	shared otherwise.
//
//------------------------------------------------------------
const QwtCurve& QwtCurve::operator=(const QwtCurve &c)
{
    copy(c);
    curveChanged();
    return *this;
}

//------------------------------------------------------------
//
//.F	QwtCurve::setStyle
//	Set the curve's drawing style
//
//.u	Syntax
//.f	void QwtCurve::setStyle(CurveStyle cs, int options)
//
//.u	Parameters
//.p	QwtCurve::CurveStyle cs
//
//.u	Description
//	Valid styles are
//.t	QwtCurve::NoCurve -- Don't draw a curve. Note: This doesn't affect
//				the symbol. 
//	QwtCurve::Lines	--   Connect the points with straight lines
//	QwtCurve::Sticks --  Draw vertical sticks from a baseline which is
//				defined by @QwtCurve::setBaseline@.
//	QwtCurve::Steps --   Connect the points with a step function. The step function
//				is drawn from the left to the right or vice versa,
//				depending on the 'Inverted' option.
//	QwtCurves::Dots --   Draw dots at the locations of the data points. Note:
//				This is different from a dotted line (see
//				@QwtCurve::setPen@).
//	QwtCurve::Spline --  Interpolate the points with a spline. The spline
//				type can be specified with @QwtCurve::setOptions@,
//				the size of the spline (= number of interpolated points)
//				can be specified with @QwtCurve::setSplineSize@.
//
//------------------------------------------------------------
void QwtCurve::setStyle(CurveStyle cs, int options)
{
    d_options = options;
    d_style = cs;
    curveChanged();
}

//------------------------------------------------------------
//
//.F	QwtCurve::setSymbol
//	Assign a symbol
//
//.u	Syntax
//.f	void QwtCurve::setSymbol(const QwtSymbol &s )
//
//.U	Parameters:
//.p	const QwtSymbol &s	--	 symbol
//
//.u	See also
//	@^QwtSymbol@
//------------------------------------------------------------
void QwtCurve::setSymbol(const QwtSymbol &s )
{
    d_sym = s;
    curveChanged();
}


//------------------------------------------------------------
//
//.F	QwtCurve::setPen
//	Assign a pen
//
//.u	Syntax
//.f	void QwtCurve::setPen(const QwtPen &p)
//
//.u	Parameters
//.p	const QPen &p	-- New pen
//
//------------------------------------------------------------
void QwtCurve::setPen(const QPen &p)
{
    d_pen = p;
    curveChanged();
}



//------------------------------------------------------------
//
//.F	QwtCurve::setData
//	Copy x-y data from specified arrays
//
//.u	Syntax
//.f	void QwtCurve::setData(double *x, double *y, int size)
//
//.u	Parameters
//.p	double *x, double *y	-- x and y data arrays
//	int size	-- size of x and y
//
//------------------------------------------------------------
void QwtCurve::setData(double *x, double *y, int size)
{
    if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
    if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());
    d_x.duplicate(x, size);
    d_y.duplicate(y, size);
    d_xraw = FALSE;
    d_yraw = FALSE;
    curveChanged();
}

//------------------------------------------------------------
//
//.F	QwtCurve::setRawData
//	Attach raw data
//
//.u	Syntax
//.f	void QwtCurve::setRawData(double *x, double *y, int size)
//
//.u	Parameters
//.p	double *x, double *y	-- x and y data arrays
//	int size	-- size of y and y
//
//.u	Description
//	setRawData is provided for efficiency. In contrast to setData,
//	it does not copy the data, so it is important to keep the pointers
//	valid while they are attached.  
//	The QwtCurve destructor does not delete the attached data, so you
//	can safely call setRawData and setData several times subsequently.
//	
//------------------------------------------------------------
void QwtCurve::setRawData(double *x, double *y, int size)
{
    if (d_xraw) d_x.resetRawData(d_x.data(), d_x.size());
    if (d_yraw) d_y.resetRawData(d_y.data(), d_y.size());
    d_x.setRawData(x, size);
    d_y.setRawData(y, size);
    d_xraw = TRUE;
    d_yraw = TRUE;
    curveChanged();
}

//------------------------------------------------------------
//
//.F QwtCurve::setTitle
//	Assign a title to a curve
//
//.u Syntax
//.f	void QwtCurve::setTitle(const char *title)
//
//.u  Parameters
//	const char *title
//------------------------------------------------------------
void QwtCurve::setTitle(const char *title)
{
    d_title = title;
    curveChanged();
}

//------------------------------------------------------------
//
//.F QwtCurve::minXValue
//	find the smallest x value
//.u Syntax
//.f double QwtCurve::minXValue() const
//------------------------------------------------------------
double QwtCurve::minXValue() const
{
    return qwtGetMin(d_x.data(),d_x.size());
}

//------------------------------------------------------------
//
//.F QwtCurve::minYValue
//	find the smallest y value
//.u Syntax
//.f double QwtCurve::minYValue() const
//
//------------------------------------------------------------
double QwtCurve::minYValue() const
{
    return qwtGetMin(d_y.data(), d_y.size());
}

//------------------------------------------------------------
//
//.F	QwtCurve::maxXValue
//	Find the largest x value
//.u	Syntax
//.f	double QwtCurve::maxXValue() const
//------------------------------------------------------------
double QwtCurve::maxXValue() const
{
    return qwtGetMax(d_x.data(), d_x.size());
}

//------------------------------------------------------------
//
//.F QwtCurve::maxYValue
//	Find the largest y value
//
//.u	Syntax
//.f	double QwtCurve::maxYValue() const
//
//------------------------------------------------------------
double QwtCurve::maxYValue() const
{
    return qwtGetMax(d_y.data(), d_y.size());
}


//------------------------------------------------------------
//
//.F QwtCurve::QwtCurve
//	Constructor
//
//.u	Syntax
//.f	QwtCurve::QwtCurve(const char *title)
//
//.u	Parameters
//.p	const char *title  -- title of the curve	
//------------------------------------------------------------
QwtCurve::QwtCurve(const char *title)
{
    init (title);
}

//------------------------------------------------------------
//
//.F	QwtCurve::verifyRange
//	Checks if a range of indices is valid and corrects
//	it if necessary
//
//.u	Syntax
//.f	int QwtCurve::verifyRange(int &i1, int &i2)
//
//.u	Parameters
//.p	int &i1, int &i2
//
//------------------------------------------------------------
int QwtCurve::verifyRange(int &i1, int &i2)
{
    int size = dataSize();

    if (size < 1) return 0;
    
    i1 = qwtLim(i1, 0, size-1);
    i2 = qwtLim(i2, 0, size-1);
    qwtSort(i1, i2, i1, i2);
    
    return (i2 - i1 + 1);
}

//------------------------------------------------------------
//
//.F QwtCurve::draw (2)
//	Assign maps and draw the curve
//
//.u	Syntax
//.f	void QwtCurve::draw(QPainter *p, const QwtDiMap &xMap, const QwtDiMap &yMap)
//
//.u Parameters
//.p QPainter *p, const QwtDiMap &xMap, const QwtDiMap &yMap
//
//.u    Note
//	This function changes the maps and overrides any previous settings
//	done by @QwtCurve::setRect@, @QwtCurve::setRange@,
//	@QwtCurve::setMap (1)@, and @QwtCurve::setMap (2)@.
//------------------------------------------------------------
void QwtCurve::draw(QPainter *p, const QwtDiMap &xMap, const QwtDiMap &yMap)
{

    d_xMap = xMap;
    d_yMap = yMap;
    draw(p);
    
}

//------------------------------------------------------------
//
//.F	QwtCurve::draw (1)
//	Draw the curve
//
//.u	Syntax
//.f	void QwtCurve::draw(QPainter *p)
//
//.u	Parameters
//.p	QPainter *p -- Painter
//
//.u	Note
//	Before drawing, you have to assign a map using
//	@QwtCurve::setMap (1)@ or @QwtCurve::setMap (2)@.
//
//------------------------------------------------------------
void QwtCurve::draw(QPainter *p)
{

    if ( dataSize() <= 0 ) return;

    switch (d_style)
    {
    case NoCurve:
	break;
    case Lines:
	drawLines(*p);
	break;
    case Sticks:
	drawSticks(*p);
	break;
    case Steps:
	drawSteps(*p);
	break;
    case Dots:
	drawDots(*p);
	break;
    case Spline:
	drawSpline(*p);
	break;
    default:
	drawDots(*p);
    }
    if (d_sym.style() != QwtSymbol::None)
    {
	drawSymbols(*p);
    }
}

//------------------------------------------------------------
//
//.F	QwtCurve::drawIntv
//	Draw a specified part of the curve
//
//.u	Syntax
//.f	void QwtCurve::drawIntv(QPainter *p, int imin, int imax)
//
//.u	Parameters
//.p	QPainter *p	--	Painter
//	int imin, int imax -- Interval boundaries as indices of the
//			      data arrays. The subset to be drawn includes both
//			      boundaries. 
//
//.u	Bugs
//	Drawing subintervals does not work with QwtCurve::Spline.
//
//------------------------------------------------------------
void QwtCurve::drawIntv(QPainter *p, int imin, int imax)
{

    if ( dataSize() <= 0 ) return;

    switch (d_style)
    {
    case NoCurve:
	break;
    case Lines:
    case Spline:
	drawLines(*p, imin, imax);
	break;
    case Sticks:
	drawSticks(*p, imin, imax);
	break;
    case Steps:
	drawSteps(*p, imin, imax);
	break;
    case Dots:
	drawDots(*p, imin, imax);
	break;
    default:
	drawDots(*p, imin, imax);
    }
    if (d_sym.style() != QwtSymbol::None)
    {
	drawSymbols(*p, imin, imax);
    }
    
}


//------------------------------------------------------------
//
//.F	QwtCurve::draw (3)
//	Assign a bounding rectangle and draw the curve
//
//.u	Syntax
//.f	void QwtCurve::draw(QPainter *p, const QRect &r)
//
//.u	Parameters
//.p	QPainter *p	-- Painter
//	const QRect &r  -- Bounding rectangle
//
//.u    Note
//	This function changes the maps and overrides previous settings
//	done by @QwtCurve::setRect@,
//	@QwtCurve::setMap (1)@, and @QwtCurve::setMap (2)@.
//------------------------------------------------------------
void QwtCurve::draw(QPainter *p, const QRect &r)
{
    setRect(r);
    draw(p);
}

//------------------------------------------------------------
//.-
// Name: QwtCurve::drawLines
//
//------------------------------------------------------------
void QwtCurve::drawLines(QPainter &p, int i1, int i2)
{
	int i, ip;
	int u1,v1,u2,v2;

	int size; 

	if (i2 < 0) i2 = dataSize() - 1; 
	size = verifyRange(i1, i2);

	if (size <= 1) return;
	
	if (d_pa.size() != uint(size))
	{
		if(!d_pa.resize(uint(size)))
		   return;
	}
	
	p.setPen(d_pen);

	transform(d_x[i1], d_y[i1], u1, v1);
	d_pa.setPoint(0,u1,v1);
	p.moveTo(u1,v1);
	for (i=i1+1,ip=1;i<=i2;i++,ip++)
	{
		transform(d_x[i], d_y[i], u2,v2);
		d_pa.setPoint(ip,u2,v2);
		u1 = u2;
		v1 = v2;
	}
	p.drawPolyline(d_pa);
	
}


//------------------------------------------------------------
//.-
//.F	QwtCurve::drawSpline
//
//
//.u	Syntax
//.f	void QwtCurve::drawSpline(QPainte *p)
//
//------------------------------------------------------------
void QwtCurve::drawSpline(QPainter &p)
{

    double xmin, xmax, ymin, ymax, delta;
    double *param;
    double dtmp;
    int i;
    int stype;
    int rc;
    int size = dataSize();
    double *txval = new double[size];
    double *tyval = new double[size];
	
	
    if ( (!txval) | (!tyval) | (!d_pa.resize(d_splineSize)) )
    {
	if (txval) delete[] txval;
	if (tyval) delete[] tyval;
	return;
    }

    //
    // Transform x and y values to window coordinates
    // to avoid a distinction between linear and
    // logarithmic scales.
    //
    for (i=0;i<size;i++)
    {
	txval[i] = d_xMap.xTransform(d_x[i]);
	tyval[i] = d_yMap.xTransform(d_y[i]);
    }

    if (! (d_options & (Yfx|Xfy|Parametric)))
    {
	
	if (qwtChkMono(txval, size))
	{
	    stype = Yfx;
	}
	else
	{
	    
	    if(qwtChkMono(tyval, size))
	    {
		stype = Xfy;
	    }
	    else
	    {
		stype = Parametric;
		if ( (d_options & Periodic) ||
		    ( (d_x[0] == d_x[size-1])
		    && (d_y[0] == d_y[size-1])))
		{
		    stype |= Periodic;  
		}
		
	    }
	}
    }
    else
    {
	stype = d_options;
    }
    
    if (stype & Parametric)
    {
	param = new double[size];
	if (param)
	{
	    //
	    // setup parameter vector
	    //
	    param[0] = 0.0;
	    for (i=1;i<size; i++)
	    {
		delta = sqrt( qwtSqr(txval[i] - txval[i-1]) + qwtSqr( tyval[i] - tyval[i-1]));
		param[i] = param[i-1] + qwtMax(delta, 1.0);
	    }
	    
	    //
	    // setup splines
	    rc = d_spx.recalc(param, txval, size, stype & Periodic);
	    if (!rc) rc = d_spy.recalc(param, tyval, size, stype & Periodic);
	    
	    if (rc)
	    {
		drawLines(p);
	    }
	    else
	    {
		
		// fill point array
		delta = param[size - 1] / double(d_splineSize-1);
		for (i=0;i<d_splineSize;i++)
		{
		    dtmp = delta * double(i);
		    d_pa.setPoint(i, int(rint(d_spx.value(dtmp))),
				  int(rint(d_spy.value(dtmp))));
		}
	    }
	    
	    
	    delete[] param;
	}
    }
    else if (stype & Xfy)
    {
	if (tyval[size-1] < tyval[0])
	{
	    qwtTwistArray(txval, size);
	    qwtTwistArray(tyval, size);
	}
	
	// 1. Calculate spline coefficients
	rc = d_spx.recalc(tyval, txval, size, stype & Periodic);
	
	if (rc)				// an error occurred
	{
	    drawLines(p);
	}
	else				// Spline OK
	{	
	    ymin = qwtGetMin(tyval, size);
	    ymax = qwtGetMax(tyval, size);
	    delta = (ymax - ymin) / double(d_splineSize - 1);
	    
	    for (i=0;i<d_splineSize;i++)
	    {
		dtmp = ymin + delta * double(i);
		d_pa.setPoint(i, int(rint(d_spx.value(dtmp))),
			      int(rint(dtmp)));
	    }
	}
    }		
    else
    {
	if (txval[size-1] < txval[0])
	{
	    qwtTwistArray(tyval, size);
	    qwtTwistArray(txval, size);
	}
	
	
	// 1. Calculate spline coefficients
	rc = d_spy.recalc(txval, tyval, size, stype & Periodic);
	
	if (rc)				// error
	{
	    drawLines(p);
	}
	else				// Spline OK
	{
	    xmin = qwtGetMin(txval, size);
	    xmax = qwtGetMax(txval, size);
	    delta = (xmax - xmin) / double(d_splineSize - 1);
	    
	    for (i=0;i<d_splineSize;i++)
	    {
		dtmp = xmin + delta * double(i);
		d_pa.setPoint(i, int(rint(dtmp)),
			      int(rint(d_spy.value(dtmp))));
	    }
	}
    }
    
    p.setPen(d_pen);
    p.drawPolyline(d_pa);
    
    delete[] txval;
    delete[] tyval;
    
}

//------------------------------------------------------------
//
//.F	QwtCurve::setOptions
//	Specify options for the drawing style	
//
//.u	Syntax
//.f	void QwtCurve::setOptions(int opt)
//
//.u	Parameters
//.p	int opt	--	new options
//
//
//.u    Description
//      The options can be used to modify the drawing style.
//	Options can be or-combined.
//	The following options are defined:
//.t
//	QwtCurve::Auto   --   The default setting. For QwtCurve::spline,
//				this means that the type of the spline is
//				determined automatically, depending on the data.
//				For all other styles, this means that y is
//				regarded as a function of x.
//	QwtCurve::Yfx    --   Draws y as a function of x (the default). The
//			      baseline is interpreted as a horizontal line
//				with y = baseline().
//	QwtCurve::Xfy    --   Draws x as a function of y. The baseline is
//				interpreted as a vertical line with x = baseline().
//	QwtCurve::Parametric  -- For QwtCurve::Spline only. Draws a parametric spline.
//	QwtCurve::Periodic   -- For QwtCurve::Spline only. Draws a periodic spline.
//	QwtCurve::Inverted  -- For QwtCurve::Steps only. Draws a step function
//				from the right to the left.
//      
//------------------------------------------------------------
void QwtCurve::setOptions(int opt)
{
    d_options = opt;
    curveChanged();
}

//------------------------------------------------------------
//
//.F	QwtCurve::setSplineSize
//	Change the number of interpolated points
//
//.u	Syntax
//.f	void QwtCurve::setSplineSize(int s)
//
//.u	Parameters
//.p	int s	--	new size
//
//.u	Note
//		The default is 250 points.
//------------------------------------------------------------
void QwtCurve::setSplineSize(int s)
{
    d_splineSize = qwtMax(s, 10);
    curveChanged();
}


//------------------------------------------------------------
//.-
//.F	QwtCurve::drawSticks
//
//
//.u	Syntax
//.f	void QwtCurve::drawSticks(QPainter &p)
//
//.u	Parameters
//.p	QPainter &p
//
//------------------------------------------------------------
void QwtCurve::drawSticks(QPainter &p, int i1, int i2)
{
    int i;
    int u,v;
    int u0, v0;
    int size;
    
    if (i2 < 0) i2 = dataSize() - 1; 
    size = verifyRange(i1, i2);

    if (size <= 0) return;
    
    p.setPen(d_pen);

    if (d_options & Xfy)
    {

	u0 = d_xMap.transform(qwtLim(d_ref, d_xMap.d1(), d_xMap.d2()));
    
	for (i=i1;i<=i2;i++)
	{
	    transform(d_x[i], d_y[i], u,v);
	    p.drawLine(u0, v, u,v);
	}
    }
    else
    {
	v0 = d_yMap.transform(qwtLim(d_ref, d_yMap.d1(), d_yMap.d2()));
    
	for (i=i1;i<=i2;i++)
	{
	    transform(d_x[i], d_y[i], u,v);
	    p.drawLine(u,v0,u,v);
	}
    }
    
}

//------------------------------------------------------------
//.-
//.F	QwtCurve::drawDots
//
//.u	Syntax
//.f	void QwtCurve::drawDots(QPainter &p)
//
//.u	Parameters
//.p	QPainter &p
//
//------------------------------------------------------------
void QwtCurve::drawDots(QPainter &p, int i1, int i2)
{
    int i;
    int u,v;
    int size;
    
    if (i2 < 0) i2 = dataSize() - 1; 
    size = verifyRange(i1, i2);

    if (size <= 0) return;
    
    p.setPen(d_pen);
    
    for (i=1;i<=i2;i++)
    {
	transform(d_x[i], d_y[i], u,v);
	p.drawPoint(u,v);
    }
    
}


//------------------------------------------------------------
//.-
//.F	QwtCurve::drawSteps
//
//.u	Syntax
//.f	void QwtCurve::drawSteps(QPainter &p)
//
//
//.u	Parameters
//.p	QPainter &p
//
//------------------------------------------------------------
void QwtCurve::drawSteps(QPainter &p, int i1, int i2)
{
    int i,ip;
    int u1,v1,u2,v2,w1,w2;
    int size;
    
    if (i2 < 0) i2 = dataSize() - 1; 
    size = verifyRange(i1, i2);

    if (size <= 1) return;
    
    if (d_pa.size() != uint (2*size - 1))
    {
	if (!d_pa.resize(uint(2*size - 1)))
	   return;
    }
    
    p.setPen(d_pen);
    
    w1 = (d_pen.width() +1) /2;
    w2 = d_pen.width() / 2;
    
    transform(d_x[i1], d_y[i1], u1, v1);
    d_pa.setPoint(0, u1,v1);   

    if (((d_options & Xfy) && (!(d_options & Inverted)))
	|| ((d_options & Inverted) && (!(d_options & Xfy))))
    {
	for (i=i1+1, ip=2;i<=i2;i++, ip+=2)
	{
	    transform(d_x[i], d_y[i], u2,v2);
	    d_pa.setPoint(ip - 1, u1,v2);   
	    d_pa.setPoint(ip, u2,v2);
	    u1 = u2;
	    v1 = v2;
	}
    }
    else
    {
	for (i=i1+1,ip=2;i<= i2;i++,ip+=2)
	{
	    transform(d_x[i], d_y[i], u2,v2);
	    d_pa.setPoint(ip - 1, u2,v1);   
	    d_pa.setPoint(ip, u2,v2);
	    u1 = u2;
	    v1 = v2;
	}
    }
    
    p.drawPolyline(d_pa);
    
    
}

//------------------------------------------------------------
//.-
//.F	QwtCurve::transform
//
//.u	Syntax
//.f	void QwtCurve::transform(double x, double y, int &u, int &v)
//
//------------------------------------------------------------
void QwtCurve::transform(double x, double y, int &u, int &v)
{
    u = d_xMap.transform(x);
    v = d_yMap.transform(y);
}


//------------------------------------------------------------
//.-
//.F	QwtCurve::drawSymbols
//
//.u	Syntax
//.f	void QwtCurve::drawSymbols(QPainter &p)
//
//
//.u	Parameters
//.p	QPainter &p
//
//------------------------------------------------------------
void QwtCurve::drawSymbols(QPainter &p, int i1, int i2)
{
    int i, u, v;
    int size;
    
    if (i2 < 0) i2 = qwtMin(d_x.size(), d_y.size()) - 1; 
    size = verifyRange(i1, i2);

    if (size < 1) return;
    int w2 = (d_sym.size().width() - 1) / 2;
    int h2 = (d_sym.size().height() - 1) / 2;
    
    for (i=i1;i<=i2;i++)
    {
	transform(d_x[i], d_y[i], u,v);
	d_sym.draw(&p, u - w2, v - h2);
    }
}


//------------------------------------------------------------
//
//.F	QwtCurve::setMap (2)
//	Specify x and y ranges
//
//.u	Syntax
//.f	void QwtCurve::setMap(const QRect &r, double x1, double x2, bool xlog, double y1, double y2, bool ylog)
//
//.u	Parameters
//.p	const QRect &r  --	bounding rectangle
//	double x1	--	left boundary of the x axis
//	double x2	--	right boundary of the y axis
//	bool xlog	--	logarithmic x division if nonzero
//	double y1	--	lower boundary of the y axis
//	double y2	--	upper boundary of the y axis
//	bool ylog	--	logarithmic y division if nonzero
//
//.u    Note
//	This function overrides the settings of @QwtCurve::setRect@
//	and @QwtCurve::setRange@.
//------------------------------------------------------------
void QwtCurve::setMap(const QRect &r, double x1, double x2, bool xlog, double y1, double y2, bool ylog)
{
    d_xMap.setDblRange(x1, x2, xlog);
    d_yMap.setDblRange(y1, y2, ylog);
    d_xMap.setIntRange(r.left(), r.right());
    d_yMap.setIntRange(r.top(), r.bottom());
}

//------------------------------------------------------------
//
//.F	QwtCurve::setRect
//	Changes the drawing region of the map.
//
//.u	Syntax
//.f	void QwtCurve::setRect(const QRect &r)
//
//.u	Parameters
//.p	const QRect &r	-- Bounding rectangle
//
//.u    Note
//	This function is intended to be used in combination
//	with @QwtCurve::setRange@. It modifies the maps and conflicts with
//	@QwtCurve::setMap (1)@ and @QwtCurve:.setMap (2)@ 
//------------------------------------------------------------
void QwtCurve::setRect(const QRect &r)
{
    d_xMap.setIntRange(r.left(), r.right());
    d_yMap.setIntRange(r.top(), r.bottom());
}


//------------------------------------------------------------
//
//.F	QwtCurve::setRange
//	Changes the data range of the map.
//
//.u	Syntax
//.f	void QwtCurve::setRange(double x1, double x2, bool xlog, double y1, double y2, bool ylog)
//
//.u	Parameters
//.p	double x1, double x2	-- X range
//	bool xlog	-- TRUE if x range is logarithmic
//	double y1, double y2 -- y range
//	bool ylog    -- TRUE if y range is logarithmic
//
//.u    Note
//	This function is intended to be used in combination
//	with @QwtCurve::setRect@. It modifies the maps and conflicts with
//	@QwtCurve::setMap (1)@ and @QwtCurve:.setMap (2)@ 
//
//------------------------------------------------------------
void QwtCurve::setRange(double x1, double x2, bool xlog, double y1, double y2, bool ylog)
{
    d_xMap.setDblRange(x1, x2, xlog);
    d_yMap.setDblRange(y1, y2, ylog);
}

//------------------------------------------------------------
//
//.F	QwtCurve::setMap (1)
//	Assign the maps for the x and y axes
//
//.u	Syntax
//.f	void QwtCurve::setMap(const QwtDiMap &xmap, const QwtDiMap &yMap)
//
//.u	Parameters
//.p	const QwtDiMap &xmap, const QwtDiMap &yMap -- Maps
//
//.u    Note
//	This function overrides the settings of @QwtCurve::setRect@
//	and @QwtCurve::setRange@.
//
//------------------------------------------------------------
void QwtCurve::setMap(const QwtDiMap &xmap, const QwtDiMap &ymap)
{
    d_xMap = xmap;
    d_yMap = ymap;
}

//------------------------------------------------------------
//
//.F	QwtCurve::setBaseline
//	Set the value of the baseline 
//
//.u	Syntax
//.f	void QwtCurve::setBaseline(double ref)
//
//.u	Parameters
//.p	double ref	--	baseline
//
//.u	Description
//	The baseline is needed for the QwtCurve::Sticks
//	drawing style. The default value is 0.0. The interpretation
//	of the baseline depends on the style options. With QwtCurve::Yfx,
//	the baseline is interpreted as a horizontal line at y = baseline(),
//	with QwtCurve::Yfy, it is interpreted as a vertical line at
//	x = baseline().
//
//.u	See also:
//	@QwtCurve::setStyle@, @QwtCurve::setOptions@
//
//------------------------------------------------------------
void QwtCurve::setBaseline(double ref)
{
    d_ref = ref;
    curveChanged();
}


//------------------------------------------------------------
//
//.F	QwtCurve::dataSize
//	Return the size of the data arrays
//
//.u	Syntax
//.f	int QwtCurve::dataSize() const
//
//------------------------------------------------------------
int QwtCurve::dataSize() const
{
    return qwtMin(d_x.size(), d_y.size());
}

//------------------------------------------------------------
//
//.F	QwtCurve::options
//	Return the current style options
//
//.u	Syntax
//.f	int QwtCurve::options()
//
//.u	See Also
//	@QwtCurve::setOptions@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtCurve::pen
//	Return the pen used to draw the lines
//
//.u	Syntax
//.f	const QPen& QwtCurve::pen()
//
//.u	See Also
//	@QwtCurve::setPen@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtCurve::baseline
//	Return the value of the baseline
//
//.u	Syntax
//.f	double QwtCurve::baseline()
//
//.u	See Also
//	@QwtCurve::setBaseline@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtCurve::splineSize
//	Return the spline size
//
//.u	Syntax
//.f	int QwtCurve::splineSize()
//
//.u	Description
//	@QwtCurve::setSplineSize@
//------------------------------------------------------------

//------------------------------------------------------------
//
//.F	QwtCurve::style
//	Return the current style
//
//.u	Syntax
//.f	CurveStyle QwtCurve::style()
//
//.u	See Also
//
//------------------------------------------------------------













