#ifndef QWT_PLOT_H
#define QWT_PLOT_H

#include <qframe.h>
#include <qwt_grid.h>
#include <qwt_autoscl.h>
#include <qwt_scldraw.h>
#include <qprinter.h>
#include <qwt_symbol.h>
#include <qwt_curve.h>
#include <qwt_marker.h>
#include <qwt_clrfltr.h>
#include <qwt_plot_classes.h>

class QwtPlotPixFrame;
class QLabel;
class QwtScale;
class QwtLegend;
class QwtCurveDict;
class QwtMarkerDict;

class QwtPlot: public QFrame
{
friend class QwtPlotItem;
friend class QwtPlotPixFrame;

    Q_OBJECT

    
public:
    
    enum {yLeft, yRight, xBottom, xTop, axisCnt};
    
private:


    QLabel *d_lblTitle;
    QwtPlotPixFrame *d_frmPlot;
    QwtScale *d_scale[axisCnt];
    QwtLegend *d_legend;
    
    bool d_titleEnabled;

    QwtAutoScale d_as[axisCnt];
    QwtDiMap d_map[axisCnt];
    QwtScaleDiv d_sdiv[axisCnt];
    QwtPlotGrid d_grid;

    int d_axisEnabled[axisCnt];
    int d_legendEnabled;
    int d_legendPos;
    
    int d_plotBorder;

    QwtCurveDict *d_curves;
    QwtMarkerDict *d_markers;

    int d_axisDist;
    int d_borderDist;
    int d_hScaleDist;
    int d_vScaleDist;
    
    int d_titleDist;
    int d_margin;
    int d_autoReplot;

    // STATIC DATA
    static const QFont DummyFont;
    static const QPen DummyPen;
    static const QwtSymbol DummySymbol;
    static const QwtScaleDiv DummyScale;
    static const QString DummyString;
    static const int DefaultXAxis;
    static const int DefaultYAxis;

    
    // PRIVATE MEMBER FUNCTIONS
    void initAxes();
    long newCurveKey();
    long newMarkerKey();
    void initPlot(const char *title = 0);
    void plotPressed(QMouseEvent *e);
    void plotReleased(QMouseEvent *e);
    void plotMoved(QMouseEvent *e);

    bool insertCurve(long key, const char *title, int xAxis = xBottom, int yAxis = yLeft);
    bool insertMarker(long key, const char *label = "", int xAxis = xBottom, int yAxis = yLeft);
    bool insertLineMarker(long key, const char *label, int axis);
    
protected:

    // PROTECTED MEMBERS
    void redrawPixmap();
    void updateAxes();
    int verifyXAxis(int axis) const;
    int verifyYAxis(int axis) const;

public:

    //
    //	Constructors
    //
    QwtPlot(QWidget *p = 0, const char *name = 0);
    QwtPlot(const char *title, QWidget *p = 0, const char *name = 0);
    virtual ~QwtPlot();


    //
    // Set General Properties
    //
    void enableOutline(bool tf);
    void setAutoReplot(bool tf = TRUE);
    void setPlotBackground (const QColor &c);
    void setPlotBorder(int bw);
    void setOutlineStyle(Qwt::Shape os);
    void setOutlinePen(const QPen &pn);
    void setTitle(const char *t);
    void setTitleFont (const QFont &f);


    //
    // Retrieve General Properties
    //
    bool autoReplot() const { return d_autoReplot; }
    bool outlineEnabled() const;
    const QPen& outlinePen() const;
    Qwt::Shape outlineStyle() const;
    const QColor& plotBackground() const;
    int plotBorder() const { return d_plotBorder; }
    const QString& title() const;
    const QFont& titleFont() const;


    //
    //	Miscellaneous
    //
    static bool axisValid(int axis);
    void clear();
    void print(QPrinter &p, const QwtColorFilter &cfilter = NoColorFilter);
    
    //
    // Insert/Remove Curves
    //
    long insertCurve(const char *title, int xAxis = xBottom, int yAxis = yLeft);
    bool removeCurve(long key);
    void removeCurves();

    //
    // Modify Curves
    //
    bool setCurveBaseline(long key, double ref);
    bool setCurveData(long key, double *x, double *y, int size);
    bool setCurveOptions(long key, int t);
    bool setCurvePen(long key, const QPen &pen);
    bool setCurveRawData(long key, double *x, double *y, int size);
    bool setCurveSplineSize(long key, int s);
    bool setCurveStyle(long key, QwtCurve::CurveStyle s, int options = 0);
    bool setCurveSymbol(long key, const QwtSymbol &s);
    bool setCurveTitle(long key, const char *s);
    bool setCurveXAxis(long key, int axis);
    bool setCurveYAxis(long key, int axis);

    //
    // Get Curve Properties
    //
    long closestCurve(int xpos, int ypos, int &dist) const;
    long closestCurve(int xpos, int ypos, int &dist, double &xval, double &yval, int &index) const;
    double curveBaseline(long key) const;
    QArray<long> curveKeys() const;
    int curveOptions(long key) const;
    const QPen &curvePen(long key) const;
    int curveSplineSize(long key) const;
    QwtCurve::CurveStyle curveStyle(long key) const;
    const QwtSymbol &curveSymbol(long key) const;
    QString curveTitle(long key) const;
    int curveXAxis(long key) const;
    int curveYAxis(long key) const;


    //
    //	Modify Grid
    //
    void enableGridX(bool tf = TRUE);
    void enableGridXMin(bool tf = TRUE);
    void enableGridY(bool tf = TRUE);
    void enableGridYMin(bool tf = TRUE);
    void setGridXAxis(int axis);
    void setGridYAxis(int axis);
    void setGridPen(const QPen &p);
    void setGridMajPen(const QPen &p);
    void setGridMinPen(const QPen &p);

    int gridXAxis() const { return d_grid.xAxis(); }
    int gridYAxis() const { return d_grid.yAxis(); }
    const QPen& gridMinPen() { return d_grid.minPen(); }
    const QPen& gridMajPen() { return d_grid.majPen(); }



    //
    //   Modify Axes
    //
    void changeAxisOptions(int axis, int opt, bool value);
    void enableAxis(int axis, bool tf = TRUE);
    void setAxisFont(int axis, const QFont &f);
    void setAxisMargins(int axis, double mlo, double mhi);
    void setAxisOptions(int axis, int opt);
    void setAxisAutoScale(int axis);
    void setAxisScale(int axis, double min, double max, double step = 0);
    void setAxisTitleFont(int axis, const QFont &f);
    void setAxisMaxMinor(int axis, int maxMinor);
    void setAxisMaxMajor(int axis, int maxMajor);
    void setAxisReference(int axis, double value);
    void setAxisTitle(int axis, const char *t);

    //
    //  Retrieve Axis Properties
    //
    bool axisAutoScale(int axis) const;
    bool axisEnabled(int axis) const;
    const QFont& axisFont(int axis);
    bool axisMargins(int axis, double &mlo, double &mhi) const;
    int axisMaxMajor(int axis) const;
    int axisMaxMinor(int axis) const;
    int axisOptions(int axis) const;
    double axisReference(int axis) const;
    const QwtScaleDiv& axisScale(int axis) const;
    const QString& axisTitle(int axis) const;
    const QFont& axisTitleFont(int axis) const;
    double invTransform(int axis, int pos) const;
    int transform(int axis, double value) const;

    //
    //  Insert/Remove Markers
    //
    long insertMarker(const char *label = 0, int xAxis = xBottom, int yAxis = yLeft);
    long insertLineMarker(const char *label, int axis);
    bool removeMarker(long key);
    void removeMarkers();

    //
    // Set Marker Properties
    //
    long closestMarker(int xpos, int ypos, int &dist) const;
    bool setMarkerFont(long key, const QFont &f);
    bool setMarkerLinePen(long key, const QPen &p);
    bool setMarkerLineStyle(long key, QwtMarker::LineStyle st);
    bool setMarkerPos(long key, double xval, double yVal);
    bool setMarkerSymbol(long key, const QwtSymbol &s);
    bool setMarkerLabel(long key, const char *txt);
    bool setMarkerLabelAlign(long key, int align);
    bool setMarkerLabelPen(long key, const QPen &p);
    bool setMarkerXPos(long key, double val);
    bool setMarkerYPos(long key, double val);
    bool setMarkerPen(long key, const QPen &p);
    bool setMarkerXAxis(long key, int axis);
    bool setMarkerYAxis(long key, int axis);

    //
    // Get Marker Properties
    //
    const QFont& markerFont(long key) const;
    QArray<long> markerKeys() const;
    const QString& markerLabel(long key) const;
    int markerLabelAlign(long key) const;
    const QPen& markerLabelPen(long key) const;
    const QPen& markerLinePen(long key) const;
    QwtMarker::LineStyle markerLineStyle(long key) const;
    void markerPos(long key, double &mx, double &my) const;
    const QwtSymbol& markerSymbol(long key) const;
    int markerXAxis(long key) const;
    int markerYAxis(long key) const;

    //
    // Set Legend Properties
    void enableLegend(bool tf);
    void setLegendPos(int pos);
    void setLegendFont(const QFont &f);
    void setLegendFrameStyle(int st);
    
    // Get Legend Properties
    bool legendEnabled() const { return d_legendEnabled; }
    int legendPos() const { return d_legendPos; }
    int legendFrameStyle() const;
    const QFont& legendFont() const;
    
protected:
    
    void resizeEvent(QResizeEvent *e);
    void autoRefresh();
    
public slots:
    
    void replot();

private slots:
    void lgdClicked(int index);
    void updatePixmap();
    
signals:

    void plotMousePressed(const QMouseEvent &e);
    void plotMouseReleased(const QMouseEvent &e);
    void plotMouseMoved(const QMouseEvent &e);

    void legendClicked(long key);

};


#endif





