/*
 * Copyright (C) 1997-1998 by Dimitri van Heesch.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation under the terms of the GNU General Public License is hereby 
 * granted. No representations are made about the suitability of this software 
 * for any purpose. It is provided "as is" without express or implied warranty.
 * See the GNU General Public License for more details.
 *
 * This file is part of QdbtTabular 0.31.
 */

#include <stdio.h>
#include <qpainter.h>
#include <qkeycode.h>
#include "qdbtsection.h"
#include "qdbtheader.h"
#include "qdbtutil.h"
#include "qdbttabular.h"

/*!
   \class QdbtSection qdbtsection.h
   \brief The QdbtSection widget represents a section of the header of 
          a QdbtTabular widget.
   
   The header of a QdbtTabular widget is divided into a number of sections, 
   one for each column.

   Each section is a specialized QdbtSection widget.
   Its position and size are controlled by the QdbtTabular widget.
   
   By inheriting this widget you can change the look and feel
   of the individual sections.
   
   You can both read and change the widget that is used for 
   the section of a column by using QdbtTabular::section() and 
   QdbtTabular::changeSection().
*/
  
/*!
   Constructs a section widget with no label.
   \warning The \a parent argument \e must be a QdbtTabular widget or
            a widget that inherits from QdbtTabular, otherwise
            a run time error will be generated.
*/

QdbtSection::QdbtSection(QWidget *parent,const char *name)
  : QPushButton(
      parent ? ( parent->inherits("QdbtTabular") ? 
                 ((QdbtTabular *)parent)->header : 0) : 0
     ,name), colSpacing(16)
                                   
{
  initMetaObject();
  if (parent==0 || !parent->inherits("QdbtTabular"))
  {
    fatal("Fatal: You must supply a QdbtTabular object as the parent "
            "of a QdbtSection.\n");
  }
  align=AlignLeft;
  setMouseTracking(TRUE);
  resizable=TRUE;
}

/*!
  Constructs a section widget with the label \a text.
  \warning The \a parent argument \e must be a QdbtTabular widget or
            a widget that inherits from QdbtTabular, otherwise
            a run time error will be generated.
*/

QdbtSection::QdbtSection(const char *text,QWidget *parent,const char *name) 
  : QPushButton(text,
      parent ? ( parent->inherits("QdbtTabular") ?
                 ((QdbtTabular *)parent)->header : 0) : 0
     ,name), colSpacing(16)
{
  initMetaObject();
  if (parent==0 || !parent->inherits("QdbtTabular"))
  {
    fatal("Fatal: You must supply a QdbtTabular object as the parent "
            "of a QdbtSection.\n");
  }
  align=AlignLeft;
  setMouseTracking(TRUE);
  resizable=TRUE;
}

/*! 
  Destroys the section widget.
*/

QdbtSection::~QdbtSection()
{
}

/*! 
  Returns the preferret width of the widget.
  That is not the actual width, but the width
  that is needed to fully show the section. 

  You probably want to reimplement this if
  you are inheriting from this widget.
  \sa heightHint(), width()
*/

int QdbtSection::widthHint() const
{
  return fontMetrics().width(text())+colSpacing;
}

/*! 
  Returns the preferret width of the widget.
  That is not the actual width, but the width
  that is needed to fully show the section. 

  You probably want to reimplement this if
  you are inheriting from this widget.
  \sa widthHint(), height()
*/
  
int QdbtSection::heightHint() const
{
  return fontMetrics().height()+8;
}

/*!
  The resize event, currently doesn't do anything
*/

void QdbtSection::resizeEvent(QResizeEvent *)
{
  //printf("QdbtSection::resizeEvent()\n");
}

/*!
  The mouse press event. 
  \warning If you plan to 
  reimplement this function please make
  sure you call QdbtSection::mousePressEvent(e)
  within the body of the function. Otherwise
  the resizing of the buttons will not work
  anymore.
*/

void QdbtSection::mousePressEvent(QMouseEvent *e)
{
  //printf("QdbtSection::mousePressEvent()\n");
  QdbtHeader *parent=(QdbtHeader *)parentWidget();
  QMouseEvent mpe(Event_MouseButtonPress,
                  mapToParent(e->pos()),e->button(),e->state());
  parent->mousePressEvent(&mpe);
  if (parent->divider==-1) QPushButton::mousePressEvent(e);
  emit clicked(this);
}

/*!
  The mouse release event. 
*/

void QdbtSection::mouseReleaseEvent(QMouseEvent *e)
{
  //printf("QdbtSection::mouseReleaseEvent()\n");
  QPushButton::mouseReleaseEvent(e);
}

/*!
  The mouse move event. 
  \warning If you plan to 
  reimplement this function please make
  sure you call QdbtSection::mouseMoveEvent(e)
  within the body of the function. Otherwise
  the resizing of the buttons will not work
  anymore.
*/

void QdbtSection::mouseMoveEvent(QMouseEvent *e)
{
  //printf("QdbtSection::mouseMoveEvent()\n");
  QdbtHeader *parent=(QdbtHeader *)parentWidget();
  QMouseEvent mpe(Event_MouseMove,
                  mapToParent(e->pos()),e->button(),e->state());
  parent->mouseMoveEvent(&mpe);
}

/*!
  This function can be reimplemented to customize
  the way the section is drawn.
  \sa drawButtonLabel
*/

void QdbtSection::drawButton(QPainter *p)
{
  //printf("QdbtSection::drawButton()\n");
  QPushButton::drawButton(p);
}

/*!
  This function can be reimplemented to customize
  the way the label of the section is drawn.
  (this is the section excluding the border)
  \sa drawButton
*/

void QdbtSection::drawButtonLabel(QPainter *p)
{
  //printf("drawButtonLabel(%d)\n",align);
  QColorGroup g = colorGroup();
  QRect r = rect();
  int x,y,w,h;
  r.rect(&x,&y,&w,&h);
  x+=4; y+=4; w-=8; h-=8;
  p->setPen(g.text());
  int a=align;
  if (a==AlignCenter) a=AlignLeft;
  //p->setFont(font());  // MAY BE REMOVED SINCE QT 1.31
  p->drawText(x,y,w,h,align|AlignVCenter,truncateString(p,text(),w,a));
}

void QdbtSection::setResizable(bool enable)
{
  resizable=enable;
  emit setResizable(this,enable);
}

/*!
  Reimplements the keyboard events. Left and Right cursor key events
  are passed on to the parent. Other events (space) are processed 
  just as QPushButton.
*/

void QdbtSection::keyPressEvent(QKeyEvent *e)
{
  QdbtHeader *parent=(QdbtHeader *)parentWidget();
  //printf("QdbtSection::keyPressEvent()\n");
  switch(e->key())
  {
    case Key_Left:
    case Key_Right:
      parent->keyPressEvent(e);
      break;
    default:
      QPushButton::keyPressEvent(e);
      break;
  } 
}



/*!
  \fn void QdbtSection::clicked(QdbtSection *)
  \internal
*/

/*!
  \fn void QdbtSection::setResizable(QdbtSection *,bool)
  \internal
*/

/*!
  \fn void QdbtSection::setAlignment(int align)
  Set the alignment of the text to \a align.
  Possible values are \c AlignLeft, \c AlignRight and \c AlignCenter.
  \sa alignment()
*/

/*!
  \fn int QdbtSection::alignment() const
  Returns the alignment that is currently used.
  \sa setAlignment()
*/

/*! 
  \fn void QdbtSection::setResizable(bool enable)
  Determines whether or not this section can be
  resized by the user.

  If \a enable is \c FALSE the section cannot be
  resized.
  
  if \a enable is \c TRUE the section can be resized.
*/

/*!
  \fn void QdbtSection::isResizable() const
  Returns \c TRUE if the section is currently
  resizable and \c FALSE otherwise.
*/
