/*
 * Copyright (C) 1997-1998 by Dimitri van Heesch.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation under the terms of the GNU General Public License is hereby 
 * granted. No representations are made about the suitability of this software 
 * for any purpose. It is provided "as is" without express or implied warranty.
 * See the GNU General Public License for more details.
 *
 * This file is part of QdbtTabular 0.31.
 */

#include <stdio.h>
#include <qpainter.h>
#include <qdrawutl.h>
#include <qkeycode.h>

#include "qdbttabular.h"
#include "qdbtheader.h"

//----------------------------------------------------------------------------

QdbtHeader::QdbtHeader(QWidget *p,const char *n) : QWidget(p,n)
{
  //sections.setAutoDelete(TRUE);
  barHeight=0;
  divider=-1;
  setMouseTracking(TRUE);
  parent=(QdbtTabular *)p;
}

//----------------------------------------------------------------------------
QdbtHeader::~QdbtHeader()
{
}

//----------------------------------------------------------------------------
// compute the height of the header. This is the maximum
// of the height of all section.

int QdbtHeader::computeHeight()
{
  int h=14;
  QdbtSection *s=sections.first();
  while (s)
  {
    int hh=s->heightHint();
    if (hh>h) h=hh;
    s=sections.next();
  }
  return h;
}

//----------------------------------------------------------------------------
// redraw the header

void QdbtHeader::updateHeader()
{
  //printf("QdbtHeader::updateButton()\n");
  if (!isVisible()) return; // not visible => no updates needed 

  // set the geometry of all sections
  QdbtSection *s=sections.first();
  while (s)
  {
    int index=sections.at();
    if (!s->isVisible()) s->show(); // make button visible if needed
    s->setGeometry(headCells[index].position,0,
                    headCells[index].width,barHeight);
    s=sections.next();
  }

  int ww=width();
  int index=headCells.size()-1;
  int barWidth;
  if (index<0) 
    barWidth=0;
  else
    barWidth=headCells[index].position+headCells[index].width;
  
  // paint the last column
  if (barWidth<ww)
  {
    QPainter p;
    QRect r1(barWidth+1,1,ww-barWidth-2,barHeight-2);
    QRect r2(barWidth,0,ww-barWidth,barHeight);
    p.begin(this);
    p.eraseRect(r1);
    if ( style() == WindowsStyle )
    {
      qDrawWinButton(&p,r2,colorGroup(),FALSE);
    }
    else // style() == MotifStyle 
    {
      qDrawShadePanel(&p,r2,colorGroup(),FALSE,2);
    }
    p.end(); 
  }
}

//----------------------------------------------------------------------------
// check if the mouse-pointer is over a divider and 
// set the appropriate cursor and divider number

void QdbtHeader::updateDividerCursor(int x)
{
  divider=-1;
  int i;
  // we start at the end of the list so that in case of overlapping
  // divider-select areas always the most right divider is selected
  for (i=headCells.size()-1;i>=0 && divider==-1;i--)
  {
    if (headCells[i].resizable)
    {
      int sx=headCells[i].position+headCells[i].width;
      if (x>sx-8 && x<sx+8) divider=i; // select area of 15 pixels around divider
    }
  }
  if (divider==-1) 
    setCursor(arrowCursor); 
  else 
    setCursor(sizeHorCursor);
}

//----------------------------------------------------------------------------

void QdbtHeader::initFocusPolicy()
{
  // Set correct focus policies for all sections.
  // Only the first section is allowed to receive focus events
  QdbtSection *s=sections.first();
  if (s)
  {
    s->setFocusPolicy(TabFocus);
    s->clearFocus();
    s=sections.next();
  }
  while (s)
  {
    s->setFocusPolicy(NoFocus);
    s->clearFocus();
    s=sections.next();
  }
}
  
//----------------------------------------------------------------------------
void QdbtHeader::insertCol(int col)
{
  ASSERT(col>=-1 && col<=(int)headCells.size());
  
  int numCells=headCells.size();
  if (col==-1) col=numCells;
  headCells.resize(numCells+1);

  QdbtSection *newSec=new QdbtSection(parentWidget());
  connect(newSec,SIGNAL(clicked(QdbtSection *)),
                 SLOT(sectionClicked(QdbtSection *)));
  connect(newSec,SIGNAL(setResizable(QdbtSection *,bool)),
                 SLOT(setResizable(QdbtSection *,bool)));
  newSec->setFont(font());
  newSec->setPalette(palette());
  sections.insert(col,newSec);
  int secWidth=newSec->widthHint();
  
  initFocusPolicy();
  
  int i;
  for (i=numCells;i>col;i--) 
  {
    headCells[i]=headCells[i-1];
    headCells[i].position+=secWidth;
  }
  headCells[col].width=secWidth;
  if (col==0) 
    headCells[col].position=0;
  else
    headCells[col].position=headCells[col-1].position+headCells[col-1].width;

  headCells[col].resizable=TRUE;
  
  barHeight=computeHeight();

  // a column has already been inserted into the table, resize it here.
  updateHeader();
  emit signalColumnWidth(col,secWidth);
  emit updateTable(); 
} 

//----------------------------------------------------------------------------
void QdbtHeader::removeCol(int col)
{
  //printf("QdbtHeader::removeCol(%d)\n",col);
  ASSERT(col>=-1 && col<=(int)headCells.size());

  int i;
  int numCells=headCells.size();
  QdbtHeadCell oldCell=headCells[col];
  int cellWidth=oldCell.width;
  for (i=col;i<numCells-1;i++) 
  {
    headCells[i]=headCells[i+1];
    headCells[i].position-=cellWidth;
  }
  headCells.resize(numCells-1);
  QdbtSection *oldSec=sections.at(col);
  sections.remove(col);
  barHeight=computeHeight();
  emit updateTable();
  delete oldSec;
}

//----------------------------------------------------------------------------
void QdbtHeader::setColumns(int number)
{
  //printf("QdbtHeader::setColumns(%d) start\n",number);
  int i,numSects=sections.count();   
  int newSects=number;
  if (numSects==newSects) 
    return;
  else if (numSects<newSects) // add cols
  {
    for (i=numSects;i<newSects;i++) insertCol(i); 
  }
  else // numSects>number
  {
    for (i=newSects;i<numSects;i++) removeCol(number);
    headCells.resize(newSects);
  }  
  barHeight=computeHeight();
  emit updateTable();
  //printf("QdbtHeader::setColumns(%d) done\n",number);
}

//----------------------------------------------------------------------------
void QdbtHeader::resizeEvent(QResizeEvent *e)
{
  //printf("QdbtHeader::resizeEvent()\n");
  int i,index=headCells.size()-1;
  if (index<0) return;
  int pos=headCells[0].position;
  int barWidth=headCells[index].position+headCells[index].width-pos;
  if (e->size().width()>barWidth && pos<0)
  {
    pos+=e->size().width()-barWidth;
    if (pos>0) pos=0;
    if (pos!=headCells[0].position) 
    {
      headCells[0].position=pos;
      for (i=1;i<(int)headCells.size();i++) 
        headCells[i].position = headCells[i-1].position + headCells[i-1].width;
    }
  }
  int barEndPos=headCells[index].position+headCells[index].width;
  if (pos==0)
    emit signalColumnWidth(index+1,0);
  else
    emit signalColumnWidth(index+1,QMAX(0,e->size().width()-barEndPos));
  emit setTableOffset(headCells[0].position);
  updateHeader();
}

//----------------------------------------------------------------------------
void QdbtHeader::mousePressEvent(QMouseEvent *e)
{
  //printf("QdbtHeader::mousePressEvent()\n");
  if (e->button()==LeftButton)
  {
    if (divider!=-1 && headCells[divider].resizable) 
    {
      int bx=headCells[divider].position;
      int x=bx+headCells[divider].width-1; 
      parent->startSlideBar(QMAX(0,bx),mapToParent(QPoint(x,0)).x());
                            
    }
  }
  else if (e->button()==MidButton)
  {
    if (divider!=-1 && headCells[divider].resizable)
    {
      //printf("QdbtHeader::mousePressEvent() MidButton\n");
      parent->setColumnWidth(divider,parent->columnWidthHint(divider));
    }
  }
}

//----------------------------------------------------------------------------
void QdbtHeader::mouseReleaseEvent(QMouseEvent *e)
{
  //printf("QdbtHeader::mouseReleaseEvent()\n");
  QWidget::mouseReleaseEvent(e);
}

//----------------------------------------------------------------------------
void QdbtHeader::mouseMoveEvent(QMouseEvent *e)
{
  //printf("QdbtHeader::mouseMoveEvent()\n");
  updateDividerCursor(e->x());
}

//----------------------------------------------------------------------------
void QdbtHeader::resizeSection(int x)
{
  uint i;
  int r=divider;
  int nw=-QMIN(headCells[r].position,0)+x+3;
  int ow=headCells[r].width;
  headCells[r].width=nw;
  int dw=nw-ow;
  int pos=headCells[0].position;
  if (dw<0 && pos<0)
  {
    headCells[0].position=QMIN(0,pos-dw);
  }
  for (i=1;i<headCells.size();i++) 
    headCells[i].position=headCells[i-1].position+headCells[i-1].width;
  updateHeader();
  emit setTableOffset(pos);
  emit signalColumnWidth(r,nw);
  emit updateTable();
}

//----------------------------------------------------------------------------
void QdbtHeader::setHeaderOffset(int x)
{
  //printf("QdbtHeader::setHeaderOffset(%d)\n",x);
  if (headCells.size()==0 || -x==headCells[0].position) return;
  int i,pos=-x;
  for (i=0;i<(int)headCells.size();i++) 
  {
    headCells[i].position=pos;
    pos+=headCells[i].width;
  }
  updateHeader();
}

//----------------------------------------------------------------------------
void QdbtHeader::setColumnWidth(int col,int width)
{
  int i;
  //printf("QdbtHeader::setColumnWidth(%d,%d)\n",col,width);
  ASSERT(col>=0 && col<=(int)headCells.size());
  if (col==(int)headCells.size()) return;
  if (headCells[col].width==width) return;
  headCells[col].width=width;
  for (i=col+1;i<(int)headCells.size();i++) 
      headCells[i].position=headCells[i-1].position+headCells[i-1].width;
  updateHeader();
  emit signalColumnWidth(col,width);
  emit updateTable();
}

//----------------------------------------------------------------------------
void QdbtHeader::paintEvent(QPaintEvent *)
{
  updateHeader();  
}

//----------------------------------------------------------------------------
void QdbtHeader::setFont(const QFont &font)
{
  //printf("QdbtHeader::setFont()\n");
  QWidget::setFont(font);
  QdbtSection *s=sections.first();
  while (s)
  {
    s->setFont(font);
    s=sections.next();
  }
  barHeight=computeHeight();
  emit updateTable();
}

//----------------------------------------------------------------------------
void QdbtHeader::setPalette(const QPalette &p)
{
  //printf("QdbtHeader::setPalette()\n");
  QWidget::setPalette(p);
  QdbtSection *s=sections.first();
  while (s)
  {
    s->setPalette(p);
    s=sections.next();
  }
  emit updateTable();
}

//----------------------------------------------------------------------------
int QdbtHeader::columnWidthHint(int col) 
{
  return sections.at(col)->widthHint();
}

//----------------------------------------------------------------------------
void QdbtHeader::changeSection(QdbtSection *sec,int col)
{
  if (!sec) return;
  ASSERT(col>=0 && col<(int)sections.count());
  QdbtSection *oldSec=sections.at(col);
  sections.remove(col); // will delete the section
  sections.insert(col,sec);
  connect(sec,SIGNAL(clicked(QdbtSection *)),
              SLOT(sectionClicked(QdbtSection *)));
  connect(sec,SIGNAL(setResizable(QdbtSection *,bool)),
              SLOT(setResizable(QdbtSection *,bool)));
  sec->setFont(font());
  sec->setPalette(palette());
  barHeight=computeHeight();
  emit updateTable();
  initFocusPolicy();
  delete oldSec;
}

//----------------------------------------------------------------------------
QdbtSection *QdbtHeader::section(int col)
{
  ASSERT(col>=0 && col<(int)sections.count());
  return sections.at(col);
}

//----------------------------------------------------------------------------
// slot that catches the click signal from a section, and
// converts it to an index. This index is then send as a signal

void QdbtHeader::sectionClicked(QdbtSection *section)
{
  int index=sections.findRef(section);
  ASSERT(index!=-1);
  emit sectionClicked(index);
}

//----------------------------------------------------------------------------
void QdbtHeader::setResizable(QdbtSection *section,bool state)
{
  int index=sections.findRef(section);
  ASSERT(index!=-1);
  headCells[index].resizable=state;
}

//----------------------------------------------------------------------------
void QdbtHeader::keyPressEvent(QKeyEvent *e)
{
  if (e->key()==Key_Left || e->key()==Key_Right)
  {
    QdbtSection *s=sections.first();
    while (s)
    {
      if (s->hasFocus())
      {
        QdbtSection *ns=0;
        if (e->key()==Key_Left && (ns=sections.prev())) 
        {
          s->setFocusPolicy(NoFocus);
          ns->setFocusPolicy(TabFocus);
          ns->setFocus();
          emit exposeColumn(sections.at());
        }
        else if (e->key()==Key_Right && (ns=sections.next())) 
        {
          s->setFocusPolicy(NoFocus);
          ns->setFocusPolicy(TabFocus);
          ns->setFocus();
          emit exposeColumn(sections.at());
        }
        break;
      }
      s=sections.next();
    }
  }
}
