/*
 * Copyright (C) 1997-1998 by Dimitri van Heesch.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation under the terms of the GNU General Public License is hereby 
 * granted. No representations are made about the suitability of this software 
 * for any purpose. It is provided "as is" without express or implied warranty.
 * See the GNU General Public License for more details.
 *
 * This file is part of QdbtTabular 0.31.
 */

#include <stdio.h>
#include <stdlib.h>

#include <qapp.h>
#include <qlayout.h>
#include <qpushbt.h>
#include <qlabel.h>
// a little hack to be able to read the text of an item of a QPopupMenu 
#define INCLUDE_MENUITEM_DEF 1
#include <qmenudta.h>

#include <qdbt/qdbttabcell.h>
#include "example2.h"


MyTabular::MyTabular(QWidget *parent,const char *name,WFlags f) :
     QdbtTabular(parent,name,f)
{
  popMenu = 0;
}

MyTabular::~MyTabular()
{
}

void MyTabular::mousePressEvent(QMouseEvent *e)
{
  if (e->button()==RightButton) // Right mouse button was clicked
  {
    // Compute the cell that was selected
    clickedRow=findRow(e->y());
    clickedCol=findCol(e->x());
    // If the cell is valid we show a popup menu at the location of the 
    // mouse press. 
    if (popMenu && clickedRow!=-1 && clickedCol!=-1) 
    {
      popMenu->popup(mapToGlobal(e->pos()),clickedRow);
      setRowSelected(clickedRow,TRUE);
    }
  }
}

void MyTabular::mouseMoveEvent(QMouseEvent *e)
{
  // compute the row and col of the cell the mouse is on
  int row=findRow(e->y()); 
  int col=findCol(e->x());
  if (row!=-1 && col!=-1) // location is valid
    emit setStatus(cell(row,col)->text());  // display the cell's text
  else
    emit setStatus("none"); 
}

void MyTabular::leaveTableEvent(QEvent *)
{
  // Set the status to `none' if the mouse leaves the table 
  setStatus("none");
}

void MyTabular::resizeEvent(QResizeEvent *)
{
  // Set the column to its maximum width
  // QdbtTabular has a frame, so we must
  // substract its width twice (left and right)
  setColumnWidth(0,width()-2*frameWidth());
}

void MyTabular::setCellText(int index)
{
  // Get the selected text
  const char *text=popMenu->findItem(index)->text();
  // Copy the cell 
  // (no check needed because we know cell() does not return null)
  QdbtTableCell c(*cell(clickedRow,clickedCol));
  // Set the selected text
  c.setText(text);
  // If the chosen text differs from the one original in the table we
  // select the row
  setRowSelected(clickedRow,index!=clickedRow);
  // Change the table so it contains the new cell
  changeCell(&c,clickedRow,clickedCol);
}

void MyTabular::buildPopupMenu()
{
  // create a popupMenu object
  popMenu=new QPopupMenu();
  // call setCellText() whenever the user activates an item
  connect(popMenu,SIGNAL(activated(int)),SLOT(setCellText(int)));
  // add an item for each cell in the table 
  int i; for (i=0;i<numRows();i++) popMenu->insertItem(cell(i,0)->text());
}

// constructor
Example::Example(QWidget *parent,const char *name, WFlags f) 
      : QWidget(parent,name,f)
{
  // Set the caption of the window
  setCaption("Example2");

  // The Layout managers
  QGridLayout *layout  = new QGridLayout(this,2,1,5);
  QGridLayout *tables  = new QGridLayout(1,3,5);
  QBoxLayout  *buttons = new QBoxLayout(QBoxLayout::LeftToRight);
  
  // create three tables
  MyTabular *tabular1=new MyTabular(this);
  MyTabular *tabular2=new MyTabular(this);
  MyTabular *tabular3=new MyTabular(this);

  // set the table size (1 column 10 rows each)
  tabular1->setDimensions(10,1);
  tabular2->setDimensions(10,1);
  tabular3->setDimensions(10,1);

  // hide the headers of the tables
  tabular1->hideHeader();
  tabular2->hideHeader();
  tabular3->hideHeader();

  int i;
  for (i=0;i<10;i++)
  {
    QString nameLeft,nameRight,nameCenter;
    nameLeft.sprintf("left%d\n",i);
    nameCenter.sprintf("center%d\n",i);
    nameRight.sprintf("right%d\n",i);
    QdbtTableCell cell;

    cell.setAlignment(AlignLeft);
    cell.setText(nameLeft);
    tabular1->changeCell(&cell,i,0);
    tabular1->setRowSelectable(i,FALSE);  

    cell.setAlignment(AlignCenter);
    cell.setText(nameCenter);
    tabular2->changeCell(&cell,i,0);
    tabular2->setRowSelectable(i,FALSE);  

    cell.setAlignment(AlignRight);
    cell.setText(nameRight);
    tabular3->changeCell(&cell,i,0);
    tabular3->setRowSelectable(i,FALSE);  
  }

  // Create a popup menu for each table
  tabular1->buildPopupMenu();
  tabular2->buildPopupMenu();
  tabular3->buildPopupMenu();

  // Make sure all cells fit
  tabular1->fitAll();
  tabular2->fitAll();
  tabular3->fitAll();

  // Set the minimum sizes of the tables
  tabular1->setMinimumSize(tabular1->sizeHint());
  tabular2->setMinimumSize(tabular2->sizeHint());
  tabular3->setMinimumSize(tabular3->sizeHint());

  // Since all cells are visible, we do not need tooltips
  tabular1->disableTooltips();
  tabular2->disableTooltips();
  tabular3->disableTooltips();
  
  // Create the control button
  QPushButton *close  = new QPushButton("Close",this);
  status = new QLabel("none    ",this);

  // Set their minimum sizes
  close ->setMinimumSize(close->sizeHint());
  status->setMinimumSize(status->sizeHint());

  // Add Widgets and layouts to the layout-manager
  layout->addLayout(tables,0,0);
  layout->addLayout(buttons,1,0);
  layout->setColStretch(0,1); // make the table strechable
  layout->setRowStretch(0,1);

  // Add Widgets to the tables layout
  tables->addWidget(tabular1,0,0);
  tables->addWidget(tabular2,0,1);
  tables->addWidget(tabular3,0,2);

  // Add Widgets to the button layout
  buttons->addWidget(status);
  buttons->addStretch(1);
  buttons->addWidget(close);

  // don't forget to activate the top layout manager
  layout->activate();

  // Let the close button quit the application
  connect(close,  SIGNAL(clicked()),qApp,SLOT(quit()));
  
  // Send the cell that is mouse is currently over to the label
  // for each of the tables.
  connect(tabular1, SIGNAL( setStatus(const char *) ),
                    SLOT(  showStatus(const char *) ) );

  connect(tabular2, SIGNAL( setStatus(const char *) ),
                    SLOT(  showStatus(const char *) ) );

  connect(tabular3, SIGNAL( setStatus(const char *) ),
                    SLOT(  showStatus(const char *) ) );

  // Resize the widget to its minimal size
  resize(layout->mainWidget()->sizeHint());
}

// destructor
Example::~Example()
{
}

void Example::showStatus(const char *text)
{
  if (strcmp(status->text(),text)) status->setText(text);
}

int main(int argc,char **argv)
{
  QApplication app(argc, argv);
  Example example;
  app.setMainWidget(&example);
  example.show();
  return app.exec();  
}
