/*
gdpc - a program for visualising molecular dynamic simulations
Copyright (C) 1999 Jonas Frantz

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include "parameters.h"

/* GTK widgets which define changes in setup window */

/* Different windows and okey-button */
GtkWidget *filew,*setupwin,*okeyb;

/* Entries, check-button, spin-buttons and so on */
GtkWidget *file_entry,*dump_entry,*dump_label,*dtcheck,*dncheck,*scol_entry,*scol_entry,*scol_label;
GtkWidget *cubespinner,*xspinner,*yspinner,*zspinner,*x2spinner,*y2spinner,*z2spinner,*ssxspinner;
GtkWidget *ssyspinner,*xcspinner,*ycspinner,*zcspinner,*tcspinner,*sleepspinner,*dspinner,*scolspinner;
GtkWidget *usetypescheck;

/* Adjustments for the spin-buttons */
GtkAdjustment *adjcube,*adjx,*adjy,*adjz,*adjxc,*adjyc,*adjzc,*adjtc,*adjssizex,*adjssizey,*adjsleep,*adjd;
GtkAdjustment *adjx2,*adjy2,*adjz2,*adjscol;

struct GlobalParams newparams;		/* New values if OK is pressed, else ignored */

gboolean usescol,usedump,filewopen;	/* Help variables to define usage of specific features */


/***************************************************/
/* This function just kills the filebrowser window */
/***************************************************/
void destroyfbw (GtkWidget *widget, gpointer data) {
    gtk_widget_destroy(filew);
    filewopen=FALSE;
}


/*****************************************************************************************/
/* This function gets called when the filebrowser window is killed, it gets the selected */
/* filename from the filebrowser and puts it in the filename entry.                      */
/*****************************************************************************************/
void file_ok_sel (GtkWidget *widget, GtkFileSelection *fs) {
    gtk_entry_set_text( GTK_ENTRY (file_entry),gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));
    destroyfbw(widget,fs);
}


/*********************************************/
/* This function just kills the setup window */
/*********************************************/
void destroy (GtkWidget *widget, gpointer data) {
//    if (!sstarted) gtk_main_quit();
    if (filewopen) gtk_widget_destroy(filew);
    gtk_widget_destroy(setupwin);
}


/****************************************************************************/
/* This function creates a filebrowser window, connects all the buttons and */
/* shows it.                                                                */
/****************************************************************************/
void filebrowser(GtkWidget *widget, gpointer data) {
        
    /* Create a new file selection widget */
    filew = gtk_file_selection_new ("File selection");
        
    gtk_signal_connect (GTK_OBJECT (filew), "destroy",
                            (GtkSignalFunc) destroyfbw, &filew);

    /* Connect the ok_button to file_ok_sel function */
    gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filew)->ok_button),
                            "clicked", (GtkSignalFunc) file_ok_sel, filew );
        
    /* Connect the cancel_button to destroy the widget */
    gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filew)->cancel_button),
                                   "clicked", (GtkSignalFunc) destroyfbw,GTK_OBJECT (filew));
    filewopen=TRUE;
    gtk_widget_show(filew);
}


/*******************************************************************************/
/* This function is called when the OK button is pressed, it gets all the      */
/* filenames and sets all the variables according to the buttons and spinners. */
/*******************************************************************************/
void okeypressed(GtkWidget *widget, struct GlobalParams *params) {
    gfloat tmpvalue;
    sprintf(params->file,"%s",gtk_entry_get_text(GTK_ENTRY (file_entry)));
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (xspinner));

    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->xmin=tmpvalue;
    }
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (x2spinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->xmax=tmpvalue;
    } 
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (yspinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->ymin=tmpvalue;
    }
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (y2spinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->ymax=tmpvalue;
    } 
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (zspinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->zmin=tmpvalue;
    }
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (z2spinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->zmax=tmpvalue;
    } 

    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (cubespinner));
    if (tmpvalue>0.0) {
	if (params->xmin==65535.0) {
	    params->xmin=-tmpvalue;
	    params->xmax=tmpvalue;
	}
	if (params->ymin==65535.0) {
	    params->ymin=-tmpvalue;
	    params->ymax=tmpvalue;
	}
	if (params->zmin==65535.0) {
	    params->zmin=-tmpvalue;
	    params->zmax=tmpvalue;
	}
    }

    params->radius = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (dspinner));
    params->xcolumn = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (xcspinner));
    params->ycolumn = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (ycspinner));
    params->zcolumn = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (zcspinner));
    params->tcolumn = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (tcspinner));
    params->absxsize = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (ssxspinner));
    params->absysize = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (ssyspinner));
    tmpvalue = (gint) 1000*gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (sleepspinner));
    if (tmpvalue<MININTERVAL) params->interval=MININTERVAL;
    else params->interval=tmpvalue;

    if (usescol) {
	sprintf(params->fstring,"%s",gtk_entry_get_text(GTK_ENTRY (scol_entry)));
	params->scol = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (scolspinner));
    }
    else {
	params->scol=0;
    }

    if (usedump) {
	sprintf(params->dumpname,"%s",gtk_entry_get_text(GTK_ENTRY (dump_entry)));
    }
    else {
	*(params->dumpname)=(char) NULL;
    }

    params->fxyz=newparams.fxyz;
    params->mode=newparams.mode;
    params->vary=newparams.vary;
    params->colorset=newparams.colorset;
    params->erase=newparams.erase;
    params->whitebg=newparams.whitebg;
    params->dumpnum=newparams.dumpnum;
    params->sort=newparams.sort;
    params->tifjpg=newparams.tifjpg;
    params->usetypes=newparams.usetypes;

    if (!params->StartedAlready) StartEverything(params);
    else SetupStartOk(params);
	destroy(widget,NULL);
}


/*****************************************************************************/
/* This function is called when the redraw button is pressed, it practically */
/* does the same thing as okeypressed, but it doesn't kill the setup window. */
/*****************************************************************************/
void redrawpressed(GtkWidget *widget, struct GlobalParams *params) {
    gfloat tmpvalue;
    sprintf(params->file,"%s",gtk_entry_get_text(GTK_ENTRY (file_entry)));
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (xspinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->xmin=tmpvalue;
    }
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (x2spinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->xmax=tmpvalue;
    } 
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (yspinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->ymin=tmpvalue;
    }
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (y2spinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->ymax=tmpvalue;
    } 
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (zspinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->zmin=tmpvalue;
    }
    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (z2spinner));
    if (tmpvalue>0.0 || tmpvalue<0.0) {
	params->zmax=tmpvalue;
    } 

    tmpvalue = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (cubespinner));
    if (tmpvalue>0.0) {
	if (params->xmin==65535.0) {
	    params->xmin=-tmpvalue;
	    params->xmax=tmpvalue;
	}
	if (params->ymin==65535.0) {
	    params->ymin=-tmpvalue;
	    params->ymax=tmpvalue;
	}
	if (params->zmin==65535.0) {
	    params->zmin=-tmpvalue;
	    params->zmax=tmpvalue;
	}
    }

    params->radius = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (dspinner));
    params->xcolumn = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (xcspinner));
    params->ycolumn = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (ycspinner));
    params->zcolumn = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (zcspinner));
    params->tcolumn = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (tcspinner));
    params->absxsize = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (ssxspinner));
    params->absysize = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (ssyspinner));
    tmpvalue = (gint) 1000*gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (sleepspinner));
    if (tmpvalue<MININTERVAL) params->interval=MININTERVAL;
    else params->interval=tmpvalue;

    if (usescol) {
	sprintf(params->fstring,"%s",gtk_entry_get_text(GTK_ENTRY (scol_entry)));
	params->scol = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (scolspinner));
    }
    else {
	params->scol=0;
    }

    if (usedump) {
	sprintf(params->dumpname,"%s",gtk_entry_get_text(GTK_ENTRY (dump_entry)));
    }
    else {
	*(params->dumpname)=(char) NULL;
    }

    params->fxyz=newparams.fxyz;
    params->mode=newparams.mode;
    params->vary=newparams.vary;
    params->colorset=newparams.colorset;
    params->erase=newparams.erase;
    params->whitebg=newparams.whitebg;
    params->dumpnum=newparams.dumpnum;
    params->sort=newparams.sort;
    
    SetupRedraw(params);
}


/**************************************************************************/
/* This function is called when the Cancel button is pressed, it discards */
/* all the buttonpresses and kills the setup window.                      */
/**************************************************************************/ 
void cancelpressed(GtkWidget *widget, struct GlobalParams *params) {
    SetupStartCancel(params);
    destroy(widget,NULL);
}


/*************************************************************************/
/* This function is called when the Quit button is pressed, it quits the */
/* main program.                                                         */
/*************************************************************************/
void quitpressed(GtkWidget *widget, gpointer data) {
        gtk_main_quit ();
}


/******************************************************************************/
/* This function is called whenever a entry is changed, it checks if the file */
/* entered exists, if the stringsearch is used and if so that the string is   */
/* longer than zero and it checks if the dumpfile option is used and its name */
/* is longer than zero. If the conditions are met it enables the OK buttons,  */
/* else it disables it.                                                       */
/******************************************************************************/
void filechange(GtkWidget *widget,gpointer data) {
    FILE *fp;
    if (strlen(gtk_entry_get_text(GTK_ENTRY (file_entry))) > 0 && (!usescol || strlen(gtk_entry_get_text(GTK_ENTRY (scol_entry))) > 0) && (!usedump || strlen(gtk_entry_get_text(GTK_ENTRY (dump_entry))) > 0) ) {
	fp=fopen(gtk_entry_get_text(GTK_ENTRY (file_entry)),"r");
	if (fp!=NULL) { 
	    gtk_widget_set_sensitive (okeyb, TRUE);
	    fclose(fp);
	}
	else gtk_widget_set_sensitive (okeyb, FALSE);
    }
    else gtk_widget_set_sensitive (okeyb, FALSE);
}


/****************************************************************************/
/* This function is called when the XYZ file format radiobutton is pressed. */
/****************************************************************************/
void toggle_checkxyz(GtkWidget *widget, gpointer data)
{
    newparams.fxyz=TRUE;
    gtk_widget_set_sensitive (usetypescheck, TRUE);
}


/**********************************************************************************/
/* This function is called when the arbitrary file format radiobutton is pressed. */
/**********************************************************************************/
void toggle_checkaff(GtkWidget *widget, gpointer data)
{
    newparams.fxyz=FALSE;
    gtk_widget_set_sensitive (usetypescheck, FALSE);
}


/**************************************************************************/
/* This function is called when the drawingmode 0 radiobutton is pressed. */
/**************************************************************************/
void toggle_checkm0(GtkWidget *widget, gpointer data)
{
    newparams.mode=0;
}


/**************************************************************************/
/* This function is called when the drawingmode 1 radiobutton is pressed. */
/**************************************************************************/
void toggle_checkm1(GtkWidget *widget, gpointer data)
{
    newparams.mode=1;
}


/**************************************************************************/
/* This function is called when the drawingmode 2 radiobutton is pressed. */
/**************************************************************************/
void toggle_checkm2(GtkWidget *widget, gpointer data)
{
    newparams.mode=2;
}


/***********************************************************************/
/* This function is called when the varymode 0 radiobutton is pressed. */
/***********************************************************************/
void toggle_checkv0(GtkWidget *widget, gpointer data)
{
    newparams.vary=0;
}


/***********************************************************************/
/* This function is called when the varymode 1 radiobutton is pressed. */
/***********************************************************************/
void toggle_checkv1(GtkWidget *widget, gpointer data)
{
    newparams.vary=1;
}


/***********************************************************************/
/* This function is called when the varymode 2 radiobutton is pressed. */
/***********************************************************************/
void toggle_checkv2(GtkWidget *widget, gpointer data)
{
    newparams.vary=2;
}


/***********************************************************************/
/* This function is called when the colorset 0 radiobutton is pressed. */
/***********************************************************************/
void toggle_checkc0(GtkWidget *widget, gpointer data)
{
    newparams.colorset=0;
}


/***********************************************************************/
/* This function is called when the colorset 1 radiobutton is pressed. */
/***********************************************************************/
void toggle_checkc1(GtkWidget *widget, gpointer data)
{
    newparams.colorset=1;
}


/***********************************************************************/
/* This function is called when the colorset 2 radiobutton is pressed. */
/***********************************************************************/
void toggle_checkc2(GtkWidget *widget, gpointer data)
{
    newparams.colorset=2;
}


/***********************************************************************/
/* This function is called when the colorset 3 radiobutton is pressed. */
/***********************************************************************/
void toggle_checkc3(GtkWidget *widget, gpointer data)
{
    newparams.colorset=3;
}


/***********************************************************************/
/* This function is called when the colorset 4 radiobutton is pressed. */
/***********************************************************************/
void toggle_checkc4(GtkWidget *widget, gpointer data)
{
    newparams.colorset=4;
}


/****************************************************************************/
/* This function is called when the dump with time radio button is pressed. */
/****************************************************************************/
void toggle_checkdt(GtkWidget *widget, gpointer data)
{
    newparams.dumpnum=FALSE;
}


/********************************************************************************/
/* This function is called when the dump with framenum radio button is pressed. */
/********************************************************************************/
void toggle_checkdn(GtkWidget *widget, gpointer data)
{
    newparams.dumpnum=TRUE;
}


/******************************************************************/
/* This function is called when the erase checkbutton is pressed. */
/******************************************************************/
void toggle_erase(GtkWidget *widget, gpointer data)
{
    newparams.erase=(GTK_TOGGLE_BUTTON (widget)->active);
}


void toggle_usetypes(GtkWidget *widget, gpointer data)
{
    newparams.usetypes=(GTK_TOGGLE_BUTTON (widget)->active);
}


/**********************************************************************/
/* This function is called when the dumpframe checkbutton is pressed. */
/* It also activates the dumpfilename entry and the other related     */
/* radiobuttons.                                                      */
/**********************************************************************/
void toggle_checkdump (GtkWidget *widget, gpointer data)
{
    usedump=FALSE;
    gtk_widget_set_sensitive (dump_entry,FALSE);
    gtk_widget_set_sensitive (dump_label,FALSE);
    gtk_widget_set_sensitive (dncheck,FALSE);
    gtk_widget_set_sensitive (dtcheck,FALSE);
    filechange(widget,data);
}

void toggle_checkdumptif (GtkWidget *widget, gpointer data)
{
    usedump=TRUE;
    gtk_widget_set_sensitive (dump_entry,TRUE);
    gtk_widget_set_sensitive (dump_label,TRUE);
    gtk_widget_set_sensitive (dncheck,TRUE);
    gtk_widget_set_sensitive (dtcheck,TRUE);
    newparams.tifjpg=TRUE;
    filechange(widget,data);
}

void toggle_checkdumpjpg (GtkWidget *widget, gpointer data)
{
    usedump=TRUE;
    gtk_widget_set_sensitive (dump_entry,TRUE);
    gtk_widget_set_sensitive (dump_label,TRUE);
    gtk_widget_set_sensitive (dncheck,TRUE);
    gtk_widget_set_sensitive (dtcheck,TRUE);
    newparams.tifjpg=FALSE;
    filechange(widget,data);
}

/********************************************************************/
/* This function is called when the whitebg checkbutton is pressed. */
/********************************************************************/
void toggle_white (GtkWidget *widget, gpointer data)
{
    newparams.whitebg=(GTK_TOGGLE_BUTTON (widget)->active);
}


/************************************************************************/
/* This function is called when the sortreverse checkbutton is pressed. */
/************************************************************************/
void toggle_sortr (GtkWidget *widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON (widget)->active) {
	newparams.sort=2;
    }
    else {
	newparams.sort=1;
    }
}


/**************************************************************************/
/* This function is called when the stringsearch checkbutton is pressed.  */
/* It also activates the string entry and the other related radiobuttons. */
/**************************************************************************/
void toggle_scol (GtkWidget *widget, gpointer data)
{
    usescol=GTK_TOGGLE_BUTTON(widget)->active;
    gtk_widget_set_sensitive (scol_label,GTK_TOGGLE_BUTTON(widget)->active);
    gtk_widget_set_sensitive (scol_entry,GTK_TOGGLE_BUTTON(widget)->active);
    gtk_widget_set_sensitive (scolspinner,GTK_TOGGLE_BUTTON(widget)->active);
    filechange(widget,data);
}


/********************************************************************************/
/* This function is called whenever the x,y or z spinbutton values are changed, */
/* it sets up the limits for them.                                              */
/********************************************************************************/
void setspinlimits(GtkWidget *widget, gpointer data) 
{
gfloat value1,value2;

    value1 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (xspinner));
    value2 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (x2spinner));
    adjx = (GtkAdjustment *) gtk_adjustment_new(value1,-10000.0,value2,1.0,5.0,0.0);
    adjx2 = (GtkAdjustment *) gtk_adjustment_new(value2,value1,10000.0,1.0,5.0,0.0);
    gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON (xspinner),adjx);
    gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON (x2spinner),adjx2);

    value1 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (yspinner));
    value2 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (y2spinner));
    adjy = (GtkAdjustment *) gtk_adjustment_new(value1,-10000.0,value2,1.0,5.0,0.0);
    adjy2 = (GtkAdjustment *) gtk_adjustment_new(value2,value1,10000.0,1.0,5.0,0.0);
    gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON (yspinner),adjy);
    gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON (y2spinner),adjy2);

    value1 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (zspinner));
    value2 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON (z2spinner));
    adjz = (GtkAdjustment *) gtk_adjustment_new(value1,-10000.0,value2,1.0,5.0,0.0);
    adjz2 = (GtkAdjustment *) gtk_adjustment_new(value2,value1,10000.0,1.0,5.0,0.0);
    gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON (zspinner),adjz);
    gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON (z2spinner),adjz2);

    gtk_signal_connect_object (GTK_OBJECT (adjx), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
    gtk_signal_connect_object (GTK_OBJECT (adjx2), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
    gtk_signal_connect_object (GTK_OBJECT (adjy), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
    gtk_signal_connect_object (GTK_OBJECT (adjy2), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
    gtk_signal_connect_object (GTK_OBJECT (adjz), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
    gtk_signal_connect_object (GTK_OBJECT (adjz2), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
}


/*******************************************************************************/
/* This function is called when the setup button is pressed in the main window */
/* or if gdpc is started without parameters. It sets up the setup window and   */
/* all its buttons and entries.                                                */
/*******************************************************************************/
void setupwindow(struct GlobalParams *params) 
{
GtkWidget *browseb,*cancelb,*redrawb,*quitb,*check,*erasetoggle,*whitetoggle,*dumpcheck,*sortrtoggle;
GtkWidget *dumptifcheck,*dumpjpgcheck;
GtkWidget *vbox_main,*hbox_main,*vbox,*hbox1,*hbox2,*hbox3,*vboxright,*vboxmostright,*vboxleft,*hboxcube;
GtkWidget *vboxcubel,*vboxcuber,*vboxcuberm,*hboxxc,*vboxxcl,*vboxxcr,*hboxssize,*hboxsleep,*hboxd,*vbox_header;
GtkWidget *hbox_cube,*hboxsc,*scoltoggle;
GtkWidget *file_label,*column_label,*ff_label,*cxyz_label,*separator;
GtkWidget *cube_label,*x_label,*y_label,*z_label,*xcol_label,*ycol_label,*zcol_label,*tcol_label,*ssize_label;
GtkWidget *ssx_label,*ssy_label,*sleep_label,*d_label,*header,*col_label,*misc_label,*draw_label,*empty_label;
GSList *group;

    newparams.mode=params->mode;
    newparams.erase=params->erase;
    newparams.whitebg=params->whitebg;
    newparams.colorset=params->colorset;
    newparams.fxyz=params->fxyz;
    newparams.sort=params->sort;
    newparams.vary=params->vary;
    newparams.dumpnum=params->dumpnum;
    newparams.tifjpg=params->tifjpg;
    newparams.dumpnum=params->dumpnum;
    newparams.usetypes=params->usetypes;

    usedump=FALSE;
    usescol=FALSE;

    setupwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (setupwin),"gdpc Setup");
    gtk_container_set_border_width (GTK_CONTAINER (setupwin), 5);

    gtk_signal_connect (GTK_OBJECT (setupwin),"destroy",(GtkSignalFunc) destroy, &setupwin);
    
/* Create boxes for layout. */
    vbox_main = gtk_vbox_new (FALSE, 10);
    vbox_header = gtk_vbox_new (FALSE, 10);
    hbox_main = gtk_hbox_new (FALSE, 30);
    vbox = gtk_vbox_new (FALSE, 3);
    hbox1 = gtk_hbox_new (FALSE, 3);
    hbox2 = gtk_hbox_new (FALSE, 30);
    hbox3 = gtk_hbox_new (FALSE, 30);
    hboxssize = gtk_hbox_new (FALSE, 0);
    hboxd = gtk_hbox_new (FALSE, 0);
    hboxsc = gtk_hbox_new (FALSE, 0);
    hbox_cube = gtk_hbox_new (FALSE, 3);
    hboxcube = gtk_hbox_new (FALSE, 3);
    vboxcubel = gtk_vbox_new (FALSE, 3);
    vboxcuber = gtk_vbox_new (FALSE, 3);
    vboxcuberm = gtk_vbox_new (FALSE, 3);
    hboxxc = gtk_hbox_new (FALSE, 0);
    vboxxcl = gtk_vbox_new (FALSE, 0);
    vboxxcr = gtk_vbox_new (FALSE, 0);
    hboxsleep = gtk_hbox_new (FALSE, 0);
    vboxleft = gtk_vbox_new (FALSE, 3);
    vboxright = gtk_vbox_new (FALSE, 3);
    vboxmostright = gtk_vbox_new (FALSE, 3);
    gtk_container_add (GTK_CONTAINER (setupwin), vbox_main);
    
    gtk_box_pack_start (GTK_BOX (vbox_main), vbox_header, FALSE, FALSE, 0);                      
    gtk_box_pack_start (GTK_BOX (vbox_main), hbox_main, FALSE, FALSE, 0);                      
    gtk_box_pack_start (GTK_BOX (vbox_main), hbox3, FALSE, FALSE, 0);                          
    gtk_box_pack_start (GTK_BOX (hbox_main), vbox, FALSE, FALSE, 0);                          
    gtk_box_pack_start (GTK_BOX (hbox_main), vboxmostright, FALSE, FALSE, 0);                          
    gtk_box_pack_start (GTK_BOX (vbox), hbox1, FALSE, FALSE, 0);                          
    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, TRUE, 3);
    gtk_widget_show (separator);
    gtk_box_pack_start (GTK_BOX (vbox), hbox2, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox2), vboxleft, FALSE, FALSE, 0);                          
    gtk_box_pack_start (GTK_BOX (hbox2), vboxright, FALSE, FALSE, 0);
                          
    header = gtk_label_new(" Setup ");
    gtk_box_pack_start (GTK_BOX (vbox_header), header, FALSE, TRUE, 0);
    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vbox_header), separator, FALSE, TRUE, 3);
    gtk_widget_show (separator);

    file_label = gtk_label_new("Input file : ");
    file_entry = gtk_entry_new_with_max_length(160);    
    gtk_signal_connect_object (GTK_OBJECT (file_entry), "changed",
                    GTK_SIGNAL_FUNC (filechange),GTK_OBJECT (setupwin));

    browseb = gtk_button_new_with_label ("Browse");
    gtk_signal_connect_object (GTK_OBJECT (browseb), "clicked",
                    GTK_SIGNAL_FUNC (filebrowser),GTK_OBJECT (setupwin));

    gtk_box_pack_start (GTK_BOX (hbox1), file_label, FALSE, FALSE, 0);                          
    gtk_box_pack_start (GTK_BOX (hbox1), file_entry, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox1), browseb, FALSE, FALSE, 0);


    ssize_label = gtk_label_new("Drawingboard size [pixels]");
    ssx_label = gtk_label_new(" X : ");
    ssy_label = gtk_label_new(" Y : ");

    adjssizex = (GtkAdjustment *) gtk_adjustment_new(params->absxsize,0.0,3000.0,10.0,5.0,0.0);
    adjssizey = (GtkAdjustment *) gtk_adjustment_new(params->absysize,0.0,3000.0,10.0,5.0,0.0);

    ssxspinner = gtk_spin_button_new (adjssizex, 0, 0);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (ssxspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (ssxspinner),
                                     GTK_SHADOW_OUT);
    ssyspinner = gtk_spin_button_new (adjssizey, 0, 0);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (ssyspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (ssyspinner),
                                     GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (vboxleft), ssize_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hboxssize), ssx_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hboxssize), ssxspinner, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hboxssize), ssy_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hboxssize), ssyspinner, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxleft), hboxssize, TRUE, TRUE, 0);


    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vboxleft), separator, FALSE, TRUE, 3);
    gtk_widget_show (separator);

    cxyz_label = gtk_label_new("Simulation box size");
    gtk_box_pack_start (GTK_BOX (vboxleft), cxyz_label, TRUE, TRUE, 3);

    gtk_box_pack_start (GTK_BOX (vboxleft), hbox_cube, TRUE, TRUE, 3);
    gtk_box_pack_start (GTK_BOX (vboxleft), hboxcube, TRUE, TRUE, 3);
    gtk_box_pack_start (GTK_BOX (hboxcube), vboxcubel, TRUE, TRUE, 3);
    gtk_box_pack_start (GTK_BOX (hboxcube), vboxcuber, TRUE, TRUE, 3);
    gtk_box_pack_start (GTK_BOX (hboxcube), vboxcuberm, TRUE, TRUE, 3);

    cube_label = gtk_label_new("Cube : ");

    adjcube = (GtkAdjustment *) gtk_adjustment_new(0.0,0.0,10000.0,1.0,5.0,0.0);

    cubespinner = gtk_spin_button_new (adjcube, 0, 2);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (cubespinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (cubespinner),
                                     GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (hbox_cube), cube_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox_cube), cubespinner, TRUE, TRUE, 0);


    x_label = gtk_label_new("X : ");

    if (params->xmin==65535.0) {
    adjx = (GtkAdjustment *) gtk_adjustment_new(0.0,-10000.0,0.0,1.0,5.0,0.0);
    adjx2 = (GtkAdjustment *) gtk_adjustment_new(0.0,0.0,10000.0,1.0,5.0,0.0);
    } else {
    adjx = (GtkAdjustment *) gtk_adjustment_new(params->xmin,-10000.0,params->xmax,1.0,5.0,0.0);
    adjx2 = (GtkAdjustment *) gtk_adjustment_new(params->xmax,params->xmin,10000.0,1.0,5.0,0.0);
    }

    xspinner = gtk_spin_button_new (adjx, 0, 2);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (xspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (xspinner),
                                     GTK_SHADOW_OUT);
    gtk_signal_connect_object (GTK_OBJECT (adjx), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
    x2spinner = gtk_spin_button_new (adjx2, 0, 2);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (x2spinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (x2spinner),
                                     GTK_SHADOW_OUT);
    gtk_signal_connect_object (GTK_OBJECT (adjx2), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxcubel), x_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxcuber), xspinner, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxcuberm), x2spinner, TRUE, TRUE, 0);


    y_label = gtk_label_new("Y : ");

    if (params->ymin==65535.0) {
    adjy = (GtkAdjustment *) gtk_adjustment_new(0.0,-10000.0,0.0,1.0,5.0,0.0);
    adjy2 = (GtkAdjustment *) gtk_adjustment_new(0.0,0.0,10000.0,1.0,5.0,0.0);
    } else {
    adjy = (GtkAdjustment *) gtk_adjustment_new(params->ymin,-10000.0,params->ymax,1.0,5.0,0.0);
    adjy2 = (GtkAdjustment *) gtk_adjustment_new(params->ymax,params->ymin,10000.0,1.0,5.0,0.0);
    }

    yspinner = gtk_spin_button_new (adjy, 0, 2);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (yspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (yspinner),
                                     GTK_SHADOW_OUT);
    gtk_signal_connect_object (GTK_OBJECT (adjy), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));

    y2spinner = gtk_spin_button_new (adjy2, 0, 2);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (y2spinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (y2spinner),
                                     GTK_SHADOW_OUT);
    gtk_signal_connect_object (GTK_OBJECT (adjy2), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxcubel), y_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxcuber), yspinner, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxcuberm), y2spinner, TRUE, TRUE, 0);


    z_label = gtk_label_new("Z : ");

    if (params->zmin==65535.0) {
    adjz = (GtkAdjustment *) gtk_adjustment_new(0.0,-10000.0,0.0,1.0,5.0,0.0);
    adjz2 = (GtkAdjustment *) gtk_adjustment_new(0.0,0.0,10000.0,1.0,5.0,0.0);
    } else {
    adjz = (GtkAdjustment *) gtk_adjustment_new(params->zmin,-10000.0,params->zmax,1.0,5.0,0.0);
    adjz2 = (GtkAdjustment *) gtk_adjustment_new(params->zmax,params->zmin,10000.0,1.0,5.0,0.0);
    }

    zspinner = gtk_spin_button_new (adjz, 0, 2);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (zspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (zspinner),
                                     GTK_SHADOW_OUT);
    gtk_signal_connect_object (GTK_OBJECT (adjz), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));

    z2spinner = gtk_spin_button_new (adjz2, 0, 2);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (z2spinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (z2spinner),
                                     GTK_SHADOW_OUT);
    gtk_signal_connect_object (GTK_OBJECT (adjz2), "value_changed",
                    GTK_SIGNAL_FUNC (setspinlimits),GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxcubel), z_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxcuber), zspinner, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxcuberm), z2spinner, TRUE, TRUE, 0);


    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vboxleft), separator, FALSE, TRUE, 3);
    gtk_widget_show (separator);

    column_label = gtk_label_new("Input data column representations");
    gtk_box_pack_start (GTK_BOX (vboxleft), column_label, TRUE, TRUE, 0);

    gtk_box_pack_start (GTK_BOX (vboxleft), hboxxc, TRUE, TRUE, 3);
    gtk_box_pack_start (GTK_BOX (hboxxc), vboxxcl, TRUE, TRUE, 3);
    gtk_box_pack_start (GTK_BOX (hboxxc), vboxxcr, TRUE, TRUE, 3);

    xcol_label = gtk_label_new("X column : ");

    adjxc = (GtkAdjustment *) gtk_adjustment_new(params->xcolumn,1.0,100.0,1.0,5.0,0.0);

    xcspinner = gtk_spin_button_new (adjxc, 0, 0);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (xcspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (xcspinner),
                                     GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (vboxxcl), xcol_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxxcr), xcspinner, TRUE, TRUE, 0);

    ycol_label = gtk_label_new("Y column : ");

    adjyc = (GtkAdjustment *) gtk_adjustment_new(params->ycolumn,1.0,100.0,1.0,5.0,0.0);

    ycspinner = gtk_spin_button_new (adjyc, 0, 0);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (ycspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (ycspinner),
                                     GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (vboxxcl), ycol_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxxcr), ycspinner, TRUE, TRUE, 0);

    zcol_label = gtk_label_new("Z column : ");

    adjzc = (GtkAdjustment *) gtk_adjustment_new(params->zcolumn,1.0,100.0,1.0,5.0,0.0);

    zcspinner = gtk_spin_button_new (adjzc, 0, 0);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (zcspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (zcspinner),
                                     GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (vboxxcl), zcol_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxxcr), zcspinner, TRUE, TRUE, 0);

    tcol_label = gtk_label_new("t column : ");

    adjtc = (GtkAdjustment *) gtk_adjustment_new(params->tcolumn,1.0,100.0,1.0,5.0,0.0);

    tcspinner = gtk_spin_button_new (adjtc, 0, 0);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (tcspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (tcspinner),
                                     GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (vboxxcl), tcol_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxxcr), tcspinner, TRUE, TRUE, 0);



    ff_label = gtk_label_new("Input file format");
    gtk_box_pack_start (GTK_BOX (vboxright), ff_label, TRUE, TRUE, 0);

    usetypescheck = gtk_check_button_new_with_label(" Use coloring by type");

    check = gtk_radio_button_new_with_label(NULL, "XYZ file format");
    gtk_box_pack_start (GTK_BOX (vboxright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "clicked",
                        GTK_SIGNAL_FUNC(toggle_checkxyz),GTK_OBJECT (setupwin));
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (check));
    if (params->fxyz) {
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    }
    gtk_widget_show(check);

    check = gtk_radio_button_new_with_label(group, "Arbitrary file format");
    gtk_box_pack_start (GTK_BOX (vboxright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "clicked",
                        GTK_SIGNAL_FUNC(toggle_checkaff),GTK_OBJECT (setupwin));
    if (!params->fxyz) {
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    }
    gtk_widget_show (check);

    gtk_signal_connect (GTK_OBJECT(usetypescheck), "toggled",
                            GTK_SIGNAL_FUNC(toggle_usetypes), GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxright), usetypescheck, TRUE, TRUE, 0);
    gtk_widget_show (usetypescheck);
    if (params->usetypes) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (usetypescheck), TRUE);
    else gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (usetypescheck), FALSE);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vboxright), separator, FALSE, TRUE, 3);
    gtk_widget_show (separator);

    draw_label = gtk_label_new(" Drawing options ");
    gtk_box_pack_start (GTK_BOX (vboxright), draw_label, TRUE, TRUE, 0);

    d_label = gtk_label_new("Size of drawn polygons :");

    adjd = (GtkAdjustment *) gtk_adjustment_new(params->radius,1.0,100.0,1.0,5.0,0.0);

    dspinner = gtk_spin_button_new (adjd, 0, 0);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (dspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (dspinner),
                                     GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (hboxd), d_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hboxd), dspinner, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxright), hboxd, TRUE, TRUE, 0);

    check = gtk_radio_button_new_with_label(NULL, "Draw as rectangles");
    gtk_box_pack_start (GTK_BOX (vboxright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkm0),GTK_OBJECT (setupwin));
    if (params->mode==0) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (check));
    gtk_widget_show (check);

    check = gtk_radio_button_new_with_label(group, "Draw as circles");
    gtk_box_pack_start (GTK_BOX (vboxright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkm1),GTK_OBJECT (setupwin));
    if (params->mode==1) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (check));
    gtk_widget_show (check);

    check = gtk_radio_button_new_with_label(group, "Draw as rendered balls");
    gtk_box_pack_start (GTK_BOX (vboxright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkm2),GTK_OBJECT (setupwin));
    if (params->mode==2) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    gtk_widget_show (check);

    empty_label = gtk_label_new("  ");
    gtk_box_pack_start (GTK_BOX (vboxright),empty_label, TRUE, TRUE, 0);

    check = gtk_radio_button_new_with_label(NULL, "Dont vary size with z");
    gtk_box_pack_start (GTK_BOX (vboxright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkv0),GTK_OBJECT (setupwin));
    if (params->vary==0) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (check)); 
    gtk_widget_show (check);
    
    check = gtk_radio_button_new_with_label(group, "Vary size with z, decreasing");
    gtk_box_pack_start (GTK_BOX (vboxright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkv1),GTK_OBJECT (setupwin));
    if (params->vary==1) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (check));
    gtk_widget_show (check);
    
    check = gtk_radio_button_new_with_label(group, "Vary size with z, increasing");   
    gtk_box_pack_start (GTK_BOX (vboxright),check, TRUE, TRUE, 0); 
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkv2),GTK_OBJECT (setupwin));
    if (params->vary==2) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    gtk_widget_show (check);


    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vboxright), separator, FALSE, TRUE, 3);
    gtk_widget_show (separator);

    scoltoggle=gtk_check_button_new_with_label("Only use inputlines with string :");
    gtk_signal_connect (GTK_OBJECT(scoltoggle), "toggled",
                            GTK_SIGNAL_FUNC(toggle_scol), GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxright), scoltoggle, TRUE, TRUE, 0);

    scol_entry = gtk_entry_new_with_max_length(60);    
    gtk_signal_connect_object (GTK_OBJECT (scol_entry), "changed",
                    GTK_SIGNAL_FUNC (filechange),GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxright), scol_entry, TRUE, TRUE, 0);

    scol_label = gtk_label_new(" in column ");
    adjscol = (GtkAdjustment *) gtk_adjustment_new(1.0,1.0,100.0,1.0,5.0,0.0);
    scolspinner = gtk_spin_button_new (adjscol, 0, 0);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (scolspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (tcspinner),
                                     GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (hboxsc), scol_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hboxsc), scolspinner, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxright), hboxsc, TRUE, TRUE, 0);

    gtk_widget_set_sensitive (scol_label, FALSE);
    gtk_widget_set_sensitive (scol_entry, FALSE);
    gtk_widget_set_sensitive (scolspinner, FALSE);


    col_label = gtk_label_new(" Color settings ");
    gtk_box_pack_start (GTK_BOX (vboxmostright),col_label, TRUE, TRUE, 0);

    check = gtk_radio_button_new_with_label(NULL, "Use default colors");
    gtk_box_pack_start (GTK_BOX (vboxmostright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkc0),GTK_OBJECT (setupwin));
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (check));
    if (params->colorset==0) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    gtk_widget_show (check);

    check = gtk_radio_button_new_with_label(group, "Use inverted colors");
    gtk_box_pack_start (GTK_BOX (vboxmostright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkc1),GTK_OBJECT (setupwin));
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (check));
    if (params->colorset==1) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    gtk_widget_show (check);

    check = gtk_radio_button_new_with_label(group, "Use cold colors");
    gtk_box_pack_start (GTK_BOX (vboxmostright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkc2),GTK_OBJECT (setupwin));
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (check));
    if (params->colorset==2) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    gtk_widget_show(check);

    check = gtk_radio_button_new_with_label(group, "Use cold colors 2");
    gtk_box_pack_start (GTK_BOX (vboxmostright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkc3),GTK_OBJECT (setupwin));
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (check));
    if (params->colorset==3) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    gtk_widget_show (check);

    check = gtk_radio_button_new_with_label(group, "Use greyscale colors");
    gtk_box_pack_start (GTK_BOX (vboxmostright),check, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(check), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkc4),GTK_OBJECT (setupwin));
    if (params->colorset==4) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
    gtk_widget_show (check);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vboxmostright), separator, FALSE, TRUE, 3);
    gtk_widget_show (separator);

    misc_label = gtk_label_new("Misc. settings");
    gtk_box_pack_start (GTK_BOX (vboxmostright),misc_label, TRUE, TRUE, 0);

    erasetoggle=gtk_check_button_new_with_label(" Erase before drawing next frame");
    gtk_signal_connect (GTK_OBJECT(erasetoggle), "toggled",
                            GTK_SIGNAL_FUNC(toggle_erase), GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxmostright), erasetoggle, TRUE, TRUE, 0);
    if (params->erase) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (erasetoggle), TRUE);

    whitetoggle=gtk_check_button_new_with_label(" Use white as backgroundcolor");
    gtk_signal_connect (GTK_OBJECT(whitetoggle), "toggled",
                            GTK_SIGNAL_FUNC(toggle_white), GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxmostright), whitetoggle, TRUE, TRUE, 0);
    if (params->whitebg) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (whitetoggle), TRUE);

    sortrtoggle=gtk_check_button_new_with_label(" Reverse sorting");
    gtk_signal_connect (GTK_OBJECT(sortrtoggle), "toggled",
                            GTK_SIGNAL_FUNC(toggle_sortr), GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxmostright), sortrtoggle, TRUE, TRUE, 0);
    if (params->sort==2) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sortrtoggle), TRUE);

    sleep_label = gtk_label_new("Delay between frames [s] : ");

    adjsleep = (GtkAdjustment *) gtk_adjustment_new((params->interval/1000),0.0,100.0,0.1,5.0,0.0);

    sleepspinner = gtk_spin_button_new (adjsleep, 0, 1);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (sleepspinner), FALSE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (sleepspinner),
                                     GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (hboxsleep), sleep_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hboxsleep), sleepspinner, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vboxmostright), hboxsleep, TRUE, TRUE, 0);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vboxmostright), separator, FALSE, TRUE, 3);
    gtk_widget_show (separator);

    dump_label = gtk_label_new(" Dumped pictures name (no extension) : ");
    dump_entry = gtk_entry_new_with_max_length(60);    
    gtk_widget_set_sensitive (dump_label, FALSE);
    gtk_widget_set_sensitive (dump_entry, FALSE);

    dtcheck = gtk_radio_button_new_with_label(NULL, " Add frame time");
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (dtcheck));
    gtk_widget_set_sensitive (dtcheck, FALSE);

    dncheck = gtk_radio_button_new_with_label(group, " Add frame number");
    gtk_widget_set_sensitive (dncheck, FALSE);

    okeyb = gtk_button_new_with_label ("Ok");

    dumpcheck = gtk_radio_button_new_with_label(NULL, " Do not dump images");
    gtk_box_pack_start (GTK_BOX (vboxmostright),dumpcheck, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(dumpcheck), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkdump),GTK_OBJECT (setupwin));
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (dumpcheck));
    if (strlen(params->dumpname)==0) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dumpcheck), TRUE);

    dumptifcheck = gtk_radio_button_new_with_label(group, " Dump a .tif of each frame");
    gtk_box_pack_start (GTK_BOX (vboxmostright),dumptifcheck, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(dumptifcheck), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkdumptif),GTK_OBJECT (setupwin));
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (dumptifcheck));
    if (strlen(params->dumpname)>0 && params->tifjpg) {
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dumptifcheck), TRUE);
	gtk_entry_set_text(GTK_ENTRY (dump_entry),params->dumpname);
    }

    dumpjpgcheck = gtk_radio_button_new_with_label(group, " Dump a .jpg of each frame");
    gtk_box_pack_start (GTK_BOX (vboxmostright),dumpjpgcheck, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(dumpjpgcheck), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkdumpjpg),GTK_OBJECT (setupwin));
    if (strlen(params->dumpname)>0 && !params->tifjpg) {
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dumpjpgcheck), TRUE);
    }

    gtk_box_pack_start (GTK_BOX (vboxmostright), dump_label, TRUE, TRUE, 0);

    gtk_signal_connect_object (GTK_OBJECT (dump_entry), "changed",
                    GTK_SIGNAL_FUNC (filechange),GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxmostright), dump_entry, TRUE, TRUE, 0);

    gtk_signal_connect (GTK_OBJECT(dtcheck), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkdt),GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxmostright),dtcheck, TRUE, TRUE, 0);
    gtk_widget_show (dtcheck);

    gtk_signal_connect (GTK_OBJECT(dncheck), "toggled",
                        GTK_SIGNAL_FUNC(toggle_checkdn),GTK_OBJECT (setupwin));
    gtk_box_pack_start (GTK_BOX (vboxmostright),dncheck, TRUE, TRUE, 0);
    gtk_widget_show (dncheck);

    if (newparams.dumpnum) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dncheck), TRUE);
    else gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dtcheck), TRUE);

    gtk_signal_connect (GTK_OBJECT (okeyb), "clicked", GTK_SIGNAL_FUNC (okeypressed),(gpointer) params);
    gtk_box_pack_start (GTK_BOX (hbox3),okeyb, TRUE, TRUE, 0);
    if (!params->StartedAlready) gtk_widget_set_sensitive (okeyb, FALSE);

    cancelb = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (cancelb), "clicked", GTK_SIGNAL_FUNC (cancelpressed),(gpointer) params);
    gtk_box_pack_start (GTK_BOX (hbox3),cancelb, TRUE, TRUE, 0);
    if (!params->StartedAlready) gtk_widget_set_sensitive (cancelb, FALSE);

    redrawb = gtk_button_new_with_label ("Redraw");
    gtk_signal_connect (GTK_OBJECT (redrawb), "clicked", GTK_SIGNAL_FUNC (redrawpressed),(gpointer) params);
    gtk_box_pack_start (GTK_BOX (hbox3),redrawb, TRUE, TRUE, 0);
    if (!params->StartedAlready) gtk_widget_set_sensitive (redrawb, FALSE);

    quitb = gtk_button_new_with_label ("Quit");
    gtk_signal_connect (GTK_OBJECT (quitb), "clicked", GTK_SIGNAL_FUNC (quitpressed),(gpointer) params);
    gtk_box_pack_start (GTK_BOX (hbox3),quitb, TRUE, TRUE, 0);

	gtk_widget_show (header);
        gtk_widget_show (vbox_header);
        gtk_widget_show (vbox_main);
        gtk_widget_show (hbox_main);
        gtk_widget_show (hbox1);
        gtk_widget_show (hbox2);
        gtk_widget_show (hbox3);
        gtk_widget_show (hboxssize);
        gtk_widget_show (hbox_cube);
        gtk_widget_show (hboxcube);
        gtk_widget_show (vboxcubel);
        gtk_widget_show (vboxcuber);
        gtk_widget_show (vboxcuberm);
        gtk_widget_show (hboxxc);
        gtk_widget_show (vboxxcl);
        gtk_widget_show (vboxxcr);
        gtk_widget_show (hboxsleep);
        gtk_widget_show (hboxd);
        gtk_widget_show (hboxsc);
        gtk_widget_show (vbox);
        gtk_widget_show (vboxleft);
        gtk_widget_show (vboxright);
        gtk_widget_show (vboxmostright);
        gtk_widget_show (ssize_label);
        gtk_widget_show (ssx_label);
        gtk_widget_show (ssy_label);
        gtk_widget_show (cube_label);
        gtk_widget_show (col_label);
        gtk_widget_show (draw_label);
        gtk_widget_show (misc_label);
        gtk_widget_show (column_label);
        gtk_widget_show (cxyz_label);
        gtk_widget_show (empty_label);
        gtk_widget_show (d_label);
        gtk_widget_show (x_label);
        gtk_widget_show (y_label);
        gtk_widget_show (z_label);
        gtk_widget_show (xcol_label);
        gtk_widget_show (ycol_label);
        gtk_widget_show (zcol_label);
        gtk_widget_show (tcol_label);
        gtk_widget_show (sleep_label);
        gtk_widget_show (file_label);
        gtk_widget_show (file_entry);
        gtk_widget_show (ff_label);
        gtk_widget_show (dump_label);
        gtk_widget_show (dump_entry);
        gtk_widget_show (quitb);
        gtk_widget_show (browseb);
        gtk_widget_show (redrawb);
        gtk_widget_show (cancelb);
        gtk_widget_show (okeyb);
        gtk_widget_show (dumpcheck);
        gtk_widget_show (dumptifcheck);
        gtk_widget_show (dumpjpgcheck);
        gtk_widget_show (whitetoggle);
        gtk_widget_show (erasetoggle);
        gtk_widget_show (sortrtoggle);
        gtk_widget_show (scoltoggle);
        gtk_widget_show (ssxspinner);
        gtk_widget_show (ssyspinner);
        gtk_widget_show (cubespinner);
        gtk_widget_show (dspinner);
        gtk_widget_show (xspinner);
        gtk_widget_show (yspinner);
        gtk_widget_show (zspinner);
        gtk_widget_show (x2spinner);
        gtk_widget_show (y2spinner);
        gtk_widget_show (z2spinner);
        gtk_widget_show (xcspinner);
        gtk_widget_show (ycspinner);
        gtk_widget_show (zcspinner);
        gtk_widget_show (tcspinner);
        gtk_widget_show (sleepspinner);
	gtk_widget_show (scol_label);
	gtk_widget_show (scol_entry);
	gtk_widget_show (scolspinner);
        gtk_widget_show (setupwin);
}
